<?php
/**
 * Uninstall Content Guard Pro
 *
 * Handles complete cleanup when the plugin is deleted from WordPress admin.
 * This file is called automatically by WordPress when the plugin is deleted.
 *
 * IMPORTANT: By default, this permanently deletes ALL plugin data including:
 * - Database tables: content_guard_pro_findings, content_guard_pro_scans, content_guard_pro_audit_log
 * - Plugin settings and options (all content_guard_pro_* entries from wp_options)
 * - Scheduled tasks (WP-Cron and Action Scheduler jobs)
 * - Transients and cached data (including daily scan quotas/limits)
 * - User meta (dismissal flags, preferences)
 * - Site transients and multisite options (if applicable)
 *
 * EXCEPTION: If "Preserve Data on Uninstall" is enabled in Settings > General,
 * NO data is deleted. This allows for plugin reinstallation without data loss.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If uninstall not called from WordPress, exit.
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

// Security: Additional check to ensure this is being called properly.
if ( ! current_user_can( 'delete_plugins' ) ) {
	exit;
}

/**
 * Main uninstall function.
 *
 * Orchestrates the complete removal of all plugin data, or preserves it
 * based on the "Preserve Data on Uninstall" setting from Settings > General tab.
 *
 * If preserve_data is TRUE:
 * - All database tables are kept (findings, scans, audit log)
 * - All settings and options are kept
 * - Scan history is preserved
 * - Daily scan quotas/limits are preserved
 * - Scheduled tasks and Action Scheduler jobs are kept
 *
 * If preserve_data is FALSE (default):
 * - Everything is deleted for a clean uninstall
 *
 * @since 1.0.0
 */
function content_guard_pro_uninstall() {
	// Check if user wants to preserve data (Settings > General > Preserve Data on Uninstall).
	$settings = get_option( 'content_guard_pro_settings', array() );
	$preserve_data = isset( $settings['preserve_data'] ) ? (bool) $settings['preserve_data'] : false;

	if ( $preserve_data ) {
		// User has explicitly enabled data preservation.
		// Do NOT delete anything - keep all tables, settings, scans, quotas, etc.
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Content Guard Pro: Uninstall requested but data preservation is enabled. Keeping all data.' );
		}
		return; // Exit without cleanup.
	}

	// User wants full cleanup (default behavior).
	// Load database class for table cleanup.
	require_once plugin_dir_path( __FILE__ ) . 'includes/class-cgp-database.php';

	// Step 1: Drop all database tables (findings, scans, audit log).
	content_guard_pro_drop_tables();

	// Step 2: Delete all plugin options and settings.
	content_guard_pro_delete_options();

	// Step 3: Clear all transients (including scan quotas/limits).
	content_guard_pro_clear_transients();

	// Step 4: Clear scheduled WP-Cron tasks.
	content_guard_pro_clear_scheduled_tasks();

	// Step 5: Clear Action Scheduler jobs.
	content_guard_pro_clear_action_scheduler_jobs();

	// Step 6: Delete user meta (dismissal flags, preferences, etc.).
	content_guard_pro_delete_user_meta();

	// Step 7: Clear any cached data.
	wp_cache_flush();

	// Log the uninstall (if debug mode is enabled).
	if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		error_log( 'Content Guard Pro: Plugin uninstalled and all data removed.' );
	}
}

/**
 * Drop all plugin database tables.
 *
 * Removes content_guard_pro_findings, content_guard_pro_scans, and content_guard_pro_audit_log tables.
 *
 * @since 1.0.0
 */
function content_guard_pro_drop_tables() {
	// Use the existing drop_tables method from CGP_Database.
	$result = CGP_Database::drop_tables();

	if ( ! $result ) {
		cgp_log( 'Content Guard Pro: Failed to drop one or more database tables during uninstall.' );
	}
}

/**
 * Delete all plugin options from wp_options table.
 *
 * Removes settings, activation flags, and version tracking.
 *
 * @since 1.0.0
 */
function content_guard_pro_delete_options() {
	// List of all plugin options.
	$options = array(
		'content_guard_pro_version',
		'content_guard_pro_settings',
		'content_guard_pro_first_activation',
		'content_guard_pro_setup_completed',
		'content_guard_pro_rule_pack_version',
		'content_guard_pro_last_rule_update',
		'content_guard_pro_allowlist_patterns_cache',
		'content_guard_pro_denylist_patterns_cache',
	);

	foreach ( $options as $option ) {
		delete_option( $option );
	}

	// Delete any options that start with 'content_guard_pro_' (catch-all).
	global $wpdb;
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->query(
		"DELETE FROM {$wpdb->options} WHERE option_name LIKE 'content_guard_pro_%'"
	);

	// For multisite, also delete site options.
	if ( is_multisite() ) {
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->query(
			"DELETE FROM {$wpdb->sitemeta} WHERE meta_key LIKE 'content_guard_pro_%'"
		);
	}
}

/**
 * Clear all plugin transients.
 *
 * Removes temporary cached data stored as transients.
 *
 * @since 1.0.0
 */
function content_guard_pro_clear_transients() {
	// List of known transients.
	$transients = array(
		'content_guard_pro_activation_redirect',
		'content_guard_pro_scan_progress',
		'content_guard_pro_scan_completed',
		'content_guard_pro_quarantine_cache',
		'content_guard_pro_allowlist_patterns',
		'content_guard_pro_reputation_cache',
		'content_guard_pro_active_scan_id',
		'content_guard_pro_manual_scans_' . gmdate( 'Y-m-d' ), // Daily scan counter (current day).
	);

	foreach ( $transients as $transient ) {
		delete_transient( $transient );
	}

	// Delete all transients that start with 'content_guard_pro_' (catch-all).
	// This includes scan counters from other days: content_guard_pro_manual_scans_YYYY-MM-DD.
	global $wpdb;
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->query(
		"DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_content_guard_pro_%' OR option_name LIKE '_transient_timeout_content_guard_pro_%'"
	);

	// Also delete site transients (for multisite compatibility).
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->query(
		"DELETE FROM {$wpdb->options} WHERE option_name LIKE '_site_transient_content_guard_pro_%' OR option_name LIKE '_site_transient_timeout_content_guard_pro_%'"
	);
}

/**
 * Clear all scheduled WP-Cron tasks.
 *
 * Removes daily digest and any other scheduled events.
 *
 * @since 1.0.0
 */
function content_guard_pro_clear_scheduled_tasks() {
	// Clear daily digest cron.
	$digest_timestamp = wp_next_scheduled( 'content_guard_pro_daily_digest' );
	if ( $digest_timestamp ) {
		wp_unschedule_event( $digest_timestamp, 'content_guard_pro_daily_digest' );
	}

	// Clear license check cron.
	wp_clear_scheduled_hook( 'content_guard_pro_daily_license_check' );

	// Clear any scheduled scans.
	$scan_timestamp = wp_next_scheduled( 'content_guard_pro_scheduled_scan' );
	if ( $scan_timestamp ) {
		wp_unschedule_event( $scan_timestamp, 'content_guard_pro_scheduled_scan' );
	}

	// Clear cleanup tasks.
	$cleanup_timestamp = wp_next_scheduled( 'content_guard_pro_cleanup_old_data' );
	if ( $cleanup_timestamp ) {
		wp_unschedule_event( $cleanup_timestamp, 'content_guard_pro_cleanup_old_data' );
	}

	// Clear all hooks for our events (catches all scheduled instances).
	wp_clear_scheduled_hook( 'content_guard_pro_daily_digest' );
	wp_clear_scheduled_hook( 'content_guard_pro_scheduled_scan' );
	wp_clear_scheduled_hook( 'content_guard_pro_cleanup_old_data' );
}

/**
 * Clear all Action Scheduler jobs.
 *
 * Removes pending scan batches and any other Action Scheduler jobs.
 *
 * @since 1.0.0
 */
function content_guard_pro_clear_action_scheduler_jobs() {
	// Check if Action Scheduler is available.
	if ( ! class_exists( 'ActionScheduler' ) ) {
		return;
	}

	try {
		// Cancel all actions in the content-guard-pro group.
		// This removes any pending scan batches.
		if ( function_exists( 'as_unschedule_all_actions' ) ) {
			// Cancel scan batch actions.
			as_unschedule_all_actions( 'content_guard_pro_scan_batch', array(), 'content-guard-pro' );
			
			// Cancel on-save scan actions.
			as_unschedule_all_actions( 'content_guard_pro_on_save_scan', array(), 'content-guard-pro' );
			
			// Cancel any other CGP actions.
			as_unschedule_all_actions( '', array(), 'content-guard-pro' );
		}
	} catch ( Exception $e ) {
		cgp_log( 'Content Guard Pro: Error clearing Action Scheduler jobs: ' . $e->getMessage() );
	}
}

/**
 * Delete all plugin user meta.
 *
 * Removes user-specific data like dismissal flags.
 *
 * @since 1.0.0
 */
function content_guard_pro_delete_user_meta() {
	global $wpdb;

	// List of known user meta keys.
	$meta_keys = array(
		'content_guard_pro_dismissed_welcome_notice',
		'content_guard_pro_dismissed_critical_notice',
		'content_guard_pro_last_dismissed_critical_count',
	);

	foreach ( $meta_keys as $meta_key ) {
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->usermeta} WHERE meta_key = %s",
				$meta_key
			)
		);
	}

	// Delete all user meta that starts with 'content_guard_pro_' (catch-all).
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->query(
		"DELETE FROM {$wpdb->usermeta} WHERE meta_key LIKE 'content_guard_pro_%'"
	);
}

// Run the uninstall process.
content_guard_pro_uninstall();

