<?php
/**
 * Scheduler Class - Action Scheduler Integration
 *
 * Wrapper class for Action Scheduler that manages all scheduled scans
 * and background jobs for Content Guard Pro.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Scheduler
 *
 * Manages scan scheduling using Action Scheduler for:
 * - Daily incremental scans
 * - On-save content checks
 * - Manual scan jobs
 * - Batch processing with throttling
 *
 * @since 1.0.0
 */
class CGP_Scheduler {

	/**
	 * Action Scheduler group name.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const GROUP = 'content-guard-pro';

	/**
	 * Hook name for daily scans.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const HOOK_DAILY_SCAN = 'content_guard_pro_daily_scan';

	/**
	 * Hook name for on-save scans.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const HOOK_ON_SAVE_SCAN = 'content_guard_pro_on_save_scan';

	/**
	 * Hook name for manual scans.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const HOOK_MANUAL_SCAN = 'content_guard_pro_manual_scan';

	/**
	 * Hook name for batch processing.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const HOOK_PROCESS_BATCH = 'content_guard_pro_process_batch';

	/**
	 * Default batch size (PRD Section 3.1).
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const DEFAULT_BATCH_SIZE = 100;

	/**
	 * Safe Mode batch size (PRD Appendix H).
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const SAFE_MODE_BATCH_SIZE = 25;

	/**
	 * Default delay between batches in seconds (PRD Appendix H).
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const DEFAULT_DELAY = 2;

	/**
	 * Safe Mode delay in seconds (PRD Appendix H).
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const SAFE_MODE_DELAY = 7;

	/**
	 * Query time threshold in milliseconds (PRD Appendix H).
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const QUERY_TIME_THRESHOLD = 300;

	/**
	 * Memory threshold percentage of WP_MEMORY_LIMIT (PRD Appendix H).
	 *
	 * @since 1.0.0
	 * @var float
	 */
	const MEMORY_THRESHOLD_PERCENT = 0.5;

	/**
	 * Batch runtime threshold in seconds (PRD Appendix H).
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const BATCH_RUNTIME_THRESHOLD = 5;

	/**
	 * Initialize the scheduler.
	 *
	 * @since 1.0.0
	 */
	public static function init() {
		$instance = new self();
		$instance->register_hooks();
	}

	/**
	 * Register WordPress and Action Scheduler hooks.
	 *
	 * @since 1.0.0
	 */
	private function register_hooks() {
		// Register Action Scheduler hooks for processing.
		add_action( self::HOOK_DAILY_SCAN, array( $this, 'process_daily_scan' ), 10, 1 );
		add_action( self::HOOK_ON_SAVE_SCAN, array( $this, 'process_on_save_scan' ), 10, 1 );
		add_action( self::HOOK_MANUAL_SCAN, array( $this, 'process_manual_scan' ), 10, 1 );
		add_action( self::HOOK_PROCESS_BATCH, array( $this, 'process_batch' ), 10, 2 );

		// WordPress content hooks for triggering on-save scans.
		add_action( 'save_post', array( $this, 'maybe_schedule_on_save_scan' ), 20, 3 );
		add_action( 'transition_post_status', array( $this, 'maybe_schedule_on_status_change' ), 10, 3 );

		// Initialize daily scan schedule if enabled.
		add_action( 'init', array( $this, 'maybe_schedule_daily_scan' ) );

		// Reschedule when settings are updated.
		add_action( 'update_option_content_guard_pro_settings', array( __CLASS__, 'on_settings_updated' ), 10, 2 );
	}

	/**
	 * Handle settings update - reschedule if schedule settings changed.
	 *
	 * @since 1.0.0
	 * @param array $old_value Old settings.
	 * @param array $new_value New settings.
	 */
	public static function on_settings_updated( $old_value, $new_value ) {
		// Check if schedule-related settings changed.
		$schedule_keys = array( 'schedule_enabled', 'schedule_time', 'schedule_frequency', 'scan_mode' );
		$changed = false;

		foreach ( $schedule_keys as $key ) {
			$old = isset( $old_value[ $key ] ) ? $old_value[ $key ] : null;
			$new = isset( $new_value[ $key ] ) ? $new_value[ $key ] : null;

			if ( $old !== $new ) {
				$changed = true;
				break;
			}
		}

		if ( $changed ) {
			cgp_log( 'Content Guard Pro: Schedule settings changed, rescheduling...' );
			self::reschedule_daily_scan();
		}
	}

	/**
	 * Check if Action Scheduler is available.
	 *
	 * @since 1.0.0
	 * @return bool True if Action Scheduler is available.
	 */
	public static function is_action_scheduler_available() {
		return function_exists( 'as_enqueue_async_action' );
	}

	/**
	 * Schedule a scan job.
	 *
	 * Enqueues a scan job to Action Scheduler for async processing.
	 * This is the main method for scheduling any type of scan.
	 *
	 * @since 1.0.0
	 * @param string $mode Scan mode: 'quick' or 'standard'.
	 * @param array  $args Additional arguments for the scan.
	 * @return int|bool Action ID on success, false on failure.
	 */
	public static function schedule_scan( $mode = 'standard', $args = array() ) {
		if ( ! self::is_action_scheduler_available() ) {
			cgp_log( 'Content Guard Pro: Action Scheduler not available' );
			return false;
		}

		// Check if this is a bonus scan (e.g., from setup wizard).
		$is_bonus_scan = isset( $args['source'] ) && 'setup_wizard' === $args['source'];

		// License guard: free tier cannot run manual scans.
		// Skip limit check for bonus scans (Fix #2).
		if ( class_exists( 'CGP_License_Manager' ) && ! $is_bonus_scan ) {
			if ( ! CGP_License_Manager::can_run_manual_scan() || ! CGP_License_Manager::can( 'quick_scan' ) ) {
				cgp_log( 'Content Guard Pro: Manual scan blocked by license tier' );
				return false;
			}
		}

		// Validate scan mode.
		$mode = in_array( $mode, array( 'quick', 'standard' ), true ) ? $mode : 'standard';

		// Prepare scan arguments.
		$scan_args = wp_parse_args(
			$args,
			array(
				'mode'       => $mode,
				'scan_type'  => 'manual',
				'initiated_by' => get_current_user_id(),
				'timestamp'  => time(),
			)
		);

		// Enqueue async action.
		$action_id = as_enqueue_async_action(
			self::HOOK_MANUAL_SCAN,
			array( $scan_args ),
			self::GROUP
		);

		if ( $action_id ) {
			// Log the scheduled scan.
			self::log_scheduled_action( 'manual_scan', $scan_args, $action_id );
			
			// Log bonus scan for tracking.
			if ( $is_bonus_scan ) {
				cgp_log( 'Content Guard Pro: Bonus scan scheduled (does not count against limit)' );
			}
		}

		return $action_id;
	}

	/**
	 * Schedule an on-save scan for a post.
	 *
	 * Triggered when content is saved. Performs quick validation
	 * before publishing (PRD Section 3.1).
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID to scan.
	 * @return int|bool Action ID on success, false on failure.
	 */
	public static function schedule_on_save_scan( $post_id ) {
		if ( ! self::is_action_scheduler_available() ) {
			return false;
		}

		// License guard: on-save scanning requires paid tier.
		if ( class_exists( 'CGP_License_Manager' ) && ! CGP_License_Manager::can( 'on_save_scanning' ) ) {
			return false;
		}

		$post_id = absint( $post_id );

		if ( ! $post_id ) {
			return false;
		}

		// Check if already scheduled (avoid duplicate scans).
		if ( self::is_post_scan_pending( $post_id ) ) {
			return false;
		}

		// Prepare scan arguments.
		$scan_args = array(
			'post_id'    => $post_id,
			'scan_type'  => 'on_save',
			'mode'       => 'quick', // On-save scans are always quick.
			'initiated_by' => get_current_user_id(),
			'timestamp'  => time(),
		);

		// Enqueue async action with high priority.
		$action_id = as_enqueue_async_action(
			self::HOOK_ON_SAVE_SCAN,
			array( $scan_args ),
			self::GROUP,
			true // Unique - prevent duplicate scheduling.
		);

		if ( $action_id ) {
			// Store transient to track pending scan.
			set_transient( 'content_guard_pro_pending_scan_' . $post_id, $action_id, 300 ); // 5 minutes.
			
			self::log_scheduled_action( 'on_save_scan', $scan_args, $action_id );
		}

		return $action_id;
	}

	/**
	 * Schedule recurring scan based on frequency setting.
	 *
	 * Sets up recurring scan at configured time and frequency (PRD Section 3.1).
	 * Supports daily, twicedaily, and weekly frequencies.
	 *
	 * @since 1.0.0
	 * @return int|bool Action ID on success, false on failure.
	 */
	public static function schedule_daily_scan() {
		if ( ! self::is_action_scheduler_available() ) {
			return false;
		}

		// License guard: skip scheduling when plan doesn't allow it.
		if ( class_exists( 'CGP_License_Manager' ) && ! CGP_License_Manager::can( 'scheduled_scans' ) ) {
			// Ensure any lingering recurring job is cleared.
			self::cancel_daily_scan();
			return false;
		}

		// Get settings.
		$settings = get_option( 'content_guard_pro_settings', array() );
		
		// Check if scheduled scans are enabled.
		if ( empty( $settings['schedule_enabled'] ) ) {
			return false;
		}

		// Get configured time (default 03:00).
		$scan_time = isset( $settings['schedule_time'] ) ? $settings['schedule_time'] : '03:00';
		
		// Validate time format (HH:MM).
		if ( ! preg_match( '/^([0-1]?[0-9]|2[0-3]):[0-5][0-9]$/', $scan_time ) ) {
			$scan_time = '03:00'; // Fallback to default if invalid.
		}

		// Get configured frequency (default daily).
		$frequency = isset( $settings['schedule_frequency'] ) ? $settings['schedule_frequency'] : 'daily';

		// Get configured scan mode (default standard).
		$scan_mode = isset( $settings['scan_mode'] ) ? $settings['scan_mode'] : 'standard';
		
		// Validate scan mode.
		if ( ! in_array( $scan_mode, array( 'quick', 'standard' ), true ) ) {
			$scan_mode = 'standard';
		}
		
		// Calculate interval based on frequency.
		switch ( $frequency ) {
			case 'twicedaily':
				$interval = 12 * HOUR_IN_SECONDS; // 12 hours.
				break;
			case 'weekly':
				$interval = WEEK_IN_SECONDS; // 7 days.
				break;
			case 'daily':
			default:
				$interval = DAY_IN_SECONDS; // 24 hours.
				break;
		}
		
		// Calculate next run time.
		$next_run = strtotime( 'today ' . $scan_time );
		
		// Check if strtotime succeeded.
		if ( false === $next_run ) {
			cgp_log( 'Content Guard Pro: Invalid scan time format: ' . $scan_time );
			return false;
		}
		
		// If time has passed today, schedule based on frequency.
		if ( $next_run <= time() ) {
			if ( 'twicedaily' === $frequency ) {
				// For twice daily, check if next 12-hour slot is still today.
				$next_run = strtotime( 'today ' . $scan_time ) + ( 12 * HOUR_IN_SECONDS );
				if ( $next_run <= time() ) {
					$next_run = strtotime( 'tomorrow ' . $scan_time );
				}
			} elseif ( 'weekly' === $frequency ) {
				// For weekly, schedule for next week.
				$next_run = strtotime( '+1 week ' . $scan_time );
			} else {
				// For daily, schedule for tomorrow.
				$next_run = strtotime( 'tomorrow ' . $scan_time );
			}
			
			// Check if strtotime succeeded.
			if ( false === $next_run ) {
				cgp_log( 'Content Guard Pro: Failed to calculate next scan time' );
				return false;
			}
		}

		// Check if already scheduled.
		if ( self::is_daily_scan_scheduled() ) {
			return false;
		}

		// Schedule recurring action with appropriate interval.
		$action_id = as_schedule_recurring_action(
			$next_run,
			$interval,
			self::HOOK_DAILY_SCAN,
			array(
				array(
					'mode'       => $scan_mode, // Use configured scan mode from settings.
					'scan_type'  => 'scheduled',
					'frequency'  => $frequency,
					'initiated_by' => 0, // System-initiated.
				),
			),
			self::GROUP,
			true // Unique.
		);

		if ( $action_id ) {
			self::log_scheduled_action(
				'scheduled_scan',
				array(
					'next_run'  => $next_run,
					'frequency' => $frequency,
					'interval'  => $interval,
					'mode'      => $scan_mode,
				),
				$action_id
			);
		}

		return $action_id;
	}

	/**
	 * Cancel daily scan schedule.
	 *
	 * @since 1.0.0
	 * @return bool True on success, false on failure.
	 */
	public static function cancel_daily_scan() {
		if ( ! self::is_action_scheduler_available() ) {
			return false;
		}

		// Unschedule all daily scans (use null to match any args).
		as_unschedule_all_actions( self::HOOK_DAILY_SCAN, null, self::GROUP );

		return true;
	}

	/**
	 * Reschedule daily scan when settings change.
	 *
	 * Called when schedule settings are updated to ensure the schedule
	 * reflects the new time and frequency settings.
	 *
	 * @since 1.0.0
	 * @return int|bool Action ID on success, false on failure.
	 */
	public static function reschedule_daily_scan() {
		// First, cancel any existing schedule.
		self::cancel_daily_scan();

		// Get current settings.
		$settings = get_option( 'content_guard_pro_settings', array() );

		// Only reschedule if enabled.
		if ( ! empty( $settings['schedule_enabled'] ) ) {
			return self::schedule_daily_scan();
		}

		return false;
	}

	/**
	 * Check if daily scan is scheduled.
	 *
	 * @since 1.0.0
	 * @return bool True if scheduled, false otherwise.
	 */
	public static function is_daily_scan_scheduled() {
		if ( ! self::is_action_scheduler_available() ) {
			return false;
		}

		// Pass null for args to match any scheduled action with this hook.
		// Using array() would only match actions with empty args (which we don't have).
		return as_next_scheduled_action( self::HOOK_DAILY_SCAN, null, self::GROUP ) !== false;
	}

	/**
	 * Check if a post scan is already pending.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if pending, false otherwise.
	 */
	private static function is_post_scan_pending( $post_id ) {
		$pending = get_transient( 'content_guard_pro_pending_scan_' . $post_id );
		return ! empty( $pending );
	}

	/**
	 * Maybe schedule on-save scan.
	 *
	 * Hook callback for save_post action.
	 *
	 * @since 1.0.0
	 * @param int     $post_id Post ID.
	 * @param WP_Post $post    Post object.
	 * @param bool    $update  Whether this is an update.
	 */
	public function maybe_schedule_on_save_scan( $post_id, $post, $update ) {
		// Check if real-time scanning is enabled in settings.
		$settings = get_option( 'content_guard_pro_settings', array() );
		if ( empty( $settings['realtime_scan_enabled'] ) ) {
			return;
		}

		// Bail if autosave or revision.
		if ( wp_is_post_autosave( $post_id ) || wp_is_post_revision( $post_id ) ) {
			return;
		}

		// Only scan relevant post types.
		$allowed_types = apply_filters( 'content_guard_pro_on_save_scan_post_types', array( 'post', 'page' ) );
		if ( ! in_array( $post->post_type, $allowed_types, true ) ) {
			return;
		}

		// Only scan if content has changed (on update).
		if ( $update && ! $this->has_content_changed( $post_id ) ) {
			return;
		}

		// Schedule the scan.
		self::schedule_on_save_scan( $post_id );
	}

	/**
	 * Maybe schedule scan on post status change.
	 *
	 * Hook callback for transition_post_status action.
	 *
	 * @since 1.0.0
	 * @param string  $new_status New post status.
	 * @param string  $old_status Old post status.
	 * @param WP_Post $post       Post object.
	 */
	public function maybe_schedule_on_status_change( $new_status, $old_status, $post ) {
		// Check if real-time scanning is enabled in settings.
		$settings = get_option( 'content_guard_pro_settings', array() );
		if ( empty( $settings['realtime_scan_enabled'] ) ) {
			return;
		}

		// Scan when transitioning to publish.
		if ( 'publish' === $new_status && 'publish' !== $old_status ) {
			self::schedule_on_save_scan( $post->ID );
		}
	}

	/**
	 * Maybe schedule daily scan on init.
	 *
	 * @since 1.0.0
	 */
	public function maybe_schedule_daily_scan() {
		// If license no longer allows scheduling, clean up any scheduled scans.
		if ( class_exists( 'CGP_License_Manager' ) && ! CGP_License_Manager::can( 'scheduled_scans' ) ) {
			self::cancel_daily_scan();
			return;
		}

		// Only schedule if not already scheduled.
		if ( ! self::is_daily_scan_scheduled() ) {
			self::schedule_daily_scan();
		}
	}

	/**
	 * Check if post content has changed.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if content changed, false otherwise.
	 */
	private function has_content_changed( $post_id ) {
		// Get current post.
		$post = get_post( $post_id );
		
		if ( ! $post ) {
			return false;
		}

		// Get previous revision.
		$revisions = wp_get_post_revisions( $post_id, array( 'posts_per_page' => 1 ) );
		
		if ( empty( $revisions ) ) {
			return true; // No previous revision, treat as changed.
		}

		$previous = array_shift( $revisions );

		// Compare content.
		return $post->post_content !== $previous->post_content;
	}

	/**
	 * Process daily scan.
	 *
	 * Action Scheduler callback for daily scans.
	 *
	 * @since 1.0.0
	 * @param array $args Scan arguments.
	 */
	public function process_daily_scan( $args ) {
		// Safety: abort if license doesn't allow scheduled scans.
		if ( class_exists( 'CGP_License_Manager' ) && ! CGP_License_Manager::can( 'scheduled_scans' ) ) {
			cgp_log( 'Content Guard Pro: Scheduled scan aborted due to license tier' );
			self::cancel_daily_scan();
			return;
		}

		cgp_log( 'Content Guard Pro: Processing daily scan' );
		cgp_log( 'Scan args: ' . wp_json_encode( $args ) );

		// Call scanner to start daily scan.
		// This creates a scan record and schedules the first batch.
		$scan_id = CGP_Scanner::start_daily_scan( $args );
		
		if ( $scan_id ) {
			cgp_log( sprintf( 'Content Guard Pro: Daily scan started with ID %d', $scan_id ) );
		} else {
			cgp_log( 'Content Guard Pro: Failed to start daily scan' );
		}
	}

	/**
	 * Process on-save scan.
	 *
	 * Action Scheduler callback for on-save scans.
	 * Quick validation scan triggered when content is saved (PRD Section 3.1).
	 * Should complete within 5 seconds per US-037.
	 *
	 * @since 1.0.0
	 * @param array $args Scan arguments including post_id.
	 */
	public function process_on_save_scan( $args ) {
		$post_id = isset( $args['post_id'] ) ? absint( $args['post_id'] ) : 0;

		if ( ! $post_id ) {
			cgp_log( 'Content Guard Pro: On-save scan called with no post_id' );
			return;
		}

		// Clear pending scan transient.
		delete_transient( 'content_guard_pro_pending_scan_' . $post_id );

		cgp_log( 'Content Guard Pro: Processing on-save scan for post ' . $post_id );

		// Call scanner to scan single post.
		// This should complete quickly (within 5 seconds).
		$result = CGP_Scanner::process_on_save_scan( $post_id, $args );
		
		if ( $result ) {
			cgp_log( sprintf( 'Content Guard Pro: On-save scan completed for post %d', $post_id ) );
		} else {
			cgp_log( sprintf( 'Content Guard Pro: On-save scan failed for post %d', $post_id ) );
		}
	}

	/**
	 * Process manual scan.
	 *
	 * Action Scheduler callback for manual scans.
	 * Initializes a user-triggered scan and schedules the first batch.
	 *
	 * @since 1.0.0
	 * @param array $args Scan arguments (mode, initiated_by, etc.).
	 */
	public function process_manual_scan( $args ) {
		cgp_log( 'Content Guard Pro: Processing manual scan' );
		cgp_log( 'Scan mode: ' . ( isset( $args['mode'] ) ? $args['mode'] : 'unknown' ) );

		// Call scanner to start manual scan.
		// This creates a scan record in content_guard_pro_scans and schedules the first batch.
		$scan_id = CGP_Scanner::start_manual_scan( $args );
		
		if ( $scan_id ) {
			cgp_log( sprintf( 'Content Guard Pro: Manual scan started with ID %d', $scan_id ) );
			
			// Store scan ID in transient for UI polling.
			set_transient( 'content_guard_pro_active_scan_id', $scan_id, HOUR_IN_SECONDS );
		} else {
			cgp_log( 'Content Guard Pro: Failed to start manual scan' );
		}
	}

	/**
	 * Process a batch of items.
	 *
	 * Action Scheduler callback for batch processing.
	 * This is the recurring job that processes items in batches.
	 *
	 * @since 1.0.0
	 * @param int   $scan_id    Scan ID.
	 * @param array $batch_args Batch arguments (target, offset, limit, etc.).
	 */
	public function process_batch( $scan_id, $batch_args ) {
		cgp_log( sprintf(
			'Content Guard Pro: Processing batch for scan %d (target: %s, offset: %d)',
			$scan_id,
			isset( $batch_args['target'] ) ? $batch_args['target'] : 'unknown',
			isset( $batch_args['offset'] ) ? absint( $batch_args['offset'] ) : 0
		) );

		// Call scanner to process this batch.
		// The scanner handles:
		// 1. Processing the batch (posts, postmeta, or options)
		// 2. Detecting threats
		// 3. Saving findings
		// 4. Tracking performance metrics
		// 5. Checking throttling
		// 6. Scheduling the next batch if needed
		$result = CGP_Scanner::process_scan_batch( $scan_id, $batch_args );
		
		if ( $result ) {
			cgp_log( sprintf( 'Content Guard Pro: Batch processed successfully for scan %d', $scan_id ) );
		} else {
			cgp_log( sprintf( 'Content Guard Pro: Batch processing failed for scan %d', $scan_id ) );
			
			// Increment error count in scan record.
			// The scanner will handle marking the scan as failed if too many errors.
		}
	}

	/**
	 * Get pending actions count.
	 *
	 * @since 1.0.0
	 * @return int Number of pending actions.
	 */
	public static function get_pending_actions_count() {
		if ( ! self::is_action_scheduler_available() ) {
			return 0;
		}

		return count(
			as_get_scheduled_actions(
				array(
					'group'    => self::GROUP,
					'status'   => 'pending',
					'per_page' => -1,
				),
				'ids'
			)
		);
	}

	/**
	 * Get running actions count.
	 *
	 * @since 1.0.0
	 * @return int Number of running actions.
	 */
	public static function get_running_actions_count() {
		if ( ! self::is_action_scheduler_available() ) {
			return 0;
		}

		return count(
			as_get_scheduled_actions(
				array(
					'group'    => self::GROUP,
					'status'   => 'in-progress',
					'per_page' => -1,
				),
				'ids'
			)
		);
	}

	/**
	 * Cancel all pending scans.
	 *
	 * Emergency stop for all scheduled scans.
	 *
	 * @since 1.0.0
	 * @return int Number of actions cancelled.
	 */
	public static function cancel_all_scans() {
		if ( ! self::is_action_scheduler_available() ) {
			return 0;
		}

		// Cancel all pending and scheduled actions.
		$cancelled = 0;

		foreach ( array( self::HOOK_MANUAL_SCAN, self::HOOK_ON_SAVE_SCAN, self::HOOK_PROCESS_BATCH, self::HOOK_DAILY_SCAN ) as $hook ) {
			as_unschedule_all_actions( $hook, array(), self::GROUP );
			$cancelled++;
		}

		return $cancelled;
	}

	/**
	 * Get next scheduled scan time.
	 *
	 * @since 1.0.0
	 * @return int|false Timestamp of next scan, false if none scheduled.
	 */
	public static function get_next_scan_time() {
		if ( ! self::is_action_scheduler_available() ) {
			return false;
		}

		// Use null to match any args (not empty array which requires exact match).
		$timestamp = as_next_scheduled_action( self::HOOK_DAILY_SCAN, null, self::GROUP );

		return $timestamp ? $timestamp : false;
	}

	/**
	 * Log scheduled action.
	 *
	 * @since 1.0.0
	 * @param string $type      Action type.
	 * @param array  $args      Action arguments.
	 * @param int    $action_id Action Scheduler action ID.
	 */
	private static function log_scheduled_action( $type, $args, $action_id ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			cgp_log(
				sprintf(
					'Content Guard Pro: Scheduled %s (Action ID: %d) - Args: %s',
					$type,
					$action_id,
					wp_json_encode( $args )
				)
			);
		}
	}

	/**
	 * Get scheduler statistics.
	 *
	 * @since 1.0.0
	 * @return array Scheduler stats.
	 */
	public static function get_stats() {
		return array(
			'action_scheduler_available' => self::is_action_scheduler_available(),
			'daily_scan_scheduled'       => self::is_daily_scan_scheduled(),
			'next_scan_time'             => self::get_next_scan_time(),
			'pending_actions'            => self::get_pending_actions_count(),
			'running_actions'            => self::get_running_actions_count(),
		);
	}

	/**
	 * Check if throttling is needed based on performance metrics.
	 *
	 * Per PRD Section 3.6 & Appendix H:
	 * - Auto-throttle when avg query time > 300ms
	 * - Auto-throttle when peak memory > 50% of WP_MEMORY_LIMIT
	 * - Auto-throttle when batch runtime > 5s
	 *
	 * @since 1.0.0
	 * @param array $performance_metrics Performance data from batch.
	 * @return bool True if throttling is needed.
	 */
	public static function should_throttle( $performance_metrics ) {
		// Check query time threshold (PRD Appendix H).
		if ( ! empty( $performance_metrics['avg_query_ms'] ) && $performance_metrics['avg_query_ms'] > self::QUERY_TIME_THRESHOLD ) {
			cgp_log( sprintf(
				'Content Guard Pro: Throttling triggered - Query time %dms exceeds threshold %dms',
				$performance_metrics['avg_query_ms'],
				self::QUERY_TIME_THRESHOLD
			) );
			return true;
		}

		// Check memory threshold (PRD Appendix H: 50% of WP_MEMORY_LIMIT).
		$memory_limit = self::get_memory_limit_bytes();
		$peak_memory  = ! empty( $performance_metrics['peak_memory'] ) ? $performance_metrics['peak_memory'] : memory_get_peak_usage( true );
		$memory_usage_percent = $memory_limit > 0 ? ( $peak_memory / $memory_limit ) : 0;

		if ( $memory_usage_percent > self::MEMORY_THRESHOLD_PERCENT ) {
			cgp_log( sprintf(
				'Content Guard Pro: Throttling triggered - Memory usage %.1f%% exceeds threshold %.1f%%',
				$memory_usage_percent * 100,
				self::MEMORY_THRESHOLD_PERCENT * 100
			) );
			return true;
		}

		// Check batch runtime threshold (PRD Appendix H: 5s).
		if ( ! empty( $performance_metrics['batch_runtime'] ) && $performance_metrics['batch_runtime'] > self::BATCH_RUNTIME_THRESHOLD ) {
			cgp_log( sprintf(
				'Content Guard Pro: Throttling triggered - Batch runtime %.2fs exceeds threshold %ds',
				$performance_metrics['batch_runtime'],
				self::BATCH_RUNTIME_THRESHOLD
			) );
			return true;
		}

		return false;
	}

	/**
	 * Calculate throttle adjustments for next batch.
	 *
	 * Returns adjusted batch size and delay based on current performance.
	 * Per PRD Appendix H: Shrink batch, add delay.
	 *
	 * @since 1.0.0
	 * @param int   $current_batch_size Current batch size.
	 * @param int   $current_delay      Current delay in seconds.
	 * @param array $performance_metrics Performance data from batch.
	 * @return array Adjusted parameters: batch_size, delay, throttle_state.
	 */
	public static function calculate_throttle_adjustments( $current_batch_size, $current_delay, $performance_metrics ) {
		// Start with current values.
		$new_batch_size = $current_batch_size;
		$new_delay      = $current_delay;
		$throttle_state = 'normal';

		// Check if Safe Mode should be enabled.
		if ( self::should_enable_safe_mode( $performance_metrics ) ) {
			$new_batch_size = self::SAFE_MODE_BATCH_SIZE;
			$new_delay      = self::SAFE_MODE_DELAY;
			$throttle_state = 'safe_mode';

			cgp_log( sprintf(
				'Content Guard Pro: Safe Mode enabled - Batch size: %d, Delay: %ds',
				$new_batch_size,
				$new_delay
			) );
		} elseif ( self::should_throttle( $performance_metrics ) ) {
			// Reduce batch size by 25% (PRD Appendix H).
			$new_batch_size = max( self::SAFE_MODE_BATCH_SIZE, (int) ceil( $current_batch_size * 0.75 ) );

			// Increase delay by 2-3 seconds (PRD Appendix H: 1-3s default + 2-3s increase = 5-10s range).
			$new_delay = min( 10, $current_delay + wp_rand( 2, 3 ) );

			$throttle_state = 'throttled';

			cgp_log( sprintf(
				'Content Guard Pro: Throttling applied - Batch size: %d → %d, Delay: %ds → %ds',
				$current_batch_size,
				$new_batch_size,
				$current_delay,
				$new_delay
			) );
		} else {
			// Performance is good - try to gradually increase back to normal.
			if ( $current_batch_size < self::DEFAULT_BATCH_SIZE ) {
				// Increase batch size by 10% but cap at default.
				$new_batch_size = min( self::DEFAULT_BATCH_SIZE, (int) ceil( $current_batch_size * 1.1 ) );
			}

			if ( $current_delay > self::DEFAULT_DELAY ) {
				// Decrease delay by 1 second but keep minimum.
				$new_delay = max( self::DEFAULT_DELAY, $current_delay - 1 );
			}

			if ( $new_batch_size !== $current_batch_size || $new_delay !== $current_delay ) {
				cgp_log( sprintf(
					'Content Guard Pro: Performance improved - Batch size: %d → %d, Delay: %ds → %ds',
					$current_batch_size,
					$new_batch_size,
					$current_delay,
					$new_delay
				) );
			}
		}

		return array(
			'batch_size'     => $new_batch_size,
			'delay'          => $new_delay,
			'throttle_state' => $throttle_state,
		);
	}

	/**
	 * Check if Safe Mode should be enabled.
	 *
	 * Per PRD Appendix H: Auto-enable on large sites or high resource usage.
	 *
	 * @since 1.0.0
	 * @param array $performance_metrics Performance data.
	 * @return bool True if Safe Mode should be enabled.
	 */
	private static function should_enable_safe_mode( $performance_metrics ) {
		global $wpdb;

		// Check user settings (can be manually enabled).
		$settings = get_option( 'content_guard_pro_settings', array() );
		if ( ! empty( $settings['safe_mode'] ) && 'always_on' === $settings['safe_mode'] ) {
			return true;
		}

		if ( ! empty( $settings['safe_mode'] ) && 'always_off' === $settings['safe_mode'] ) {
			return false;
		}

		// Auto-detect: Check if memory usage is critically high (>70%).
		$memory_limit = self::get_memory_limit_bytes();
		$peak_memory  = ! empty( $performance_metrics['peak_memory'] ) ? $performance_metrics['peak_memory'] : memory_get_peak_usage( true );
		$memory_usage_percent = $memory_limit > 0 ? ( $peak_memory / $memory_limit ) : 0;

		if ( $memory_usage_percent > 0.7 ) {
			cgp_log( sprintf(
				'Content Guard Pro: Safe Mode auto-enabled - Memory usage %.1f%% exceeds critical threshold 70%%',
				$memory_usage_percent * 100
			) );
			return true;
		}

		// Auto-detect: Check if query time is critically high (>500ms).
		if ( ! empty( $performance_metrics['avg_query_ms'] ) && $performance_metrics['avg_query_ms'] > 500 ) {
			cgp_log( sprintf(
				'Content Guard Pro: Safe Mode auto-enabled - Query time %dms exceeds critical threshold 500ms',
				$performance_metrics['avg_query_ms']
			) );
			return true;
		}

		// Auto-detect: Check if database is very large (PRD Appendix H: >2M content rows or >2-5 GB).
		// Note: This is a one-time check, cached for 24 hours.
		$db_size_check = get_transient( 'content_guard_pro_db_size_check' );
		if ( false === $db_size_check ) {
			// Count total rows in posts + postmeta + options.
			$total_rows = $wpdb->get_var(
				"SELECT
					(SELECT COUNT(*) FROM `{$wpdb->posts}`) +
					(SELECT COUNT(*) FROM `{$wpdb->postmeta}`) +
					(SELECT COUNT(*) FROM `{$wpdb->options}`) AS total"
			);

			$db_size_check = array(
				'total_rows' => absint( $total_rows ),
				'safe_mode'  => $total_rows > 2000000, // 2M rows threshold.
			);

			set_transient( 'content_guard_pro_db_size_check', $db_size_check, DAY_IN_SECONDS );
		}

		if ( ! empty( $db_size_check['safe_mode'] ) ) {
			cgp_log( sprintf(
				'Content Guard Pro: Safe Mode auto-enabled - Database has %s rows (exceeds 2M threshold)',
				number_format_i18n( $db_size_check['total_rows'] )
			) );
			return true;
		}

		return false;
	}

	/**
	 * Schedule next batch with throttling.
	 *
	 * This is the critical method that implements auto-throttling.
	 * It schedules the next batch with adjusted parameters based on performance.
	 *
	 * @since 1.0.0
	 * @param int   $scan_id            Scan ID.
	 * @param array $batch_args         Batch arguments.
	 * @param array $performance_metrics Performance data from completed batch.
	 * @return int|bool Action ID on success, false on failure.
	 */
	public static function schedule_next_batch( $scan_id, $batch_args, $performance_metrics = array() ) {
		if ( ! self::is_action_scheduler_available() ) {
			return false;
		}

		// Get current batch parameters.
		$current_batch_size = ! empty( $batch_args['batch_size'] ) ? $batch_args['batch_size'] : self::DEFAULT_BATCH_SIZE;
		$current_delay      = ! empty( $batch_args['delay'] ) ? $batch_args['delay'] : self::DEFAULT_DELAY;

		// Calculate throttle adjustments (PRD Section 3.6 & Appendix H).
		$adjustments = self::calculate_throttle_adjustments(
			$current_batch_size,
			$current_delay,
			$performance_metrics
		);

		// Update batch args with new parameters.
		$next_batch_args = array_merge(
			$batch_args,
			array(
				'batch_size'     => $adjustments['batch_size'],
				'delay'          => $adjustments['delay'],
				'throttle_state' => $adjustments['throttle_state'],
			)
		);

		// Update scan record with throttle state.
		self::update_scan_throttle_state( $scan_id, $adjustments['throttle_state'] );

		// Schedule next batch with delay (PRD Appendix H: 1-3s default, 5-10s if throttled).
		$next_run = time() + $adjustments['delay'];

		$action_id = as_schedule_single_action(
			$next_run,
			self::HOOK_PROCESS_BATCH,
			array( $scan_id, $next_batch_args ),
			self::GROUP
		);

		if ( $action_id ) {
			cgp_log( sprintf(
				'Content Guard Pro: Next batch scheduled for scan %d in %ds (Batch size: %d, State: %s)',
				$scan_id,
				$adjustments['delay'],
				$adjustments['batch_size'],
				$adjustments['throttle_state']
			) );
		}

		return $action_id;
	}

	/**
	 * Update scan throttle state in database.
	 *
	 * @since 1.0.0
	 * @param int    $scan_id        Scan ID.
	 * @param string $throttle_state Throttle state.
	 */
	private static function update_scan_throttle_state( $scan_id, $throttle_state ) {
		global $wpdb;

		$scan_id = absint( $scan_id );
		if ( ! $scan_id ) {
			return;
		}

		$table_name = $wpdb->prefix . 'content_guard_pro_scans';

		$wpdb->update(
			$table_name,
			array( 'throttle_state' => $throttle_state ),
			array( 'scan_id' => $scan_id ),
			array( '%s' ),
			array( '%d' )
		);
	}

	/**
	 * Get memory limit in bytes.
	 *
	 * Converts WP_MEMORY_LIMIT to bytes for comparison.
	 *
	 * @since 1.0.0
	 * @return int Memory limit in bytes.
	 */
	private static function get_memory_limit_bytes() {
		$memory_limit = WP_MEMORY_LIMIT;

		// Convert to bytes.
		$value = (int) $memory_limit;
		$unit  = strtoupper( substr( $memory_limit, -1 ) );

		switch ( $unit ) {
			case 'G':
				$value *= 1024 * 1024 * 1024;
				break;
			case 'M':
				$value *= 1024 * 1024;
				break;
			case 'K':
				$value *= 1024;
				break;
		}

		return $value;
	}

	/**
	 * Get current performance metrics for a batch.
	 *
	 * This should be called at the end of batch processing to gather metrics.
	 *
	 * @since 1.0.0
	 * @param float $batch_start_time Batch start microtime.
	 * @param int   $queries_count    Number of queries executed.
	 * @param float $total_query_time Total query time in seconds.
	 * @return array Performance metrics.
	 */
	public static function get_batch_performance_metrics( $batch_start_time, $queries_count = 0, $total_query_time = 0 ) {
		$batch_end_time = microtime( true );
		$batch_runtime  = $batch_end_time - $batch_start_time;

		// Calculate average query time in milliseconds.
		$avg_query_ms = $queries_count > 0 ? ( $total_query_time / $queries_count ) * 1000 : 0;

		// Get peak memory usage.
		$peak_memory = memory_get_peak_usage( true );

		// Convert peak memory to MB for readability.
		$peak_mem_mb = round( $peak_memory / 1024 / 1024, 2 );

		return array(
			'batch_runtime'  => $batch_runtime,
			'avg_query_ms'   => round( $avg_query_ms, 2 ),
			'peak_memory'    => $peak_memory,
			'peak_mem_mb'    => $peak_mem_mb,
			'queries_count'  => $queries_count,
		);
	}
}

