<?php
/**
 * Scans List Table
 *
 * Extends WP_List_Table to display scan history from the content_guard_pro_scans table.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Load WP_List_Table if not already loaded.
if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Class CGP_Scans_List_Table
 *
 * Displays scan history with columns for:
 * - Scan ID
 * - Mode (Quick/Standard)
 * - Started At
 * - Duration
 * - Items Checked
 * - Findings
 * - Status
 * - Performance Metrics
 *
 * @since 1.0.0
 */
class CGP_Scans_List_Table extends WP_List_Table {

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		parent::__construct(
			array(
				'singular' => 'scan',
				'plural'   => 'scans',
				'ajax'     => false,
			)
		);
	}

	/**
	 * Get columns.
	 *
	 * @since 1.0.0
	 * @return array Column headers.
	 */
	public function get_columns() {
		return array(
			'scan_id'        => __( 'Scan ID', 'content-guard-pro' ),
			'mode'           => __( 'Mode', 'content-guard-pro' ),
			'started_at'     => __( 'Started', 'content-guard-pro' ),
			'duration'       => __( 'Duration', 'content-guard-pro' ),
			'totals_checked' => __( 'Items Checked', 'content-guard-pro' ),
			'totals_flagged' => __( 'Findings', 'content-guard-pro' ),
			'status'         => __( 'Status', 'content-guard-pro' ),
			'performance'    => __( 'Performance', 'content-guard-pro' ),
		);
	}

	/**
	 * Get sortable columns.
	 *
	 * @since 1.0.0
	 * @return array Sortable columns.
	 */
	protected function get_sortable_columns() {
		return array(
			'scan_id'        => array( 'scan_id', true ),
			'mode'           => array( 'mode', false ),
			'started_at'     => array( 'started_at', true ),
			'totals_checked' => array( 'totals_checked', false ),
			'totals_flagged' => array( 'totals_flagged', false ),
		);
	}

	/**
	 * Get table classes.
	 *
	 * Adds custom class for JavaScript targeting.
	 *
	 * @since 1.0.1
	 * @return array Table classes.
	 */
	protected function get_table_classes() {
		$classes = parent::get_table_classes();
		$classes[] = 'content-guard-pro-scans-list-table';
		return $classes;
	}

	/**
	 * Default column renderer.
	 *
	 * @since 1.0.0
	 * @param array  $item        Scan record.
	 * @param string $column_name Column name.
	 * @return string Column content.
	 */
	protected function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			case 'scan_id':
			case 'totals_checked':
			case 'totals_flagged':
				return esc_html( $item[ $column_name ] );
			default:
				return '—';
		}
	}

	/**
	 * Render scan_id column.
	 *
	 * @since 1.0.0
	 * @param array $item Scan record.
	 * @return string Column content.
	 */
	protected function column_scan_id( $item ) {
		$scan_id = absint( $item['scan_id'] );
		
		// Build row actions.
		$actions = array();
		
		// View details action - show for all scans.
		// Uses totals_flagged (accurate count) + approximate breakdown with clear note when needed.
		$view_url = wp_nonce_url(
			add_query_arg(
				array(
					'page'    => 'content-guard-pro-scans',
					'action'  => 'view',
					'scan_id' => $scan_id,
				),
				admin_url( 'admin.php' )
			),
			'view_scan_' . $scan_id
		);
		$actions['view'] = sprintf(
			'<a href="%s" class="cgp-view-scan-details" data-scan-id="%d">%s</a>',
			esc_url( $view_url ),
			esc_attr( $scan_id ),
			__( 'View Details', 'content-guard-pro' )
		);

		// Delete action (if completed or failed).
		if ( ! empty( $item['finished_at'] ) ) {
			$actions['delete'] = sprintf(
				'<a href="%s" class="delete" onclick="return confirm(\'%s\')">%s</a>',
				esc_url(
					wp_nonce_url(
						add_query_arg(
							array(
								'page'    => 'content-guard-pro-scans',
								'action'  => 'delete',
								'scan_id' => $scan_id,
							),
							admin_url( 'admin.php' )
						),
						'delete_scan_' . $scan_id
					)
				),
				esc_js( __( 'Are you sure you want to delete this scan record?', 'content-guard-pro' ) ),
				__( 'Delete', 'content-guard-pro' )
			);
		}

		return sprintf(
			'<strong>#%s</strong>%s',
			$scan_id,
			$this->row_actions( $actions )
		);
	}

	/**
	 * Render mode column.
	 *
	 * @since 1.0.0
	 * @param array $item Scan record.
	 * @return string Column content.
	 */
	protected function column_mode( $item ) {
		$mode = esc_html( $item['mode'] );
		$class = 'quick' === $item['mode'] ? 'content-guard-pro-badge-quick' : 'content-guard-pro-badge-standard';
		
		return sprintf(
			'<span class="content-guard-pro-badge %s">%s</span>',
			esc_attr( $class ),
			ucfirst( $mode )
		);
	}

	/**
	 * Render started_at column.
	 *
	 * @since 1.0.0
	 * @param array $item Scan record.
	 * @return string Column content.
	 */
	protected function column_started_at( $item ) {
		if ( empty( $item['started_at'] ) ) {
			return '—';
		}

		$timestamp = strtotime( $item['started_at'] );
		
		// Check if strtotime() failed.
		if ( false === $timestamp ) {
			return esc_html( $item['started_at'] );
		}
		
		$time_diff = human_time_diff( $timestamp, current_time( 'timestamp' ) );

		return sprintf(
			'<span title="%s">%s ago</span>',
			esc_attr( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $timestamp ) ),
			esc_html( $time_diff )
		);
	}

	/**
	 * Render duration column.
	 *
	 * @since 1.0.0
	 * @param array $item Scan record.
	 * @return string Column content.
	 */
	protected function column_duration( $item ) {
		if ( empty( $item['started_at'] ) ) {
			return '—';
		}

		// If not finished, show "Running...".
		if ( empty( $item['finished_at'] ) ) {
			return '<span class="content-guard-pro-status-running">' . esc_html__( 'Running...', 'content-guard-pro' ) . '</span>';
		}

		$start = strtotime( $item['started_at'] );
		$end   = strtotime( $item['finished_at'] );
		
		// Check if strtotime() failed for either date.
		if ( false === $start || false === $end ) {
			return '—';
		}
		
		$duration = $end - $start;

		// Format duration.
		if ( $duration < 60 ) {
			/* translators: %d: Duration in seconds. */
			return sprintf( __( '%d seconds', 'content-guard-pro' ), $duration );
		} elseif ( $duration < 3600 ) {
			$minutes = floor( $duration / 60 );
			$seconds = $duration % 60;
			/* translators: 1: Duration in minutes, 2: Duration in seconds. */
			return sprintf( __( '%1$d min %2$d sec', 'content-guard-pro' ), $minutes, $seconds );
		} else {
			$hours = floor( $duration / 3600 );
			$minutes = floor( ( $duration % 3600 ) / 60 );
			/* translators: 1: Duration in hours, 2: Duration in minutes. */
			return sprintf( __( '%1$d hr %2$d min', 'content-guard-pro' ), $hours, $minutes );
		}
	}

	/**
	 * Render totals_flagged column.
	 *
	 * @since 1.0.0
	 * @param array $item Scan record.
	 * @return string Column content.
	 */
	protected function column_totals_flagged( $item ) {
		$flagged = absint( $item['totals_flagged'] );

		if ( $flagged === 0 ) {
			return '<span class="content-guard-pro-findings-none">0</span>';
		}

		// Link to findings filtered by this scan.
		return sprintf(
			'<a href="%s"><strong>%d</strong></a>',
			esc_url(
				add_query_arg(
					array(
						'page'    => 'content-guard-pro-findings',
						'scan_id' => $item['scan_id'],
					),
					admin_url( 'admin.php' )
				)
			),
			$flagged
		);
	}

	/**
	 * Render status column.
	 *
	 * @since 1.0.0
	 * @param array $item Scan record.
	 * @return string Column content.
	 */
	protected function column_status( $item ) {
		// Check for explicit status first (pending, running, cancelled, paused).
		if ( ! empty( $item['status'] ) && empty( $item['finished_at'] ) ) {
			// Use explicit status for unfinished scans.
			switch ( $item['status'] ) {
				case 'pending':
					$label = __( 'Pending', 'content-guard-pro' );
					$class = 'content-guard-pro-status-pending';
					break;
				case 'running':
					$label = __( 'Running', 'content-guard-pro' );
					$class = 'content-guard-pro-status-running';
					break;
				case 'paused':
					$label = __( 'Paused', 'content-guard-pro' );
					$class = 'content-guard-pro-status-paused';
					break;
				case 'cancelled':
					$label = __( 'Cancelled', 'content-guard-pro' );
					$class = 'content-guard-pro-status-cancelled';
					break;
				default:
					$label = __( 'Running', 'content-guard-pro' );
					$class = 'content-guard-pro-status-running';
			}
		} elseif ( empty( $item['finished_at'] ) ) {
			// Fallback for scans without explicit status.
			$label = __( 'Running', 'content-guard-pro' );
			$class = 'content-guard-pro-status-running';
		} elseif ( ! empty( $item['errors'] ) && $item['errors'] > 0 ) {
			// Completed with errors.
			/* translators: %d: Number of errors. */
			$label = sprintf( __( 'Failed (%d errors)', 'content-guard-pro' ), $item['errors'] );
			$class = 'content-guard-pro-status-failed';
		} else {
			// Completed successfully.
			$label = __( 'Completed', 'content-guard-pro' );
			$class = 'content-guard-pro-status-completed';
		}

		return sprintf(
			'<span class="content-guard-pro-status-badge %s">%s</span>',
			esc_attr( $class ),
			esc_html( $label )
		);
	}

	/**
	 * Render performance column.
	 *
	 * @since 1.0.0
	 * @param array $item Scan record.
	 * @return string Column content.
	 */
	protected function column_performance( $item ) {
		if ( empty( $item['finished_at'] ) ) {
			return '—';
		}

		$metrics = array();

		// Average query time.
		if ( ! empty( $item['avg_query_ms'] ) ) {
			$metrics[] = sprintf(
				'<span title="%s">%d ms</span>',
				esc_attr__( 'Average query time', 'content-guard-pro' ),
				absint( $item['avg_query_ms'] )
			);
		}

		// Peak memory.
		if ( ! empty( $item['peak_mem_mb'] ) ) {
			$metrics[] = sprintf(
				'<span title="%s">%d MB</span>',
				esc_attr__( 'Peak memory usage', 'content-guard-pro' ),
				absint( $item['peak_mem_mb'] )
			);
		}

		// Throttle state.
		if ( ! empty( $item['throttle_state'] ) && 'normal' !== $item['throttle_state'] ) {
			$metrics[] = sprintf(
				'<span class="content-guard-pro-throttle-badge" title="%s">%s</span>',
				esc_attr__( 'Throttle state', 'content-guard-pro' ),
				esc_html( ucfirst( $item['throttle_state'] ) )
			);
		}

		return ! empty( $metrics ) ? implode( ' ', $metrics ) : '—';
	}

	/**
	 * Generate row ID for each scan.
	 *
	 * Ensures proper row ID for JavaScript targeting.
	 *
	 * @since 1.0.1
	 * @param array $item Scan record.
	 */
	public function single_row( $item ) {
		echo '<tr id="scan-' . absint( $item['scan_id'] ) . '">';
		$this->single_row_columns( $item );
		echo '</tr>';
	}

	/**
	 * Prepare items for display.
	 *
	 * Fetches scan records from the database with pagination and sorting.
	 *
	 * @since 1.0.0
	 */
	public function prepare_items() {
		global $wpdb;

		// Set up columns.
		$columns  = $this->get_columns();
		$hidden   = array();
		$sortable = $this->get_sortable_columns();
		$this->_column_headers = array( $columns, $hidden, $sortable );

		// Pagination.
		$per_page     = 20;
		$current_page = $this->get_pagenum();
		$offset       = ( $current_page - 1 ) * $per_page;

		// Sorting.
		$orderby = isset( $_GET['orderby'] ) ? sanitize_sql_orderby( $_GET['orderby'] ) : 'started_at';
		$order   = isset( $_GET['order'] ) && 'asc' === strtolower( $_GET['order'] ) ? 'ASC' : 'DESC';

		// Validate orderby.
		$allowed_orderby = array( 'scan_id', 'mode', 'started_at', 'totals_checked', 'totals_flagged' );
		if ( ! in_array( $orderby, $allowed_orderby, true ) ) {
			$orderby = 'started_at';
		}

		// Build query.
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		$blog_id    = get_current_blog_id();

		// Count total items.
		$total_items = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$table_name}` WHERE blog_id = %d",
				$blog_id
			)
		);

		// Fetch items.
		// Note: $orderby is validated against whitelist, $order is validated as ASC/DESC.
		$this->items = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT * FROM `{$table_name}` 
				WHERE blog_id = %d 
				ORDER BY `{$orderby}` {$order} 
				LIMIT %d OFFSET %d",
				$blog_id,
				$per_page,
				$offset
			),
			ARRAY_A
		);

		// Set pagination.
		$this->set_pagination_args(
			array(
				'total_items' => $total_items,
				'per_page'    => $per_page,
				'total_pages' => ceil( $total_items / $per_page ),
			)
		);
	}

	/**
	 * Display message when no scans found.
	 *
	 * @since 1.0.0
	 */
	public function no_items() {
		esc_html_e( 'No scans found. Run your first scan to get started!', 'content-guard-pro' );
	}
}

