<?php
/**
 * Scanner Class - Content Scanning Engine
 *
 * Handles all content scanning operations including batch processing,
 * progress tracking, and scan execution.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Scanner
 *
 * Main scanning engine that processes content in batches:
 * - wp_posts (with Gutenberg block parsing)
 * - wp_postmeta
 * - wp_options (allowlisted keys only)
 *
 * Implements batching, throttling, and progress tracking as per PRD Section 3.1.
 *
 * @since 1.0.0
 */
class CGP_Scanner {

	/**
	 * Batch size for posts (configurable via settings).
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const DEFAULT_BATCH_SIZE = 100;

	/**
	 * Default delay between batches in seconds.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const DEFAULT_BATCH_DELAY = 2;

	/**
	 * Memory threshold percentage for throttling.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const MEMORY_THRESHOLD = 50;

	/**
	 * Query time threshold in milliseconds for throttling.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const QUERY_TIME_THRESHOLD = 300;

	/**
	 * Initialize the scanner.
	 *
	 * @since 1.0.0
	 */
	public static function init() {
		// Register hooks for processing actions triggered by scheduler.
		add_action( 'content_guard_pro_start_manual_scan', array( __CLASS__, 'start_manual_scan' ), 10, 1 );
		add_action( 'content_guard_pro_start_daily_scan', array( __CLASS__, 'start_daily_scan' ), 10, 1 );
		add_action( 'content_guard_pro_process_on_save_scan', array( __CLASS__, 'process_on_save_scan' ), 10, 2 );
		add_action( 'content_guard_pro_process_batch', array( __CLASS__, 'process_scan_batch' ), 10, 2 );
		
		// Register hooks for immediate detection of deleted/trashed posts.
		add_action( 'wp_trash_post', array( __CLASS__, 'on_post_trashed' ), 10, 1 );
		add_action( 'before_delete_post', array( __CLASS__, 'on_post_deleted' ), 10, 1 );
		add_action( 'untrash_post', array( __CLASS__, 'on_post_untrashed' ), 10, 1 );
	}

	/**
	 * Start a manual scan.
	 *
	 * Initializes a new scan record (or uses existing one) and enqueues the first batch.
	 *
	 * @since 1.0.0
	 * @param array $args Scan arguments including mode, scan_id (optional), initiated_by, etc.
	 * @return int|false Scan ID on success, false on failure.
	 */
	public static function start_manual_scan( $args ) {
		$mode = isset( $args['mode'] ) ? $args['mode'] : 'standard';
		
		cgp_log( 'Content Guard Pro: Starting manual scan in ' . $mode . ' mode' );
		
		// Check if scan_id was provided (scan record already created in UI).
		if ( isset( $args['scan_id'] ) && $args['scan_id'] > 0 ) {
			$scan_id = absint( $args['scan_id'] );
			
			// Verify scan exists and update status.
			global $wpdb;
			$table_name = $wpdb->prefix . 'content_guard_pro_scans';
			$scan_exists = $wpdb->get_var(
				$wpdb->prepare(
					// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
					"SELECT COUNT(*) FROM `{$table_name}` WHERE scan_id = %d",
					$scan_id
				)
			);
			
			if ( ! $scan_exists ) {
				cgp_log( 'Content Guard Pro: Provided scan_id does not exist: ' . $scan_id );
				// Fall through to create new scan record.
			} else {
				// Scan record exists, update it with estimated total and set status to 'running'.
				$total_items = self::estimate_total_items( $mode );
				
				self::update_scan_record( $scan_id, array(
					'status' => 'running', // Update from 'pending' to 'running' when scan actually starts.
					'notes' => wp_json_encode( array(
						'current_target'   => 'posts',
						'current_offset'   => 0,
						'estimated_total'  => $total_items,
						'initiated_by'     => isset( $args['initiated_by'] ) ? $args['initiated_by'] : get_current_user_id(),
						'scan_type'        => isset( $args['scan_type'] ) ? $args['scan_type'] : 'manual',
					) ),
				) );
				
				cgp_log( sprintf(
					'Content Guard Pro: Using existing scan %d - mode: %s, estimated items: %d',
					$scan_id,
					$mode,
					$total_items
				) );
				
				// Skip to scheduling first batch.
				$scheduled = self::schedule_next_batch( $scan_id, 'posts', 0, 'normal' );
				
				if ( ! $scheduled ) {
					cgp_log( 'Content Guard Pro: Failed to schedule first batch for scan ' . $scan_id );
					return false;
				}
				
				do_action( 'content_guard_pro_scan_started', $scan_id, $args );
				return $scan_id;
			}
		}
		
		// Step 1: Create scan record in content_guard_pro_scans table (fallback if not already created).
		$scan_id = self::create_scan_record( array(
			'mode'           => $mode,
			'started_at'     => current_time( 'mysql' ),
			'status'         => 'running', // Set to 'running' immediately since scan is starting.
			'throttle_state' => 'normal',
			'totals_checked' => 0,
			'totals_flagged' => 0,
			'notes'          => wp_json_encode( array(
				'current_target' => 'posts',
				'current_offset' => 0,
			) ),
		) );
		
		if ( ! $scan_id ) {
			cgp_log( 'Content Guard Pro: Failed to create scan record' );
			return false;
		}
		
		// Step 2: Estimate total items to scan based on mode.
		$total_items = self::estimate_total_items( $mode );
		
		cgp_log( sprintf(
			'Content Guard Pro: Scan %d initialized - mode: %s, estimated items: %d',
			$scan_id,
			$mode,
			$total_items
		) );
		
		// Step 3: Update scan record with estimated total.
		self::update_scan_record( $scan_id, array(
			'notes' => wp_json_encode( array(
				'current_target'   => 'posts',
				'current_offset'   => 0,
				'estimated_total'  => $total_items,
			) ),
		) );
		
		// Step 4: Schedule the very first batch (posts, offset 0).
		$scheduled = self::schedule_next_batch( $scan_id, 'posts', 0, 'normal' );
		
		if ( ! $scheduled ) {
			cgp_log( 'Content Guard Pro: Failed to schedule first batch for scan ' . $scan_id );
			return false;
		}
		
		cgp_log( sprintf(
			'Content Guard Pro: Successfully started scan %d, first batch scheduled',
			$scan_id
		) );
		
		// Trigger action for integrations.
		do_action( 'content_guard_pro_scan_started', $scan_id, $args );
		
		return $scan_id;
	}

	/**
	 * Start a daily scheduled scan.
	 *
	 * Similar to manual scan but triggered by scheduler.
	 *
	 * @since 1.0.0
	 * @param array $args Scan arguments including mode.
	 * @return int|false Scan ID on success, false on failure.
	 */
	public static function start_daily_scan( $args ) {
		cgp_log( 'Content Guard Pro: Starting daily scheduled scan' );
		
		// Daily scans use the same initialization logic as manual scans.
		// Simply delegate to start_manual_scan.
		return self::start_manual_scan( $args );
	}

	/**
	 * Process on-save scan for a single post.
	 *
	 * Quick validation scan triggered when content is saved (PRD Section 3.1).
	 * Should complete within 5 seconds per US-037.
	 *
	 * @since 1.0.0
	 * @param int   $post_id Post ID to scan.
	 * @param array $args    Scan arguments.
	 * @return bool True on success, false on failure.
	 */
	public static function process_on_save_scan( $post_id, $args ) {
		cgp_log( 'Content Guard Pro: Processing on-save scan for post ' . $post_id );
		
		// Simply call scan_post() which handles all the scanning logic.
		$result = self::scan_post( $post_id, array( 'save_findings' => true ) );
		
		if ( ! $result['success'] ) {
			cgp_log( 'Content Guard Pro: On-save scan failed for post ' . $post_id . ' - ' . $result['error'] );
			return false;
		}
		
		// Trigger action for integrations.
		do_action( 'content_guard_pro_on_save_scan_completed', $post_id, array(
			'findings_count' => $result['findings_count'],
			'elapsed'        => $result['elapsed'],
		) );
		
		return true;
	}

	/**
	 * Process a scan batch.
	 *
	 * Main batch processing dispatcher. Routes to appropriate batch handler
	 * based on scan state and mode.
	 *
	 * @since 1.0.0
	 * @param int   $scan_id    Scan ID.
	 * @param array $batch_args Batch arguments (target, offset, limit, etc.).
	 * @return bool True on success, false on failure.
	 */
	public static function process_scan_batch( $scan_id, $batch_args ) {
		global $wpdb;
		
		$target = isset( $batch_args['target'] ) ? $batch_args['target'] : 'posts';
		$offset = isset( $batch_args['offset'] ) ? absint( $batch_args['offset'] ) : 0;
		$limit  = isset( $batch_args['limit'] ) ? absint( $batch_args['limit'] ) : self::DEFAULT_BATCH_SIZE;
		
		// CRITICAL: Use transient lock to prevent concurrent processing of same batch.
		$lock_key = "content_guard_pro_batch_lock_{$scan_id}_{$target}_{$offset}";
		$lock_value = time();
		
		// Try to acquire lock (expires after 60 seconds as safety).
		if ( false === get_transient( $lock_key ) ) {
			set_transient( $lock_key, $lock_value, 60 );
		} else {
			cgp_log( sprintf(
				'Content Guard Pro: Batch already processing (scan_id: %d, target: %s, offset: %d) - skipping duplicate',
				$scan_id,
				$target,
				$offset
			) );
			return false;
		}
		
		cgp_log( sprintf(
			'Content Guard Pro: Processing batch (scan_id: %d, target: %s, offset: %d, limit: %d)',
			$scan_id,
			$target,
			$offset,
			$limit
		) );
		
		// Step 1: Get scan record and verify it's still active.
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		$scan = $wpdb->get_row(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				"SELECT * FROM `{$table_name}` WHERE scan_id = %d",
				$scan_id
			)
		);
		
		if ( ! $scan ) {
			cgp_log( 'Content Guard Pro: Scan not found: ' . $scan_id );
			delete_transient( $lock_key );
			return false;
		}
		
		// Check if scan is paused or cancelled.
		$status = isset( $scan->status ) ? $scan->status : null;
		if ( 'paused' === $status ) {
			cgp_log( 'Content Guard Pro: Scan is paused, skipping batch' );
			delete_transient( $lock_key );
			return false;
		}
		if ( 'cancelled' === $status ) {
			cgp_log( 'Content Guard Pro: Scan is cancelled, skipping batch' );
			delete_transient( $lock_key );
			return false;
		}
		if ( ! empty( $scan->finished_at ) ) {
			cgp_log( 'Content Guard Pro: Scan already completed, skipping batch' );
			delete_transient( $lock_key );
			return false;
		}
		
		// Step 2: Update scan record with current state.
		$notes = array();
		if ( isset( $scan->notes ) && ! empty( $scan->notes ) ) {
			$decoded = json_decode( $scan->notes, true );
			if ( null !== $decoded && JSON_ERROR_NONE === json_last_error() && is_array( $decoded ) ) {
				$notes = $decoded;
			}
		}
		$notes['current_target'] = $target;
		$notes['current_offset'] = $offset;
		
		self::update_scan_record( $scan_id, array(
			'status' => 'running',
			'notes'  => wp_json_encode( $notes ),
		) );
		
		// Step 3: Call the appropriate worker method based on target.
		$has_more_items = false;
		$batch_result = array(
			'items_processed' => 0,
			'findings_count' => 0,
		);
		
		switch ( $target ) {
			case 'posts':
				$result = self::perform_posts_batch( $scan_id, $batch_args );
				$has_more_items = $result['has_more'];
				$batch_result = $result;
				break;
			case 'postmeta':
				$result = self::perform_postmeta_batch( $scan_id, $batch_args );
				$has_more_items = $result['has_more'];
				$batch_result = $result;
				break;
			case 'options':
				$result = self::perform_options_batch( $scan_id, $batch_args );
				$has_more_items = $result['has_more'];
				$batch_result = $result;
				break;
			default:
				cgp_log( 'Content Guard Pro: Unknown batch target: ' . $target );
				delete_transient( $lock_key );
				return false;
		}
		
		// Step 3.5: Update scan totals incrementally.
		if ( $batch_result['items_processed'] > 0 ) {
			$current_totals_checked = isset( $scan->totals_checked ) ? absint( $scan->totals_checked ) : 0;
			$current_totals_flagged = isset( $scan->totals_flagged ) ? absint( $scan->totals_flagged ) : 0;
			
			self::update_scan_record( $scan_id, array(
				'totals_checked' => $current_totals_checked + $batch_result['items_processed'],
				'totals_flagged' => $current_totals_flagged + $batch_result['findings_count'],
			) );
		}
		
		// Step 4: Get current throttle state for batch scheduling.
		$throttle_state = isset( $scan->throttle_state ) ? $scan->throttle_state : 'normal';
		
		// Step 5: Schedule next batch or move to next target.
		if ( $has_more_items ) {
			// More items in current target, schedule next batch with updated offset.
			$new_offset = $offset + $limit;
			
			cgp_log( sprintf(
				'Content Guard Pro: More items in %s, scheduling next batch at offset %d',
				$target,
				$new_offset
			) );
			
			self::schedule_next_batch( $scan_id, $target, $new_offset, $throttle_state );
		} else {
			// Current target is complete, determine next target.
			$next_target = self::get_next_target( $target, $scan->mode );
			
			if ( $next_target ) {
				// Move to next target, starting at offset 0.
				cgp_log( sprintf(
					'Content Guard Pro: Target %s complete, moving to %s',
					$target,
					$next_target
				) );
				
				self::schedule_next_batch( $scan_id, $next_target, 0, $throttle_state );
			} else {
				// All targets complete, finalize scan.
				cgp_log( sprintf(
					'Content Guard Pro: All targets complete, finalizing scan %d',
					$scan_id
				) );
				
				self::finalize_scan( $scan_id );
			}
		}
		
		// Release the batch lock.
		delete_transient( $lock_key );
		
		return true;
	}

	/**
	 * Perform batch scan of wp_posts.
	 *
	 * Scans posts in batches, parsing Gutenberg blocks and detecting threats.
	 * Query from PRD Appendix B.
	 *
	 * @since 1.0.0
	 * @param int   $scan_id    Scan ID.
	 * @param array $batch_args Batch arguments (offset, limit).
	 * @return bool True on success, false on failure.
	 */
	public static function perform_posts_batch( $scan_id, $batch_args ) {
		$offset = isset( $batch_args['offset'] ) ? absint( $batch_args['offset'] ) : 0;
		$limit  = isset( $batch_args['limit'] ) ? absint( $batch_args['limit'] ) : self::DEFAULT_BATCH_SIZE;
		
		// Set flag to prevent auto-resolution during batch scans.
		// Batch scans are for discovery - they should NOT trigger auto-resolution.
		// Only post-save scans and manual single-post scans should trigger auto-resolution.
		if ( ! defined( 'CGP_BATCH_SCAN_IN_PROGRESS' ) ) {
			define( 'CGP_BATCH_SCAN_IN_PROGRESS', true );
		}
		
		cgp_log( sprintf(
			'Content Guard Pro: Processing posts batch (scan_id: %d, offset: %d, limit: %d)',
			$scan_id,
			$offset,
			$limit
		) );
		
		$start_time = microtime( true );
		$start_memory = memory_get_usage( true );
		
		// Get candidate posts using pre-filters from PRD Appendix B.
		$posts = self::get_posts_for_scanning( $offset, $limit );
		
		if ( empty( $posts ) ) {
			cgp_log( 'Content Guard Pro: No posts found in batch' );
			return false; // No posts to process, target complete.
		}
		
		$findings_count = 0;
		$items_processed = count( $posts );
		
		foreach ( $posts as $post ) {
			$content = $post->post_content;
			
			// Parse Gutenberg blocks if content contains block markers.
			// Per PRD Section 3.1: parse post_content with parse_blocks().
			if ( has_blocks( $content ) ) {
				$blocks = parse_blocks( $content );
				
				// Reconstruct content from blocks for comprehensive scanning.
				// This ensures we scan both block content and HTML within blocks.
				$block_content = '';
				foreach ( $blocks as $block ) {
					if ( ! empty( $block['innerHTML'] ) ) {
						$block_content .= $block['innerHTML'] . "\n";
					}
					// Also check block attributes which may contain malicious data.
					if ( ! empty( $block['attrs'] ) ) {
						$block_content .= wp_json_encode( $block['attrs'] ) . "\n";
					}
				}
				
				// Scan both original content and parsed block content.
				$content_to_scan = $content . "\n\n" . $block_content;
			} else {
				$content_to_scan = $content;
			}
			
			// Run detection patterns via CGP_Detector.
			// Returns array of findings with rule_id, severity, confidence, matched_text, etc.
			$issues = self::find_issues_in_content(
				$content_to_scan,
				$post->ID,
				'post',
				'post_content'
			);
			
			// Save findings with fingerprint-based deduplication.
			if ( ! empty( $issues ) ) {
				foreach ( $issues as $issue ) {
					$finding_id = self::save_finding( $issue );
					if ( $finding_id ) {
						$findings_count++;
						
						// Trigger notification for critical findings.
						if ( 'critical' === $issue['severity'] ) {
							do_action( 'content_guard_pro_critical_finding_detected', $finding_id, $issue );
						}
					}
				}
			}
		}
		
		// Track performance metrics per PRD Section 3.6.
		$end_time = microtime( true );
		$elapsed_ms = ( $end_time - $start_time ) * 1000;
		$peak_memory_mb = ( memory_get_usage( true ) - $start_memory ) / 1024 / 1024;
		
		cgp_log( sprintf(
			'Content Guard Pro: Posts batch complete - %d posts scanned, %d findings, %.2f ms, %.2f MB',
			count( $posts ),
			$findings_count,
			$elapsed_ms,
			$peak_memory_mb
		) );
		
		// Check if throttling is needed based on performance metrics.
		$throttle_state = self::check_throttling( array(
			'query_time'    => $elapsed_ms,
			'memory'        => $peak_memory_mb,
			'batch_runtime' => $elapsed_ms / 1000, // Convert to seconds.
		) );
		
		// Store metrics for scan summary.
		do_action( 'content_guard_pro_batch_completed', $scan_id, array(
			'target'         => 'posts',
			'items_scanned'  => $items_processed,
			'findings_count' => $findings_count,
			'elapsed_ms'     => $elapsed_ms,
			'peak_memory_mb' => $peak_memory_mb,
			'throttle_state' => $throttle_state,
		) );
		
		// Return batch results including whether there are more items.
		// CRITICAL: Only return has_more=true if we processed EXACTLY the limit (full batch).
		// If we got less than a full batch, we've reached the end.
		// Using === instead of >= prevents infinite loops.
		return array(
			'has_more' => $items_processed === $limit,
			'items_processed' => $items_processed,
			'findings_count' => $findings_count,
		);
	}

	/**
	 * Perform batch scan of wp_postmeta.
	 *
	 * Scans post meta values for suspicious content.
	 * Includes special handling for Elementor page builder data.
	 * Query from PRD Appendix B.
	 *
	 * @since 1.0.0
	 * @param int   $scan_id    Scan ID.
	 * @param array $batch_args Batch arguments (offset, limit).
	 * @return bool True on success, false on failure.
	 */
	public static function perform_postmeta_batch( $scan_id, $batch_args ) {
		$offset = isset( $batch_args['offset'] ) ? absint( $batch_args['offset'] ) : 0;
		$limit  = isset( $batch_args['limit'] ) ? absint( $batch_args['limit'] ) : self::DEFAULT_BATCH_SIZE;
		
		cgp_log( sprintf(
			'Content Guard Pro: Processing postmeta batch (scan_id: %d, offset: %d, limit: %d)',
			$scan_id,
			$offset,
			$limit
		) );
		
		$start_time = microtime( true );
		$start_memory = memory_get_usage( true );
		
		// Get candidate postmeta using pre-filters from PRD Appendix B.
		$meta_entries = self::get_postmeta_for_scanning( $offset, $limit );
		
		if ( empty( $meta_entries ) ) {
			cgp_log( 'Content Guard Pro: No postmeta found in batch' );
			return false; // No meta to process, target complete.
		}
		
		$findings_count = 0;
		$items_processed = count( $meta_entries );
		$detector = new CGP_Detector();
		
		foreach ( $meta_entries as $meta ) {
			$content = $meta->meta_value;
			$issues = array();
			
			// Special handling for Elementor data.
			if ( '_elementor_data' === $meta->meta_key ) {
				// Elementor stores data as JSON array.
				$elementor_data = json_decode( $content, true );
				if ( null !== $elementor_data && JSON_ERROR_NONE === json_last_error() && is_array( $elementor_data ) ) {
					// Use dedicated Elementor scanner.
					$issues = $detector->scan_elementor_data(
						$elementor_data,
						$meta->post_id,
						'_elementor_data'
					);
					
					cgp_log( sprintf(
						'Content Guard Pro: Scanned Elementor data for post %d - %d findings',
						$meta->post_id,
						count( $issues )
					) );
				}
			} else {
				// Standard postmeta scanning.
				// Attempt to decode JSON if the meta value looks like JSON.
				// Some plugins store Gutenberg blocks or structured data in postmeta.
				if ( is_string( $content ) && strlen( $content ) > 2 ) {
					$first_char = $content[0];
					if ( '{' === $first_char || '[' === $first_char ) {
						$decoded = json_decode( $content, true );
						if ( null !== $decoded && JSON_ERROR_NONE === json_last_error() && is_array( $decoded ) ) {
							// Scan both the original and decoded content.
							$content = $content . "\n\n" . wp_json_encode( $decoded );
						}
					}
				}
				
				// Run detection patterns via CGP_Detector.
				$issues = self::find_issues_in_content(
					$content,
					$meta->post_id,
					'postmeta',
					'meta_value[' . $meta->meta_key . ']'
				);
			}
			
			// Save findings with fingerprint-based deduplication.
			if ( ! empty( $issues ) ) {
				foreach ( $issues as $issue ) {
					// Add meta_key to extra metadata for context.
					if ( ! isset( $issue['extra'] ) ) {
						$issue['extra'] = array();
					}
					if ( is_string( $issue['extra'] ) ) {
						$decoded_extra = json_decode( $issue['extra'], true );
						if ( null !== $decoded_extra && JSON_ERROR_NONE === json_last_error() && is_array( $decoded_extra ) ) {
							$issue['extra'] = $decoded_extra;
						} else {
							$issue['extra'] = array();
						}
					}
					$issue['extra']['meta_key'] = $meta->meta_key;
					$issue['extra']['meta_id'] = $meta->meta_id;
					
					// Mark Elementor source for UI context.
					if ( '_elementor_data' === $meta->meta_key ) {
						$issue['extra']['source'] = 'elementor';
					}
					
					$finding_id = self::save_finding( $issue );
					if ( $finding_id ) {
						$findings_count++;
						
						// Trigger notification for critical findings.
						if ( 'critical' === $issue['severity'] ) {
							do_action( 'content_guard_pro_critical_finding_detected', $finding_id, $issue );
						}
					}
				}
			}
		}
		
		// Track performance metrics per PRD Section 3.6.
		$end_time = microtime( true );
		$elapsed_ms = ( $end_time - $start_time ) * 1000;
		$peak_memory_mb = ( memory_get_usage( true ) - $start_memory ) / 1024 / 1024;
		
		cgp_log( sprintf(
			'Content Guard Pro: Postmeta batch complete - %d entries scanned, %d findings, %.2f ms, %.2f MB',
			count( $meta_entries ),
			$findings_count,
			$elapsed_ms,
			$peak_memory_mb
		) );
		
		// Check if throttling is needed based on performance metrics.
		$throttle_state = self::check_throttling( array(
			'query_time'    => $elapsed_ms,
			'memory'        => $peak_memory_mb,
			'batch_runtime' => $elapsed_ms / 1000,
		) );
		
		// Store metrics for scan summary.
		do_action( 'content_guard_pro_batch_completed', $scan_id, array(
			'target'         => 'postmeta',
			'items_scanned'  => $items_processed,
			'findings_count' => $findings_count,
			'elapsed_ms'     => $elapsed_ms,
			'peak_memory_mb' => $peak_memory_mb,
			'throttle_state' => $throttle_state,
		) );
		
		// Return batch results including whether there are more items.
		// CRITICAL: Only return has_more=true if we processed EXACTLY the limit (full batch).
		// If we got less than a full batch, we've reached the end.
		// Using === instead of >= prevents infinite loops.
		return array(
			'has_more' => $items_processed === $limit,
			'items_processed' => $items_processed,
			'findings_count' => $findings_count,
		);
	}

	/**
	 * Perform batch scan of wp_options (allowlisted keys only).
	 *
	 * Scans only specific option keys as defined in PRD Section 3.1:
	 * - widget_text
	 * - widget_custom_html
	 * - widget_block
	 *
	 * @since 1.0.0
	 * @param int   $scan_id    Scan ID.
	 * @param array $batch_args Batch arguments.
	 * @return bool True on success, false on failure.
	 */
	public static function perform_options_batch( $scan_id, $batch_args ) {
		cgp_log( sprintf(
			'Content Guard Pro: Processing options batch (scan_id: %d)',
			$scan_id
		) );
		
		$start_time = microtime( true );
		$start_memory = memory_get_usage( true );
		
		// Get allowlisted options from PRD Appendix B.
		// Options batch scans all at once since it's a small, fixed set.
		$options = self::get_options_for_scanning();
		
		if ( empty( $options ) ) {
			cgp_log( 'Content Guard Pro: No options found in batch' );
			return false; // No options to process, target complete.
		}
		
		$findings_count = 0;
		$items_processed = count( $options );
		
		foreach ( $options as $option ) {
			$option_name = $option->option_name;
			$option_value = $option->option_value;
			
			// Parse serialized data (widgets are typically serialized arrays).
			$parsed_value = maybe_unserialize( $option_value );
			
			if ( is_array( $parsed_value ) ) {
				// Process each widget instance in the array.
				foreach ( $parsed_value as $widget_index => $widget_data ) {
					if ( ! is_array( $widget_data ) ) {
						continue;
					}
					
					// For widget_block, parse Gutenberg blocks per PRD Section 3.1.
					if ( 'widget_block' === $option_name && ! empty( $widget_data['content'] ) ) {
						$block_content = $widget_data['content'];
						
						if ( has_blocks( $block_content ) ) {
							$blocks = parse_blocks( $block_content );
							
							// Reconstruct content from blocks.
							$block_html = '';
							foreach ( $blocks as $block ) {
								if ( ! empty( $block['innerHTML'] ) ) {
									$block_html .= $block['innerHTML'] . "\n";
								}
								if ( ! empty( $block['attrs'] ) ) {
									$block_html .= wp_json_encode( $block['attrs'] ) . "\n";
								}
							}
							
							$content_to_scan = $block_content . "\n\n" . $block_html;
						} else {
							$content_to_scan = $block_content;
						}
					} elseif ( ! empty( $widget_data['text'] ) ) {
						// For widget_text and widget_custom_html, scan the text field.
						$content_to_scan = $widget_data['text'];
					} else {
						// Generic scan of widget data.
						$content_to_scan = wp_json_encode( $widget_data );
					}
					
					// Run detection patterns via CGP_Detector.
					// Use widget index as surrogate object_id since options don't have numeric IDs.
					$object_id = crc32( $option_name . '_' . $widget_index );
					
					$issues = self::find_issues_in_content(
						$content_to_scan,
						$object_id,
						'option',
						$option_name . '[' . $widget_index . ']'
					);
					
					// Save findings with fingerprint-based deduplication.
					if ( ! empty( $issues ) ) {
						foreach ( $issues as $issue ) {
							// Add option context to extra metadata.
							if ( ! isset( $issue['extra'] ) ) {
								$issue['extra'] = array();
							}
							if ( is_string( $issue['extra'] ) ) {
								$decoded_extra = json_decode( $issue['extra'], true );
								if ( null !== $decoded_extra && JSON_ERROR_NONE === json_last_error() && is_array( $decoded_extra ) ) {
									$issue['extra'] = $decoded_extra;
								} else {
									$issue['extra'] = array();
								}
							}
							$issue['extra']['option_name'] = $option_name;
							$issue['extra']['widget_index'] = $widget_index;
							
							$finding_id = self::save_finding( $issue );
							if ( $finding_id ) {
								$findings_count++;
								
								// Trigger notification for critical findings.
								if ( 'critical' === $issue['severity'] ) {
									do_action( 'content_guard_pro_critical_finding_detected', $finding_id, $issue );
								}
							}
						}
					}
				}
			} else {
				// Non-array option value, scan as-is.
				$object_id = crc32( $option_name );
				
				$issues = self::find_issues_in_content(
					$option_value,
					$object_id,
					'option',
					$option_name
				);
				
				if ( ! empty( $issues ) ) {
					foreach ( $issues as $issue ) {
						if ( ! isset( $issue['extra'] ) ) {
							$issue['extra'] = array();
						}
						if ( is_string( $issue['extra'] ) ) {
							$decoded_extra = json_decode( $issue['extra'], true );
							if ( null !== $decoded_extra && JSON_ERROR_NONE === json_last_error() && is_array( $decoded_extra ) ) {
								$issue['extra'] = $decoded_extra;
							} else {
								$issue['extra'] = array();
							}
						}
						$issue['extra']['option_name'] = $option_name;
						
						$finding_id = self::save_finding( $issue );
						if ( $finding_id ) {
							$findings_count++;
							
							if ( 'critical' === $issue['severity'] ) {
								do_action( 'content_guard_pro_critical_finding_detected', $finding_id, $issue );
							}
						}
					}
				}
			}
		}
		
		// Track performance metrics per PRD Section 3.6.
		$end_time = microtime( true );
		$elapsed_ms = ( $end_time - $start_time ) * 1000;
		$peak_memory_mb = ( memory_get_usage( true ) - $start_memory ) / 1024 / 1024;
		
		cgp_log( sprintf(
			'Content Guard Pro: Options batch complete - %d options scanned, %d findings, %.2f ms, %.2f MB',
			count( $options ),
			$findings_count,
			$elapsed_ms,
			$peak_memory_mb
		) );
		
		// Check if throttling is needed based on performance metrics.
		$throttle_state = self::check_throttling( array(
			'query_time'    => $elapsed_ms,
			'memory'        => $peak_memory_mb,
			'batch_runtime' => $elapsed_ms / 1000,
		) );
		
		// Store metrics for scan summary.
		do_action( 'content_guard_pro_batch_completed', $scan_id, array(
			'target'         => 'options',
			'items_scanned'  => $items_processed,
			'findings_count' => $findings_count,
			'elapsed_ms'     => $elapsed_ms,
			'peak_memory_mb' => $peak_memory_mb,
			'throttle_state' => $throttle_state,
		) );
		
		// Options are processed all at once (no pagination).
		// Always return has_more=false indicating target is complete.
		return array(
			'has_more' => false,
			'items_processed' => $items_processed,
			'findings_count' => $findings_count,
		);
	}

	/**
	 * Scan a single post.
	 *
	 * Used for on-save scans and individual post validation.
	 * Scans the post content and saves any findings to the database.
	 *
	 * @since 1.0.0
	 * @param int   $post_id Post ID to scan.
	 * @param array $options Scan options (save_findings defaults to true).
	 * @return array Scan results with findings.
	 */
	public static function scan_post( $post_id, $options = array() ) {
		cgp_log( 'Content Guard Pro: Scanning post ' . $post_id );
		
		$start_time = microtime( true );
		$save_findings = isset( $options['save_findings'] ) ? $options['save_findings'] : true;
		
		// Step 1: Get post object.
		$post = get_post( $post_id );
		
		if ( ! $post ) {
			return array(
				'success' => false,
				'error'   => 'Post not found',
			);
		}
		
		// Step 2: Parse post_content with parse_blocks() - same logic as perform_posts_batch.
		$content = $post->post_content;
		
		if ( has_blocks( $content ) ) {
			$blocks = parse_blocks( $content );
			
			// Reconstruct content from blocks for comprehensive scanning.
			$block_content = '';
			foreach ( $blocks as $block ) {
				if ( ! empty( $block['innerHTML'] ) ) {
					$block_content .= $block['innerHTML'] . "\n";
				}
				if ( ! empty( $block['attrs'] ) ) {
					$block_content .= wp_json_encode( $block['attrs'] ) . "\n";
				}
			}
			
			$content_to_scan = $content . "\n\n" . $block_content;
		} else {
			$content_to_scan = $content;
		}
		
		// Step 3: Call find_issues_in_content() to run detection patterns.
		$issues = self::find_issues_in_content(
			$content_to_scan,
			$post_id,
			'post',
			'post_content'
		);
		
		// Step 4: Loop findings and save them with save_finding().
		$findings_count = 0;
		$findings = array();
		
		if ( ! empty( $issues ) ) {
			foreach ( $issues as $issue ) {
				if ( $save_findings ) {
					$finding_id = self::save_finding( $issue );
					if ( $finding_id ) {
						$findings_count++;
						
						// Trigger notification for critical findings.
						if ( 'critical' === $issue['severity'] ) {
							do_action( 'content_guard_pro_critical_finding_detected', $finding_id, $issue );
						}
					}
				}
				
				// Add to findings array for return value.
				$findings[] = array(
					'rule_id'         => $issue['rule_id'],
					'severity'        => $issue['severity'],
					'confidence'      => $issue['confidence'],
					'matched_text'    => $issue['matched_text'],
					'matched_excerpt' => wp_strip_all_tags( substr( $issue['matched_text'], 0, 200 ) ),
				);
			}
		}
		
		$elapsed = microtime( true ) - $start_time;
		
		cgp_log( sprintf(
			'Content Guard Pro: Scanned post %d - %d findings in %.2f seconds',
			$post_id,
			count( $findings ),
			$elapsed
		) );
		
		return array(
			'success'        => true,
			'post_id'        => $post_id,
			'post_type'      => $post->post_type,
			'post_status'    => $post->post_status,
			'findings'       => $findings,
			'findings_count' => $findings_count,
			'elapsed'        => $elapsed,
		);
	}

	/**
	 * Get posts for scanning with pre-filters.
	 *
	 * Uses LIKE queries to pre-select candidates as per PRD Appendix B.
	 *
	 * @since 1.0.0
	 * @param int $offset Offset for pagination.
	 * @param int $limit  Limit for pagination.
	 * @return array Array of post objects.
	 */
	private static function get_posts_for_scanning( $offset = 0, $limit = 100 ) {
		global $wpdb;
		
		// Candidate selection query from PRD Appendix B.
		// Pre-filter using LIKE to reduce the dataset before PHP regex scanning.
		// Note: post_type and post_status values are hardcoded (not user input) but wrapped in prepare for consistency.
		$query = $wpdb->prepare(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT ID, post_type, post_status, post_content
			FROM `{$wpdb->posts}`
			WHERE post_type IN (%s, %s)
			  AND post_status IN (%s, %s, %s, %s, %s)
			  AND (
			    post_content LIKE %s OR
			    post_content LIKE %s OR
			    post_content LIKE %s OR
			    post_content LIKE %s OR
			    post_content LIKE %s
			  )
			LIMIT %d OFFSET %d",
			'post',
			'page',
			'publish',
			'future',
			'draft',
			'pending',
			'private',
			'%http%',
			'%<script%',
			'%<iframe%',
			'%display:%',
			'%visibility:%',
			$limit,
            $offset
        );
        
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is prepared with $wpdb->prepare() above
        $results = $wpdb->get_results( $query );
        
        if ( $wpdb->last_error ) {
            cgp_log( 'Content Guard Pro: Database error in get_posts_for_scanning - ' . $wpdb->last_error );
            return array();
            }
		
		return $results ? $results : array();
	}

	/**
	 * Get postmeta entries for scanning with pre-filters.
	 *
	 * Includes Elementor page builder data (_elementor_data).
	 *
	 * @since 1.0.0
	 * @param int $offset Offset for pagination.
	 * @param int $limit  Limit for pagination.
	 * @return array Array of meta objects.
	 */
	private static function get_postmeta_for_scanning( $offset = 0, $limit = 100 ) {
		global $wpdb;
		
		// Candidate selection query from PRD Appendix B.
		// Pre-filter using LIKE to reduce the dataset before PHP regex scanning.
		// Also includes Elementor data (_elementor_data) for page builder support.
		$query = $wpdb->prepare(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT meta_id, post_id, meta_key, meta_value
			FROM `{$wpdb->postmeta}`
			WHERE LENGTH(meta_value) > 0
			  AND (
			    meta_value LIKE %s OR
			    meta_value LIKE %s OR
			    meta_value LIKE %s OR
			    meta_key = %s
			  )
			LIMIT %d OFFSET %d",
			'%http%',
			'%<script%',
			'%<iframe%',
			'_elementor_data',
			$limit,
		    $offset
        );
        
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is prepared with $wpdb->prepare() above
        $results = $wpdb->get_results( $query );
        
        if ( $wpdb->last_error ) {
            cgp_log( 'Content Guard Pro: Database error in get_postmeta_for_scanning - ' . $wpdb->last_error );
            return array();
		}
		
		return $results ? $results : array();
	}

	/**
	 * Get allowlisted options for scanning.
	 *
	 * @since 1.0.0
	 * @return array Array of option objects.
	 */
	private static function get_options_for_scanning() {
		global $wpdb;
		
		// Allowlisted options query from PRD Appendix B.
		// Only scan specific widget option keys per PRD Section 3.1.
		$query = $wpdb->prepare(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT option_name, option_value
			FROM `{$wpdb->options}`
			WHERE option_name IN (%s, %s, %s)",
			'widget_text',
			'widget_custom_html',
		    'widget_block'
        );
	
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is prepared with $wpdb->prepare() above
        $results = $wpdb->get_results( $query );
        
        if ( $wpdb->last_error ) {
            cgp_log( 'Content Guard Pro: Database error in get_options_for_scanning - ' . $wpdb->last_error );
            return array();
            }
		
		return $results ? $results : array();
	}

	/**
	 * Find security issues in content.
	 *
	 * Delegates to CGP_Detector for pattern matching and threat detection.
	 *
	 * @since 1.0.0
	 * @param string $content     Content to scan (HTML, blocks, etc.).
	 * @param int    $object_id   Object ID (post ID, meta ID, etc.).
	 * @param string $object_type Object type (post, postmeta, option).
	 * @param string $field       Field name (default: post_content).
	 * @return array Array of findings.
	 */
	private static function find_issues_in_content( $content, $object_id, $object_type, $field = 'post_content' ) {
		// Use the dedicated detector class for content analysis.
		$detector = new CGP_Detector();
		return $detector->scan_content( $content, $object_id, $object_type, $field );
	}

	/**
	 * Save or update a finding in the database.
	 *
	 * Implements fingerprint-based deduplication per PRD Section 3.1 and Appendix E.
	 * If the finding fingerprint already exists, updates last_seen timestamp.
	 * Otherwise, inserts a new finding record.
	 *
	 * @since 1.0.0
	 * @param array $finding_data Finding data including:
	 *                            - object_type (post, postmeta, option)
	 *                            - object_id (post ID or option surrogate)
	 *                            - field (e.g., post_content, meta_value)
	 *                            - rule_id
	 *                            - severity (critical, suspicious, review)
	 *                            - confidence (0-100)
	 *                            - matched_text (the actual matched content)
	 *                            - extra (optional JSON/array metadata)
	 * @return int|false Finding ID on success, false on failure.
	 */
	/**
	 * Save a finding to the database.
	 *
	 * Made public to allow synchronous scans to save findings immediately.
	 *
	 * @since 1.0.0
	 * @param array $finding_data Finding data from detector.
	 * @return int|false Finding ID on success, false on failure.
	 */
	public static function save_finding( $finding_data ) {
		global $wpdb;

		// Validate required fields.
		$required = array( 'object_type', 'object_id', 'field', 'rule_id', 'severity', 'confidence', 'matched_text' );
		foreach ( $required as $field ) {
			if ( ! isset( $finding_data[ $field ] ) ) {
				cgp_log( 'Content Guard Pro: Missing required field "' . $field . '" in save_finding()' );
				return false;
			}
		}

		// Get blog_id for multisite support.
		$blog_id = get_current_blog_id();

		// Generate fingerprint per PRD Appendix E.
		$fingerprint = self::generate_fingerprint(
			$blog_id,
			$finding_data['object_type'],
			$finding_data['object_id'],
			$finding_data['field'],
			$finding_data['rule_id'],
			$finding_data['matched_text']
		);

		// Prepare matched excerpt (limit to 500 chars).
		$matched_excerpt = wp_strip_all_tags( $finding_data['matched_text'] );
		if ( strlen( $matched_excerpt ) > 500 ) {
			$matched_excerpt = substr( $matched_excerpt, 0, 500 ) . '...';
		}

		// Prepare extra metadata.
		$extra = isset( $finding_data['extra'] ) ? $finding_data['extra'] : array();
		if ( is_array( $extra ) ) {
			$extra = wp_json_encode( $extra );
		}

		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		// Check if finding already exists by fingerprint.
		$existing = $wpdb->get_row(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				"SELECT id, first_seen, status FROM `{$table_name}` WHERE fingerprint = %s",
				$fingerprint
			)
		);

		$current_time = current_time( 'mysql' );

		if ( $existing ) {
			// Update existing finding: update last_seen timestamp.
			// IMPORTANT: Preserve user-set statuses (ignored, quarantined) when detected again.
			// Re-open findings that were previously 'resolved' or 'deleted' (vulnerability reappeared).
			// Findings that were 'open' will remain 'open' (just update timestamp).
			$preserved_statuses = array( 'ignored', 'quarantined' );
			$new_status = 'open'; // Default to open for new detections.
			
			// If finding was previously ignored or quarantined, preserve that status.
			// If it was resolved or deleted, it will be reopened (set to 'open').
			if ( isset( $existing->status ) && in_array( $existing->status, $preserved_statuses, true ) ) {
				$new_status = $existing->status;
			}
			
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$updated = $wpdb->update(
				$table_name,
				array(
					'last_seen'       => $current_time,
					'confidence'      => absint( $finding_data['confidence'] ),
					'matched_excerpt' => $matched_excerpt,
					'extra'           => $extra,
					'status'          => $new_status,
				),
				array( 'id' => $existing->id ),
				array( '%s', '%d', '%s', '%s', '%s' ),
				array( '%d' )
			);

			if ( false === $updated ) {
				cgp_log( 'Content Guard Pro: Failed to update finding ID ' . $existing->id );
				return false;
			}

			// Log if we're re-opening a previously resolved or deleted finding.
			// This indicates the vulnerability has reappeared after being fixed/removed.
			if ( isset( $existing->status ) && in_array( $existing->status, array( 'resolved', 'deleted' ), true ) ) {
				cgp_log( sprintf(
					'Content Guard Pro: Re-opened finding ID %d (was: %s, rule: %s, object: %s #%d) - vulnerability reappeared',
					$existing->id,
					$existing->status,
					$finding_data['rule_id'],
					$finding_data['object_type'],
					$finding_data['object_id']
				) );
			}

			// Trigger finding saved action (for updates).
			do_action( 'content_guard_pro_finding_saved', $existing->id, $finding_data );

			return $existing->id;
		} else {
			// Insert new finding.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$inserted = $wpdb->insert(
				$table_name,
				array(
					'blog_id'         => $blog_id,
					'object_type'     => $finding_data['object_type'],
					'object_id'       => absint( $finding_data['object_id'] ),
					'field'           => $finding_data['field'],
					'fingerprint'     => $fingerprint,
					'rule_id'         => $finding_data['rule_id'],
					'severity'        => $finding_data['severity'],
					'confidence'      => absint( $finding_data['confidence'] ),
					'matched_excerpt' => $matched_excerpt,
					'first_seen'      => $current_time,
					'last_seen'       => $current_time,
					'status'          => 'open',
					'extra'           => $extra,
				),
				array(
					'%d', // blog_id
					'%s', // object_type
					'%d', // object_id
					'%s', // field
					'%s', // fingerprint
					'%s', // rule_id
					'%s', // severity
					'%d', // confidence
					'%s', // matched_excerpt
					'%s', // first_seen
					'%s', // last_seen
					'%s', // status
					'%s', // extra
				)
			);

			if ( false === $inserted ) {
				cgp_log( 'Content Guard Pro: Failed to insert finding - ' . $wpdb->last_error );
				return false;
			}

			$finding_id = $wpdb->insert_id;

			// Trigger finding saved action (for new findings).
			do_action( 'content_guard_pro_finding_saved', $finding_id, $finding_data );

			return $finding_id;
		}
	}

	/**
	 * Generate finding fingerprint per PRD Appendix E.
	 *
	 * Canonical form: sha256( blog_id + object_type + object_id + field + rule_id + normalized_match )
	 *
	 * @since 1.0.0
	 * @param int    $blog_id     Blog ID (multisite).
	 * @param string $object_type Object type (post, postmeta, option).
	 * @param int    $object_id   Object ID.
	 * @param string $field       Field name.
	 * @param string $rule_id     Rule ID.
	 * @param string $matched_text Matched text to normalize and hash.
	 * @return string SHA-256 fingerprint (64 chars hex).
	 */
	private static function generate_fingerprint( $blog_id, $object_type, $object_id, $field, $rule_id, $matched_text ) {
		// Normalize the matched text per Appendix E.
		$normalized = self::normalize_match( $matched_text );

		// Build canonical string.
		$canonical = $blog_id . '|' . $object_type . '|' . $object_id . '|' . $field . '|' . $rule_id . '|' . $normalized;

		// Generate SHA-256 hash.
		return hash( 'sha256', $canonical );
	}

	/**
	 * Normalize matched text for fingerprint generation.
	 *
	 * Per PRD Appendix E:
	 * - Lowercased
	 * - Stripped HTML
	 * - Domain only for external URLs (ignore querystrings)
	 * - Simplified CSS properties
	 * - Limited to first 200 chars
	 *
	 * @since 1.0.0
	 * @param string $text Matched text.
	 * @return string Normalized text.
	 */
	private static function normalize_match( $text ) {
		// Strip HTML tags.
		$text = wp_strip_all_tags( $text );

		// Lowercase.
		$text = strtolower( $text );

		// Extract domain and path from URLs (ignore query strings only, per PRD Appendix E).
		// Match http(s):// URLs and extract domain + path (without query string).
		// This ensures different URLs from the same domain have different fingerprints.
		$replaced = preg_replace_callback(
			'/https?:\/\/([^\s?#]+)/i',
			function( $matches ) {
				// Return domain + path (query string already excluded by regex).
				// This allows different URLs from same domain to have unique fingerprints.
				return $matches[1];
			},
			$text
		);
		
		// Check for preg_replace_callback errors.
		if ( null !== $replaced ) {
			$text = $replaced;
		} else {
			cgp_log( 'Content Guard Pro: preg_replace_callback failed in normalize_match' );
		}

		// Simplify CSS properties: remove extra spaces, normalize common patterns.
		$text = preg_replace( '/\s+/', ' ', $text );
		$text = preg_replace( '/:\s+/', ':', $text );
		$text = preg_replace( '/;\s+/', ';', $text );

		// Trim and limit to 200 chars for fingerprint consistency.
		$text = trim( $text );
		if ( strlen( $text ) > 200 ) {
			$text = substr( $text, 0, 200 );
		}

		return $text;
	}

	/**
	 * Create or update scan record.
	 *
	 * @since 1.0.0
	 * @param array $data Scan data including mode, started_at, etc.
	 * @return int|false Scan ID on success, false on failure.
	 */
	private static function create_scan_record( $data ) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		$blog_id = get_current_blog_id();
		
		// Determine targets based on mode if not provided.
		$mode = isset( $data['mode'] ) ? $data['mode'] : 'standard';
		if ( ! isset( $data['targets'] ) ) {
			if ( 'quick' === $mode ) {
				$targets = 'posts';
			} else {
				// Standard mode: posts, postmeta, options.
				$targets = 'posts,postmeta,options';
			}
		} else {
			$targets = $data['targets'];
		}
		
		// Prepare data for insertion per PRD Appendix A schema.
		$insert_data = array(
			'blog_id'         => $blog_id,
			'started_at'      => isset( $data['started_at'] ) ? $data['started_at'] : current_time( 'mysql' ),
			'mode'            => $mode,
			'targets'         => $targets,
			'status'          => isset( $data['status'] ) ? $data['status'] : 'running',
			'totals_checked'  => isset( $data['totals_checked'] ) ? absint( $data['totals_checked'] ) : 0,
			'totals_flagged'  => isset( $data['totals_flagged'] ) ? absint( $data['totals_flagged'] ) : 0,
			'avg_query_ms'    => isset( $data['avg_query_ms'] ) ? absint( $data['avg_query_ms'] ) : 0,
			'peak_mem_mb'     => isset( $data['peak_mem_mb'] ) ? absint( $data['peak_mem_mb'] ) : 0,
			'throttle_state'  => isset( $data['throttle_state'] ) ? $data['throttle_state'] : 'normal',
			'errors'          => isset( $data['errors'] ) ? absint( $data['errors'] ) : 0,
			'notes'           => isset( $data['notes'] ) ? $data['notes'] : '',
		);
		
		$format = array(
			'%d', // blog_id
			'%s', // started_at
			'%s', // mode
			'%s', // targets
			'%s', // status
			'%d', // totals_checked
			'%d', // totals_flagged
			'%d', // avg_query_ms
			'%d', // peak_mem_mb
			'%s', // throttle_state
			'%d', // errors
			'%s', // notes
		);
		
		$inserted = $wpdb->insert( $table_name, $insert_data, $format );
		
		if ( false === $inserted ) {
			cgp_log( 'Content Guard Pro: Failed to create scan record - ' . $wpdb->last_error );
			return false;
		}
		
		$scan_id = $wpdb->insert_id;
		
		cgp_log( sprintf(
			'Content Guard Pro: Created scan record (scan_id: %d, mode: %s)',
			$scan_id,
			$insert_data['mode']
		) );
		
		return $scan_id;
	}

	/**
	 * Update scan record with progress.
	 *
	 * @since 1.0.0
	 * @param int   $scan_id Scan ID.
	 * @param array $data    Data to update (finished_at, totals_checked, etc.).
	 * @return bool True on success, false on failure.
	 */
	private static function update_scan_record( $scan_id, $data ) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		
		// Build update data array with only provided fields.
		$update_data = array();
		$format = array();
		
		if ( isset( $data['finished_at'] ) ) {
			$update_data['finished_at'] = $data['finished_at'];
			$format[] = '%s';
		}
		
		if ( isset( $data['totals_checked'] ) ) {
			$update_data['totals_checked'] = absint( $data['totals_checked'] );
			$format[] = '%d';
		}
		
		if ( isset( $data['totals_flagged'] ) ) {
			$update_data['totals_flagged'] = absint( $data['totals_flagged'] );
			$format[] = '%d';
		}
		
		if ( isset( $data['avg_query_ms'] ) ) {
			$update_data['avg_query_ms'] = absint( $data['avg_query_ms'] );
			$format[] = '%d';
		}
		
		if ( isset( $data['peak_mem_mb'] ) ) {
			$update_data['peak_mem_mb'] = absint( $data['peak_mem_mb'] );
			$format[] = '%d';
		}
		
		if ( isset( $data['throttle_state'] ) ) {
			$update_data['throttle_state'] = $data['throttle_state'];
			$format[] = '%s';
		}
		
		if ( isset( $data['errors'] ) ) {
			$update_data['errors'] = absint( $data['errors'] );
			$format[] = '%d';
		}
		
		if ( isset( $data['notes'] ) ) {
			$update_data['notes'] = $data['notes'];
			$format[] = '%s';
		}
		
		if ( isset( $data['status'] ) ) {
			$update_data['status'] = $data['status'];
			$format[] = '%s';
		}
		
		if ( empty( $update_data ) ) {
			cgp_log( 'Content Guard Pro: No data provided for scan record update' );
			return false;
		}
		
		$updated = $wpdb->update(
			$table_name,
			$update_data,
			array( 'scan_id' => $scan_id ),
			$format,
			array( '%d' )
		);
		
		if ( false === $updated ) {
			cgp_log( 'Content Guard Pro: Failed to update scan record ' . $scan_id . ' - ' . $wpdb->last_error );
			return false;
		}
		
		return true;
	}

	/**
	 * Get scan progress.
	 *
	 * @since 1.0.0
	 * @param int $scan_id Scan ID.
	 * @return array Progress data (percentage, items scanned, ETA, etc.).
	 */
	public static function get_scan_progress( $scan_id ) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		
		// Query the scan record from content_guard_pro_scans table.
		$scan = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$table_name}` WHERE scan_id = %d",
				$scan_id
			)
		);
		
		if ( ! $scan ) {
			return array(
				'scan_id'        => $scan_id,
				'percentage'     => 0,
				'items_scanned'  => 0,
				'items_total'    => 0,
				'current_target' => 'unknown',
				'eta_seconds'    => 0,
				'memory_usage'   => 0,
				'status'         => 'not_found',
			);
		}
		
		// Determine status based on timestamps and state.
		$status = 'running';
		if ( ! empty( $scan->finished_at ) ) {
			$status = 'completed';
		} elseif ( isset( $scan->status ) && 'paused' === $scan->status ) {
			$status = 'paused';
		} elseif ( isset( $scan->status ) && 'cancelled' === $scan->status ) {
			$status = 'cancelled';
		} elseif ( isset( $scan->status ) && 'failed' === $scan->status ) {
			$status = 'failed';
		}
		
		// Calculate items scanned and total (estimated).
		// For now, use totals_checked from the scan record.
		// In a full implementation, you'd track batch progress in scan metadata.
		$items_scanned = isset( $scan->totals_checked ) ? absint( $scan->totals_checked ) : 0;
		
		// Estimate total items based on mode.
		// This is a rough estimate; real implementation would calculate from DB.
		$items_total = self::estimate_total_items( $scan->mode );
		
		// Calculate percentage.
		$percentage = $items_total > 0 ? min( 100, floor( ( $items_scanned / $items_total ) * 100 ) ) : 0;
		
		// Calculate ETA based on elapsed time and progress.
		$eta_seconds = 0;
		if ( $percentage > 0 && $percentage < 100 && 'running' === $status ) {
			$started = strtotime( $scan->started_at );
			if ( false !== $started ) {
				$elapsed = time() - $started;
				$eta_seconds = floor( ( $elapsed / $percentage ) * ( 100 - $percentage ) );
			}
		}
		
		// Get current memory usage.
		$memory_usage = memory_get_usage( true ) / 1024 / 1024;
		
		// Determine current target (would come from scan metadata in real implementation).
		$current_target = 'posts'; // Default, would be tracked in notes/metadata.
		if ( isset( $scan->notes ) && ! empty( $scan->notes ) ) {
			$decoded_notes = json_decode( $scan->notes, true );
			if ( null !== $decoded_notes && JSON_ERROR_NONE === json_last_error() && is_array( $decoded_notes ) ) {
				$notes = $decoded_notes;
				if ( isset( $notes['current_target'] ) ) {
					$current_target = $notes['current_target'];
				}
			}
		}
		
		return array(
			'scan_id'        => $scan_id,
			'percentage'     => $percentage,
			'items_scanned'  => $items_scanned,
			'items_total'    => $items_total,
			'current_target' => $current_target,
			'eta_seconds'    => $eta_seconds,
			'memory_usage'   => round( $memory_usage, 2 ),
			'peak_memory_mb' => isset( $scan->peak_mem_mb ) ? absint( $scan->peak_mem_mb ) : 0,
			'avg_query_ms'   => isset( $scan->avg_query_ms ) ? absint( $scan->avg_query_ms ) : 0,
			'throttle_state' => isset( $scan->throttle_state ) ? $scan->throttle_state : 'normal',
			'status'         => $status,
			'started_at'     => $scan->started_at,
			'finished_at'    => isset( $scan->finished_at ) ? $scan->finished_at : null,
		);
	}
	
	/**
	 * Estimate total items to scan based on mode.
	 *
	 * @since 1.0.0
	 * @param string $mode Scan mode (quick, standard).
	 * @return int Estimated total items.
	 */
	private static function estimate_total_items( $mode ) {
		global $wpdb;
		
		$total = 0;
		
		// Quick mode: posts only.
		// Standard mode: posts + postmeta + options.
		if ( 'quick' === $mode || 'standard' === $mode ) {
			// Count posts.
			// Note: post_type and post_status values are hardcoded (not user input) but wrapped in prepare for consistency.
			$posts_count = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM `{$wpdb->posts}`
					WHERE post_type IN (%s, %s)
					  AND post_status IN (%s, %s, %s, %s, %s)",
					'post',
					'page',
					'publish',
					'future',
					'draft',
					'pending',
					'private'
				)
			);
			$total += absint( $posts_count );
		}
		
        if ( 'standard' === $mode ) {
            // Count postmeta entries.
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $meta_count = $wpdb->get_var(
                "SELECT COUNT(*) FROM `{$wpdb->postmeta}`
                WHERE LENGTH(meta_value) > 0"
            );
            $total += absint( $meta_count );
			
			// Options are a fixed small set, add nominal count.
			$total += 10; // widget_text, widget_custom_html, widget_block instances.
		}
		
		return $total;
	}

	/**
	 * Check if throttling is needed.
	 *
	 * Monitors performance metrics and returns throttle state.
	 * PRD Section 3.6 and Appendix H - Auto-throttling.
	 *
	 * @since 1.0.0
	 * @param array $metrics Performance metrics (query_time, memory, batch_runtime).
	 * @return string Throttle state: 'normal', 'throttled', 'safe_mode'.
	 */
	private static function check_throttling( $metrics ) {
		$query_time = isset( $metrics['query_time'] ) ? floatval( $metrics['query_time'] ) : 0;
		$memory = isset( $metrics['memory'] ) ? floatval( $metrics['memory'] ) : 0;
		$batch_runtime = isset( $metrics['batch_runtime'] ) ? floatval( $metrics['batch_runtime'] ) : 0;
		
		// Get WP_MEMORY_LIMIT for threshold calculations.
		// Default is 40M, but can be overridden in wp-config.php.
		$memory_limit = WP_MEMORY_LIMIT;
		if ( is_string( $memory_limit ) ) {
			// Convert string like "128M" to numeric MB.
			$memory_limit = intval( $memory_limit );
		}
		
		// Calculate 50% threshold per PRD Appendix H.
		$memory_threshold = $memory_limit * 0.5;
		
		// Count threshold violations.
		$violations = 0;
		$reasons = array();
		
		// Check avg query time > 300ms per PRD Appendix H.
		if ( $query_time > 300 ) {
			$violations++;
			$reasons[] = sprintf( 'Query time %.2fms exceeds 300ms threshold', $query_time );
		}
		
		// Check peak memory > 50% of WP_MEMORY_LIMIT per PRD Appendix H.
		if ( $memory > $memory_threshold ) {
			$violations++;
			$reasons[] = sprintf( 'Memory %.2fMB exceeds %.2fMB threshold (50%% of limit)', $memory, $memory_threshold );
		}
		
		// Check batch runtime exceeds 5s per PRD Appendix H.
		if ( $batch_runtime > 5 ) {
			$violations++;
			$reasons[] = sprintf( 'Batch runtime %.2fs exceeds 5s threshold', $batch_runtime );
		}
		
		// Determine throttle state based on violations.
		$throttle_state = 'normal';
		
		if ( $violations >= 2 ) {
			// Multiple violations: enable Safe Mode.
			// Per PRD Appendix H: batch=25, delay=5-10s.
			$throttle_state = 'safe_mode';
			cgp_log( 'Content Guard Pro: Safe Mode activated - ' . implode( ', ', $reasons ) );
		} elseif ( $violations === 1 ) {
			// Single violation: throttle.
			// Per PRD Appendix H: reduce batch by 50%, increase delay to 5s.
			$throttle_state = 'throttled';
			cgp_log( 'Content Guard Pro: Throttling activated - ' . implode( ', ', $reasons ) );
		}
		
		// Log throttle state changes.
		if ( 'normal' !== $throttle_state ) {
			do_action( 'content_guard_pro_throttle_state_changed', $throttle_state, $metrics, $reasons );
		}
		
		return $throttle_state;
	}

	/**
	 * Calculate batch size based on throttle state.
	 *
	 * Dynamic batch sizing per PRD Appendix H:
	 * - normal: use configured batch size (100)
	 * - throttled: reduce by 50%
	 * - safe_mode: use minimum (25)
	 *
	 * @since 1.0.0
	 * @param string $throttle_state Current throttle state.
	 * @return int Batch size.
	 */
	private static function get_batch_size( $throttle_state = 'normal' ) {
		$settings   = get_option( 'content_guard_pro_settings', array() );
		$base_size  = isset( $settings['batch_size'] ) ? absint( $settings['batch_size'] ) : self::DEFAULT_BATCH_SIZE;
		
		switch ( $throttle_state ) {
			case 'safe_mode':
				return 25;
			case 'throttled':
				return max( 25, floor( $base_size / 2 ) );
			default:
				return $base_size;
		}
	}

	/**
	 * Calculate delay between batches.
	 *
	 * Dynamic delay calculation per PRD Appendix H:
	 * - normal: use configured delay (2s)
	 * - throttled: increase to 5s
	 * - safe_mode: increase to 10s
	 *
	 * @since 1.0.0
	 * @param string $throttle_state Current throttle state.
	 * @return int Delay in seconds.
	 */
	private static function get_batch_delay( $throttle_state = 'normal' ) {
		$settings   = get_option( 'content_guard_pro_settings', array() );
		$base_delay = isset( $settings['batch_delay'] ) ? absint( $settings['batch_delay'] ) : self::DEFAULT_BATCH_DELAY;
		
		switch ( $throttle_state ) {
			case 'safe_mode':
				return 10;
			case 'throttled':
				return 5;
			default:
				return $base_delay;
		}
	}

	/**
	 * Schedule next batch.
	 *
	 * Enqueues the next batch via Action Scheduler with appropriate delay.
	 * Prevents duplicate scheduling by checking for existing pending actions.
	 *
	 * @since 1.0.0
	 * @param int    $scan_id    Scan ID.
	 * @param string $target     Target to scan (posts, postmeta, options).
	 * @param int    $offset     Next offset.
	 * @param string $throttle_state Current throttle state.
	 * @return int|false Action ID on success, false on failure.
	 */
	private static function schedule_next_batch( $scan_id, $target, $offset, $throttle_state = 'normal' ) {
		$delay = self::get_batch_delay( $throttle_state );
		$limit = self::get_batch_size( $throttle_state );
		
		if ( ! function_exists( 'as_schedule_single_action' ) || ! function_exists( 'as_next_scheduled_action' ) ) {
			cgp_log( 'Content Guard Pro: Action Scheduler not available' );
			return false;
		}
		
		// CRITICAL: Check if this exact batch is already scheduled to prevent duplicates.
		$batch_args = array(
			$scan_id,
			array(
				'target' => $target,
				'offset' => $offset,
				'limit'  => $limit,
			),
		);
		
		$existing_action = as_next_scheduled_action( 'content_guard_pro_process_batch', $batch_args, 'content-guard-pro' );
		
		if ( false !== $existing_action ) {
			cgp_log( sprintf(
				'Content Guard Pro: Batch already scheduled (scan_id: %d, target: %s, offset: %d) - skipping duplicate',
				$scan_id,
				$target,
				$offset
			) );
			return $existing_action;
		}
		
		$timestamp = time() + $delay;
		
		cgp_log( sprintf(
			'Content Guard Pro: Scheduling next batch (scan_id: %d, target: %s, offset: %d, delay: %ds)',
			$scan_id,
			$target,
			$offset,
			$delay
		) );
		
		return as_schedule_single_action(
			$timestamp,
			'content_guard_pro_process_batch',
			$batch_args,
			'content-guard-pro'
		);
	}

	/**
	 * Get next scan target based on current target and mode.
	 *
	 * @since 1.0.0
	 * @param string $current_target Current target (posts, postmeta, options).
	 * @param string $mode           Scan mode (quick, standard).
	 * @return string|false Next target, or false if all targets complete.
	 */
	private static function get_next_target( $current_target, $mode ) {
		// Quick mode: posts only.
		// Standard mode: posts -> postmeta -> options.
		
		if ( 'quick' === $mode ) {
			// Quick mode only scans posts.
			return false;
		}
		
		// Standard mode: progress through all targets.
		switch ( $current_target ) {
			case 'posts':
				return 'postmeta';
			case 'postmeta':
				return 'options';
			case 'options':
				return false; // All targets complete.
			default:
				return false;
		}
	}

	/**
	 * Finalize a completed scan.
	 *
	 * Updates scan record with completion time and final statistics.
	 *
	 * @since 1.0.0
	 * @param int $scan_id Scan ID to finalize.
	 * @return bool True on success, false on failure.
	 */
	private static function finalize_scan( $scan_id ) {
		global $wpdb;
		
		$findings_table = $wpdb->prefix . 'content_guard_pro_findings';
		$scans_table = $wpdb->prefix . 'content_guard_pro_scans';
		
		// Get scan record.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$scan = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$scans_table}` WHERE scan_id = %d",
				$scan_id
			)
		);
		
		if ( ! $scan ) {
			cgp_log( 'Content Guard Pro: Cannot finalize scan - scan not found' );
			return false;
		}
		
		// Use the totals that were tracked incrementally during batch processing.
		// No need to recount findings - they were already tracked.
		$current_totals_flagged = isset( $scan->totals_flagged ) ? absint( $scan->totals_flagged ) : 0;
		
		// Clean up findings for deleted/trashed content.
		// This marks findings as 'deleted' when the associated content no longer exists or is in trash.
		$deleted_count = self::cleanup_deleted_content_findings();
		if ( $deleted_count > 0 ) {
			cgp_log( sprintf(
				'Content Guard Pro: Marked %d findings as deleted (content no longer exists or in trash)',
				$deleted_count
			) );
		}
		
		// Calculate actual actionable findings for notification (exclude ignored/resolved/deleted).
		// PRD Scenario 3: "Issues Found" should not count ignored items.
		$notification_count = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$findings_table}` 
				WHERE last_seen >= %s 
				AND status = %s",
				$scan->started_at,
				'open'
			)
		);
		
		// Get critical count for notification.
		$notification_critical = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$findings_table}` 
				WHERE last_seen >= %s 
				AND status = %s 
				AND severity = %s",
				$scan->started_at,
				'open',
				'critical'
			)
		);

		// Update scan record with completion data.
		$updated = self::update_scan_record( $scan_id, array(
			'finished_at' => current_time( 'mysql' ),
			'status'      => 'completed',
		) );
		
		if ( ! $updated ) {
			cgp_log( 'Content Guard Pro: Failed to finalize scan record' );
			return false;
		}
		
		// IMPORTANT: Clear the API cache so JavaScript gets the updated scan immediately
		delete_transient( 'content_guard_pro_active_scan_progress' );
		delete_transient( 'content_guard_pro_scan_progress_' . $scan_id );
		
		cgp_log( sprintf(
			'Content Guard Pro: Scan %d completed - %d findings (Notification: %d open)',
			$scan_id,
			$current_totals_flagged,
			$notification_count
		) );
		
		// Trigger completion action for integrations and notifications.
		do_action( 'content_guard_pro_scan_completed', $scan_id, array(
			'findings_count' => $notification_count, // Use open count for notifications
			'started_at'     => $scan->started_at,
			'finished_at'    => current_time( 'mysql' ),
		) );
		
		// Set transient for UI notification.
		// We use the open/actionable count here so the user isn't alerted about ignored items.
		set_transient( 'content_guard_pro_scan_completed', array( // Fixed: Removed ID suffix to match Admin consumer
			'scan_id'        => $scan_id,
			'findings'       => $notification_count,
			'critical'       => $notification_critical,
			'completed_at'   => current_time( 'mysql' ),
		), DAY_IN_SECONDS );
		
		return true;
	}

	/**
	 * Pause a running scan.
	 *
	 * @since 1.0.0
	 * @param int $scan_id Scan ID to pause.
	 * @return bool True on success, false on failure.
	 */
	public static function pause_scan( $scan_id ) {
		global $wpdb;
		
		cgp_log( sprintf( 'Content Guard Pro: Pausing scan %d', $scan_id ) );
		
		// Verify scan exists and is running.
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$scan = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$table_name}` WHERE scan_id = %d",
				$scan_id
			)
		);
		
		if ( ! $scan ) {
			cgp_log( 'Content Guard Pro: Cannot pause scan - scan not found' );
			return false;
		}
		
		if ( ! empty( $scan->finished_at ) ) {
			cgp_log( 'Content Guard Pro: Cannot pause scan - already completed' );
			return false;
		}
		
		// Update scan status to 'paused'.
		$updated = self::update_scan_record( $scan_id, array(
			'status' => 'paused',
			'notes'  => wp_json_encode( array(
				'paused_at' => current_time( 'mysql' ),
				'reason'    => 'User requested pause',
			) ),
		) );
		
		if ( ! $updated ) {
			return false;
		}
		
		// Cancel all pending batch actions using Action Scheduler.
		if ( function_exists( 'as_unschedule_all_actions' ) ) {
			// Unschedule all content_guard_pro_process_batch actions for this scan.
			as_unschedule_all_actions( 'content_guard_pro_process_batch', array( $scan_id ), 'content-guard-pro' );
			
			cgp_log( sprintf( 'Content Guard Pro: Cancelled pending batch actions for scan %d', $scan_id ) );
		} else {
			cgp_log( 'Content Guard Pro: Action Scheduler not available for unscheduling' );
		}
		
		// Trigger action for integrations.
		do_action( 'content_guard_pro_scan_paused', $scan_id );
		
		cgp_log( sprintf( 'Content Guard Pro: Successfully paused scan %d', $scan_id ) );
		
		return true;
	}

	/**
	 * Resume a paused scan.
	 *
	 * @since 1.0.0
	 * @param int $scan_id Scan ID to resume.
	 * @return bool True on success, false on failure.
	 */
	public static function resume_scan( $scan_id ) {
		global $wpdb;
		
		cgp_log( sprintf( 'Content Guard Pro: Resuming scan %d', $scan_id ) );
		
		// Verify scan exists and is paused.
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$scan = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$table_name}` WHERE scan_id = %d",
				$scan_id
			)
		);
		
		if ( ! $scan ) {
			cgp_log( 'Content Guard Pro: Cannot resume scan - scan not found' );
			return false;
		}
		
		// Check if scan is paused.
		$status = isset( $scan->status ) ? $scan->status : 'unknown';
		if ( 'paused' !== $status ) {
			cgp_log( sprintf( 'Content Guard Pro: Cannot resume scan - status is %s, not paused', $status ) );
			return false;
		}
		
		// Update scan status to 'running'.
		$updated = self::update_scan_record( $scan_id, array(
			'status' => 'running',
			'notes'  => wp_json_encode( array(
				'resumed_at' => current_time( 'mysql' ),
				'reason'     => 'User requested resume',
			) ),
		) );
		
		if ( ! $updated ) {
			return false;
		}
		
		// Get scan state from notes to determine where to resume.
		$notes = array();
		if ( isset( $scan->notes ) && ! empty( $scan->notes ) ) {
			$decoded_notes = json_decode( $scan->notes, true );
			if ( null !== $decoded_notes && JSON_ERROR_NONE === json_last_error() && is_array( $decoded_notes ) ) {
				$notes = $decoded_notes;
			}
		}
		$current_target = isset( $notes['current_target'] ) ? $notes['current_target'] : 'posts';
		$current_offset = isset( $notes['current_offset'] ) ? absint( $notes['current_offset'] ) : 0;
		
		// Determine throttle state for batch sizing.
		$throttle_state = isset( $scan->throttle_state ) ? $scan->throttle_state : 'normal';
		
		// Schedule next batch immediately to resume scanning.
		$scheduled = self::schedule_next_batch( $scan_id, $current_target, $current_offset, $throttle_state );
		
		if ( ! $scheduled ) {
			cgp_log( 'Content Guard Pro: Failed to schedule next batch for resumed scan' );
			return false;
		}
		
		// Trigger action for integrations.
		do_action( 'content_guard_pro_scan_resumed', $scan_id );
		
		cgp_log( sprintf(
			'Content Guard Pro: Successfully resumed scan %d (target: %s, offset: %d)',
			$scan_id,
			$current_target,
			$current_offset
		) );
		
		return true;
	}

	/**
	 * Cancel a running or paused scan.
	 *
	 * @since 1.0.0
	 * @param int $scan_id Scan ID to cancel.
	 * @return bool True on success, false on failure.
	 */
	public static function cancel_scan( $scan_id ) {
		global $wpdb;
		
		cgp_log( sprintf( 'Content Guard Pro: Cancelling scan %d', $scan_id ) );
		
		// Verify scan exists.
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$scan = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$table_name}` WHERE scan_id = %d",
				$scan_id
			)
		);
		
		if ( ! $scan ) {
			cgp_log( 'Content Guard Pro: Cannot cancel scan - scan not found' );
			return false;
		}
		
		// Check if scan is already completed.
		if ( ! empty( $scan->finished_at ) ) {
			cgp_log( 'Content Guard Pro: Cannot cancel scan - already completed' );
			return false;
		}
		
		// Cancel all pending batch actions using Action Scheduler.
		if ( function_exists( 'as_unschedule_all_actions' ) ) {
			// Unschedule all content_guard_pro_process_batch actions for this scan.
			as_unschedule_all_actions( 'content_guard_pro_process_batch', array( $scan_id ), 'content-guard-pro' );
			
			cgp_log( sprintf( 'Content Guard Pro: Cancelled all pending batch actions for scan %d', $scan_id ) );
		} else {
			cgp_log( 'Content Guard Pro: Action Scheduler not available for unscheduling' );
		}
		
		// Update scan status to 'cancelled' and set finished_at timestamp.
		$updated = self::update_scan_record( $scan_id, array(
			'status'      => 'cancelled',
			'finished_at' => current_time( 'mysql' ),
			'notes'       => wp_json_encode( array(
				'cancelled_at' => current_time( 'mysql' ),
				'reason'       => 'User requested cancellation',
			) ),
		) );
		
		if ( ! $updated ) {
			return false;
		}
		
		// Trigger action for integrations and cleanup.
		do_action( 'content_guard_pro_scan_cancelled', $scan_id );
		
		cgp_log( sprintf( 'Content Guard Pro: Successfully cancelled scan %d', $scan_id ) );
		
		return true;
	}

	/**
	 * Get active scan.
	 *
	 * Returns the currently running or paused scan, if any.
	 *
	 * @since 1.0.0
	 * @return int|false Active scan ID, or false if none.
	 */
	public static function get_active_scan() {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		
		// Query for scans that are pending, running, or paused (not finished).
		// Include 'pending' status so newly created scans appear immediately.
		// Note: status values are hardcoded (not user input) but wrapped in prepare for consistency.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$scan_id = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT scan_id FROM `{$table_name}`
				WHERE finished_at IS NULL
				  AND (status = %s OR status = %s OR status = %s OR status IS NULL)
				ORDER BY started_at DESC
				LIMIT 1",
				'pending',
				'running',
				'paused'
			)
		);
		
		if ( $scan_id ) {
			return absint( $scan_id );
		}
		
		return false;
	}

	/**
	 * Get scan statistics.
	 *
	 * Retrieves comprehensive statistics for a completed or running scan.
	 *
	 * @since 1.0.0
	 * @param int $scan_id Scan ID.
	 * @return array Scan statistics.
	 */
	public static function get_scan_stats( $scan_id ) {
		global $wpdb;
		
		$scans_table = $wpdb->prefix . 'content_guard_pro_scans';
		$findings_table = $wpdb->prefix . 'content_guard_pro_findings';
		
		// Get scan record.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$scan = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$scans_table}` WHERE scan_id = %d",
				$scan_id
			)
		);
		
		if ( ! $scan ) {
			return array(
				'scan_id'              => $scan_id,
				'items_scanned'        => 0,
				'findings_total'       => 0,
				'findings_critical'    => 0,
				'findings_suspicious'  => 0,
				'findings_review'      => 0,
				'duration_seconds'     => 0,
				'avg_query_ms'         => 0,
				'peak_mem_mb'          => 0,
				'errors'               => 0,
				'status'               => 'not_found',
			);
		}
		
		// Calculate scan duration.
		$duration_seconds = 0;
		if ( ! empty( $scan->finished_at ) ) {
			$start = strtotime( $scan->started_at );
			$end = strtotime( $scan->finished_at );
			if ( false !== $start && false !== $end ) {
				$duration_seconds = $end - $start;
			}
		} elseif ( ! empty( $scan->started_at ) ) {
			// Scan still running, calculate elapsed time.
			$start = strtotime( $scan->started_at );
			if ( false !== $start ) {
				$duration_seconds = time() - $start;
			}
		}
		
		// Get findings counts by severity.
		// Count findings created during this scan timeframe.
		// Note: severity values are hardcoded (not user input) but wrapped in prepare for consistency.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$findings_stats = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT
					COUNT(*) as total,
					SUM(CASE WHEN severity = %s THEN 1 ELSE 0 END) as critical,
					SUM(CASE WHEN severity = %s THEN 1 ELSE 0 END) as suspicious,
					SUM(CASE WHEN severity = %s THEN 1 ELSE 0 END) as review
				FROM `{$findings_table}`
				WHERE first_seen >= %s
				  AND first_seen <= %s",
				'critical',
				'suspicious',
				'review',
				$scan->started_at,
				! empty( $scan->finished_at ) ? $scan->finished_at : current_time( 'mysql' )
			)
		);
		
		// Determine status.
		$status = 'running';
		if ( ! empty( $scan->finished_at ) ) {
			$status = isset( $scan->status ) && 'cancelled' === $scan->status ? 'cancelled' : 'completed';
		} elseif ( isset( $scan->status ) && 'paused' === $scan->status ) {
			$status = 'paused';
		}
		
		return array(
			'scan_id'              => $scan_id,
			'mode'                 => isset( $scan->mode ) ? $scan->mode : 'unknown',
			'status'               => $status,
			'items_scanned'        => isset( $scan->totals_checked ) ? absint( $scan->totals_checked ) : 0,
			'findings_total'       => $findings_stats ? absint( $findings_stats->total ) : 0,
			'findings_critical'    => $findings_stats ? absint( $findings_stats->critical ) : 0,
			'findings_suspicious'  => $findings_stats ? absint( $findings_stats->suspicious ) : 0,
			'findings_review'      => $findings_stats ? absint( $findings_stats->review ) : 0,
			'duration_seconds'     => $duration_seconds,
			'duration_formatted'   => self::format_duration( $duration_seconds ),
			'avg_query_ms'         => isset( $scan->avg_query_ms ) ? absint( $scan->avg_query_ms ) : 0,
			'peak_mem_mb'          => isset( $scan->peak_mem_mb ) ? absint( $scan->peak_mem_mb ) : 0,
			'throttle_state'       => isset( $scan->throttle_state ) ? $scan->throttle_state : 'normal',
			'errors'               => isset( $scan->errors ) ? absint( $scan->errors ) : 0,
			'started_at'           => $scan->started_at,
			'finished_at'          => isset( $scan->finished_at ) ? $scan->finished_at : null,
		);
	}
	
	/**
	 * Format duration in seconds to human-readable string.
	 *
	 * @since 1.0.0
	 * @param int $seconds Duration in seconds.
	 * @return string Formatted duration.
	 */
	private static function format_duration( $seconds ) {
		if ( $seconds < 60 ) {
			return sprintf( '%d seconds', $seconds );
		} elseif ( $seconds < 3600 ) {
			$minutes = floor( $seconds / 60 );
			$secs = $seconds % 60;
			return sprintf( '%d min %d sec', $minutes, $secs );
		} else {
			$hours = floor( $seconds / 3600 );
			$minutes = floor( ( $seconds % 3600 ) / 60 );
			return sprintf( '%d hr %d min', $hours, $minutes );
		}
	}

	/**
	 * Clean up findings for deleted or trashed content.
	 *
	 * Checks all open, quarantined findings where object_type is 'post' and marks them
	 * as 'deleted' if the post no longer exists or is in trash.
	 * This runs during scan finalization to keep findings accurate.
	 *
	 * @since 1.0.0
	 * @return int Number of findings marked as deleted.
	 */
	private static function cleanup_deleted_content_findings() {
		global $wpdb;
		
		$findings_table = $wpdb->prefix . 'content_guard_pro_findings';
		$blog_id = get_current_blog_id();
		
		// Get all active findings (open, quarantined) for posts.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$findings = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT id, object_id, object_type 
				FROM `{$findings_table}` 
				WHERE blog_id = %d 
				AND object_type = %s 
				AND status IN (%s, %s)",
				$blog_id,
				'post',
				'open',
				'quarantined'
			),
			ARRAY_A
		);
		
		if ( empty( $findings ) ) {
			return 0;
		}
		
		$deleted_count = 0;
		$current_time = current_time( 'mysql' );
		
		foreach ( $findings as $finding ) {
			$post_id = absint( $finding['object_id'] );
			$post = get_post( $post_id );
			
			// Check if post is deleted (null) or in trash.
			$should_mark_deleted = false;
			if ( ! $post ) {
				$should_mark_deleted = true;
				$reason = 'Post permanently deleted or does not exist';
			} elseif ( 'trash' === $post->post_status ) {
				$should_mark_deleted = true;
				$reason = 'Post moved to trash';
			}
			
			if ( $should_mark_deleted ) {
				// Update finding to 'deleted' status.
				// Store metadata about when/why it was marked deleted.
				$extra_data = array(
					'deleted_at' => $current_time,
					'deleted_reason' => $reason,
					'deleted_by' => 'auto_scan_cleanup',
				);
				
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$updated = $wpdb->update(
					$findings_table,
					array(
						'status' => 'deleted',
						'extra'  => wp_json_encode( $extra_data ),
					),
					array(
						'id'      => $finding['id'],
						'blog_id' => $blog_id,
					),
					array( '%s', '%s' ),
					array( '%d', '%d' )
				);
				
				if ( false !== $updated && $updated > 0 ) {
					$deleted_count++;
					
					// Trigger action for logging/integrations.
					do_action( 'content_guard_pro_finding_marked_deleted', $finding['id'], $post_id, $reason );
				}
			}
		}
		
		return $deleted_count;
	}

	/**
	 * Handle post trashed event.
	 *
	 * Immediately marks findings as 'deleted' when a post is moved to trash.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID being trashed.
	 */
	public static function on_post_trashed( $post_id ) {
		self::mark_post_findings_deleted( $post_id, 'Post moved to trash' );
	}

	/**
	 * Handle post deleted event.
	 *
	 * Immediately marks findings as 'deleted' when a post is permanently deleted.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID being deleted.
	 */
	public static function on_post_deleted( $post_id ) {
		self::mark_post_findings_deleted( $post_id, 'Post permanently deleted' );
	}

	/**
	 * Handle post untrashed event.
	 *
	 * Re-opens findings when a post is restored from trash.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID being restored.
	 */
	public static function on_post_untrashed( $post_id ) {
		global $wpdb;
		
		$findings_table = $wpdb->prefix . 'content_guard_pro_findings';
		$blog_id = get_current_blog_id();
		
		// Find all 'deleted' findings for this post.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$findings = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT id FROM `{$findings_table}` 
				WHERE blog_id = %d 
				AND object_type = %s 
				AND object_id = %d 
				AND status = %s",
				$blog_id,
				'post',
				$post_id,
				'deleted'
			),
			ARRAY_A
		);
		
		if ( empty( $findings ) ) {
			return;
		}
		
		$reopened_count = 0;
		foreach ( $findings as $finding ) {
			// Re-open the finding (set back to 'open' status).
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$updated = $wpdb->update(
				$findings_table,
				array( 'status' => 'open' ),
				array(
					'id'      => $finding['id'],
					'blog_id' => $blog_id,
				),
				array( '%s' ),
				array( '%d', '%d' )
			);
			
			if ( false !== $updated && $updated > 0 ) {
				$reopened_count++;
				do_action( 'content_guard_pro_finding_reopened', $finding['id'], $post_id );
			}
		}
		
		if ( $reopened_count > 0 ) {
			cgp_log( sprintf(
				'Content Guard Pro: Re-opened %d findings for post %d (restored from trash)',
				$reopened_count,
				$post_id
			) );
		}
	}

	/**
	 * Mark all findings for a post as deleted.
	 *
	 * Helper method to mark findings as 'deleted' with appropriate metadata.
	 *
	 * @since 1.0.0
	 * @param int    $post_id Post ID.
	 * @param string $reason  Reason for deletion.
	 */
	private static function mark_post_findings_deleted( $post_id, $reason ) {
		global $wpdb;
		
		$findings_table = $wpdb->prefix . 'content_guard_pro_findings';
		$blog_id = get_current_blog_id();
		$current_time = current_time( 'mysql' );
		
		// Find all active findings (open, quarantined) for this post.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$findings = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT id FROM `{$findings_table}` 
				WHERE blog_id = %d 
				AND object_type = %s 
				AND object_id = %d 
				AND status IN (%s, %s)",
				$blog_id,
				'post',
				$post_id,
				'open',
				'quarantined'
			),
			ARRAY_A
		);
		
		if ( empty( $findings ) ) {
			return;
		}
		
		$deleted_count = 0;
		foreach ( $findings as $finding ) {
			// Store metadata about deletion.
			$extra_data = array(
				'deleted_at' => $current_time,
				'deleted_reason' => $reason,
				'deleted_by' => 'wp_hook',
			);
			
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$updated = $wpdb->update(
				$findings_table,
				array(
					'status' => 'deleted',
					'extra'  => wp_json_encode( $extra_data ),
				),
				array(
					'id'      => $finding['id'],
					'blog_id' => $blog_id,
				),
				array( '%s', '%s' ),
				array( '%d', '%d' )
			);
			
			if ( false !== $updated && $updated > 0 ) {
				$deleted_count++;
				do_action( 'content_guard_pro_finding_marked_deleted', $finding['id'], $post_id, $reason );
			}
		}
		
		if ( $deleted_count > 0 ) {
			cgp_log( sprintf(
				'Content Guard Pro: Marked %d findings as deleted for post %d (%s)',
				$deleted_count,
				$post_id,
				$reason
			) );
		}
	}
}

