<?php
/**
 * REST API Class
 *
 * Provides read-only REST API endpoints for accessing findings data.
 * Per PRD Section 3.7 and Appendix D.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_REST_API
 *
 * Implements REST API endpoints for Content Guard Pro.
 *
 * Endpoints:
 * - GET /wp-json/content-guard-pro/v1/findings
 *
 * Per PRD Section 3.7: REST API (read-only) for findings with pagination and auth.
 * Per PRD Section 3.8: edit_posts capability required to view findings.
 *
 * @since 1.0.0
 */
class CGP_REST_API {

	/**
	 * API namespace.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const NAMESPACE = 'content-guard-pro/v1';

	/**
	 * Initialize the REST API.
	 *
	 * @since 1.0.0
	 */
	public static function init() {
		add_action( 'rest_api_init', array( __CLASS__, 'register_routes' ) );
	}

	/**
	 * Register REST API routes.
	 *
	 * @since 1.0.0
	 */
	public static function register_routes() {
		// GET /wp-json/content-guard-pro/v1/findings
		register_rest_route(
			self::NAMESPACE,
			'/findings',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( __CLASS__, 'get_findings_callback' ),
				'permission_callback' => array( __CLASS__, 'check_permission' ),
				'args'                => self::get_findings_query_args(),
			)
		);

		// GET /wp-json/content-guard-pro/v1/findings/{id}
		register_rest_route(
			self::NAMESPACE,
			'/findings/(?P<id>\d+)',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( __CLASS__, 'get_finding_callback' ),
				'permission_callback' => array( __CLASS__, 'check_permission' ),
				'args'                => array(
					'id' => array(
						'description'       => __( 'Finding ID', 'content-guard-pro' ),
						'type'              => 'integer',
						'required'          => true,
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return is_numeric( $param ) && $param > 0;
						},
					),
					'include_excerpt' => array(
						'description'       => __( 'Include matched excerpt', 'content-guard-pro' ),
						'type'              => 'boolean',
						'default'           => false,
						'sanitize_callback' => 'rest_sanitize_boolean',
					),
				),
			)
		);

		// GET /wp-json/content-guard-pro/v1/stats
		register_rest_route(
			self::NAMESPACE,
			'/stats',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( __CLASS__, 'get_stats_callback' ),
				'permission_callback' => array( __CLASS__, 'check_permission' ),
			)
		);

		// GET /wp-json/content-guard-pro/v1/scans/active - Real-time progress for active scans.
		register_rest_route(
			self::NAMESPACE,
			'/scans/active',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( __CLASS__, 'get_active_scan_callback' ),
				'permission_callback' => array( __CLASS__, 'check_permission' ),
			)
		);

		// GET /wp-json/content-guard-pro/v1/scans/{scan_id}/progress - Progress for specific scan.
		register_rest_route(
			self::NAMESPACE,
			'/scans/(?P<scan_id>\d+)/progress',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( __CLASS__, 'get_scan_progress_callback' ),
				'permission_callback' => array( __CLASS__, 'check_permission' ),
				'args'                => array(
					'scan_id' => array(
						'description'       => __( 'Scan ID', 'content-guard-pro' ),
						'type'              => 'integer',
						'required'          => true,
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return is_numeric( $param ) && $param > 0;
						},
					),
				),
			)
		);

		// GET /wp-json/content-guard-pro/v1/scans/{scan_id}/details - Detailed scan information.
		register_rest_route(
			self::NAMESPACE,
			'/scans/(?P<scan_id>\d+)/details',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( __CLASS__, 'get_scan_details_callback' ),
				'permission_callback' => array( __CLASS__, 'check_permission' ),
				'args'                => array(
					'scan_id' => array(
						'description'       => __( 'Scan ID', 'content-guard-pro' ),
						'type'              => 'integer',
						'required'          => true,
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return is_numeric( $param ) && $param > 0;
						},
					),
				),
			)
		);

		// POST /wp-json/content-guard-pro/v1/scan-post - Quick synchronous scan for a post (US-037).
		register_rest_route(
			self::NAMESPACE,
			'/scan-post',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( __CLASS__, 'quick_scan_post_callback' ),
				'permission_callback' => array( __CLASS__, 'check_edit_permission' ),
				'args'                => array(
					'post_id' => array(
						'description'       => __( 'Post ID to scan', 'content-guard-pro' ),
						'type'              => 'integer',
						'required'          => true,
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return is_numeric( $param ) && $param > 0;
						},
					),
					'content' => array(
						'description'       => __( 'Optional content to scan (if not saved yet)', 'content-guard-pro' ),
						'type'              => 'string',
						'required'          => false,
						// Note: We intentionally do NOT sanitize this content because we need to
						// scan the raw content for malicious patterns. The content is not stored,
						// only analyzed. Using wp_kses_post would strip the exact patterns we're
						// trying to detect (scripts, iframes, event handlers, etc.).
						'sanitize_callback' => function( $value ) {
							// Only do basic sanitization - remove null bytes and normalize line endings.
							$value = str_replace( chr( 0 ), '', $value );
							$value = str_replace( array( "\r\n", "\r" ), "\n", $value );
							return $value;
						},
					),
					'manual' => array(
						'description'       => __( 'Whether this is a manual scan (bypasses auto-scan setting)', 'content-guard-pro' ),
						'type'              => 'boolean',
						'required'          => false,
						'default'           => false,
						'sanitize_callback' => 'rest_sanitize_boolean',
					),
				),
			)
		);

		// GET /wp-json/content-guard-pro/v1/post-findings/{post_id} - Get findings for a specific post.
		register_rest_route(
			self::NAMESPACE,
			'/post-findings/(?P<post_id>\d+)',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( __CLASS__, 'get_post_findings_callback' ),
				'permission_callback' => array( __CLASS__, 'check_edit_permission' ),
				'args'                => array(
					'post_id' => array(
						'description'       => __( 'Post ID', 'content-guard-pro' ),
						'type'              => 'integer',
						'required'          => true,
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return is_numeric( $param ) && $param > 0;
						},
					),
				),
			)
		);
	}

	/**
	 * Permission callback for REST endpoints.
	 *
	 * Per PRD Section 3.8: edit_posts capability required to view findings.
	 *
	 * @since 1.0.0
	 * @return bool True if user has permission, false otherwise.
	 */
	public static function check_permission() {
		// Check if user can edit posts (Editors+).
		return current_user_can( 'edit_posts' );
	}

	/**
	 * Permission callback for edit endpoints.
	 *
	 * Checks if user can edit the specific post.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Request object.
	 * @return bool True if user has permission, false otherwise.
	 */
	public static function check_edit_permission( $request ) {
		$post_id = $request->get_param( 'post_id' );
		if ( ! $post_id ) {
			return current_user_can( 'edit_posts' );
		}
		return current_user_can( 'edit_post', $post_id );
	}

	/**
	 * Get query arguments for findings endpoint.
	 *
	 * @since 1.0.0
	 * @return array Query arguments.
	 */
	private static function get_findings_query_args() {
		return array(
			'status'          => array(
				'description'       => __( 'Filter by status', 'content-guard-pro' ),
				'type'              => 'string',
				'enum'              => array( 'open', 'quarantined', 'ignored', 'resolved', 'deleted' ),
				'sanitize_callback' => 'sanitize_text_field',
				'validate_callback' => function( $param ) {
					return in_array( $param, array( 'open', 'quarantined', 'ignored', 'resolved', 'deleted' ), true );
				},
			),
			'severity'        => array(
				'description'       => __( 'Filter by severity', 'content-guard-pro' ),
				'type'              => 'string',
				'enum'              => array( 'critical', 'suspicious', 'review' ),
				'sanitize_callback' => 'sanitize_text_field',
				'validate_callback' => function( $param ) {
					return in_array( $param, array( 'critical', 'suspicious', 'review' ), true );
				},
			),
			'object_type'     => array(
				'description'       => __( 'Filter by object type', 'content-guard-pro' ),
				'type'              => 'string',
				'enum'              => array( 'post', 'postmeta', 'option' ),
				'sanitize_callback' => 'sanitize_text_field',
				'validate_callback' => function( $param ) {
					return in_array( $param, array( 'post', 'postmeta', 'option' ), true );
				},
			),
			'object_id'       => array(
				'description'       => __( 'Filter by object ID', 'content-guard-pro' ),
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
			),
			'rule_id'         => array(
				'description'       => __( 'Filter by rule ID', 'content-guard-pro' ),
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
			),
			'page'            => array(
				'description'       => __( 'Current page', 'content-guard-pro' ),
				'type'              => 'integer',
				'default'           => 1,
				'minimum'           => 1,
				'sanitize_callback' => 'absint',
			),
			'per_page'        => array(
				'description'       => __( 'Results per page', 'content-guard-pro' ),
				'type'              => 'integer',
				'default'           => 50,
				'minimum'           => 1,
				'maximum'           => 100,
				'sanitize_callback' => 'absint',
			),
			'orderby'         => array(
				'description'       => __( 'Order by field', 'content-guard-pro' ),
				'type'              => 'string',
				'enum'              => array( 'id', 'severity', 'confidence', 'first_seen', 'last_seen' ),
				'default'           => 'last_seen',
				'sanitize_callback' => 'sanitize_text_field',
			),
			'order'           => array(
				'description'       => __( 'Order direction', 'content-guard-pro' ),
				'type'              => 'string',
				'enum'              => array( 'asc', 'desc' ),
				'default'           => 'desc',
				'sanitize_callback' => 'sanitize_text_field',
			),
			'include_excerpt' => array(
				'description'       => __( 'Include matched excerpts', 'content-guard-pro' ),
				'type'              => 'boolean',
				'default'           => false,
				'sanitize_callback' => 'rest_sanitize_boolean',
			),
		);
	}

	/**
	 * Get findings callback.
	 *
	 * GET /wp-json/content-guard-pro/v1/findings
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response Response object.
	 */
	public static function get_findings_callback( $request ) {
		global $wpdb;

		// Get parameters.
		$status          = $request->get_param( 'status' );
		$severity        = $request->get_param( 'severity' );
		$object_type     = $request->get_param( 'object_type' );
		$object_id       = $request->get_param( 'object_id' );
		$rule_id         = $request->get_param( 'rule_id' );
		$page            = $request->get_param( 'page' );
		$per_page        = $request->get_param( 'per_page' );
		$orderby         = $request->get_param( 'orderby' );
		$order           = $request->get_param( 'order' );
		$include_excerpt = $request->get_param( 'include_excerpt' );

		// Build query.
		$where_clauses = array( '1=1' );
		$where_values  = array();

		if ( $status ) {
			$where_clauses[] = 'status = %s';
			$where_values[]  = $status;
		}

		if ( $severity ) {
			$where_clauses[] = 'severity = %s';
			$where_values[]  = $severity;
		}

		if ( $object_type ) {
			$where_clauses[] = 'object_type = %s';
			$where_values[]  = $object_type;
		}

		if ( $object_id ) {
			$where_clauses[] = 'object_id = %d';
			$where_values[]  = $object_id;
		}

		if ( $rule_id ) {
			$where_clauses[] = 'rule_id = %s';
			$where_values[]  = $rule_id;
		}

		$where_sql = implode( ' AND ', $where_clauses );
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		// Get total count for pagination.
		$count_query = "SELECT COUNT(*) FROM `" . $table_name . "` WHERE " . $where_sql;

		if ( ! empty( $where_values ) ) {
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query template is built safely above.
			$count_query = $wpdb->prepare( $count_query, $where_values );
		}

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is prepared above.
		$total_items = (int) $wpdb->get_var( $count_query );

		// Calculate pagination (protect against division by zero).
		$per_page    = max( 1, $per_page );
		$total_pages = ceil( $total_items / $per_page );
		$offset      = ( $page - 1 ) * $per_page;

		// Determine which fields to select.
		if ( $include_excerpt ) {
			$select_fields = '*';
		} else {
			// Omit matched_excerpt per PRD Appendix D.
			$select_fields = 'id, blog_id, object_type, object_id, field, fingerprint, rule_id, severity, confidence, first_seen, last_seen, status, extra';
		}

		// Build main query.
		$query = "SELECT " . $select_fields . " FROM `" . $table_name . "` WHERE " . $where_sql;

		// Add ORDER BY with backticks for column names.
		$allowed_orderby = array( 'id', 'severity', 'confidence', 'first_seen', 'last_seen' );
		$orderby         = in_array( $orderby, $allowed_orderby, true ) ? $orderby : 'last_seen';
		$order           = 'asc' === strtolower( $order ) ? 'ASC' : 'DESC';
		$query          .= " ORDER BY `{$orderby}` {$order}";

		// Add LIMIT.
		$query .= $wpdb->prepare( ' LIMIT %d OFFSET %d', $per_page, $offset );

		// Execute query with prepared statement if we have values.
		if ( ! empty( $where_values ) ) {
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query template is built safely above.
			$findings = $wpdb->get_results( $wpdb->prepare( $query, $where_values ) );
		} else {
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is built safely with validated parameters above.
			$findings = $wpdb->get_results( $query );
		}

		// Format findings for API response.
		$formatted_findings = array();
		foreach ( $findings as $finding ) {
			$formatted_finding = self::format_finding( $finding, $include_excerpt );
			$formatted_findings[] = $formatted_finding;
		}

		// Build response.
		$response = rest_ensure_response( $formatted_findings );

		// Add pagination headers (WordPress REST API standard).
		$response->header( 'X-WP-Total', $total_items );
		$response->header( 'X-WP-TotalPages', $total_pages );

		// Add Link header for pagination navigation (RFC 5988).
		$base_url = rest_url( self::NAMESPACE . '/findings' );
		$links    = array();

		if ( $page > 1 ) {
			$prev_url   = add_query_arg( array_merge( $request->get_query_params(), array( 'page' => $page - 1 ) ), $base_url );
			$links[]    = '<' . esc_url_raw( $prev_url ) . '>; rel="prev"';
		}

		if ( $page < $total_pages ) {
			$next_url = add_query_arg( array_merge( $request->get_query_params(), array( 'page' => $page + 1 ) ), $base_url );
			$links[]  = '<' . esc_url_raw( $next_url ) . '>; rel="next"';
		}

		if ( ! empty( $links ) ) {
			$response->header( 'Link', implode( ', ', $links ) );
		}

		return $response;
	}

	/**
	 * Get single finding callback.
	 *
	 * GET /wp-json/content-guard-pro/v1/findings/{id}
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error Response object or error.
	 */
	public static function get_finding_callback( $request ) {
		global $wpdb;

		$id              = $request->get_param( 'id' );
		$include_excerpt = $request->get_param( 'include_excerpt' );
		$table_name      = $wpdb->prefix . 'content_guard_pro_findings';

		// Determine which fields to select.
		if ( $include_excerpt ) {
			$select_fields = '*';
		} else {
			$select_fields = 'id, blog_id, object_type, object_id, field, fingerprint, rule_id, severity, confidence, first_seen, last_seen, status, extra';
		}

		// Query for finding.
		$finding = $wpdb->get_row(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
				"SELECT " . $select_fields . " FROM `" . $table_name . "` WHERE id = %d",
				$id
			)
		);

		if ( ! $finding ) {
			return new WP_Error(
				'content_guard_pro_finding_not_found',
				__( 'Finding not found', 'content-guard-pro' ),
				array( 'status' => 404 )
			);
		}

		// Format and return.
		$formatted_finding = self::format_finding( $finding, $include_excerpt );

		return rest_ensure_response( $formatted_finding );
	}

	/**
	 * Get stats callback.
	 *
	 * GET /wp-json/content-guard-pro/v1/stats
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response Response object.
	 */
	public static function get_stats_callback( $request ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		// Get counts by severity.
		// phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
		$severity_counts = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT severity, COUNT(*) as count
				FROM `" . $table_name . "`
				WHERE status = %s
				GROUP BY severity",
				'open'
			)
		);
		// phpcs:enable WordPress.DB.PreparedSQL.NotPrepared

		$stats = array(
			'total'      => 0,
			'critical'   => 0,
			'suspicious' => 0,
			'review'     => 0,
		);

		// Valid severity levels.
		$valid_severities = array( 'critical', 'suspicious', 'review' );

		foreach ( $severity_counts as $row ) {
			// Only add counts for valid severity levels.
			if ( in_array( $row->severity, $valid_severities, true ) ) {
				$stats[ $row->severity ] = absint( $row->count );
				$stats['total']         += absint( $row->count );
			}
		}

		// Get counts by status.
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared
		$status_counts = $wpdb->get_results(
			"SELECT status, COUNT(*) as count
			FROM `" . $table_name . "`
			GROUP BY status"
		);
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared

		$stats['by_status'] = array();
		foreach ( $status_counts as $row ) {
			$stats['by_status'][ $row->status ] = absint( $row->count );
		}

		// Get last scan info.
		$scans_table = $wpdb->prefix . 'content_guard_pro_scans';
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared
		$last_scan = $wpdb->get_row(
			"SELECT * FROM `" . $scans_table . "`
			WHERE finished_at IS NOT NULL
			ORDER BY started_at DESC
			LIMIT 1"
		);
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared

		if ( $last_scan ) {
			$stats['last_scan'] = array(
				'scan_id'        => absint( $last_scan->scan_id ),
				'started_at'     => $last_scan->started_at,
				'finished_at'    => $last_scan->finished_at,
				'mode'           => $last_scan->mode,
				'items_checked'  => absint( $last_scan->totals_checked ),
				'items_flagged'  => absint( $last_scan->totals_flagged ),
			);
		} else {
			$stats['last_scan'] = null;
		}

		return rest_ensure_response( $stats );
	}

	/**
	 * Get active scan callback.
	 *
	 * GET /wp-json/content-guard-pro/v1/scans/active
	 *
	 * Returns progress information for currently active scans with caching.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response Response object.
	 */
	public static function get_active_scan_callback( $request ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';

		// Check transient cache first (2 second TTL).
		$cache_key     = 'content_guard_pro_active_scan_progress';
		$cached_result = get_transient( $cache_key );

		if ( false !== $cached_result ) {
			return rest_ensure_response( $cached_result );
		}

		// Query for active scans (pending, running, or recently completed).
		// Include recently completed scans (last 60 seconds) so JavaScript can do final update.
		// Use UTC time for comparison to avoid timezone issues.
		// phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
		$active_scans = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT *
				FROM `" . $table_name . "`
				WHERE status IN (%s, %s)
				   OR (status IN (%s, %s, %s) AND finished_at IS NOT NULL AND finished_at > %s)
				   OR (status IN (%s, %s, %s) AND finished_at IS NULL AND started_at > %s)
				ORDER BY started_at DESC
				LIMIT 5",
				'pending',
				'running',
				'completed',
				'failed',
				'cancelled',
				gmdate( 'Y-m-d H:i:s', time() - 60 ), // 60 seconds ago in GMT
				'completed',
				'failed',
				'cancelled',
				gmdate( 'Y-m-d H:i:s', time() - 300 ) // 5 minutes ago in GMT
			)
		);
		// phpcs:enable WordPress.DB.PreparedSQL.NotPrepared

		$formatted_scans = array();
		$has_truly_active = false;

		foreach ( $active_scans as $scan ) {
			$formatted_scans[] = self::format_scan_progress( $scan );
			
			// Check if there are any truly active scans (pending or running).
			if ( in_array( $scan->status, array( 'pending', 'running' ), true ) ) {
				$has_truly_active = true;
			}
		}

		$response = array(
			'active_scans' => $formatted_scans,
			'has_active'   => $has_truly_active,
		);

		// Cache for 2 seconds to reduce database load.
		set_transient( $cache_key, $response, 2 );

		return rest_ensure_response( $response );
	}

	/**
	 * Get scan progress callback.
	 *
	 * GET /wp-json/content-guard-pro/v1/scans/{scan_id}/progress
	 *
	 * Returns progress information for a specific scan with caching.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error Response object or error.
	 */
	public static function get_scan_progress_callback( $request ) {
		global $wpdb;

		$scan_id    = $request->get_param( 'scan_id' );
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';

		// Check transient cache first (2 second TTL).
		$cache_key     = 'content_guard_pro_scan_progress_' . $scan_id;
		$cached_result = get_transient( $cache_key );

		if ( false !== $cached_result ) {
			return rest_ensure_response( $cached_result );
		}

		// Query for scan.
		$scan = $wpdb->get_row(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
				"SELECT * FROM `" . $table_name . "` WHERE scan_id = %d",
				$scan_id
			)
		);

		if ( ! $scan ) {
			return new WP_Error(
				'content_guard_pro_scan_not_found',
				__( 'Scan not found', 'content-guard-pro' ),
				array( 'status' => 404 )
			);
		}

		$formatted_scan = self::format_scan_progress( $scan );

		// Cache for 2 seconds to reduce database load.
		set_transient( $cache_key, $formatted_scan, 2 );

		return rest_ensure_response( $formatted_scan );
	}

	/**
	 * Get scan details callback.
	 *
	 * GET /wp-json/content-guard-pro/v1/scans/{scan_id}/details
	 *
	 * Returns detailed information about a specific scan including findings summary.
	 *
	 * @since 1.0.1
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error Response object or error.
	 */
	public static function get_scan_details_callback( $request ) {
		global $wpdb;

		$scan_id = absint( $request->get_param( 'scan_id' ) );

		$scans_table = $wpdb->prefix . 'content_guard_pro_scans';
		$findings_table = $wpdb->prefix . 'content_guard_pro_findings';

		// Get scan record.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		$scan = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$scans_table}` WHERE scan_id = %d AND blog_id = %d",
				$scan_id,
				get_current_blog_id()
			)
		);

		if ( ! $scan ) {
			return new WP_Error(
				'content_guard_pro_scan_not_found',
				__( 'Scan not found', 'content-guard-pro' ),
				array( 'status' => 404 )
			);
		}

		// Determine status.
		$status = 'running';
		if ( ! empty( $scan->finished_at ) ) {
			$status = isset( $scan->status ) && 'cancelled' === $scan->status ? 'cancelled' : 'completed';
		} elseif ( isset( $scan->status ) && 'paused' === $scan->status ) {
			$status = 'paused';
		} elseif ( isset( $scan->status ) && 'pending' === $scan->status ) {
			$status = 'pending';
		}

		// Calculate duration.
		$duration_seconds = 0;
		$duration_formatted = __( 'N/A', 'content-guard-pro' );
		if ( ! empty( $scan->started_at ) ) {
			$started = strtotime( $scan->started_at );
			$finished = ! empty( $scan->finished_at ) ? strtotime( $scan->finished_at ) : time();
			if ( false !== $started && false !== $finished ) {
				$duration_seconds = $finished - $started;
				if ( $duration_seconds < 60 ) {
					/* translators: %d: Number of seconds */
					$duration_formatted = sprintf( __( '%d seconds', 'content-guard-pro' ), $duration_seconds );
				} elseif ( $duration_seconds < 3600 ) {
					$minutes = floor( $duration_seconds / 60 );
					$seconds = $duration_seconds % 60;
					/* translators: 1: Number of minutes, 2: Number of seconds */
					$duration_formatted = sprintf( __( '%1$d min %2$d sec', 'content-guard-pro' ), $minutes, $seconds );
				} else {
					$hours = floor( $duration_seconds / 3600 );
					$minutes = floor( ( $duration_seconds % 3600 ) / 60 );
					/* translators: 1: Number of hours, 2: Number of minutes */
					$duration_formatted = sprintf( __( '%1$d hr %2$d min', 'content-guard-pro' ), $hours, $minutes );
				}
			}
		}

		// Get findings summary for this scan.
		// Strategy: Use totals_flagged (accurate count) + breakdown estimate from findings first seen in this scan.
		$findings_stats = array(
			'total'      => absint( $scan->totals_flagged ), // Use accurate count from scan record.
			'critical'   => 0,
			'suspicious' => 0,
			'review'     => 0,
			'is_approximate' => false, // Flag to indicate if breakdown is approximate.
		);

		if ( ! empty( $scan->started_at ) ) {
			// For completed scans, use finished_at. For running scans, use current time.
			$finished_time = ! empty( $scan->finished_at ) ? $scan->finished_at : current_time( 'mysql' );
			
			// Query findings that were FIRST SEEN during this scan period.
			// This gives us an estimate of severity breakdown for findings encountered in this scan.
			// Note: This is approximate because:
			// - Existing findings re-encountered in this scan won't be counted (their first_seen is earlier)
			// - But totals_flagged is accurate (counted during scan)
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
			$findings_result = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT
						COUNT(*) as total,
						SUM(CASE WHEN severity = %s THEN 1 ELSE 0 END) as critical,
						SUM(CASE WHEN severity = %s THEN 1 ELSE 0 END) as suspicious,
						SUM(CASE WHEN severity = %s THEN 1 ELSE 0 END) as review
					FROM `{$findings_table}`
					WHERE blog_id = %d
					  AND first_seen >= %s
					  AND first_seen <= %s",
					'critical',
					'suspicious',
					'review',
					get_current_blog_id(),
					$scan->started_at,
					$finished_time
				)
			);

			if ( $findings_result ) {
				$breakdown_total = absint( $findings_result->total );
				$breakdown_critical = absint( $findings_result->critical );
				$breakdown_suspicious = absint( $findings_result->suspicious );
				$breakdown_review = absint( $findings_result->review );
				
				// If breakdown total doesn't match totals_flagged, the breakdown is approximate.
				// This happens when existing findings were re-encountered in this scan.
				$is_approximate = ( $breakdown_total !== $findings_stats['total'] );
				
				// If totals_flagged is greater than breakdown, scale the breakdown proportionally.
				// This gives a better estimate of severity distribution.
				if ( $findings_stats['total'] > 0 && $breakdown_total > 0 ) {
					$scale_factor = $findings_stats['total'] / $breakdown_total;
					$findings_stats['critical'] = round( $breakdown_critical * $scale_factor );
					$findings_stats['suspicious'] = round( $breakdown_suspicious * $scale_factor );
					$findings_stats['review'] = round( $breakdown_review * $scale_factor );
					
					// Adjust to ensure total matches (rounding may cause small differences).
					$calculated_total = $findings_stats['critical'] + $findings_stats['suspicious'] + $findings_stats['review'];
					if ( $calculated_total !== $findings_stats['total'] ) {
						// Distribute difference to largest category.
						$diff = $findings_stats['total'] - $calculated_total;
						if ( $findings_stats['critical'] >= $findings_stats['suspicious'] && $findings_stats['critical'] >= $findings_stats['review'] ) {
							$findings_stats['critical'] += $diff;
						} elseif ( $findings_stats['suspicious'] >= $findings_stats['review'] ) {
							$findings_stats['suspicious'] += $diff;
						} else {
							$findings_stats['review'] += $diff;
						}
					}
				} else {
					// No breakdown available, use zeros (will show note).
					$findings_stats['critical'] = 0;
					$findings_stats['suspicious'] = 0;
					$findings_stats['review'] = 0;
					$is_approximate = true;
				}
				
				$findings_stats['is_approximate'] = $is_approximate;
			} else {
				// No findings first seen in this scan period.
				// Breakdown is approximate (existing findings were re-encountered).
				$findings_stats['is_approximate'] = true;
			}
		} else {
			// No scan start time available.
			$findings_stats['is_approximate'] = true;
		}

		// Format response.
		$response = array(
			'scan_id'           => $scan_id,
			'status'            => $status,
			'mode'              => isset( $scan->mode ) ? $scan->mode : 'unknown',
			'started_at'        => ! empty( $scan->started_at ) ? mysql2date( 'c', $scan->started_at, false ) : null,
			'started_at_formatted' => ! empty( $scan->started_at ) ? mysql2date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $scan->started_at ) : null,
			'finished_at'       => ! empty( $scan->finished_at ) ? mysql2date( 'c', $scan->finished_at, false ) : null,
			'finished_at_formatted' => ! empty( $scan->finished_at ) ? mysql2date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $scan->finished_at ) : null,
			'duration_seconds'  => $duration_seconds,
			'duration_formatted' => $duration_formatted,
			'items_checked'     => absint( $scan->totals_checked ),
			'items_flagged'     => absint( $scan->totals_flagged ),
			'findings'          => $findings_stats,
			'avg_query_ms'      => isset( $scan->avg_query_ms ) ? absint( $scan->avg_query_ms ) : 0,
			'peak_mem_mb'        => isset( $scan->peak_mem_mb ) ? absint( $scan->peak_mem_mb ) : 0,
			'throttle_state'    => isset( $scan->throttle_state ) ? $scan->throttle_state : 'normal',
			'errors'             => isset( $scan->errors ) ? absint( $scan->errors ) : 0,
		);

		return rest_ensure_response( $response );
	}

	/**
	 * Format scan progress for API response.
	 *
	 * Calculates progress percentage, elapsed time, and other metrics.
	 *
	 * @since 1.0.0
	 * @param object $scan Scan object from database.
	 * @return array Formatted scan progress.
	 */
	private static function format_scan_progress( $scan ) {
		$scan_id   = absint( $scan->scan_id );
		$status    = $scan->status;
		$mode      = $scan->mode;
		$targets   = ! empty( $scan->targets ) ? explode( ',', $scan->targets ) : array();
		$progress  = ! empty( $scan->progress ) ? json_decode( $scan->progress, true ) : array();

		// Get current counts.
		$items_checked = absint( $scan->totals_checked );
		$items_flagged = absint( $scan->totals_flagged );

		// Estimate total items based on progress.
		$estimated_total = self::estimate_total_items( $scan );
		$progress_percentage = 0;

		if ( $estimated_total > 0 && $items_checked > 0 ) {
			$progress_percentage = min( 100, round( ( $items_checked / $estimated_total ) * 100, 1 ) );
		} elseif ( 'completed' === $status ) {
			$progress_percentage = 100;
		}

		// Calculate elapsed time.
		$started_at      = ! empty( $scan->started_at ) ? strtotime( $scan->started_at ) : null;
		$finished_at     = ! empty( $scan->finished_at ) ? strtotime( $scan->finished_at ) : null;
		$elapsed_seconds = 0;

		if ( $started_at ) {
			if ( $finished_at ) {
				$elapsed_seconds = $finished_at - $started_at;
			} else {
				$elapsed_seconds = time() - $started_at;
			}
		}

		// Determine current target and offset from progress.
		$current_target = '';
		$current_offset = 0;

		if ( ! empty( $progress ) && is_array( $progress ) ) {
			// Find the last target that has progress.
			foreach ( $progress as $target => $data ) {
				if ( isset( $data['offset'] ) && $data['offset'] > 0 ) {
					$current_target = $target;
					$current_offset = absint( $data['offset'] );
				}
			}
		}

		// Format response.
		$formatted = array(
			'scan_id'             => $scan_id,
			'status'              => $status,
			'mode'                => $mode,
			'progress_percentage' => $progress_percentage,
			'items_checked'       => $items_checked,
			'items_total'         => $estimated_total,
			'findings'            => $items_flagged,
			'current_target'      => $current_target,
			'current_offset'      => $current_offset,
			'started_at'          => ! empty( $scan->started_at ) ? mysql2date( 'c', $scan->started_at, false ) : null,
			'finished_at'         => ! empty( $scan->finished_at ) ? mysql2date( 'c', $scan->finished_at, false ) : null,
			'elapsed_seconds'     => $elapsed_seconds,
			'elapsed_display'     => self::format_elapsed_time( $elapsed_seconds ),
			'targets'             => $targets,
			'errors'              => ! empty( $scan->errors ) ? $scan->errors : '',
		);

		return $formatted;
	}

	/**
	 * Estimate total items for a scan.
	 *
	 * @since 1.0.0
	 * @param object $scan Scan object from database.
	 * @return int Estimated total items.
	 */
	private static function estimate_total_items( $scan ) {
		global $wpdb;

		$mode    = $scan->mode;
		$targets = ! empty( $scan->targets ) ? explode( ',', $scan->targets ) : array();

		$total = 0;

		if ( in_array( 'posts', $targets, true ) ) {
			// Count posts using same filters as scanner.
			$total += (int) $wpdb->get_var(
				"SELECT COUNT(*) FROM `" . $wpdb->posts . "`
				WHERE post_status IN ('publish', 'draft', 'private', 'pending')
				AND post_type NOT IN ('revision', 'nav_menu_item', 'attachment')"
			);
		}

		if ( 'standard' === $mode ) {
			if ( in_array( 'postmeta', $targets, true ) ) {
				// Count postmeta only for valid posts (matches scanner behavior).
				$total += (int) $wpdb->get_var(
					"SELECT COUNT(pm.meta_id)
					FROM `" . $wpdb->postmeta . "` pm
					INNER JOIN `" . $wpdb->posts . "` p ON pm.post_id = p.ID
					WHERE p.post_status IN ('publish', 'draft', 'private', 'pending')
					AND p.post_type NOT IN ('revision', 'nav_menu_item', 'attachment')"
				);
			}

			if ( in_array( 'options', $targets, true ) ) {
				// Count allowlisted options (matches scanner exactly).
				$allowlist = array( 'blogname', 'blogdescription', 'home', 'siteurl', 'admin_email' );
				$total    += count( $allowlist );
			}
		}

		return max( 1, $total ); // Avoid division by zero.
	}

	/**
	 * Format elapsed time in human-readable format.
	 *
	 * @since 1.0.0
	 * @param int $seconds Elapsed seconds.
	 * @return string Formatted time string.
	 */
	private static function format_elapsed_time( $seconds ) {
		if ( $seconds < 60 ) {
			/* translators: %d: number of seconds */
			return sprintf( _n( '%d second', '%d seconds', $seconds, 'content-guard-pro' ), $seconds );
		} elseif ( $seconds < 3600 ) {
			$minutes = floor( $seconds / 60 );
			$secs    = $seconds % 60;
			/* translators: %d: number of minutes */
			return sprintf( _n( '%d minute', '%d minutes', $minutes, 'content-guard-pro' ), $minutes ) . ' ' . sprintf( _n( '%d second', '%d seconds', $secs, 'content-guard-pro' ), $secs );
		} else {
			$hours   = floor( $seconds / 3600 );
			$minutes = floor( ( $seconds % 3600 ) / 60 );
			/* translators: %d: number of hours */
			return sprintf( _n( '%d hour', '%d hours', $hours, 'content-guard-pro' ), $hours ) . ' ' . sprintf( _n( '%d minute', '%d minutes', $minutes, 'content-guard-pro' ), $minutes );
		}
	}

	/**
	 * Format finding for API response.
	 *
	 * @since 1.0.0
	 * @param object $finding         Finding object from database.
	 * @param bool   $include_excerpt Whether to include matched excerpt.
	 * @return array Formatted finding.
	 */
	private static function format_finding( $finding, $include_excerpt = false ) {
		$formatted = array(
			'id'          => absint( $finding->id ),
			'blog_id'     => absint( $finding->blog_id ),
			'object'      => array(
				'type'  => $finding->object_type,
				'id'    => absint( $finding->object_id ),
				'field' => $finding->field,
			),
			'fingerprint' => $finding->fingerprint,
			'rule_id'     => $finding->rule_id,
			'severity'    => $finding->severity,
			'confidence'  => absint( $finding->confidence ),
			'first_seen'  => mysql2date( 'c', $finding->first_seen, false ),
			'last_seen'   => mysql2date( 'c', $finding->last_seen, false ),
			'status'      => $finding->status,
		);

		// Include matched excerpt if requested.
		if ( $include_excerpt && isset( $finding->matched_excerpt ) ) {
			$formatted['matched_excerpt'] = $finding->matched_excerpt;
		}

		// Parse extra field (JSON).
		if ( ! empty( $finding->extra ) ) {
			$extra = json_decode( $finding->extra, true );
			if ( json_last_error() === JSON_ERROR_NONE ) {
				$formatted['extra'] = $extra;
			}
		}

		// Add action URLs.
		$formatted['actions'] = array(
			'review_url' => admin_url( 'admin.php?page=content-guard-pro-findings&finding=' . $finding->id ),
			'edit_url'   => self::get_edit_url( $finding ),
		);

		return $formatted;
	}

	/**
	 * Get edit URL for finding's object.
	 *
	 * @since 1.0.0
	 * @param object $finding Finding object.
	 * @return string|null Edit URL or null.
	 */
	private static function get_edit_url( $finding ) {
		if ( 'post' === $finding->object_type ) {
			return get_edit_post_link( $finding->object_id, 'raw' );
		}

		// For postmeta, link to the post edit page.
		if ( 'postmeta' === $finding->object_type ) {
			global $wpdb;
			$post_id = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT post_id FROM `" . $wpdb->postmeta . "` WHERE meta_id = %d",
					$finding->object_id
				)
			);
			if ( $post_id ) {
				return get_edit_post_link( $post_id, 'raw' );
			}
		}

		return null;
	}

	/**
	 * Quick scan post callback.
	 *
	 * POST /wp-json/content-guard-pro/v1/scan-post
	 *
	 * Performs a synchronous quick scan for immediate feedback (US-037, US-038).
	 * Should complete within 5 seconds per PRD requirements.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error Response object or error.
	 */
	public static function quick_scan_post_callback( $request ) {
		$post_id = $request->get_param( 'post_id' );
		$content = $request->get_param( 'content' );
		$manual  = $request->get_param( 'manual' );

		// Validate post exists.
		$post = get_post( $post_id );
		if ( ! $post ) {
			return new WP_Error(
				'content_guard_pro_post_not_found',
				__( 'Post not found', 'content-guard-pro' ),
				array( 'status' => 404 )
			);
		}

		// Check if real-time scanning is enabled (only for auto-scans, not manual).
		// Manual scans (user clicked "Scan" button) should always work.
		$settings = get_option( 'content_guard_pro_settings', array() );
		if ( ! $manual && isset( $settings['realtime_scan_enabled'] ) && ! $settings['realtime_scan_enabled'] ) {
			return rest_ensure_response(
				array(
					'success'  => true,
					'enabled'  => false,
					'message'  => __( 'Real-time scanning is disabled.', 'content-guard-pro' ),
					'findings' => array(),
				)
			);
		}

		// Set time limit for quick scan (5 seconds per PRD US-037).
		$start_time = microtime( true );
		$time_limit = 5;

		// Use provided content or get from post.
		$scan_content = ! empty( $content ) ? $content : $post->post_content;

		// Perform synchronous scan using detector.
		$findings     = array();
		$raw_findings = array(); // Store raw findings for selective auto-resolve logic.
		$error        = null;

		// Debug logging.
		cgp_log( 'Content Guard Pro: Quick scan starting for post ' . $post_id );
		cgp_log( 'Content Guard Pro: Content length: ' . strlen( $scan_content ) );

		try {
			if ( class_exists( 'CGP_Detector' ) ) {
				$detector     = new CGP_Detector();
				$raw_findings = $detector->scan_content( $scan_content, $post_id, 'post', 'post_content' );

				cgp_log( 'Content Guard Pro: Raw findings from content: ' . count( $raw_findings ) );

				// Also scan title and excerpt.
				$title_findings   = $detector->scan_content( $post->post_title, $post_id, 'post', 'post_title' );
				$excerpt_findings = $detector->scan_content( $post->post_excerpt, $post_id, 'post', 'post_excerpt' );

				cgp_log( 'Content Guard Pro: Title findings: ' . count( $title_findings ) . ', Excerpt findings: ' . count( $excerpt_findings ) );

				$raw_findings = array_merge( $raw_findings, $title_findings, $excerpt_findings );

				cgp_log( 'Content Guard Pro: Total raw findings: ' . count( $raw_findings ) );

				// Format findings for response and save to database.
				foreach ( $raw_findings as $finding ) {
					$findings[] = array(
						'rule_id'        => isset( $finding['rule_id'] ) ? $finding['rule_id'] : 'unknown',
						'severity'       => isset( $finding['severity'] ) ? $finding['severity'] : 'review',
						'confidence'     => isset( $finding['confidence'] ) ? absint( $finding['confidence'] ) : 0,
						'matched_excerpt' => isset( $finding['matched_excerpt'] ) ? wp_trim_words( $finding['matched_excerpt'], 20, '...' ) : '',
						'field'          => isset( $finding['field'] ) ? $finding['field'] : 'post_content',
						'description'    => self::get_rule_description( isset( $finding['rule_id'] ) ? $finding['rule_id'] : '' ),
					);

					// Save finding immediately so it appears on Findings page right away.
					// CGP_Scanner::save_finding handles deduplication via fingerprint.
					if ( class_exists( 'CGP_Scanner' ) ) {
						CGP_Scanner::save_finding( $finding );
					}
				}
			} else {
				cgp_log( 'Content Guard Pro: CGP_Detector class not found!' );
			}
		} catch ( Exception $e ) {
			$error = $e->getMessage();
			cgp_log( 'Content Guard Pro: Quick scan error - ' . $error );
		}

		cgp_log( 'Content Guard Pro: Final formatted findings: ' . count( $findings ) );

		$elapsed = microtime( true ) - $start_time;

		// Sort findings by severity.
		usort(
			$findings,
			function ( $a, $b ) {
				$severity_order = array(
					'critical'   => 0,
					'suspicious' => 1,
					'review'     => 2,
				);
				$a_order = isset( $severity_order[ $a['severity'] ] ) ? $severity_order[ $a['severity'] ] : 3;
				$b_order = isset( $severity_order[ $b['severity'] ] ) ? $severity_order[ $b['severity'] ] : 3;
				return $a_order - $b_order;
			}
		);

		// Count by severity.
		$counts = array(
			'critical'   => 0,
			'suspicious' => 0,
			'review'     => 0,
		);
		foreach ( $findings as $finding ) {
			if ( isset( $counts[ $finding['severity'] ] ) ) {
				$counts[ $finding['severity'] ]++;
			}
		}

		// Auto-resolve findings when scanning saved content.
		// Determine if we're scanning saved content or unsaved editor content:
		// - If content parameter is empty, we use saved post content -> safe to auto-resolve.
		// - If content parameter matches saved post content, it's saved content -> safe to auto-resolve.
		// - If content parameter differs from saved post content, it's unsaved editor content -> don't auto-resolve.
		$auto_resolved_count = 0;
		$is_scanning_saved_content = empty( $content ) || $content === $post->post_content;
		
		cgp_log( sprintf( 'Content Guard Pro: Auto-resolve check - content empty: %s, content matches saved: %s, is_scanning_saved: %s',
			empty( $content ) ? 'yes' : 'no',
			( ! empty( $content ) && $content === $post->post_content ) ? 'yes' : 'no',
			$is_scanning_saved_content ? 'yes' : 'no'
		) );

		if ( $is_scanning_saved_content && class_exists( 'CGP_Realtime_Scanner' ) ) {
			// Scanning saved content - safe to auto-resolve findings that are no longer detected.
			if ( empty( $findings ) ) {
				// No findings detected - resolve all open findings for this post.
				$auto_resolved_count = CGP_Realtime_Scanner::auto_resolve_findings( $post_id, 'manual_scan_saved_content' );
				if ( $auto_resolved_count > 0 ) {
					cgp_log( sprintf( 'Content Guard Pro: Auto-resolved %d findings for post %d via manual scan (no findings detected)', $auto_resolved_count, $post_id ) );
				}
			} else {
				// Some findings detected - resolve only findings that are no longer in scan results.
				// This handles the case where user removes one vulnerability but others remain.
				$auto_resolved_count = CGP_Realtime_Scanner::auto_resolve_missing_findings( $post_id, $raw_findings ?? array(), 'manual_scan_saved_content' );
				if ( $auto_resolved_count > 0 ) {
					cgp_log( sprintf( 'Content Guard Pro: Auto-resolved %d missing findings for post %d via manual scan (selective resolve)', $auto_resolved_count, $post_id ) );
				}
			}
		} elseif ( ! $is_scanning_saved_content ) {
			cgp_log( sprintf( 'Content Guard Pro: Skipping auto-resolve for post %d - scanning unsaved editor content', $post_id ) );
		}

		$response = array(
			'success'       => ( null === $error ),
			'enabled'       => true,
			'post_id'       => $post_id,
			'findings'      => $findings,
			'counts'        => $counts,
			'total'         => count( $findings ),
			'has_critical'  => $counts['critical'] > 0,
			'has_issues'    => count( $findings ) > 0,
			'elapsed_ms'    => round( $elapsed * 1000, 2 ),
			'timeout'       => $elapsed > $time_limit,
			'auto_resolved' => $auto_resolved_count,
		);

		// If scanning unsaved content, store content hash and findings for post-save auto-resolution.
		// This allows auto-resolution when user saves after scanning unsaved content.
		if ( ! $is_scanning_saved_content && ! empty( $content ) ) {
			$response['pending_auto_resolve'] = true;
			$response['scanned_content_hash'] = md5( $content );
			// Store raw findings for selective auto-resolution (handles partial fixes).
			if ( ! empty( $raw_findings ) ) {
				$response['scanned_raw_findings'] = $raw_findings;
			}
		}

		if ( $error ) {
			$response['error'] = $error;
		}

		// Store result in transient for post-save notice (30 seconds TTL).
		$transient_key = 'cgp_scan_result_' . $post_id . '_' . get_current_user_id();
		set_transient( $transient_key, $response, 30 );

		return rest_ensure_response( $response );
	}

	/**
	 * Get findings for a specific post.
	 *
	 * GET /wp-json/content-guard-pro/v1/post-findings/{post_id}
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public static function get_post_findings_callback( $request ) {
		global $wpdb;

		$post_id    = $request->get_param( 'post_id' );
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		// Get open findings for this post.
		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		$findings = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT * FROM `{$table_name}`
				WHERE object_type = 'post'
				AND object_id = %d
				AND status = 'open'
				ORDER BY severity ASC, confidence DESC",
				$post_id
			)
		);

		// Also check for quarantined findings.
		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		$quarantined_count = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$table_name}`
				WHERE object_type = 'post'
				AND object_id = %d
				AND status = 'quarantined'",
				$post_id
			)
		);

		$formatted_findings = array();
		foreach ( $findings as $finding ) {
			$formatted_findings[] = array(
				'id'              => absint( $finding->id ),
				'rule_id'         => $finding->rule_id,
				'severity'        => $finding->severity,
				'confidence'      => absint( $finding->confidence ),
				'field'           => $finding->field,
				'matched_excerpt' => isset( $finding->matched_excerpt ) ? wp_trim_words( $finding->matched_excerpt, 20, '...' ) : '',
				'first_seen'      => mysql2date( 'c', $finding->first_seen, false ),
				'status'          => $finding->status,
				'description'     => self::get_rule_description( $finding->rule_id ),
			);
		}

		// Count by severity.
		$counts = array(
			'critical'   => 0,
			'suspicious' => 0,
			'review'     => 0,
		);
		foreach ( $formatted_findings as $finding ) {
			if ( isset( $counts[ $finding['severity'] ] ) ) {
				$counts[ $finding['severity'] ]++;
			}
		}

		return rest_ensure_response(
			array(
				'post_id'         => $post_id,
				'findings'        => $formatted_findings,
				'counts'          => $counts,
				'total'           => count( $formatted_findings ),
				'has_critical'    => $counts['critical'] > 0,
				'has_issues'      => count( $formatted_findings ) > 0,
				'quarantined_count' => absint( $quarantined_count ),
				'has_quarantined' => absint( $quarantined_count ) > 0,
			)
		);
	}

	/**
	 * Get human-readable description for a rule ID.
	 *
	 * @since 1.0.0
	 * @param string $rule_id Rule identifier.
	 * @return string Human-readable description.
	 */
	private static function get_rule_description( $rule_id ) {
		$descriptions = array(
			'url_shortener'            => __( 'URL shortener detected - may be used to hide malicious destinations', 'content-guard-pro' ),
			'hidden_external_link'     => __( 'Hidden element contains external link - potential SEO spam', 'content-guard-pro' ),
			'hidden_external_content'  => __( 'Hidden element contains suspicious external content', 'content-guard-pro' ),
			'external_script'          => __( 'External script from non-allowlisted domain', 'content-guard-pro' ),
			'external_iframe'          => __( 'External iframe from non-allowlisted domain', 'content-guard-pro' ),
			'obfuscated_js'            => __( 'Obfuscated JavaScript detected - may hide malicious code', 'content-guard-pro' ),
			'seo_spam'                 => __( 'SEO spam keywords detected', 'content-guard-pro' ),
			'seo_spam_pharma'          => __( 'Pharmaceutical spam keywords detected', 'content-guard-pro' ),
			'seo_spam_casino'          => __( 'Gambling/casino spam keywords detected', 'content-guard-pro' ),
			'seo_spam_adult'           => __( 'Adult content spam keywords detected', 'content-guard-pro' ),
			'inline_event_handler'     => __( 'Inline JavaScript event handler detected', 'content-guard-pro' ),
			'document_write'           => __( 'document.write() call detected - may inject malicious content', 'content-guard-pro' ),
			'javascript_uri'           => __( 'JavaScript URI detected - potential XSS vector', 'content-guard-pro' ),
			'object_embed'             => __( 'Object/Embed tag with external source detected', 'content-guard-pro' ),
			'meta_refresh'             => __( 'Meta refresh redirect detected - may redirect to malicious site', 'content-guard-pro' ),
			'php_function'             => __( 'Dangerous PHP function pattern in serialized data', 'content-guard-pro' ),
			'css_cloaking'             => __( 'CSS cloaking technique detected - may hide malicious content', 'content-guard-pro' ),
			'svg_script'               => __( 'SVG with embedded script detected', 'content-guard-pro' ),
			'crypto_miner'             => __( 'Cryptocurrency miner script detected', 'content-guard-pro' ),
			'js_redirect'              => __( 'JavaScript redirect detected - may redirect visitors', 'content-guard-pro' ),
			'anomalous_link_profile'   => __( 'Unusual number of external links detected', 'content-guard-pro' ),
			'reputation_hit'           => __( 'URL flagged by reputation service (Safe Browsing/PhishTank)', 'content-guard-pro' ),
		);

		return isset( $descriptions[ $rule_id ] ) ? $descriptions[ $rule_id ] : __( 'Potential security issue detected', 'content-guard-pro' );
	}
}

