<?php
/**
 * Real-time Scanner Class
 *
 * Monitors content changes in real-time and triggers on-save scans
 * when posts are created, updated, or published (PRD Section 3.1).
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Realtime_Scanner
 *
 * Hooks into WordPress save actions to detect content changes and
 * trigger high-priority on-save scans via the scheduler.
 *
 * Per PRD Section 3.1:
 * - Enqueue on save_post
 * - Enqueue on transition_post_status (Gutenberg)
 * - Quick validation before publishing
 * - Content change detection
 *
 * @since 1.0.0
 */
class CGP_Realtime_Scanner {

	/**
	 * Scan mode for on-save scans.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const SCAN_MODE = 'quick';

	/**
	 * Minimum time between scans for the same post (in seconds).
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const SCAN_THROTTLE = 60;

	/**
	 * Initialize the real-time scanner.
	 *
	 * @since 1.0.0
	 */
	public static function init() {
		$instance = new self();
		$instance->register_hooks();
	}

	/**
	 * Register WordPress hooks.
	 *
	 * @since 1.0.0
	 */
	private function register_hooks() {
		// Hook into save_post (PRD Section 3.1).
		add_action( 'save_post', array( $this, 'on_post_save' ), 20, 3 );

		// Hook into transition_post_status for Gutenberg (PRD Section 3.1).
		add_action( 'transition_post_status', array( $this, 'on_post_status_change' ), 10, 3 );

		// Hook into post updates (additional safety net).
		add_action( 'post_updated', array( $this, 'on_post_updated' ), 10, 3 );

		// Settings hook to enable/disable real-time scanning.
		add_filter( 'content_guard_pro_realtime_scan_enabled', array( $this, 'is_realtime_scan_enabled' ) );
	}

	/**
	 * Handle post save action.
	 *
	 * Triggered when a post is saved. Performs a synchronous quick scan for immediate
	 * feedback (US-037, US-038) and schedules a background scan for deeper analysis.
	 *
	 * @since 1.0.0
	 * @param int     $post_id Post ID.
	 * @param WP_Post $post    Post object.
	 * @param bool    $update  Whether this is an update.
	 */
	public function on_post_save( $post_id, $post, $update ) {
		$this->log_debug( sprintf( 'on_post_save hook triggered for post %d (update: %s)', $post_id, $update ? 'yes' : 'no' ) );
		
		// IMPORTANT: Skip auto-resolution if we're in the middle of a batch scan.
		// Batch scans should NOT trigger auto-resolution - they're for discovery, not for resolving findings.
		// Only post-save scans and manual single-post scans should trigger auto-resolution.
		if ( defined( 'CGP_BATCH_SCAN_IN_PROGRESS' ) && CGP_BATCH_SCAN_IN_PROGRESS ) {
			$this->log_debug( sprintf( 'Skipping auto-resolution for post %d - batch scan in progress', $post_id ) );
			return;
		}
		
		// Check if real-time scanning is enabled.
		$realtime_enabled = apply_filters( 'content_guard_pro_realtime_scan_enabled', true );
		
		// Even if real-time scanning is disabled, check for pending auto-resolution from manual scan.
		// This handles the case where user scanned unsaved content, then saved.
		if ( ! $realtime_enabled ) {
			$transient_key = 'cgp_scan_result_' . $post_id . '_' . get_current_user_id();
			$scan_result = get_transient( $transient_key );
			
			if ( $scan_result && is_array( $scan_result ) && ! empty( $scan_result['pending_auto_resolve'] ) && ! empty( $scan_result['scanned_content_hash'] ) ) {
				// User scanned unsaved content - check if saved content matches.
				$saved_content_hash = md5( $post->post_content );
				if ( $saved_content_hash === $scan_result['scanned_content_hash'] ) {
					// Saved content matches scanned content - safe to auto-resolve.
					$this->log_debug( sprintf( 'Auto-resolve: Post-save content matches scanned unsaved content for post %d - resolving findings', $post_id ) );
					
					if ( empty( $scan_result['scanned_raw_findings'] ) || ( isset( $scan_result['has_issues'] ) && ! $scan_result['has_issues'] ) ) {
						// No findings detected - resolve all open findings.
						$auto_resolved_count = self::auto_resolve_findings( $post_id, 'manual_scan_saved_content' );
						if ( $auto_resolved_count > 0 ) {
							$this->log_debug( sprintf( 'Auto-resolved %d findings for post %d after save (from manual scan of unsaved content - no findings)', $auto_resolved_count, $post_id ) );
						}
					} elseif ( ! empty( $scan_result['scanned_raw_findings'] ) ) {
						// Some findings detected - selectively resolve findings that are no longer detected.
						$auto_resolved_count = self::auto_resolve_missing_findings( $post_id, $scan_result['scanned_raw_findings'], 'manual_scan_saved_content' );
						if ( $auto_resolved_count > 0 ) {
							$this->log_debug( sprintf( 'Selectively auto-resolved %d missing findings for post %d after save (from manual scan of unsaved content)', $auto_resolved_count, $post_id ) );
						}
					}
					
					// Clear the transient since we've processed it.
					delete_transient( $transient_key );
				} else {
					$this->log_debug( sprintf( 'Auto-resolve: Saved content does not match scanned content for post %d - skipping (content changed after scan)', $post_id ) );
				}
			}
			
			$this->log_debug( sprintf( 'Skipping scan for post %d - real-time scanning disabled', $post_id ) );
			return;
		}

		// Validate post object.
		if ( ! $post instanceof WP_Post ) {
			$this->log_debug( sprintf( 'Skipping scan for post %d - invalid post object', $post_id ) );
			return;
		}

		// Bail if autosave.
		if ( wp_is_post_autosave( $post_id ) ) {
			$this->log_debug( sprintf( 'Skipping scan for post %d - autosave', $post_id ) );
			return;
		}

		// Bail if revision.
		if ( wp_is_post_revision( $post_id ) ) {
			$this->log_debug( sprintf( 'Skipping scan for post %d - revision', $post_id ) );
			return;
		}

		// Check if post type should be scanned.
		if ( ! $this->should_scan_post_type( $post->post_type ) ) {
			$this->log_debug( sprintf( 'Skipping scan for post %d - post type %s not scannable', $post_id, $post->post_type ) );
			return;
		}

		// Check if post status should be scanned.
		if ( ! $this->should_scan_post_status( $post->post_status ) ) {
			$this->log_debug( sprintf( 'Skipping scan for post %d - post status %s not scannable', $post_id, $post->post_status ) );
			return;
		}

		// On update, check if content has actually changed.
		if ( $update && ! $this->has_content_changed( $post_id ) ) {
			$this->log_debug( sprintf( 'Skipping scan for post %d - content unchanged', $post_id ) );
			return;
		}

		// Check throttle - prevent scanning too frequently.
		if ( $this->is_scan_throttled( $post_id ) ) {
			$this->log_debug( sprintf( 'Skipping scan for post %d - throttled', $post_id ) );
			return;
		}

		// Perform synchronous quick scan for immediate feedback (US-037, US-038).
		// This runs in the same request as the save for immediate UI feedback.
		$this->log_debug( sprintf( 'on_post_save: Starting scan for post %d', $post_id ) );
		$scan_result = $this->perform_synchronous_scan( $post );
		$this->log_debug( sprintf( 'on_post_save: Scan completed for post %d, result type: %s', $post_id, gettype( $scan_result ) ) );
		if ( is_array( $scan_result ) ) {
			$this->log_debug( sprintf( 'on_post_save: Scan result for post %d - total: %d, has_issues: %s, findings count: %d', 
				$post_id, 
				isset( $scan_result['total'] ) ? $scan_result['total'] : 'N/A',
				isset( $scan_result['has_issues'] ) ? ( $scan_result['has_issues'] ? 'true' : 'false' ) : 'N/A',
				isset( $scan_result['findings'] ) ? count( $scan_result['findings'] ) : 0
			) );
		}

		// Auto-resolve: If scan found no issues on SAVE, mark existing open findings as resolved.
		// This is safe because we're scanning the actual saved content, not unsaved editor content.
		$auto_resolved_count = 0;
		
		// Check if scan completed successfully and found no issues.
		// Handle both array results and check multiple conditions to be safe.
		$has_no_findings = false;
		if ( $scan_result && is_array( $scan_result ) ) {
			// Check if findings array is empty or total is 0.
			$has_no_findings = (
				( isset( $scan_result['findings'] ) && empty( $scan_result['findings'] ) ) ||
				( isset( $scan_result['total'] ) && 0 === absint( $scan_result['total'] ) ) ||
				( isset( $scan_result['has_issues'] ) && false === $scan_result['has_issues'] )
			);
		}
		
		if ( $has_no_findings ) {
			// Scan found NO issues - resolve ALL open findings for this post.
			$this->log_debug( sprintf( 'Auto-resolve: Scan found no issues for post %d, checking for existing findings to resolve', $post_id ) );
			$auto_resolved_count = self::auto_resolve_findings( $post_id, 'auto_scan_on_save' );
			if ( $auto_resolved_count > 0 ) {
				$this->log_debug( sprintf( 'Auto-resolved %d findings for post %d on save', $auto_resolved_count, $post_id ) );
			} else {
				$this->log_debug( sprintf( 'Auto-resolve: No open findings found to resolve for post %d', $post_id ) );
			}
		} elseif ( $scan_result && is_array( $scan_result ) && isset( $scan_result['raw_findings'] ) ) {
			// Scan found SOME issues - selectively resolve findings that are no longer detected.
			// This handles partial fixes (e.g., user removed vulnerability A but B and C remain).
			$raw_findings = $scan_result['raw_findings'];
			$findings_count = isset( $scan_result['total'] ) ? $scan_result['total'] : count( $raw_findings );
			$this->log_debug( sprintf( 'Auto-resolve: Scan found %d issues for post %d, checking for removed findings to selectively resolve', $findings_count, $post_id ) );
			
			$auto_resolved_count = self::auto_resolve_missing_findings( $post_id, $raw_findings, 'auto_scan_on_save' );
			if ( $auto_resolved_count > 0 ) {
				$this->log_debug( sprintf( 'Selectively auto-resolved %d missing findings for post %d on save', $auto_resolved_count, $post_id ) );
			} else {
				$this->log_debug( sprintf( 'Auto-resolve: No missing findings to resolve for post %d', $post_id ) );
			}
		} else {
			if ( $scan_result && is_array( $scan_result ) ) {
				$findings_count = isset( $scan_result['total'] ) ? $scan_result['total'] : ( isset( $scan_result['findings'] ) ? count( $scan_result['findings'] ) : 'unknown' );
				$this->log_debug( sprintf( 'Auto-resolve: Could not process for post %d - scan found %s issues but no raw_findings available', $post_id, $findings_count ) );
			} else {
				$this->log_debug( sprintf( 'Auto-resolve: Skipped for post %d - scan result invalid or scan failed', $post_id ) );
			}
		}

		// Store result in transient for post-save admin notice display.
		if ( $scan_result ) {
			$scan_result['auto_resolved'] = $auto_resolved_count;
			$transient_key = 'cgp_scan_result_' . $post_id . '_' . get_current_user_id();
			set_transient( $transient_key, $scan_result, 30 ); // 30 seconds TTL.
			
			$this->log_debug( sprintf(
				'Stored scan result for post %d: %d findings, %d auto-resolved',
				$post_id,
				isset( $scan_result['total'] ) ? $scan_result['total'] : 0,
				$auto_resolved_count
			) );
		}

		// Also schedule a background scan for deeper analysis (async).
		$this->schedule_post_scan( $post_id, 'save_post' );
	}

	/**
	 * Perform synchronous quick scan for immediate feedback.
	 *
	 * Runs a quick scan during the save request to provide immediate
	 * feedback to the user. Should complete within 5 seconds per US-037.
	 *
	 * @since 1.0.0
	 * @param WP_Post $post Post object to scan.
	 * @return array|false Scan result array or false on failure.
	 */
	private function perform_synchronous_scan( $post ) {
		// Set time limit for quick scan (5 seconds per PRD US-037).
		$start_time = microtime( true );
		$time_limit = 5;

		$findings = array();
		$error    = null;

		try {
			if ( class_exists( 'CGP_Detector' ) ) {
				$detector = new CGP_Detector();

				// Scan post content.
				$content_findings = $detector->scan_content( $post->post_content, $post->ID, 'post', 'post_content' );

				// Scan title.
				$title_findings = $detector->scan_content( $post->post_title, $post->ID, 'post', 'post_title' );

				// Scan excerpt.
				$excerpt_findings = $detector->scan_content( $post->post_excerpt, $post->ID, 'post', 'post_excerpt' );

				// Merge findings.
				$raw_findings = array_merge( $content_findings, $title_findings, $excerpt_findings );

				// Format findings.
				foreach ( $raw_findings as $finding ) {
					$findings[] = array(
						'rule_id'        => isset( $finding['rule_id'] ) ? $finding['rule_id'] : 'unknown',
						'severity'       => isset( $finding['severity'] ) ? $finding['severity'] : 'review',
						'confidence'     => isset( $finding['confidence'] ) ? absint( $finding['confidence'] ) : 0,
						'matched_excerpt' => isset( $finding['matched_excerpt'] ) ? wp_trim_words( $finding['matched_excerpt'], 20, '...' ) : '',
						'field'          => isset( $finding['field'] ) ? $finding['field'] : 'post_content',
						'description'    => $this->get_rule_description( isset( $finding['rule_id'] ) ? $finding['rule_id'] : '' ),
					);
				}

				// Save findings immediately so they appear on Findings page right away.
				// CGP_Scanner::save_finding handles deduplication via fingerprint.
				if ( ! empty( $raw_findings ) && class_exists( 'CGP_Scanner' ) ) {
					foreach ( $raw_findings as $finding ) {
						CGP_Scanner::save_finding( $finding );
					}
				}
			} else {
				$this->log_error( 'CGP_Detector class not available for synchronous scan' );
				return false;
			}
		} catch ( Exception $e ) {
			$error = $e->getMessage();
			$this->log_error( 'Synchronous scan error: ' . $error );
			// Fail-open: don't block on errors.
			return false;
		}

		$elapsed = microtime( true ) - $start_time;

		// Sort findings by severity.
		usort(
			$findings,
			function ( $a, $b ) {
				$severity_order = array(
					'critical'   => 0,
					'suspicious' => 1,
					'review'     => 2,
				);
				$a_order = isset( $severity_order[ $a['severity'] ] ) ? $severity_order[ $a['severity'] ] : 3;
				$b_order = isset( $severity_order[ $b['severity'] ] ) ? $severity_order[ $b['severity'] ] : 3;
				return $a_order - $b_order;
			}
		);

		// Count by severity.
		$counts = array(
			'critical'   => 0,
			'suspicious' => 0,
			'review'     => 0,
		);
		foreach ( $findings as $finding ) {
			if ( isset( $counts[ $finding['severity'] ] ) ) {
				$counts[ $finding['severity'] ]++;
			}
		}

		return array(
			'success'       => true,
			'post_id'       => $post->ID,
			'findings'      => $findings,
			'raw_findings'  => isset( $raw_findings ) ? $raw_findings : array(), // For auto-resolve logic.
			'counts'        => $counts,
			'total'         => count( $findings ),
			'has_critical'  => $counts['critical'] > 0,
			'has_issues'    => count( $findings ) > 0,
			'elapsed_ms'    => round( $elapsed * 1000, 2 ),
			'timeout'       => $elapsed > $time_limit,
		);
	}

	/**
	 * Get human-readable description for a rule ID.
	 *
	 * @since 1.0.0
	 * @param string $rule_id Rule identifier.
	 * @return string Human-readable description.
	 */
	private function get_rule_description( $rule_id ) {
		$descriptions = array(
			'url_shortener'            => __( 'URL shortener detected - may hide malicious destinations', 'content-guard-pro' ),
			'hidden_external_link'     => __( 'Hidden element contains external link - potential SEO spam', 'content-guard-pro' ),
			'hidden_external_content'  => __( 'Hidden element contains suspicious external content', 'content-guard-pro' ),
			'external_script'          => __( 'External script from non-allowlisted domain', 'content-guard-pro' ),
			'external_iframe'          => __( 'External iframe from non-allowlisted domain', 'content-guard-pro' ),
			'obfuscated_js'            => __( 'Obfuscated JavaScript detected - may hide malicious code', 'content-guard-pro' ),
			'seo_spam'                 => __( 'SEO spam keywords detected', 'content-guard-pro' ),
			'seo_spam_pharma'          => __( 'Pharmaceutical spam keywords detected', 'content-guard-pro' ),
			'seo_spam_casino'          => __( 'Gambling/casino spam keywords detected', 'content-guard-pro' ),
			'seo_spam_adult'           => __( 'Adult content spam keywords detected', 'content-guard-pro' ),
			'inline_event_handler'     => __( 'Inline JavaScript event handler detected', 'content-guard-pro' ),
			'document_write'           => __( 'document.write() call detected - may inject malicious content', 'content-guard-pro' ),
			'javascript_uri'           => __( 'JavaScript URI detected - potential XSS vector', 'content-guard-pro' ),
			'object_embed'             => __( 'Object/Embed tag with external source detected', 'content-guard-pro' ),
			'meta_refresh'             => __( 'Meta refresh redirect detected - may redirect to malicious site', 'content-guard-pro' ),
			'php_function'             => __( 'Dangerous PHP function pattern in serialized data', 'content-guard-pro' ),
			'css_cloaking'             => __( 'CSS cloaking technique detected - may hide malicious content', 'content-guard-pro' ),
			'svg_script'               => __( 'SVG with embedded script detected', 'content-guard-pro' ),
			'crypto_miner'             => __( 'Cryptocurrency miner script detected', 'content-guard-pro' ),
			'js_redirect'              => __( 'JavaScript redirect detected - may redirect visitors', 'content-guard-pro' ),
			'anomalous_link_profile'   => __( 'Unusual number of external links detected', 'content-guard-pro' ),
			'reputation_hit'           => __( 'URL flagged by reputation service (Safe Browsing/PhishTank)', 'content-guard-pro' ),
		);

		return isset( $descriptions[ $rule_id ] ) ? $descriptions[ $rule_id ] : __( 'Potential security issue detected', 'content-guard-pro' );
	}

	/**
	 * Handle post status transition.
	 *
	 * Triggered when post status changes. Particularly important for
	 * Gutenberg which uses this for publishing.
	 *
	 * @since 1.0.0
	 * @param string  $new_status New post status.
	 * @param string  $old_status Old post status.
	 * @param WP_Post $post       Post object.
	 */
	public function on_post_status_change( $new_status, $old_status, $post ) {
		// Check if real-time scanning is enabled.
		if ( ! apply_filters( 'content_guard_pro_realtime_scan_enabled', true ) ) {
			return;
		}

		// Validate post object.
		if ( ! $post instanceof WP_Post ) {
			return;
		}

		// Priority: Scan when transitioning TO publish.
		if ( 'publish' === $new_status && 'publish' !== $old_status ) {
			// Check if post type should be scanned.
			if ( ! $this->should_scan_post_type( $post->post_type ) ) {
				return;
			}

			// Check throttle.
			if ( $this->is_scan_throttled( $post->ID ) ) {
				$this->log_debug( sprintf( 'Skipping scan for post %d - throttled (status change)', $post->ID ) );
				return;
			}

			// Schedule high-priority scan before publishing.
			$this->schedule_post_scan( $post->ID, 'publish', true );
		}

		// Also scan when moving from trash (content might have been modified).
		if ( 'trash' === $old_status && in_array( $new_status, array( 'publish', 'draft', 'pending' ), true ) ) {
			if ( $this->should_scan_post_type( $post->post_type ) ) {
				$this->schedule_post_scan( $post->ID, 'untrash' );
			}
		}
	}

	/**
	 * Handle post updated action.
	 *
	 * Additional safety net for catching content changes.
	 *
	 * @since 1.0.0
	 * @param int     $post_id      Post ID.
	 * @param WP_Post $post_after   Post object after update.
	 * @param WP_Post $post_before  Post object before update.
	 */
	public function on_post_updated( $post_id, $post_after, $post_before ) {
		// Validate post objects.
		if ( ! $post_after instanceof WP_Post || ! $post_before instanceof WP_Post ) {
			return;
		}

		// This is a safety net - only trigger if content actually changed
		// and we haven't already scanned via save_post.
		// Note: This method primarily serves as a logging mechanism.
		// Actual scanning is triggered by save_post and transition_post_status hooks.
		if ( $post_after->post_content !== $post_before->post_content ) {
			// Check if we already scheduled a scan recently.
			if ( ! $this->is_scan_throttled( $post_id ) ) {
				$this->log_debug( sprintf( 'Content change detected for post %d via post_updated', $post_id ) );
			}
		}
	}

	/**
	 * Schedule a scan for a specific post.
	 *
	 * Delegates to CGP_Scheduler::schedule_on_save_scan().
	 *
	 * @since 1.0.0
	 * @param int    $post_id      Post ID to scan.
	 * @param string $trigger      What triggered the scan (save_post, publish, etc.).
	 * @param bool   $high_priority Whether this is high priority (default false).
	 * @return int|false Action ID on success, false on failure.
	 */
	private function schedule_post_scan( $post_id, $trigger = 'save_post', $high_priority = false ) {
		// Validate post ID.
		$post_id = absint( $post_id );
		if ( ! $post_id ) {
			$this->log_error( 'Invalid post ID provided to schedule_post_scan' );
			return false;
		}

		// Ensure scheduler is available.
		if ( ! class_exists( 'CGP_Scheduler' ) ) {
			$this->log_error( 'CGP_Scheduler class not available' );
			return false;
		}

		// Schedule the scan via the scheduler.
		$action_id = CGP_Scheduler::schedule_on_save_scan( $post_id );

		if ( $action_id ) {
			// Set throttle to prevent immediate re-scan.
			$this->set_scan_throttle( $post_id );

			// Log the scheduled scan.
			$this->log_debug(
				sprintf(
					'Scheduled on-save scan for post %d (trigger: %s, action_id: %d, priority: %s)',
					$post_id,
					$trigger,
					$action_id,
					$high_priority ? 'high' : 'normal'
				)
			);

			/**
			 * Action hook after scheduling a real-time scan.
			 *
			 * @since 1.0.0
			 * @param int    $post_id   Post ID.
			 * @param string $trigger   What triggered the scan.
			 * @param int    $action_id Action Scheduler action ID.
			 */
			do_action( 'content_guard_pro_realtime_scan_scheduled', $post_id, $trigger, $action_id );

			return $action_id;
		} else {
			$this->log_error( sprintf( 'Failed to schedule scan for post %d', $post_id ) );
			return false;
		}
	}

	/**
	 * Check if a post type should be scanned.
	 *
	 * @since 1.0.0
	 * @param string $post_type Post type.
	 * @return bool True if should scan, false otherwise.
	 */
	private function should_scan_post_type( $post_type ) {
		// Default scannable post types.
		$default_types = array( 'post', 'page' );

		/**
		 * Filter the post types that should be scanned in real-time.
		 *
		 * @since 1.0.0
		 * @param array $post_types Array of post type slugs.
		 */
		$scannable_types = apply_filters( 'content_guard_pro_realtime_scan_post_types', $default_types );

		return in_array( $post_type, $scannable_types, true );
	}

	/**
	 * Check if a post status should trigger scanning.
	 *
	 * @since 1.0.0
	 * @param string $post_status Post status.
	 * @return bool True if should scan, false otherwise.
	 */
	private function should_scan_post_status( $post_status ) {
		// Default statuses to scan.
		$default_statuses = array( 'publish', 'draft', 'pending', 'private', 'future' );

		/**
		 * Filter the post statuses that should trigger real-time scans.
		 *
		 * @since 1.0.0
		 * @param array $statuses Array of post status slugs.
		 */
		$scannable_statuses = apply_filters( 'content_guard_pro_realtime_scan_post_statuses', $default_statuses );

		return in_array( $post_status, $scannable_statuses, true );
	}

	/**
	 * Check if post content has changed.
	 *
	 * Compares current post content with the previous revision to detect changes.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if content changed, false otherwise.
	 */
	private function has_content_changed( $post_id ) {
		// Validate post ID.
		$post_id = absint( $post_id );
		if ( ! $post_id ) {
			return false;
		}

		// Get current post.
		$post = get_post( $post_id );

		if ( ! $post ) {
			return false;
		}

		// Get previous revision.
		$revisions = wp_get_post_revisions(
			$post_id,
			array(
				'posts_per_page' => 1,
				'fields'         => 'ids',
			)
		);

		// If no revisions, consider it changed (new post or first save).
		if ( empty( $revisions ) ) {
			return true;
		}

		$previous_revision_id = array_shift( $revisions );
		$previous_revision = get_post( $previous_revision_id );

		if ( ! $previous_revision ) {
			return true;
		}

		// Compare content.
		$content_changed = ( $post->post_content !== $previous_revision->post_content );
		$title_changed = ( $post->post_title !== $previous_revision->post_title );
		$excerpt_changed = ( $post->post_excerpt !== $previous_revision->post_excerpt );

		// Consider it changed if any of these fields changed.
		return ( $content_changed || $title_changed || $excerpt_changed );
	}

	/**
	 * Check if a post scan is currently throttled.
	 *
	 * Prevents scanning the same post too frequently.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if throttled, false otherwise.
	 */
	private function is_scan_throttled( $post_id ) {
		$post_id = absint( $post_id );
		if ( ! $post_id ) {
			return false;
		}

		$throttle_key = 'content_guard_pro_scan_throttle_' . $post_id;
		$last_scan = get_transient( $throttle_key );

		return ( false !== $last_scan );
	}

	/**
	 * Set scan throttle for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 */
	private function set_scan_throttle( $post_id ) {
		$post_id = absint( $post_id );
		if ( ! $post_id ) {
			return;
		}

		$throttle_key = 'content_guard_pro_scan_throttle_' . $post_id;
		set_transient( $throttle_key, time(), self::SCAN_THROTTLE );
	}

	/**
	 * Check if real-time scanning is enabled.
	 *
	 * @since 1.0.0
	 * @param bool $enabled Default enabled state.
	 * @return bool True if enabled, false otherwise.
	 */
	public function is_realtime_scan_enabled( $enabled = true ) {
		// Get settings.
		$settings = get_option( 'content_guard_pro_settings', array() );

		// License guard: Single post scanning is available to all tiers (including free).
		// Full site on-save scanning (background, scheduled) requires paid tier.
		// For single post scans, check single_post_scanning capability.
		// For full site scans, check on_save_scanning capability.
		// Since this method is called for single post on-save scans, we check single_post_scanning.
		if ( class_exists( 'CGP_License_Manager' ) ) {
			// Single post scanning is available to all tiers.
			if ( ! CGP_License_Manager::can( 'single_post_scanning' ) ) {
				return false;
			}
		}

		// Check if real-time scanning is explicitly disabled in settings.
		if ( isset( $settings['realtime_scan_enabled'] ) ) {
			return (bool) $settings['realtime_scan_enabled'];
		}

		// Default to enabled.
		return $enabled;
	}

	/**
	 * Log debug message.
	 *
	 * @since 1.0.0
	 * @param string $message Debug message.
	 */
	private function log_debug( $message ) {
		$full_message = 'Content Guard Pro [Realtime Scanner]: ' . $message;
		cgp_log( $full_message );
		// Also log directly to error_log for debugging (when WP_DEBUG_LOG is enabled).
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( $full_message );
		}
	}

	/**
	 * Log error message.
	 *
	 * @since 1.0.0
	 * @param string $message Error message.
	 */
	private function log_error( $message ) {
		$full_message = 'Content Guard Pro [Realtime Scanner ERROR]: ' . $message;
		cgp_log( $full_message );
		// Also log directly to error_log for debugging (when WP_DEBUG_LOG is enabled).
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( $full_message );
		}
	}

	/**
	 * Get real-time scan statistics.
	 *
	 * @since 1.0.0
	 * @return array Statistics.
	 */
	public static function get_stats() {
		// Get settings.
		$settings = get_option( 'content_guard_pro_settings', array() );

		return array(
			'enabled'           => isset( $settings['realtime_scan_enabled'] ) ? (bool) $settings['realtime_scan_enabled'] : true,
			'scan_mode'         => self::SCAN_MODE,
			'throttle_seconds'  => self::SCAN_THROTTLE,
			'scannable_types'   => apply_filters( 'content_guard_pro_realtime_scan_post_types', array( 'post', 'page' ) ),
			'scannable_statuses' => apply_filters( 'content_guard_pro_realtime_scan_post_statuses', array( 'publish', 'draft', 'pending', 'private', 'future' ) ),
		);
	}

	/**
	 * Manually trigger a scan for a post.
	 *
	 * Public method for programmatically triggering scans.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return int|false Action ID on success, false on failure.
	 */
	public static function trigger_manual_scan( $post_id ) {
		// Validate post ID.
		$post_id = absint( $post_id );
		if ( ! $post_id ) {
			return false;
		}

		$instance = new self();
		return $instance->schedule_post_scan( $post_id, 'manual', true );
	}

	/**
	 * Auto-resolve open findings for a post when scan finds no issues.
	 *
	 * When a scan completes with no findings, any existing 'open' findings for
	 * that post are automatically marked as 'resolved'.
	 *
	 * **Important:** Only call this when scanning SAVED content (not unsaved
	 * editor content) to prevent the edge case where user scans unsaved content,
	 * sees "resolved", but doesn't save - leaving the vulnerability in the database.
	 *
	 * Findings are not deleted - status is changed to 'resolved' with metadata
	 * indicating they were auto-resolved, preserving audit trail.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID to auto-resolve findings for.
	 * @param string $method Resolution method identifier (default: 'auto_scan_on_save').
	 * @return int Number of findings that were auto-resolved.
	 */
	public static function auto_resolve_findings( $post_id, $method = 'auto_scan_on_save' ) {
		global $wpdb;

		$post_id    = absint( $post_id );
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		// Get count of open findings for this post before updating.
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$open_count = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$table_name}`
				WHERE object_type = 'post'
				AND object_id = %d
				AND status = 'open'",
				$post_id
			)
		);

		if ( ! $open_count || $open_count < 1 ) {
			return 0;
		}

		// Update all open findings for this post to resolved.
		// Store metadata about auto-resolution in the extra field.
		$current_time = current_time( 'mysql' );
		$resolved_by  = wp_json_encode(
			array(
				'method'    => $method,
				'timestamp' => $current_time,
				'user_id'   => get_current_user_id(),
			)
		);

		// Log before update for debugging (only if WP_DEBUG is enabled).
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( sprintf(
				'Content Guard Pro [Realtime Scanner]: Auto-resolve: Attempting to update findings for post %d (object_type=post, status=open, method=%s)',
				$post_id,
				$method
			) );
		}

		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$updated = $wpdb->query(
			$wpdb->prepare(
				"UPDATE `{$table_name}`
				SET status = 'resolved',
				    last_seen = %s,
				    extra = JSON_SET(COALESCE(extra, '{}'), '$.resolved_by', %s)
				WHERE object_type = 'post'
				AND object_id = %d
				AND status = 'open'",
				$current_time,
				$resolved_by,
				$post_id
			)
		);

		// Log result for debugging (only if WP_DEBUG is enabled).
		if ( false === $updated ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( sprintf(
					'Content Guard Pro [Realtime Scanner ERROR]: Auto-resolve: Database update failed for post %d - %s',
					$post_id,
					$wpdb->last_error
				) );
			}
			return 0;
		}

		$updated_count = absint( $updated );
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( sprintf(
				'Content Guard Pro [Realtime Scanner]: Auto-resolve: Successfully updated %d findings for post %d (method=%s)',
				$updated_count,
				$post_id,
				$method
			) );
		}

		// Verify the update actually worked by checking if any open findings remain.
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$remaining_open = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$table_name}`
				WHERE object_type = 'post'
				AND object_id = %d
				AND status = 'open'",
				$post_id
			)
		);

		if ( $remaining_open > 0 && defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( sprintf(
				'Content Guard Pro [Realtime Scanner]: Auto-resolve: Warning - %d open findings still remain for post %d after update',
				$remaining_open,
				$post_id
			) );
		}

		return $updated_count;
	}

	/**
	 * Auto-resolve findings that are no longer detected in scan results.
	 *
	 * Compares scan result fingerprints with existing open findings for a post.
	 * If an existing finding's fingerprint is NOT in the scan results, it means
	 * the vulnerability has been removed and should be auto-resolved.
	 *
	 * **Important:** Only call this when scanning SAVED content (not unsaved
	 * editor content) to prevent false positives.
	 *
	 * @since 1.0.0
	 * @param int   $post_id Post ID.
	 * @param array $scan_findings Array of finding data from scan (must include fingerprint or data to generate it).
	 * @param string $method Resolution method identifier (default: 'manual_scan_saved_content').
	 * @return int Number of findings that were auto-resolved.
	 */
	public static function auto_resolve_missing_findings( $post_id, $scan_findings, $method = 'manual_scan_saved_content' ) {
		global $wpdb;

		$post_id    = absint( $post_id );
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		// Get all existing open findings for this post.
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$existing_findings = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT id, fingerprint FROM `{$table_name}`
				WHERE object_type = 'post'
				AND object_id = %d
				AND status = 'open'",
				$post_id
			)
		);

		if ( empty( $existing_findings ) ) {
			return 0;
		}

		// Generate fingerprints for scan results.
		$scan_fingerprints = array();
		$blog_id = get_current_blog_id();

		foreach ( $scan_findings as $finding ) {
			// If fingerprint is already in finding data, use it.
			if ( ! empty( $finding['fingerprint'] ) ) {
				$scan_fingerprints[] = $finding['fingerprint'];
				continue;
			}

			// Otherwise, generate fingerprint from finding data.
			if ( class_exists( 'CGP_Scanner' ) ) {
				// Use reflection to access private method, or make it public.
				// For now, we'll query the database for fingerprints of recently saved findings.
				// Actually, better approach: query database for findings saved in last few seconds.
			}
		}

		// If we don't have fingerprints from scan results, query database for recently updated findings.
		// This works because save_finding() updates last_seen when fingerprint matches.
		if ( empty( $scan_fingerprints ) ) {
			$recent_time = current_time( 'mysql', true );
			// Get findings updated in last 5 seconds (should cover the scan we just ran).
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$recent_findings = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT fingerprint FROM `{$table_name}`
					WHERE object_type = 'post'
					AND object_id = %d
					AND status = 'open'
					AND last_seen >= DATE_SUB(%s, INTERVAL 5 SECOND)
					ORDER BY last_seen DESC",
					$post_id,
					$recent_time
				)
			);

			foreach ( $recent_findings as $finding ) {
				$scan_fingerprints[] = $finding->fingerprint;
			}
		}

		// Find existing findings that are NOT in scan results.
		$to_resolve = array();
		foreach ( $existing_findings as $existing ) {
			if ( ! in_array( $existing->fingerprint, $scan_fingerprints, true ) ) {
				$to_resolve[] = $existing->id;
			}
		}

		if ( empty( $to_resolve ) ) {
			return 0;
		}

		// Auto-resolve findings that are no longer detected.
		$current_time = current_time( 'mysql' );
		$resolved_by  = wp_json_encode(
			array(
				'method'    => $method,
				'timestamp' => $current_time,
				'user_id'   => get_current_user_id(),
			)
		);

		// Build query with proper placeholders.
		$placeholders = implode( ',', array_fill( 0, count( $to_resolve ), '%d' ) );
		$query = sprintf(
			"UPDATE `{$table_name}`
			SET status = 'resolved',
			    last_seen = %%s,
			    extra = JSON_SET(COALESCE(extra, '{}'), '$.resolved_by', %%s)
			WHERE id IN (%s)",
			$placeholders
		);

		// Prepare query with values.
		$prepared_values = array_merge( array( $current_time, $resolved_by ), $to_resolve );
		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query template is built safely above.
		$prepared_query = $wpdb->prepare( $query, $prepared_values );

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is prepared above.
		$updated = $wpdb->query( $prepared_query );

		if ( false === $updated ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( sprintf(
					'Content Guard Pro [Realtime Scanner ERROR]: Auto-resolve missing findings: Database update failed for post %d - %s',
					$post_id,
					$wpdb->last_error
				) );
			}
			return 0;
		}

		$updated_count = absint( $updated );
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( sprintf(
				'Content Guard Pro [Realtime Scanner]: Auto-resolve missing findings: Resolved %d findings for post %d (method=%s)',
				$updated_count,
				$post_id,
				$method
			) );
		}

		return $updated_count;
	}
}

