<?php
/**
 * Quarantine Class - Non-destructive Content Neutralization
 *
 * Implements render-time quarantine that neutralizes malicious content
 * without modifying the database (PRD Section 3.4 & Appendix C).
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Quarantine
 *
 * Provides non-destructive quarantine functionality:
 * - Hooks into the_content, the_excerpt, widget_text filters
 * - Checks for quarantined findings in real-time
 * - Neutralizes content by stripping scripts/iframes and defanging links
 * - Database remains unchanged; only render output is affected
 *
 * @since 1.0.0
 */
class CGP_Quarantine {

	/**
	 * Cache for quarantine status to avoid repeated DB queries.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	private static $quarantine_cache = array();

	/**
	 * Initialize the quarantine system.
	 *
	 * @since 1.0.0
	 */
	public static function init() {
		$instance = new self();
		$instance->register_hooks();
	}

	/**
	 * Register WordPress hooks for content filtering.
	 *
	 * @since 1.0.0
	 */
	private function register_hooks() {
		// Post content filters (PRD Appendix C).
		add_filter( 'the_content', array( $this, 'neutralize_post_content' ), 999 );
		add_filter( 'the_excerpt', array( $this, 'neutralize_post_excerpt' ), 999 );

		// Widget filters (legacy text widget filter approach).
		add_filter( 'widget_text', array( $this, 'neutralize_widget_text' ), 999, 2 );
		add_filter( 'widget_custom_html', array( $this, 'neutralize_widget_html' ), 999, 2 );

		// Widget option filters (intercept before widgets are loaded).
		// Per PRD Section 3.1 - allowlisted wp_options keys.
		add_filter( 'pre_option_widget_text', array( $this, 'filter_widget_text_option' ), 10, 1 );
		add_filter( 'pre_option_widget_custom_html', array( $this, 'filter_widget_custom_html_option' ), 10, 1 );
		add_filter( 'pre_option_widget_block', array( $this, 'filter_widget_block_option' ), 10, 1 );

		// Clear cache on finding status changes.
		add_action( 'content_guard_pro_finding_status_changed', array( __CLASS__, 'clear_cache' ), 10, 2 );

		// Flush external caches (WP Rocket, W3TC, etc.) on finding status changes.
		add_action( 'content_guard_pro_finding_status_changed', array( $this, 'flush_external_caches' ), 10, 2 );
	}

	/**
	 * Flush external caches when a finding status changes.
	 *
	 * Triggers clean_post_cache and hooks for popular caching plugins
	 * (WP Rocket, W3 Total Cache) to ensure neutralized content is served.
	 *
	 * @since 1.0.0
	 * @param int    $finding_id Finding ID.
	 * @param string $new_status New status.
	 */
	public function flush_external_caches( $finding_id, $new_status ) {
		global $wpdb;
		
		// Get finding details to identify the object.
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';
		$finding = $wpdb->get_row(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
				"SELECT object_type, object_id FROM `" . $table_name . "` WHERE id = %d",
				$finding_id
			)
		);

		if ( ! $finding || 'post' !== $finding->object_type ) {
			return;
		}

		$post_id = (int) $finding->object_id;

		// 1. Core WordPress Cache.
		clean_post_cache( $post_id );

		// 2. WP Rocket.
		if ( function_exists( 'rocket_clean_post' ) ) {
			rocket_clean_post( $post_id );
		}

		// 3. W3 Total Cache.
		if ( function_exists( 'w3tc_flush_post' ) ) {
			w3tc_flush_post( $post_id );
		}
		
		// 4. Autoptimize (if it has a specific API, otherwise generic cache clear often suffices).
		// Many other plugins hook into clean_post_cache, so explicit calls are only needed for those that don't.
	}

	/**
	 * Neutralize post content.
	 *
	 * Filters the_content to neutralize quarantined posts.
	 *
	 * @since 1.0.0
	 * @param string $content Post content.
	 * @return string Neutralized content if quarantined, original content otherwise.
	 */
	public function neutralize_post_content( $content ) {
		// Get the current post ID.
		$post_id = get_the_ID();

		if ( ! $post_id ) {
			return $content;
		}

		// Check if this post has quarantined findings.
		if ( ! $this->has_quarantined_findings( 'post', $post_id, 'post_content' ) ) {
			return $content;
		}

		// Neutralize and return.
		return $this->neutralize_content( $content, 'post', $post_id );
	}

	/**
	 * Neutralize post excerpt.
	 *
	 * Filters the_excerpt to neutralize quarantined posts.
	 *
	 * @since 1.0.0
	 * @param string $excerpt Post excerpt.
	 * @return string Neutralized excerpt if quarantined, original excerpt otherwise.
	 */
	public function neutralize_post_excerpt( $excerpt ) {
		// Get the current post ID.
		$post_id = get_the_ID();

		if ( ! $post_id ) {
			return $excerpt;
		}

		// Check if this post has quarantined findings.
		if ( ! $this->has_quarantined_findings( 'post', $post_id, 'post_content' ) ) {
			return $excerpt;
		}

		// Neutralize and return.
		return $this->neutralize_content( $excerpt, 'post', $post_id );
	}

	/**
	 * Neutralize widget text.
	 *
	 * Filters widget_text to neutralize quarantined widgets.
	 * This is a secondary filter; primary filtering happens at option level.
	 *
	 * @since 1.0.0
	 * @param string $text     Widget text.
	 * @param array  $instance Widget instance.
	 * @return string Neutralized text if quarantined, original text otherwise.
	 */
	public function neutralize_widget_text( $text, $instance ) {
		// Primary filtering happens at option level (pre_option_widget_text).
		// This is kept for backwards compatibility and edge cases.
		return $text;
	}

	/**
	 * Neutralize custom HTML widget.
	 *
	 * Filters widget_custom_html to neutralize quarantined widgets.
	 * This is a secondary filter; primary filtering happens at option level.
	 *
	 * @since 1.0.0
	 * @param string $content  Widget HTML content.
	 * @param array  $instance Widget instance.
	 * @return string Neutralized content if quarantined, original content otherwise.
	 */
	public function neutralize_widget_html( $content, $instance ) {
		// Primary filtering happens at option level (pre_option_widget_custom_html).
		// This is kept for backwards compatibility and edge cases.
		return $content;
	}
	
	/**
	 * Filter widget_text option before it's loaded.
	 *
	 * Intercepts the widget_text option and neutralizes quarantined widgets.
	 * Per PRD Section 3.1 - allowlisted wp_options keys.
	 *
	 * @since 1.0.0
	 * @param mixed $value The option value (false if not set).
	 * @return mixed Modified option value with neutralized widgets.
	 */
	public function filter_widget_text_option( $value ) {
		// If value is false, let WordPress handle the default.
		if ( false === $value ) {
			return $value;
		}
		
		// Get the actual option value from database.
		global $wpdb;
		$option_value = $wpdb->get_var( $wpdb->prepare(
			"SELECT option_value FROM `{$wpdb->options}` WHERE option_name = %s",
			'widget_text'
		) );
		
		if ( ! $option_value ) {
			return $value;
		}
		
		// Unserialize the widgets array.
		$widgets = maybe_unserialize( $option_value );
		
		if ( ! is_array( $widgets ) ) {
			return $value;
		}
		
		// Loop through each widget instance.
		foreach ( $widgets as $widget_index => $widget_data ) {
			if ( ! is_array( $widget_data ) ) {
				continue;
			}
			
			// Check if this widget instance is quarantined.
			// Object ID is generated using crc32 of option_name + widget_index (as per scanner logic).
			$object_id = crc32( 'widget_text_' . $widget_index );
			
			if ( $this->has_quarantined_findings( 'option', $object_id, 'widget_text[' . $widget_index . ']' ) ) {
				// Neutralize the widget text.
				if ( ! empty( $widget_data['text'] ) ) {
					$widgets[ $widget_index ]['text'] = $this->neutralize_content(
						$widget_data['text'],
						'option',
						$object_id
					);
				}
			}
		}
		
		return $widgets;
	}
	
	/**
	 * Filter widget_custom_html option before it's loaded.
	 *
	 * Intercepts the widget_custom_html option and neutralizes quarantined widgets.
	 * Per PRD Section 3.1 - allowlisted wp_options keys.
	 *
	 * @since 1.0.0
	 * @param mixed $value The option value (false if not set).
	 * @return mixed Modified option value with neutralized widgets.
	 */
	public function filter_widget_custom_html_option( $value ) {
		// If value is false, let WordPress handle the default.
		if ( false === $value ) {
			return $value;
		}
		
		// Get the actual option value from database.
		global $wpdb;
		$option_value = $wpdb->get_var( $wpdb->prepare(
			"SELECT option_value FROM `{$wpdb->options}` WHERE option_name = %s",
			'widget_custom_html'
		) );
		
		if ( ! $option_value ) {
			return $value;
		}
		
		// Unserialize the widgets array.
		$widgets = maybe_unserialize( $option_value );
		
		if ( ! is_array( $widgets ) ) {
			return $value;
		}
		
		// Loop through each widget instance.
		foreach ( $widgets as $widget_index => $widget_data ) {
			if ( ! is_array( $widget_data ) ) {
				continue;
			}
			
			// Check if this widget instance is quarantined.
			$object_id = crc32( 'widget_custom_html_' . $widget_index );
			
			if ( $this->has_quarantined_findings( 'option', $object_id, 'widget_custom_html[' . $widget_index . ']' ) ) {
				// Neutralize the widget content (custom HTML field).
				if ( ! empty( $widget_data['content'] ) ) {
					$widgets[ $widget_index ]['content'] = $this->neutralize_content(
						$widget_data['content'],
						'option',
						$object_id
					);
				}
			}
		}
		
		return $widgets;
	}
	
	/**
	 * Filter widget_block option before it's loaded.
	 *
	 * Intercepts the widget_block option and neutralizes quarantined block widgets.
	 * Per PRD Section 3.1 - allowlisted wp_options keys.
	 * Handles Gutenberg block parsing per PRD Section 3.1.
	 *
	 * @since 1.0.0
	 * @param mixed $value The option value (false if not set).
	 * @return mixed Modified option value with neutralized widgets.
	 */
	public function filter_widget_block_option( $value ) {
		// If value is false, let WordPress handle the default.
		if ( false === $value ) {
			return $value;
		}
		
		// Get the actual option value from database.
		global $wpdb;
		$option_value = $wpdb->get_var( $wpdb->prepare(
			"SELECT option_value FROM `{$wpdb->options}` WHERE option_name = %s",
			'widget_block'
		) );
		
		if ( ! $option_value ) {
			return $value;
		}
		
		// Unserialize the widgets array.
		$widgets = maybe_unserialize( $option_value );
		
		if ( ! is_array( $widgets ) ) {
			return $value;
		}
		
		// Loop through each widget instance.
		foreach ( $widgets as $widget_index => $widget_data ) {
			if ( ! is_array( $widget_data ) ) {
				continue;
			}
			
			// Check if this widget instance is quarantined.
			$object_id = crc32( 'widget_block_' . $widget_index );
			
			if ( $this->has_quarantined_findings( 'option', $object_id, 'widget_block[' . $widget_index . ']' ) ) {
				// Neutralize the block content.
				if ( ! empty( $widget_data['content'] ) ) {
					$block_content = $widget_data['content'];
					
					// Parse Gutenberg blocks if present.
					if ( has_blocks( $block_content ) ) {
						// Parse the blocks.
						$blocks = parse_blocks( $block_content );
						
						// Neutralize each block's content.
						foreach ( $blocks as $block_key => $block ) {
							if ( ! empty( $block['innerHTML'] ) ) {
								$blocks[ $block_key ]['innerHTML'] = $this->neutralize_content(
									$block['innerHTML'],
									'option',
									$object_id
								);
							}
							
							// Also neutralize block attributes if they contain content.
							if ( ! empty( $block['attrs'] ) && is_array( $block['attrs'] ) ) {
								foreach ( $block['attrs'] as $attr_key => $attr_value ) {
									if ( is_string( $attr_value ) && strlen( $attr_value ) > 0 ) {
										$blocks[ $block_key ]['attrs'][ $attr_key ] = $this->neutralize_content(
											$attr_value,
											'option',
											$object_id
										);
									}
								}
							}
						}
						
						// Reconstruct the block content.
						$block_content = serialize_blocks( $blocks );
					} else {
						// Not block content, neutralize as plain HTML.
						$block_content = $this->neutralize_content(
							$block_content,
							'option',
							$object_id
						);
					}
					
					$widgets[ $widget_index ]['content'] = $block_content;
				}
			}
		}
		
		return $widgets;
	}

	/**
	 * Check if an object has quarantined findings.
	 *
	 * Queries the content_guard_pro_findings table for quarantined status.
	 * Results are cached to avoid repeated DB queries.
	 *
	 * @since 1.0.0
	 * @param string $object_type Object type (post, postmeta, option).
	 * @param int    $object_id   Object ID.
	 * @param string $field       Field name (optional).
	 * @return bool True if quarantined findings exist, false otherwise.
	 */
	private function has_quarantined_findings( $object_type, $object_id, $field = '' ) {
		// Build cache key.
		$cache_key = $object_type . '_' . $object_id . '_' . $field;

		// Check cache first.
		if ( isset( self::$quarantine_cache[ $cache_key ] ) ) {
			return self::$quarantine_cache[ $cache_key ];
		}

		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';
		$blog_id    = get_current_blog_id();

		// Build query.
		// If field is specified, include it in the query.
		if ( ! empty( $field ) ) {
			$query = $wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
				"SELECT COUNT(*) FROM `" . $table_name . "` 
				WHERE blog_id = %d 
				AND object_type = %s 
				AND object_id = %d 
				AND field = %s 
				AND status = %s",
				$blog_id,
				$object_type,
				$object_id,
				$field,
				'quarantined'
			);
		} else {
			$query = $wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
				"SELECT COUNT(*) FROM `" . $table_name . "` 
				WHERE blog_id = %d 
				AND object_type = %s 
				AND object_id = %d 
				AND status = %s",
				$blog_id,
				$object_type,
				$object_id,
				'quarantined'
			);
		}

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		$count = $wpdb->get_var( $query );
		$has_quarantined = ( $count > 0 );

		// Cache the result.
		self::$quarantine_cache[ $cache_key ] = $has_quarantined;

		return $has_quarantined;
	}

	/**
	 * Neutralize content.
	 *
	 * Implements the neutralization logic per PRD Appendix C:
	 * - Strip <script> and <iframe> tags
	 * - Convert <a> tags to <span> tags (preserving text)
	 * - Add title attribute "neutralized by Content Guard Pro"
	 * - Set rel="nofollow ugc" on any remaining links
	 *
	 * @since 1.0.0
	 * @param string $content     Content to neutralize.
	 * @param string $object_type Object type (for context).
	 * @param int    $object_id   Object ID (for context).
	 * @return string Neutralized content.
	 */
	private function neutralize_content( $content, $object_type, $object_id ) {
		// If content is empty, return as-is.
		if ( empty( $content ) ) {
			return $content;
		}

		// Step 1: Strip <script> tags and their content.
		$result = preg_replace(
			'/<script\b[^>]*>.*?<\/script>/is',
			'<!-- script removed by Content Guard Pro -->',
			$content
		);
		
		// Check for preg_replace error.
		if ( null === $result ) {
			cgp_log( 'Content Guard Pro: preg_replace failed for script tags in neutralize_content' );
		} else {
			$content = $result;
		}

		// Step 2: Strip <iframe> tags and their content.
		$result = preg_replace(
			'/<iframe\b[^>]*>.*?<\/iframe>/is',
			'<!-- iframe removed by Content Guard Pro -->',
			$content
		);
		
		// Check for preg_replace error.
		if ( null === $result ) {
			cgp_log( 'Content Guard Pro: preg_replace failed for iframe tags in neutralize_content' );
		} else {
			$content = $result;
		}

		// Step 3: Convert <a> tags to <span> tags (defang links).
		// This preserves the text but removes the link functionality.
		$result = preg_replace_callback(
			'/<a\b([^>]*)>(.*?)<\/a>/is',
			function( $matches ) {
				$attributes = $matches[1];
				$link_text  = $matches[2];

				// Extract href for display (optional).
				$href = '';
				if ( preg_match( '/href=["\']([^"\']+)["\']/i', $attributes, $href_match ) ) {
					$href = esc_url( $href_match[1] );
				}

				// Convert to span with title note.
				return sprintf(
					'<span class="content-guard-pro-neutralized-link" title="%s" data-original-href="%s">%s</span>',
					esc_attr__( 'Link neutralized by Content Guard Pro', 'content-guard-pro' ),
					esc_attr( $href ),
					$link_text
				);
			},
			$content
		);
		
		// Check for preg_replace_callback error.
		if ( null === $result ) {
			cgp_log( 'Content Guard Pro: preg_replace_callback failed for link defanging in neutralize_content' );
		} else {
			$content = $result;
		}

		// Step 4: Add rel="nofollow ugc" to any remaining links (edge case).
		// This shouldn't normally be needed since we converted all <a> tags above,
		// but we'll keep it as a safety measure.
		$result = preg_replace_callback(
			'/<a\b([^>]*)>/is',
			function( $matches ) {
				$attributes = $matches[1];

				// Check if rel attribute already exists.
				if ( preg_match( '/\brel=["\']([^"\']+)["\']/i', $attributes, $rel_match ) ) {
					// Append to existing rel.
					$new_rel = $rel_match[1] . ' nofollow ugc';
					$attributes = preg_replace(
						'/\brel=["\'][^"\']+["\']/i',
						'rel="' . esc_attr( $new_rel ) . '"',
						$attributes
					);
				} else {
					// Add new rel attribute.
					$attributes .= ' rel="nofollow ugc"';
				}

				return '<a' . $attributes . '>';
			},
			$content
		);
		
		// Check for preg_replace_callback error.
		if ( null === $result ) {
			cgp_log( 'Content Guard Pro: preg_replace_callback failed for rel attribute in neutralize_content' );
		} else {
			$content = $result;
		}

		// Step 5: Add a visual indicator at the top of the content (admin view only).
		if ( is_admin() || current_user_can( 'edit_posts' ) ) {
			$notice = sprintf(
				'<div class="content-guard-pro-quarantine-notice" style="background: #ffebcc; border-left: 4px solid #ff922b; padding: 12px; margin-bottom: 15px;">
					<strong>%s</strong> %s
				</div>',
				esc_html__( 'Content Guard Pro:', 'content-guard-pro' ),
				esc_html__( 'This content has been quarantined and neutralized. Suspicious scripts and links have been removed or disabled.', 'content-guard-pro' )
			);

			$content = $notice . $content;
		}

		/**
		 * Filter the neutralized content.
		 *
		 * Allows developers to customize the neutralization process.
		 *
		 * @since 1.0.0
		 * @param string $content     Neutralized content.
		 * @param string $object_type Object type.
		 * @param int    $object_id   Object ID.
		 */
		return apply_filters( 'content_guard_pro_neutralized_content', $content, $object_type, $object_id );
	}

	/**
	 * Clear the quarantine cache.
	 *
	 * Called when a finding's status changes to ensure cache is fresh.
	 *
	 * @since 1.0.0
	 * @param int    $finding_id Finding ID (unused, for hook compatibility).
	 * @param string $new_status New status (unused, for hook compatibility).
	 */
	public static function clear_cache( $finding_id = 0, $new_status = '' ) {
		self::$quarantine_cache = array();
	}

	/**
	 * Get quarantine status for a post (public method).
	 *
	 * Useful for displaying quarantine indicators in admin.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if post is quarantined, false otherwise.
	 */
	public static function is_post_quarantined( $post_id ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';
		$blog_id    = get_current_blog_id();

		$count = $wpdb->get_var(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
				"SELECT COUNT(*) FROM `" . $table_name . "` 
				WHERE blog_id = %d 
				AND object_type = %s 
				AND object_id = %d 
				AND status = %s",
				$blog_id,
				'post',
				$post_id,
				'quarantined'
			)
		);

		return ( $count > 0 );
	}

	/**
	 * Quarantine a finding.
	 *
	 * Updates the finding status to 'quarantined' in the database.
	 *
	 * @since 1.0.0
	 * @param int $finding_id Finding ID.
	 * @return bool True on success, false on failure.
	 */
	public static function quarantine_finding( $finding_id ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		$updated = $wpdb->update(
			$table_name,
			array( 'status' => 'quarantined' ),
			array( 'id' => $finding_id ),
			array( '%s' ),
			array( '%d' )
		);

		if ( $updated ) {
			// Clear cache.
			do_action( 'content_guard_pro_finding_status_changed', $finding_id, 'quarantined' );

			// Log the action.
			self::log_quarantine_action( $finding_id, 'quarantine' );

			return true;
		}

		return false;
	}

	/**
	 * Un-quarantine a finding.
	 *
	 * Updates the finding status back to 'open'.
	 *
	 * @since 1.0.0
	 * @param int $finding_id Finding ID.
	 * @return bool True on success, false on failure.
	 */
	public static function unquarantine_finding( $finding_id ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		$updated = $wpdb->update(
			$table_name,
			array( 'status' => 'open' ),
			array( 'id' => $finding_id ),
			array( '%s' ),
			array( '%d' )
		);

		if ( $updated ) {
			// Clear cache.
			do_action( 'content_guard_pro_finding_status_changed', $finding_id, 'open' );

			// Log the action.
			self::log_quarantine_action( $finding_id, 'unquarantine' );

			return true;
		}

		return false;
	}

	/**
	 * Bulk quarantine findings.
	 *
	 * @since 1.0.0
	 * @param array $finding_ids Array of finding IDs.
	 * @return int Number of findings quarantined.
	 */
	public static function bulk_quarantine( $finding_ids ) {
		if ( empty( $finding_ids ) || ! is_array( $finding_ids ) ) {
			return 0;
		}

		$count = 0;
		foreach ( $finding_ids as $finding_id ) {
			if ( self::quarantine_finding( absint( $finding_id ) ) ) {
				$count++;
			}
		}

		return $count;
	}

	/**
	 * Bulk un-quarantine findings.
	 *
	 * @since 1.0.0
	 * @param array $finding_ids Array of finding IDs.
	 * @return int Number of findings un-quarantined.
	 */
	public static function bulk_unquarantine( $finding_ids ) {
		if ( empty( $finding_ids ) || ! is_array( $finding_ids ) ) {
			return 0;
		}

		$count = 0;
		foreach ( $finding_ids as $finding_id ) {
			if ( self::unquarantine_finding( absint( $finding_id ) ) ) {
				$count++;
			}
		}

		return $count;
	}

	/**
	 * Log quarantine action to audit log.
	 *
	 * @since 1.0.0
	 * @param int    $finding_id Finding ID.
	 * @param string $action     Action performed (quarantine, unquarantine).
	 */
	private static function log_quarantine_action( $finding_id, $action ) {
		global $wpdb;

		// Get finding details.
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';
		$finding = $wpdb->get_row(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
				"SELECT * FROM `" . $table_name . "` WHERE id = %d",
				$finding_id
			),
			ARRAY_A
		);

		if ( ! $finding ) {
			return;
		}

		// Insert audit log entry.
		$audit_table = $wpdb->prefix . 'content_guard_pro_audit_log';
		$wpdb->insert(
			$audit_table,
			array(
				'blog_id'    => get_current_blog_id(),
				'finding_id' => $finding_id,
				'user_id'    => get_current_user_id(),
				'action'     => $action,
				'object_type' => $finding['object_type'],
				'object_id'  => $finding['object_id'],
				'field'      => $finding['field'],
				'old_value'  => $action === 'quarantine' ? 'open' : 'quarantined',
				'new_value'  => $action === 'quarantine' ? 'quarantined' : 'open',
				'metadata'   => wp_json_encode( array( 'rule_id' => $finding['rule_id'] ) ),
				'ip_address' => self::get_client_ip(),
				'created_at' => current_time( 'mysql' ),
			),
			array( '%d', '%d', '%d', '%s', '%s', '%d', '%s', '%s', '%s', '%s', '%s', '%s' )
		);
	}

	/**
	 * Get client IP address.
	 *
	 * Prioritizes REMOTE_ADDR (most reliable) and only uses X-Forwarded-For
	 * if REMOTE_ADDR is a private/reserved IP (indicating proxy).
	 *
	 * @since 1.0.0
	 * @return string Client IP address.
	 */
	private static function get_client_ip() {
		$ip = '';

		// REMOTE_ADDR is the most reliable source (cannot be spoofed by client).
		if ( ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
			$ip = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
		}

		// Validate the IP address.
		$validated_ip = filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE );

		// If REMOTE_ADDR is private/reserved (indicating proxy), check X-Forwarded-For.
		if ( ! $validated_ip && ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$forwarded_ips = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
			
			// HTTP_X_FORWARDED_FOR can contain multiple IPs (comma-separated).
			// Format: client, proxy1, proxy2
			$ip_list = array_map( 'trim', explode( ',', $forwarded_ips ) );
			
			// Get the first valid public IP.
			foreach ( $ip_list as $forwarded_ip ) {
				$validated = filter_var( $forwarded_ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE );
				if ( $validated ) {
					return $validated;
				}
			}
		}

		// Return validated REMOTE_ADDR or fallback.
		return $validated_ip ? $validated_ip : '0.0.0.0';
	}
}

