<?php
/**
 * Quarantine List Table Class
 *
 * Extends WP_List_Table to display quarantined findings with bulk actions
 * for un-quarantine and delete operations.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Load WP_List_Table if not already loaded.
if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Class CGP_Quarantine_List_Table
 *
 * Displays quarantined findings in a native WordPress list table format.
 * Filtered to show only findings with status = 'quarantined'.
 *
 * @since 1.0.0
 */
class CGP_Quarantine_List_Table extends WP_List_Table {

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		parent::__construct(
			array(
				'singular' => 'quarantined_item',
				'plural'   => 'quarantined_items',
				'ajax'     => false,
			)
		);
	}

	/**
	 * Get table columns.
	 *
	 * @since 1.0.0
	 * @return array Column names and labels.
	 */
	public function get_columns() {
		return array(
			'cb'              => '<input type="checkbox" />',
			'severity'        => __( 'Severity', 'content-guard-pro' ),
			'object_type'     => __( 'Type', 'content-guard-pro' ),
			'object_id'       => __( 'Location', 'content-guard-pro' ),
			'rule_id'         => __( 'Rule', 'content-guard-pro' ),
			'confidence'      => __( 'Confidence', 'content-guard-pro' ),
			'matched_excerpt' => __( 'Match', 'content-guard-pro' ),
			'first_seen'      => __( 'First Seen', 'content-guard-pro' ),
			'last_seen'       => __( 'Last Seen', 'content-guard-pro' ),
			'actions'         => __( 'Actions', 'content-guard-pro' ),
		);
	}

	/**
	 * Get sortable columns.
	 *
	 * @since 1.0.0
	 * @return array Sortable column names.
	 */
	protected function get_sortable_columns() {
		return array(
			'severity'   => array( 'severity', false ),
			'confidence' => array( 'confidence', true ),
			'first_seen' => array( 'first_seen', true ),
			'last_seen'  => array( 'last_seen', true ),
		);
	}

	/**
	 * Get bulk actions.
	 *
	 * @since 1.0.0
	 * @return array Bulk actions.
	 */
	protected function get_bulk_actions() {
		return array(
			'unquarantine' => __( 'Un-quarantine', 'content-guard-pro' ),
			'delete'       => __( 'Delete Permanently', 'content-guard-pro' ),
		);
	}

	/**
	 * Column checkbox for bulk actions.
	 *
	 * @since 1.0.0
	 * @param object $item Finding item.
	 * @return string Checkbox HTML.
	 */
	protected function column_cb( $item ) {
		return sprintf( '<input type="checkbox" name="finding_ids[]" value="%d" />', absint( $item->id ) );
	}

	/**
	 * Column severity with color coding.
	 *
	 * @since 1.0.0
	 * @param object $item Finding item.
	 * @return string Severity HTML.
	 */
	protected function column_severity( $item ) {
		$severity_labels = array(
			'critical'   => __( 'Critical', 'content-guard-pro' ),
			'suspicious' => __( 'Suspicious', 'content-guard-pro' ),
			'review'     => __( 'Review', 'content-guard-pro' ),
		);

		$label = isset( $severity_labels[ $item->severity ] ) ? $severity_labels[ $item->severity ] : $item->severity;

		return sprintf(
			'<span class="content-guard-pro-severity content-guard-pro-severity-%s">%s</span>',
			esc_attr( $item->severity ),
			esc_html( $label )
		);
	}

	/**
	 * Column object type.
	 *
	 * @since 1.0.0
	 * @param object $item Finding item.
	 * @return string Object type HTML.
	 */
	protected function column_object_type( $item ) {
		$type_labels = array(
			'post'     => __( 'Post', 'content-guard-pro' ),
			'postmeta' => __( 'Post Meta', 'content-guard-pro' ),
			'option'   => __( 'Option', 'content-guard-pro' ),
		);

		$label = isset( $type_labels[ $item->object_type ] ) ? $type_labels[ $item->object_type ] : $item->object_type;

		return esc_html( $label );
	}

	/**
	 * Column object ID with link to edit page.
	 *
	 * @since 1.0.0
	 * @param object $item Finding item.
	 * @return string Object ID HTML.
	 */
	protected function column_object_id( $item ) {
		if ( 'post' === $item->object_type ) {
			$edit_link = get_edit_post_link( $item->object_id );
			$post      = get_post( $item->object_id );
			$title     = $post ? get_the_title( $post ) : __( 'Post', 'content-guard-pro' );

			if ( $edit_link ) {
				return sprintf(
					'<a href="%s" target="_blank">%s <span class="content-guard-pro-object-id">(ID: %d)</span></a>',
					esc_url( $edit_link ),
					esc_html( $title ),
					absint( $item->object_id )
				);
			}
		}

		return sprintf( 'ID: %d', absint( $item->object_id ) );
	}

	/**
	 * Column rule ID.
	 *
	 * @since 1.0.0
	 * @param object $item Finding item.
	 * @return string Rule ID HTML.
	 */
	protected function column_rule_id( $item ) {
		return sprintf( '<code>%s</code>', esc_html( $item->rule_id ) );
	}

	/**
	 * Column confidence with progress bar.
	 *
	 * @since 1.0.0
	 * @param object $item Finding item.
	 * @return string Confidence HTML.
	 */
	protected function column_confidence( $item ) {
		$confidence = absint( $item->confidence );

		// Determine color based on confidence level.
		$color = 'green';
		if ( $confidence < 50 ) {
			$color = 'orange';
		} elseif ( $confidence >= 80 ) {
			$color = 'red';
		}

		return sprintf(
			'<div class="content-guard-pro-confidence">
				<div class="content-guard-pro-confidence-bar">
					<div class="content-guard-pro-confidence-fill content-guard-pro-confidence-%s" style="width: %d%%;"></div>
				</div>
				<span class="content-guard-pro-confidence-text">%d%%</span>
			</div>',
			esc_attr( $color ),
			$confidence,
			$confidence
		);
	}

	/**
	 * Column matched excerpt with tooltip.
	 *
	 * @since 1.0.0
	 * @param object $item Finding item.
	 * @return string Matched excerpt HTML.
	 */
	protected function column_matched_excerpt( $item ) {
		$excerpt = wp_strip_all_tags( $item->matched_excerpt );
		$short   = strlen( $excerpt ) > 100 ? substr( $excerpt, 0, 100 ) . '...' : $excerpt;

		return sprintf(
			'<span class="content-guard-pro-matched-excerpt" title="%s">%s</span>',
			esc_attr( $excerpt ),
			esc_html( $short )
		);
	}

	/**
	 * Column first seen with human-readable time.
	 *
	 * @since 1.0.0
	 * @param object $item Finding item.
	 * @return string First seen HTML.
	 */
	protected function column_first_seen( $item ) {
		$timestamp = strtotime( $item->first_seen );
		
		// Check if strtotime() failed.
		if ( false === $timestamp ) {
			return esc_html( $item->first_seen );
		}
		
		$time_ago  = human_time_diff( $timestamp, current_time( 'timestamp' ) );

		return sprintf(
			'<span title="%s">%s ago</span>',
			esc_attr( $item->first_seen ),
			esc_html( $time_ago )
		);
	}

	/**
	 * Column last seen with human-readable time.
	 *
	 * @since 1.0.0
	 * @param object $item Finding item.
	 * @return string Last seen HTML.
	 */
	protected function column_last_seen( $item ) {
		$timestamp = strtotime( $item->last_seen );
		
		// Check if strtotime() failed.
		if ( false === $timestamp ) {
			return esc_html( $item->last_seen );
		}
		
		$time_ago  = human_time_diff( $timestamp, current_time( 'timestamp' ) );

		return sprintf(
			'<span title="%s">%s ago</span>',
			esc_attr( $item->last_seen ),
			esc_html( $time_ago )
		);
	}

	/**
	 * Column actions with row-specific links.
	 *
	 * @since 1.0.0
	 * @param object $item Finding item.
	 * @return string Actions HTML.
	 */
	protected function column_actions( $item ) {
		$actions = array();

		// Un-quarantine action.
		$unquarantine_url = wp_nonce_url(
			admin_url( 'admin.php?page=content-guard-pro-quarantine&action=unquarantine&finding_id=' . $item->id ),
			'content_guard_pro_unquarantine_' . $item->id
		);
		$actions['unquarantine'] = sprintf(
			'<a href="%s" class="content-guard-pro-action-unquarantine">%s</a>',
			esc_url( $unquarantine_url ),
			__( 'Un-quarantine', 'content-guard-pro' )
		);

		// View/Edit post action (if applicable).
		if ( 'post' === $item->object_type ) {
			$edit_link = get_edit_post_link( $item->object_id );
			if ( $edit_link ) {
				$actions['edit'] = sprintf(
					'<a href="%s" target="_blank">%s</a>',
					esc_url( $edit_link ),
					__( 'Edit Post', 'content-guard-pro' )
				);
			}
		}

		// Delete action.
		$delete_url = wp_nonce_url(
			admin_url( 'admin.php?page=content-guard-pro-quarantine&action=delete&finding_id=' . $item->id ),
			'content_guard_pro_delete_' . $item->id
		);
		$actions['delete'] = sprintf(
			'<a href="%s" class="content-guard-pro-action-delete" onclick="return confirm(\'%s\');">%s</a>',
			esc_url( $delete_url ),
			esc_js( __( 'Are you sure you want to permanently delete this finding?', 'content-guard-pro' ) ),
			__( 'Delete', 'content-guard-pro' )
		);

		return $this->row_actions( $actions );
	}

	/**
	 * Prepare items for display.
	 *
	 * Queries the database for quarantined findings only (status = 'quarantined').
	 *
	 * @since 1.0.0
	 */
	public function prepare_items() {
		global $wpdb;

		// Register columns.
		$columns  = $this->get_columns();
		$hidden   = array();
		$sortable = $this->get_sortable_columns();

		$this->_column_headers = array( $columns, $hidden, $sortable );

		// Handle bulk actions.
		$this->process_bulk_action();

		// Pagination settings.
		$per_page     = 20;
		$current_page = $this->get_pagenum();
		$offset       = ( $current_page - 1 ) * $per_page;

		// Get sorting parameters.
		$orderby = ! empty( $_REQUEST['orderby'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['orderby'] ) ) : 'last_seen';
		$order   = ! empty( $_REQUEST['order'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['order'] ) ) : 'DESC';

		// Validate orderby.
		$allowed_orderby = array( 'severity', 'confidence', 'first_seen', 'last_seen' );
		if ( ! in_array( $orderby, $allowed_orderby, true ) ) {
			$orderby = 'last_seen';
		}

		// Validate order.
		$order = strtoupper( $order ) === 'ASC' ? 'ASC' : 'DESC';

		// Build query - ONLY quarantined items.
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		// Get total count (for pagination).
		$total_items = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$table_name}` WHERE status = %s",
				'quarantined'
			)
		);

		// Get items for current page.
		// Note: $orderby is validated against whitelist, $order is validated as ASC/DESC.
		$this->items = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT * FROM `{$table_name}`
				WHERE status = %s
				ORDER BY `{$orderby}` {$order}
				LIMIT %d OFFSET %d",
				'quarantined',
				$per_page,
				$offset
			)
		);

		// Set pagination.
		$this->set_pagination_args(
			array(
				'total_items' => $total_items,
				'per_page'    => $per_page,
				'total_pages' => ceil( $total_items / $per_page ),
			)
		);
	}

	/**
	 * Process bulk actions.
	 *
	 * @since 1.0.0
	 */
	protected function process_bulk_action() {
		// Check nonce.
		if ( ! isset( $_REQUEST['_wpnonce'] ) ) {
			return;
		}

		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'content-guard-pro' ) );
		}

		$nonce  = sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] ) );
		$action = $this->current_action();

		if ( ! wp_verify_nonce( $nonce, 'bulk-' . $this->_args['plural'] ) ) {
			return;
		}

		// Get selected finding IDs.
		$finding_ids = isset( $_REQUEST['finding_ids'] ) ? array_map( 'absint', $_REQUEST['finding_ids'] ) : array();

		if ( empty( $finding_ids ) ) {
			return;
		}

		// Process actions.
		switch ( $action ) {
			case 'unquarantine':
				$this->bulk_unquarantine( $finding_ids );
				break;

			case 'delete':
				$this->bulk_delete( $finding_ids );
				break;
		}
	}

	/**
	 * Bulk un-quarantine findings.
	 *
	 * @since 1.0.0
	 * @param array $finding_ids Array of finding IDs.
	 */
	private function bulk_unquarantine( $finding_ids ) {
		if ( ! class_exists( 'CGP_Quarantine' ) ) {
			return;
		}

		$count = CGP_Quarantine::bulk_unquarantine( $finding_ids );

		// Set admin notice.
		set_transient( 'content_guard_pro_admin_notice', array(
			'type'    => 'success',
			'message' => sprintf(
				/* translators: %d: number of findings un-quarantined */
				_n(
					'%d finding un-quarantined successfully.',
					'%d findings un-quarantined successfully.',
					$count,
					'content-guard-pro'
				),
				$count
			),
		), 30 );

		// Redirect to avoid resubmission.
		wp_safe_redirect( admin_url( 'admin.php?page=content-guard-pro-quarantine' ) );
		exit;
	}

	/**
	 * Bulk delete findings.
	 *
	 * @since 1.0.0
	 * @param array $finding_ids Array of finding IDs.
	 */
	private function bulk_delete( $finding_ids ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'content_guard_pro_findings';
		$count      = 0;

		foreach ( $finding_ids as $finding_id ) {
			// Log the deletion to audit log before deleting.
			$this->log_finding_action( $finding_id, 'delete' );

			// Delete the finding.
			$deleted = $wpdb->delete(
				$table_name,
				array( 'id' => $finding_id ),
				array( '%d' )
			);

			if ( false !== $deleted ) {
				$count++;
			}
		}

		// Clear quarantine cache.
		if ( class_exists( 'CGP_Quarantine' ) ) {
			CGP_Quarantine::clear_cache();
		}

		// Set admin notice.
		set_transient( 'content_guard_pro_admin_notice', array(
			'type'    => 'success',
			'message' => sprintf(
				/* translators: %d: number of findings deleted */
				_n(
					'%d finding deleted successfully.',
					'%d findings deleted successfully.',
					$count,
					'content-guard-pro'
				),
				$count
			),
		), 30 );

		// Redirect to avoid resubmission.
		wp_safe_redirect( admin_url( 'admin.php?page=content-guard-pro-quarantine' ) );
		exit;
	}

	/**
	 * Log finding action to audit log.
	 *
	 * @since 1.0.0
	 * @param int    $finding_id Finding ID.
	 * @param string $action     Action performed.
	 */
	private function log_finding_action( $finding_id, $action ) {
		global $wpdb;

		// Get finding details before action.
		$findings_table = $wpdb->prefix . 'content_guard_pro_findings';
		$finding = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$findings_table}` WHERE id = %d",
				$finding_id
			)
		);

		if ( ! $finding ) {
			return;
		}

		// Prepare audit log entry.
		$audit_data = array(
			'blog_id'     => get_current_blog_id(),
			'finding_id'  => $finding_id,
			'user_id'     => get_current_user_id(),
			'action'      => $action,
			'object_type' => $finding->object_type,
			'object_id'   => $finding->object_id,
			'field'       => $finding->field,
			'old_value'   => $finding->status,
			'new_value'   => ( 'delete' === $action ) ? 'deleted' : 'open',
			'metadata'    => wp_json_encode( array(
				'severity'    => $finding->severity,
				'rule_id'     => $finding->rule_id,
				'confidence'  => $finding->confidence,
				'fingerprint' => $finding->fingerprint,
			) ),
			'ip_address'  => $this->get_client_ip(),
			'created_at'  => current_time( 'mysql' ),
		);

		// Insert into audit log.
		$inserted = $wpdb->insert(
			$wpdb->prefix . 'content_guard_pro_audit_log',
			$audit_data,
			array( '%d', '%d', '%d', '%s', '%s', '%d', '%s', '%s', '%s', '%s', '%s', '%s' )
		);

		// Log error if insert failed (optional, for debugging).
		if ( false === $inserted ) {
			// Could log to error_log or WordPress debug log if needed.
			// cgp_log( 'CGP: Failed to insert audit log entry for finding ' . $finding_id );
		}
	}

	/**
	 * Get client IP address.
	 *
	 * Uses REMOTE_ADDR as primary source to prevent IP spoofing.
	 * Falls back to X-Forwarded-For only if REMOTE_ADDR is a known proxy/local IP.
	 *
	 * @since 1.0.0
	 * @return string IP address.
	 */
	private function get_client_ip() {
		$ip = '';

		// REMOTE_ADDR is the most reliable source (cannot be spoofed by client).
		if ( ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
			$ip = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
		}

		// Validate the IP address.
		$validated_ip = filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE );

		// If REMOTE_ADDR is private/reserved (indicating proxy), check X-Forwarded-For.
		if ( ! $validated_ip && ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$forwarded_ips = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
			
			// HTTP_X_FORWARDED_FOR can contain multiple IPs (comma-separated).
			// Format: client, proxy1, proxy2
			$ip_list = array_map( 'trim', explode( ',', $forwarded_ips ) );
			
			// Get the first valid public IP.
			foreach ( $ip_list as $forwarded_ip ) {
				$validated = filter_var( $forwarded_ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE );
				if ( $validated ) {
					return $validated;
				}
			}
		}

		// Return validated REMOTE_ADDR or fallback.
		return $validated_ip ? $validated_ip : '0.0.0.0';
	}

	/**
	 * Display message when no items found.
	 *
	 * @since 1.0.0
	 */
	public function no_items() {
		esc_html_e( 'No quarantined findings. All clear!', 'content-guard-pro' );
	}
}

