<?php
/**
 * Notifications Class
 *
 * Handles all notification and alerting functionality for Content Guard Pro.
 * Implements PRD Section 3.5: Notifications and Alerts.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Notifications
 *
 * Manages notifications for security findings:
 * - Admin notices for Critical findings
 * - Admin bar badge with count
 * - Email alerts (immediate or digest)
 * - Webhook notifications
 *
 * @since 1.0.0
 */
class CGP_Notifications {

	/**
	 * Transient key for tracking notified findings.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const NOTIFIED_TRANSIENT = 'content_guard_pro_notified_findings';

	/**
	 * Initialize the notifications system.
	 *
	 * @since 1.0.0
	 */
	public static function init() {
		$instance = new self();
		$instance->register_hooks();
	}

	/**
	 * Register WordPress hooks.
	 *
	 * @since 1.0.0
	 */
	private function register_hooks() {
		// Hook into finding saved action.
		add_action( 'content_guard_pro_finding_saved', array( $this, 'on_finding_saved' ), 10, 2 );

		// Admin notices.
		add_action( 'admin_notices', array( $this, 'display_admin_notices' ) );

		// Admin bar badge.
		add_action( 'admin_bar_menu', array( $this, 'add_admin_bar_badge' ), 100 );

		// Admin bar styles.
		add_action( 'admin_head', array( $this, 'admin_bar_styles' ) );
		add_action( 'wp_head', array( $this, 'admin_bar_styles' ) );

		// Daily digest cron.
		add_action( 'content_guard_pro_daily_digest', array( $this, 'send_daily_digest' ) );
	}

	/**
	 * Handle finding saved event.
	 *
	 * Triggered when a finding is saved or updated.
	 *
	 * @since 1.0.0
	 * @param int   $finding_id   Finding ID.
	 * @param array $finding_data Finding data.
	 */
	public function on_finding_saved( $finding_id, $finding_data ) {
		// Only process Critical severity.
		if ( ! isset( $finding_data['severity'] ) || 'critical' !== $finding_data['severity'] ) {
			return;
		}

		// Check if this is a new finding (not an update).
		if ( ! $this->is_new_finding( $finding_id ) ) {
			return;
		}

		// Mark as notified to avoid duplicate notifications.
		$this->mark_as_notified( $finding_id );

		// Trigger notifications.
		$this->trigger_admin_notice( $finding_id, $finding_data );
		$this->trigger_email_alert( $finding_id, $finding_data );
		$this->trigger_webhook( $finding_id, $finding_data );
	}

	/**
	 * Trigger admin notice for Critical finding.
	 *
	 * Stores notice in transient for display on next admin page load.
	 *
	 * @since 1.0.0
	 * @param int   $finding_id   Finding ID.
	 * @param array $finding_data Finding data.
	 */
	public function trigger_admin_notice( $finding_id, $finding_data ) {
		$notices = get_transient( 'content_guard_pro_admin_notices' );
		if ( ! is_array( $notices ) ) {
			$notices = array();
		}

		// Limit to 5 most recent notices.
		if ( count( $notices ) >= 5 ) {
			array_shift( $notices );
		}

		// Add new notice.
		$notices[] = array(
			'finding_id' => $finding_id,
			'severity'   => $finding_data['severity'],
			'rule_id'    => isset( $finding_data['rule_id'] ) ? $finding_data['rule_id'] : '',
			'object_id'  => isset( $finding_data['object_id'] ) ? $finding_data['object_id'] : 0,
			'object_type' => isset( $finding_data['object_type'] ) ? $finding_data['object_type'] : '',
			'timestamp'  => time(),
		);

		// Store for 1 hour.
		set_transient( 'content_guard_pro_admin_notices', $notices, HOUR_IN_SECONDS );
	}

	/**
	 * Display admin notices.
	 *
	 * Shows stored notices for Critical findings.
	 *
	 * @since 1.0.0
	 */
	public function display_admin_notices() {
		// Only show to users who can manage options.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Check if admin notices are enabled in settings.
		$settings = get_option( 'content_guard_pro_settings', array() );
		if ( empty( $settings['notifications_admin'] ) ) {
			return;
		}

		$notices = get_transient( 'content_guard_pro_admin_notices' );
		if ( ! is_array( $notices ) || empty( $notices ) ) {
			return;
		}

		foreach ( $notices as $notice ) {
			$finding_id  = isset( $notice['finding_id'] ) ? absint( $notice['finding_id'] ) : 0;
			$rule_id     = isset( $notice['rule_id'] ) ? $notice['rule_id'] : '';
			$object_id   = isset( $notice['object_id'] ) ? absint( $notice['object_id'] ) : 0;
			$object_type = isset( $notice['object_type'] ) ? $notice['object_type'] : '';

			// Get location display (title + ID) for better UX.
			$location = $this->get_location_display( $object_id, $object_type );

			// Build message.
			$message = sprintf(
				/* translators: %1$s: rule ID, %2$s: location (title + ID) */
				__( 'Critical security finding detected: %1$s in %2$s. ', 'content-guard-pro' ),
				'<strong>' . esc_html( $rule_id ) . '</strong>',
				$location
			);

			// Add review link.
			$review_url = admin_url( 'admin.php?page=content-guard-pro-findings' );
			$message   .= sprintf(
				'<a href="%s">%s</a>',
				esc_url( $review_url ),
				__( 'Review findings &rarr;', 'content-guard-pro' )
			);

			printf(
				'<div class="notice notice-error is-dismissible content-guard-pro-critical-notice">
					<p>%s</p>
				</div>',
				wp_kses_post( $message )
			);
		}

		// Clear notices after display.
		delete_transient( 'content_guard_pro_admin_notices' );
	}

	/**
	 * Get location display string with title and ID.
	 *
	 * Returns a formatted location string like "Article Title (ID: 123)".
	 *
	 * @since 1.0.0
	 * @param int    $object_id   Object ID.
	 * @param string $object_type Object type (post, option, etc.).
	 * @return string Formatted location string.
	 */
	private function get_location_display( $object_id, $object_type ) {
		if ( 'post' === $object_type ) {
			$post = get_post( $object_id );
			if ( $post ) {
				$title = get_the_title( $post );
				// Truncate long titles.
				if ( strlen( $title ) > 40 ) {
					$title = substr( $title, 0, 37 ) . '...';
				}
				if ( $title ) {
					return sprintf(
						'<strong>%s</strong> (ID: %d)',
						esc_html( $title ),
						$object_id
					);
				}
			}
		}

		// Fallback for non-post objects or if title not available.
		$type_label = ucfirst( $object_type );
		return sprintf( '%s (ID: %d)', esc_html( $type_label ), $object_id );
	}

	/**
	 * Get location display string for email notifications.
	 *
	 * Returns a formatted location string like "Article Title (ID: 123)" for emails.
	 *
	 * @since 1.0.0
	 * @param int    $object_id   Object ID.
	 * @param string $object_type Object type (post, option, etc.).
	 * @return string Formatted location string for email.
	 */
	private function get_email_location_display( $object_id, $object_type ) {
		if ( 'post' === $object_type ) {
			$post = get_post( $object_id );
			if ( $post ) {
				$title = get_the_title( $post );
				// Truncate long titles for email.
				if ( strlen( $title ) > 50 ) {
					$title = substr( $title, 0, 47 ) . '...';
				}
				if ( $title ) {
					return sprintf(
						'<strong>%s</strong> (ID: %d)',
						esc_html( $title ),
						$object_id
					);
				}
			}
		}

		// Fallback for non-post objects or if title not available.
		$type_label = ucfirst( $object_type );
		return sprintf( '%s (ID: %d)', esc_html( $type_label ), $object_id );
	}

	/**
	 * Get location display string for digest emails (plain text format).
	 *
	 * Returns a formatted location string like "Article Title (ID: 123)" for digest.
	 *
	 * @since 1.0.0
	 * @param int    $object_id   Object ID.
	 * @param string $object_type Object type (post, option, etc.).
	 * @return string Formatted location string for digest.
	 */
	private function get_digest_location_display( $object_id, $object_type ) {
		if ( 'post' === $object_type ) {
			$post = get_post( $object_id );
			if ( $post ) {
				$title = get_the_title( $post );
				// Truncate long titles for digest.
				if ( strlen( $title ) > 40 ) {
					$title = substr( $title, 0, 37 ) . '...';
				}
				if ( $title ) {
					return sprintf(
						'%s (ID: %d)',
						esc_html( $title ),
						$object_id
					);
				}
			}
		}

		// Fallback for non-post objects or if title not available.
		$type_label = ucfirst( $object_type );
		return sprintf( '%s (ID: %d)', esc_html( $type_label ), $object_id );
	}

	/**
	 * Add admin bar badge.
	 *
	 * Shows count of open Critical findings with dropdown.
	 *
	 * @since 1.0.0
	 * @param WP_Admin_Bar $wp_admin_bar Admin bar object.
	 */
	public function add_admin_bar_badge( $wp_admin_bar ) {
		// Only show to users who can manage options.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Check if admin bar badge is enabled in settings.
		$settings = get_option( 'content_guard_pro_settings', array() );
		if ( empty( $settings['notifications_admin'] ) ) {
			return;
		}

		// Get count of open Critical findings.
		$count = $this->get_critical_findings_count();

		if ( $count === 0 ) {
			return;
		}

		// Add parent menu.
		$wp_admin_bar->add_node(
			array(
				'id'    => 'content-guard-pro-alerts',
				'title' => sprintf(
					'<span class="dashicons dashicons-shield-alt"></span> <span class="content-guard-pro-admin-bar-count">%d</span>',
					absint( $count )
				),
				'href'  => admin_url( 'admin.php?page=content-guard-pro-findings&severity=critical' ),
				'meta'  => array(
					'title' => sprintf(
						/* translators: %d: number of critical findings */
						_n(
							'%d Critical Finding',
							'%d Critical Findings',
							$count,
							'content-guard-pro'
						),
						$count
					),
				),
			)
		);

		// Get recent Critical findings for dropdown.
		$recent_findings = $this->get_recent_critical_findings( 5 );

		if ( ! empty( $recent_findings ) ) {
			foreach ( $recent_findings as $finding ) {
				$wp_admin_bar->add_node(
					array(
						'parent' => 'content-guard-pro-alerts',
						'id'     => 'content-guard-pro-finding-' . $finding->id,
						'title'  => $this->format_finding_title( $finding ),
						'href'   => admin_url( 'admin.php?page=content-guard-pro-findings&finding_id=' . $finding->id ),
					)
				);
			}

			// Add "View All" link.
			$wp_admin_bar->add_node(
				array(
					'parent' => 'content-guard-pro-alerts',
					'id'     => 'content-guard-pro-view-all',
					'title'  => __( 'View All Findings &rarr;', 'content-guard-pro' ),
					'href'   => admin_url( 'admin.php?page=content-guard-pro-findings&severity=critical' ),
				)
			);
		}
	}

	/**
	 * Add admin bar styles.
	 *
	 * @since 1.0.0
	 */
	public function admin_bar_styles() {
		if ( ! is_admin_bar_showing() || ! current_user_can( 'manage_options' ) ) {
			return;
		}

		?>
		<style>
			#wpadminbar .content-guard-pro-admin-bar-icon {
				font-size: 18px;
				margin-top: 3px;
			}
			#wpadminbar .content-guard-pro-admin-bar-count {
				background: #d63638;
				color: #fff;
				border-radius: 10px;
				padding: 2px 6px;
				font-size: 11px;
				font-weight: 600;
				margin-left: 4px;
			}
			#wpadminbar #wp-admin-bar-content-guard-pro-alerts .ab-item {
				color: #d63638 !important;
			}
			#wpadminbar #wp-admin-bar-content-guard-pro-alerts:hover .ab-item {
				background: rgba(214, 54, 56, 0.1) !important;
			}
		</style>
		<?php
	}

	/**
	 * Trigger email alert for Critical finding.
	 *
	 * Sends immediate email notification to configured recipients.
	 *
	 * @since 1.0.0
	 * @param int   $finding_id   Finding ID.
	 * @param array $finding_data Finding data.
	 */
	public function trigger_email_alert( $finding_id, $finding_data ) {
		// Get email settings.
		$settings = get_option( 'content_guard_pro_settings', array() );

		// Check if email alerts are enabled.
		if ( ! isset( $settings['email_alerts_enabled'] ) || ! $settings['email_alerts_enabled'] ) {
			return;
		}

		// Get email mode (immediate or digest).
		$email_mode = isset( $settings['email_alert_mode'] ) ? $settings['email_alert_mode'] : 'immediate';

		// For digest mode, findings are queued and sent daily.
		if ( 'digest' === $email_mode ) {
			$this->queue_for_digest( $finding_id, $finding_data );
			return;
		}

		// Immediate mode: send email now.
		$this->send_finding_email( $finding_id, $finding_data, $settings );
	}

	/**
	 * Send email for a single finding.
	 *
	 * @since 1.0.0
	 * @param int   $finding_id   Finding ID.
	 * @param array $finding_data Finding data.
	 * @param array $settings     Plugin settings (optional).
	 * @return bool True on success, false on failure.
	 */
	private function send_finding_email( $finding_id, $finding_data, $settings = array() ) {
		// Get recipients.
		$recipients = $this->get_email_recipients();
		if ( empty( $recipients ) ) {
			return false;
		}

		// Build email subject.
		$subject = sprintf(
			/* translators: %s: site name */
			__( '[Content Guard Pro] Critical Finding Detected on %s', 'content-guard-pro' ),
			get_bloginfo( 'name' )
		);

		// Build email body.
		$body = $this->build_finding_email_body( $finding_id, $finding_data );

		// Email headers.
		$headers = array(
			'Content-Type: text/html; charset=UTF-8',
			'From: ' . get_bloginfo( 'name' ) . ' <' . get_option( 'admin_email' ) . '>',
		);

		// Send email.
		$sent = wp_mail( $recipients, $subject, $body, $headers );

		// Log email send status.
		if ( $sent ) {
			cgp_log( 'Content Guard Pro: Email alert sent for finding ' . $finding_id );
		} else {
			cgp_log( 'Content Guard Pro: Failed to send email alert for finding ' . $finding_id );
		}

		return $sent;
	}

	/**
	 * Build email body for finding notification.
	 *
	 * @since 1.0.0
	 * @param int   $finding_id   Finding ID.
	 * @param array $finding_data Finding data.
	 * @return string HTML email body.
	 */
	private function build_finding_email_body( $finding_id, $finding_data ) {
		$rule_id      = isset( $finding_data['rule_id'] ) ? $finding_data['rule_id'] : __( 'Unknown', 'content-guard-pro' );
		$severity     = isset( $finding_data['severity'] ) ? $finding_data['severity'] : 'unknown';
		$confidence   = isset( $finding_data['confidence'] ) ? absint( $finding_data['confidence'] ) : 0;
		$object_type  = isset( $finding_data['object_type'] ) ? $finding_data['object_type'] : 'unknown';
		$object_id    = isset( $finding_data['object_id'] ) ? absint( $finding_data['object_id'] ) : 0;
		$matched_text = isset( $finding_data['matched_text'] ) ? wp_strip_all_tags( substr( $finding_data['matched_text'], 0, 200 ) ) : '';

		// Get location display (title + ID) for email.
		$location_display = $this->get_email_location_display( $object_id, $object_type );

		// Review URL.
		$review_url = admin_url( 'admin.php?page=content-guard-pro-findings&finding_id=' . $finding_id );

		// Edit URL (if post).
		$edit_url = '';
		if ( 'post' === $object_type ) {
			$edit_url = get_edit_post_link( $object_id, 'raw' );
		}

		// Build HTML email.
		ob_start();
		?>
		<!DOCTYPE html>
		<html>
		<head>
			<meta charset="UTF-8">
			<style>
				body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
				.container { max-width: 600px; margin: 0 auto; padding: 20px; }
				.header { background: #d63638; color: #fff; padding: 20px; border-radius: 5px 5px 0 0; }
				.content { background: #f9f9f9; padding: 20px; border: 1px solid #ddd; border-top: none; }
				.footer { background: #f0f0f0; padding: 15px; text-align: center; border-radius: 0 0 5px 5px; font-size: 12px; }
				.button { display: inline-block; background: #2271b1; color: #fff; padding: 10px 20px; text-decoration: none; border-radius: 3px; margin: 10px 5px; }
				.detail { margin: 10px 0; padding: 10px; background: #fff; border-left: 4px solid #d63638; }
				.detail strong { display: inline-block; width: 120px; }
				.severity-critical { color: #d63638; font-weight: bold; }
			</style>
		</head>
		<body>
			<div class="container">
				<div class="header">
					<h1>⚠️ <?php esc_html_e( 'Critical Security Finding Detected', 'content-guard-pro' ); ?></h1>
				</div>
				<div class="content">
					<p><?php esc_html_e( 'Content Guard Pro has detected a critical security issue on your WordPress site.', 'content-guard-pro' ); ?></p>

					<div class="detail">
						<strong><?php esc_html_e( 'Rule ID:', 'content-guard-pro' ); ?></strong>
						<code><?php echo esc_html( $rule_id ); ?></code>
					</div>

					<div class="detail">
						<strong><?php esc_html_e( 'Severity:', 'content-guard-pro' ); ?></strong>
						<span class="severity-critical"><?php echo esc_html( ucfirst( $severity ) ); ?></span>
					</div>

					<div class="detail">
						<strong><?php esc_html_e( 'Confidence:', 'content-guard-pro' ); ?></strong>
						<?php echo absint( $confidence ); ?>%
					</div>

					<div class="detail">
						<strong><?php esc_html_e( 'Location:', 'content-guard-pro' ); ?></strong>
						<?php echo wp_kses_post( $location_display ); ?>
					</div>

					<?php if ( $matched_text ) : ?>
					<div class="detail">
						<strong><?php esc_html_e( 'Match:', 'content-guard-pro' ); ?></strong>
						<?php echo esc_html( $matched_text ); ?>...
					</div>
					<?php endif; ?>

					<div style="margin: 20px 0;">
						<a href="<?php echo esc_url( $review_url ); ?>" class="button">
							<?php esc_html_e( 'Review Finding', 'content-guard-pro' ); ?>
						</a>

						<?php if ( $edit_url ) : ?>
						<a href="<?php echo esc_url( $edit_url ); ?>" class="button">
							<?php esc_html_e( 'Edit Content', 'content-guard-pro' ); ?>
						</a>
						<?php endif; ?>
					</div>

					<p><strong><?php esc_html_e( 'Recommended Action:', 'content-guard-pro' ); ?></strong></p>
					<ol>
						<li><?php esc_html_e( 'Review the finding in your WordPress admin', 'content-guard-pro' ); ?></li>
						<li><?php esc_html_e( 'Quarantine the content if malicious', 'content-guard-pro' ); ?></li>
						<li><?php esc_html_e( 'Or mark as false positive if legitimate', 'content-guard-pro' ); ?></li>
					</ol>
				</div>
				<div class="footer">
					<p>
						<?php
						printf(
							/* translators: %s: site URL */
							esc_html__( 'This email was sent by Content Guard Pro monitoring %s', 'content-guard-pro' ),
							'<a href="' . esc_url( home_url() ) . '">' . esc_html( get_bloginfo( 'name' ) ) . '</a>'
						);
						?>
					</p>
					<p><?php esc_html_e( 'To stop receiving these alerts, disable email notifications in Content Guard Pro settings.', 'content-guard-pro' ); ?></p>
				</div>
			</div>
		</body>
		</html>
		<?php
		return ob_get_clean();
	}

	/**
	 * Queue finding for daily digest.
	 *
	 * @since 1.0.0
	 * @param int   $finding_id   Finding ID.
	 * @param array $finding_data Finding data.
	 */
	private function queue_for_digest( $finding_id, $finding_data ) {
		$queue = get_option( 'content_guard_pro_digest_queue', array() );

		$queue[] = array(
			'finding_id'   => $finding_id,
			'finding_data' => $finding_data,
			'queued_at'    => time(),
		);

		update_option( 'content_guard_pro_digest_queue', $queue, false );
	}

	/**
	 * Send daily digest email.
	 *
	 * Triggered by cron job.
	 *
	 * @since 1.0.0
	 */
	public function send_daily_digest() {
		$queue = get_option( 'content_guard_pro_digest_queue', array() );

		if ( empty( $queue ) ) {
			return;
		}

		$settings   = get_option( 'content_guard_pro_settings', array() );
		$recipients = $this->get_email_recipients();

		if ( empty( $recipients ) ) {
			return;
		}

		// Build digest email.
		$subject = sprintf(
			/* translators: %s: site name */
			__( '[Content Guard Pro] Daily Security Digest for %s', 'content-guard-pro' ),
			get_bloginfo( 'name' )
		);

		$body = $this->build_digest_email_body( $queue );

		// Email headers.
		$headers = array(
			'Content-Type: text/html; charset=UTF-8',
			'From: ' . get_bloginfo( 'name' ) . ' <' . get_option( 'admin_email' ) . '>',
		);

		// Send email.
		$sent = wp_mail( $recipients, $subject, $body, $headers );

		if ( $sent ) {
			// Clear queue after sending.
			delete_option( 'content_guard_pro_digest_queue' );
			cgp_log( 'Content Guard Pro: Daily digest sent with ' . count( $queue ) . ' findings' );
		}
	}

	/**
	 * Build daily digest email body.
	 *
	 * @since 1.0.0
	 * @param array $queue Queue of findings.
	 * @return string HTML email body.
	 */
	private function build_digest_email_body( $queue ) {
		$count = count( $queue );

		ob_start();
		?>
		<!DOCTYPE html>
		<html>
		<head>
			<meta charset="UTF-8">
			<style>
				body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
				.container { max-width: 600px; margin: 0 auto; padding: 20px; }
				.header { background: #d63638; color: #fff; padding: 20px; border-radius: 5px 5px 0 0; }
				.content { background: #f9f9f9; padding: 20px; border: 1px solid #ddd; border-top: none; }
				.footer { background: #f0f0f0; padding: 15px; text-align: center; border-radius: 0 0 5px 5px; font-size: 12px; }
				.finding { margin: 15px 0; padding: 15px; background: #fff; border-left: 4px solid #d63638; }
				.button { display: inline-block; background: #2271b1; color: #fff; padding: 10px 20px; text-decoration: none; border-radius: 3px; margin: 10px 0; }
			</style>
		</head>
		<body>
			<div class="container">
				<div class="header">
					<h1>
						📊 <?php
						printf(
							/* translators: %d: number of findings */
							esc_html__( 'Daily Security Digest - %d Findings', 'content-guard-pro' ),
							absint( $count )
						);
						?>
					</h1>
				</div>
				<div class="content">
					<p>
						<?php
						printf(
							/* translators: %d: number of findings */
							esc_html( _n(
								'Content Guard Pro detected %d critical security issue in the last 24 hours.',
								'Content Guard Pro detected %d critical security issues in the last 24 hours.',
								$count,
								'content-guard-pro'
							) ),
							absint( $count )
						);
						?>
					</p>

					<?php foreach ( $queue as $item ) : ?>
						<?php
						$finding_id   = $item['finding_id'];
						$finding_data = $item['finding_data'];
						$rule_id      = isset( $finding_data['rule_id'] ) ? $finding_data['rule_id'] : 'Unknown';
						$object_type  = isset( $finding_data['object_type'] ) ? $finding_data['object_type'] : 'unknown';
						$object_id    = isset( $finding_data['object_id'] ) ? absint( $finding_data['object_id'] ) : 0;
						$location     = $this->get_digest_location_display( $object_id, $object_type );
						?>
						<div class="finding">
							<strong><?php echo esc_html( $rule_id ); ?></strong><br>
							<?php
							printf(
								/* translators: %s: location (title + ID) */
								esc_html__( 'Location: %s', 'content-guard-pro' ),
								esc_html( $location )
							);
							?>
						</div>
					<?php endforeach; ?>

					<div style="margin: 20px 0; text-align: center;">
						<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-findings&severity=critical' ) ); ?>" class="button">
							<?php esc_html_e( 'Review All Findings', 'content-guard-pro' ); ?>
						</a>
					</div>
				</div>
				<div class="footer">
					<p>
						<?php
						printf(
							/* translators: %s: site URL */
							esc_html__( 'This email was sent by Content Guard Pro monitoring %s', 'content-guard-pro' ),
							'<a href="' . esc_url( home_url() ) . '">' . esc_html( get_bloginfo( 'name' ) ) . '</a>'
						);
						?>
					</p>
				</div>
			</div>
		</body>
		</html>
		<?php
		return ob_get_clean();
	}

	/**
	 * Trigger webhook notification.
	 *
	 * Sends JSON payload to configured webhook URL per PRD Appendix D.
	 *
	 * @since 1.0.0
	 * @param int   $finding_id   Finding ID.
	 * @param array $finding_data Finding data.
	 */
	public function trigger_webhook( $finding_id, $finding_data ) {
		$settings = get_option( 'content_guard_pro_settings', array() );

		// Check if webhook is enabled and URL is configured.
		if ( ! isset( $settings['webhook_enabled'] ) || ! $settings['webhook_enabled'] ) {
			return;
		}

		$webhook_url = isset( $settings['webhook_url'] ) ? $settings['webhook_url'] : '';
		if ( empty( $webhook_url ) || ! filter_var( $webhook_url, FILTER_VALIDATE_URL ) ) {
			return;
		}

		// Build payload per PRD Appendix D.
		$payload = $this->build_webhook_payload( $finding_id, $finding_data );

		// Prepare request headers.
		$headers = array(
			'Content-Type' => 'application/json',
		);

		// Add HMAC signature if secret is configured.
		$webhook_secret = isset( $settings['webhook_secret'] ) ? $settings['webhook_secret'] : '';
		if ( ! empty( $webhook_secret ) ) {
			$signature = hash_hmac( 'sha256', wp_json_encode( $payload ), $webhook_secret );
			$headers['X-Content-Guard-Pro-Signature'] = $signature;
		}

		// Send webhook.
		$response = wp_remote_post(
			$webhook_url,
			array(
				'headers' => $headers,
				'body'    => wp_json_encode( $payload ),
				'timeout' => 15,
			)
		);

		// Log webhook result.
		if ( is_wp_error( $response ) ) {
			cgp_log( 'Content Guard Pro: Webhook failed - ' . $response->get_error_message() );
		} else {
			$status_code = wp_remote_retrieve_response_code( $response );
			cgp_log( 'Content Guard Pro: Webhook sent for finding ' . $finding_id . ' (status: ' . $status_code . ')' );

			// If failed, queue for retry.
			if ( $status_code < 200 || $status_code >= 300 ) {
				$this->queue_webhook_retry( $finding_id, $finding_data, $webhook_url );
			}
		}
	}

	/**
	 * Build webhook payload per PRD Appendix D.
	 *
	 * @since 1.0.0
	 * @param int   $finding_id   Finding ID.
	 * @param array $finding_data Finding data.
	 * @return array Webhook payload.
	 */
	private function build_webhook_payload( $finding_id, $finding_data ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		// Get full finding details from database.
		$finding = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$table_name}` WHERE id = %d",
				$finding_id
			)
		);

		if ( ! $finding ) {
			// Fallback to provided data.
			$finding = (object) $finding_data;
		}

		// Build payload structure from PRD Appendix D.
		$payload = array(
			'site' => array(
				'url'     => home_url(),
				'blog_id' => get_current_blog_id(),
			),
			'event' => 'critical_finding',
			'finding' => array(
				'id'          => $finding_id,
				'fingerprint' => isset( $finding->fingerprint ) ? $finding->fingerprint : '',
				'severity'    => isset( $finding->severity ) ? $finding->severity : 'critical',
				'confidence'  => isset( $finding->confidence ) ? absint( $finding->confidence ) : 0,
				'rule_id'     => isset( $finding->rule_id ) ? $finding->rule_id : '',
				'object'      => array(
					'type'  => isset( $finding->object_type ) ? $finding->object_type : '',
					'id'    => isset( $finding->object_id ) ? absint( $finding->object_id ) : 0,
					'field' => isset( $finding->field ) ? $finding->field : '',
				),
				'summary'     => $this->build_finding_summary( $finding ),
				'first_seen'  => $this->format_datetime_for_webhook( $finding->first_seen ?? '' ),
				'last_seen'   => $this->format_datetime_for_webhook( $finding->last_seen ?? '' ),
				'actions'     => array(
					'review_url' => admin_url( 'admin.php?page=content-guard-pro-findings&finding=' . $finding_id ),
				),
			),
		);

		return $payload;
	}

	/**
	 * Build finding summary for webhook/notifications.
	 *
	 * @since 1.0.0
	 * @param object $finding Finding object.
	 * @return string Summary text.
	 */
	private function build_finding_summary( $finding ) {
		$rule_id     = isset( $finding->rule_id ) ? $finding->rule_id : 'unknown';
		$object_type = isset( $finding->object_type ) ? $finding->object_type : 'unknown';

		// Build human-readable summary based on rule_id.
		$summaries = array(
			'ext_script_non_allowlist' => __( 'External script from non-allowlisted domain', 'content-guard-pro' ),
			'ext_iframe_non_allowlist' => __( 'External iframe from non-allowlisted domain', 'content-guard-pro' ),
			'hidden_external_link'     => __( 'Hidden element with external link', 'content-guard-pro' ),
			'url_shortener'            => __( 'Suspicious URL shortener detected', 'content-guard-pro' ),
			'obfuscation_js'           => __( 'JavaScript obfuscation detected', 'content-guard-pro' ),
			'obfuscation_base64_data_url' => __( 'Base64 data URL detected', 'content-guard-pro' ),
		);

		$summary = isset( $summaries[ $rule_id ] ) ? $summaries[ $rule_id ] : ucwords( str_replace( '_', ' ', $rule_id ) );

		return sprintf(
			/* translators: 1: rule summary, 2: object type */
			__( '%1$s in %2$s', 'content-guard-pro' ),
			$summary,
			$object_type
		);
	}

	/**
	 * Format datetime for webhook payload.
	 *
	 * @since 1.0.0
	 * @param string $datetime Datetime string.
	 * @return string ISO 8601 formatted datetime or empty string.
	 */
	private function format_datetime_for_webhook( $datetime ) {
		if ( empty( $datetime ) ) {
			return '';
		}

		$timestamp = strtotime( $datetime );
		if ( false === $timestamp ) {
			return '';
		}

		return gmdate( 'c', $timestamp );
	}

	/**
	 * Queue webhook for retry.
	 *
	 * Uses exponential backoff (future enhancement).
	 *
	 * @since 1.0.0
	 * @param int    $finding_id   Finding ID.
	 * @param array  $finding_data Finding data.
	 * @param string $webhook_url  Webhook URL.
	 */
	private function queue_webhook_retry( $finding_id, $finding_data, $webhook_url ) {
		// Store in transient for retry (expires in 1 hour).
		$retry_queue = get_transient( 'content_guard_pro_webhook_retry_queue' );
		if ( ! is_array( $retry_queue ) ) {
			$retry_queue = array();
		}

		$retry_queue[] = array(
			'finding_id'   => $finding_id,
			'finding_data' => $finding_data,
			'webhook_url'  => $webhook_url,
			'attempts'     => 1,
			'next_retry'   => time() + 300, // Retry in 5 minutes.
		);

		set_transient( 'content_guard_pro_webhook_retry_queue', $retry_queue, HOUR_IN_SECONDS );
	}

	/**
	 * Get count of open Critical findings.
	 *
	 * @since 1.0.0
	 * @return int Count.
	 */
	private function get_critical_findings_count() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		// Cache for 5 minutes.
		$cache_key = 'content_guard_pro_critical_count';
		$count     = wp_cache_get( $cache_key );

		if ( false === $count ) {
			$count = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM `{$table_name}`
					WHERE severity = %s AND status = %s",
					'critical',
					'open'
				)
			);

			wp_cache_set( $cache_key, $count, '', 300 );
		}

		return absint( $count );
	}

	/**
	 * Get recent Critical findings.
	 *
	 * @since 1.0.0
	 * @param int $limit Limit.
	 * @return array Findings.
	 */
	private function get_recent_critical_findings( $limit = 5 ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		// Validate limit parameter.
		$limit = absint( $limit );
		if ( $limit < 1 ) {
			$limit = 5;
		}

		return $wpdb->get_results(
			$wpdb->prepare(
				"SELECT * FROM `{$table_name}`
				WHERE severity = %s AND status = %s
				ORDER BY last_seen DESC
				LIMIT %d",
				'critical',
				'open',
				$limit
			)
		);
	}

	/**
	 * Format finding title for admin bar dropdown.
	 *
	 * @since 1.0.0
	 * @param object $finding Finding object.
	 * @return string Formatted title.
	 */
	private function format_finding_title( $finding ) {
		$rule_label = ucwords( str_replace( '_', ' ', $finding->rule_id ) );

		if ( 'post' === $finding->object_type ) {
			$post = get_post( $finding->object_id );
			if ( $post ) {
				$title = get_the_title( $post );
				return sprintf( '%s: %s', $rule_label, wp_trim_words( $title, 5 ) );
			}
		}

		return sprintf(
			'%s (%s ID: %d)',
			$rule_label,
			ucfirst( $finding->object_type ),
			$finding->object_id
		);
	}

	/**
	 * Get email recipients from settings.
	 *
	 * @since 1.0.0
	 * @return array Email addresses.
	 */
	private function get_email_recipients() {
		$settings = get_option( 'content_guard_pro_settings', array() );

		$recipients = isset( $settings['email_recipients'] ) ? $settings['email_recipients'] : '';

		// Default to admin email if not configured.
		if ( empty( $recipients ) ) {
			return array( get_option( 'admin_email' ) );
		}

		// Parse comma-separated list.
		$recipients = array_map( 'trim', explode( ',', $recipients ) );
		$recipients = array_filter( $recipients, 'is_email' );

		return $recipients;
	}

	/**
	 * Check if finding is new (not an update).
	 *
	 * @since 1.0.0
	 * @param int $finding_id Finding ID.
	 * @return bool True if new, false if update.
	 */
	private function is_new_finding( $finding_id ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		$finding = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT first_seen, last_seen FROM `{$table_name}` WHERE id = %d",
				$finding_id
			)
		);

		if ( ! $finding ) {
			return true;
		}

		// Consider it new if first_seen equals last_seen (within 1 second).
		$first_seen = strtotime( $finding->first_seen );
		$last_seen  = strtotime( $finding->last_seen );

		// Check if strtotime() failed.
		if ( false === $first_seen || false === $last_seen ) {
			return true; // Treat as new if timestamps cannot be parsed.
		}

		return abs( $first_seen - $last_seen ) < 2;
	}

	/**
	 * Mark finding as notified.
	 *
	 * Prevents duplicate notifications for the same finding.
	 *
	 * @since 1.0.0
	 * @param int $finding_id Finding ID.
	 */
	private function mark_as_notified( $finding_id ) {
		$notified = get_transient( self::NOTIFIED_TRANSIENT );
		if ( ! is_array( $notified ) ) {
			$notified = array();
		}

		$notified[] = $finding_id;

		// Store for 24 hours.
		set_transient( self::NOTIFIED_TRANSIENT, $notified, DAY_IN_SECONDS );
	}
}

