<?php
/**
 * License Manager Class
 *
 * Handles license validation, feature gating, and tier management
 * for Content Guard Pro. Implements the pricing tiers as defined in the PRD.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_License_Manager
 *
 * Manages license validation and feature availability based on plan tier.
 *
 * Plan Tiers (from PRD Section 5):
 * - Free: Basic scanning, limited features
 * - Solo Guard: Full features for 1 site
 * - Agency Lite: Multi-site (5), export, API
 * - Agency Pro: Multi-site (25), priority support
 * - Enterprise: Unlimited, white-label, SLA
 *
 * @since 1.0.0
 */
class CGP_License_Manager {

	/**
	 * Option key for license data.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const OPTION_LICENSE_DATA = 'content_guard_pro_license_data';

	/**
	 * Transient key for cached license validation.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const TRANSIENT_LICENSE_STATUS = 'content_guard_pro_license_status';

	/**
	 * Available license tiers.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	const TIERS = array(
		'free'         => 0,
		'solo_guard'   => 1,
		'agency_lite'  => 2,
		'agency_pro'   => 3,
		'enterprise'   => 4,
	);

	/**
	 * Tier display names.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	const TIER_NAMES = array(
		'free'         => 'Free',
		'solo_guard'   => 'Solo Guard',
		'agency_lite'  => 'Agency Lite',
		'agency_pro'   => 'Agency Pro',
		'enterprise'   => 'Enterprise',
	);

	/**
	 * Cached license data.
	 *
	 * @since 1.0.0
	 * @var array|null
	 */
	private static $license_cache = null;

	/**
	 * Initialize the license manager.
	 *
	 * @since 1.0.0
	 */
	public static function init() {
		// Schedule daily license check.
		add_action( 'content_guard_pro_daily_license_check', array( __CLASS__, 'daily_license_check' ) );

		// Register cron event if not scheduled.
		if ( ! wp_next_scheduled( 'content_guard_pro_daily_license_check' ) ) {
			wp_schedule_event( time(), 'daily', 'content_guard_pro_daily_license_check' );
		}

		// AJAX handlers for license management.
		add_action( 'wp_ajax_cgp_validate_license', array( __CLASS__, 'ajax_validate_license' ) );
		add_action( 'wp_ajax_cgp_deactivate_license', array( __CLASS__, 'ajax_deactivate_license' ) );
		add_action( 'wp_ajax_cgp_check_api_health', array( __CLASS__, 'ajax_check_api_health' ) );
	}

	/**
	 * Get the current license key.
	 *
	 * @since 1.0.0
	 * @return string License key or empty string.
	 */
	public static function get_license_key() {
		$settings = get_option( 'content_guard_pro_settings', array() );
		return isset( $settings['license_key'] ) ? sanitize_text_field( $settings['license_key'] ) : '';
	}

	/**
	 * Get the current license tier.
	 *
	 * @since 1.0.0
	 * @return string Tier slug (free, solo_guard, agency_lite, agency_pro, enterprise).
	 */
	public static function get_tier() {
		$license_data = self::get_license_data();
		
		if ( ! $license_data || ! isset( $license_data['valid'] ) || ! $license_data['valid'] ) {
			return 'free';
		}

		return isset( $license_data['license']['tier'] ) ? $license_data['license']['tier'] : 'free';
	}

	/**
	 * Get the tier display name.
	 *
	 * @since 1.0.0
	 * @param string $tier Optional tier slug. Uses current tier if not provided.
	 * @return string Tier display name.
	 */
	public static function get_tier_name( $tier = null ) {
		if ( null === $tier ) {
			$tier = self::get_tier();
		}
		return isset( self::TIER_NAMES[ $tier ] ) ? self::TIER_NAMES[ $tier ] : __( 'Free', 'content-guard-pro' );
	}

	/**
	 * Check if user is on a paid tier.
	 *
	 * @since 1.0.0
	 * @return bool True if paid, false if free.
	 */
	public static function is_paid() {
		return 'free' !== self::get_tier();
	}

	/**
	 * Check if user is on at least a specific tier.
	 *
	 * @since 1.0.0
	 * @param string $required_tier Required tier slug.
	 * @return bool True if user's tier is >= required tier.
	 */
	public static function has_tier( $required_tier ) {
		$current_tier = self::get_tier();
		$current_level = isset( self::TIERS[ $current_tier ] ) ? self::TIERS[ $current_tier ] : 0;
		$required_level = isset( self::TIERS[ $required_tier ] ) ? self::TIERS[ $required_tier ] : 0;

		return $current_level >= $required_level;
	}

	/**
	 * Get cached license data.
	 *
	 * @since 1.0.0
	 * @param bool $force_refresh Force API validation, bypassing cache.
	 * @return array|null License data or null.
	 */
	public static function get_license_data( $force_refresh = false ) {
		// Return cached data if available.
		if ( null !== self::$license_cache && ! $force_refresh ) {
			return self::$license_cache;
		}

		// Try to get from transient first.
		if ( ! $force_refresh ) {
			$cached = get_transient( self::TRANSIENT_LICENSE_STATUS );
			if ( false !== $cached ) {
				self::$license_cache = $cached;
				return $cached;
			}
		}

		// Get stored license data.
		$stored = get_option( self::OPTION_LICENSE_DATA );
		if ( $stored && ! $force_refresh ) {
			self::$license_cache = $stored;
			return $stored;
		}

		// Validate with API if we have a license key.
		$license_key = self::get_license_key();
		if ( ! empty( $license_key ) ) {
			$validation = CGP_API_Client::validate_license( $license_key, $force_refresh );
			
			if ( ! is_wp_error( $validation ) ) {
				// Store license data.
				update_option( self::OPTION_LICENSE_DATA, $validation );
				
				// Cache in transient.
				$cache_duration = isset( $validation['hints']['cache_seconds'] ) 
					? absint( $validation['hints']['cache_seconds'] ) 
					: DAY_IN_SECONDS;
				set_transient( self::TRANSIENT_LICENSE_STATUS, $validation, $cache_duration );
				
				self::$license_cache = $validation;
				return $validation;
			}
		}

		// Return free tier data if no valid license.
		$free_data = self::get_free_tier_data();
		self::$license_cache = $free_data;
		return $free_data;
	}

	/**
	 * Get default data for free tier.
	 *
	 * @since 1.0.0
	 * @return array Free tier license data.
	 */
	private static function get_free_tier_data() {
		return array(
			'valid'    => true,
			'license'  => array(
				'tier'   => 'free',
				'status' => 'active',
			),
			'features' => self::get_tier_features( 'free' ),
			'limits'   => self::get_tier_limits( 'free' ),
		);
	}

	/**
	 * Get features for a specific tier.
	 *
	 * Per PRD Section 5.3 - Feature Matrix.
	 *
	 * @since 1.0.0
	 * @param string $tier Tier slug.
	 * @return array Feature availability flags.
	 */
	public static function get_tier_features( $tier = null ) {
		if ( null === $tier ) {
			$tier = self::get_tier();
		}

		$tier_level = isset( self::TIERS[ $tier ] ) ? self::TIERS[ $tier ] : 0;

		return array(
			// Scanning Capabilities.
			'quick_scan'              => true,
			'standard_scan'           => $tier_level >= self::TIERS['solo_guard'],
			'gutenberg_scanning'      => true,
			'elementor_scanning'      => $tier_level >= self::TIERS['solo_guard'],
			'single_post_scanning'    => true, // Available to all tiers - single post scans (on-save and manual) don't count against quota.
			'on_save_scanning'        => $tier_level >= self::TIERS['solo_guard'], // Full site on-save scanning (background, scheduled).
			'scheduled_scans'         => $tier_level >= self::TIERS['solo_guard'],

			// Detection & Findings.
			'core_patterns'           => true,
			'enhanced_patterns'       => true,
			'seo_spam_lexicon'        => true,
			'confidence_scoring'      => true,
			'severity_classification' => true,
			'full_findings_access'    => $tier_level >= self::TIERS['solo_guard'],
			'finding_details'         => $tier_level >= self::TIERS['solo_guard'],

			// Reputation & Rules.
			'reputation_apis'         => $tier_level >= self::TIERS['solo_guard'],
			'immediate_rule_updates'  => $tier_level >= self::TIERS['solo_guard'],
			'full_allowlist'          => $tier_level >= self::TIERS['solo_guard'],
			'false_positive_reporting'=> true,

			// Remediation & Quarantine.
			'quarantine'              => $tier_level >= self::TIERS['solo_guard'],
			'bulk_quarantine'         => $tier_level >= self::TIERS['solo_guard'],
			'restore_quarantine'      => $tier_level >= self::TIERS['solo_guard'],
			'revision_rollback'       => $tier_level >= self::TIERS['solo_guard'],
			'one_click_edit'          => true,
			'ignore_finding'          => true,
			'cache_clearing'          => $tier_level >= self::TIERS['solo_guard'],
			'audit_log'               => $tier_level >= self::TIERS['solo_guard'],

			// Notifications & Alerts.
			'admin_notices'           => true,
			'admin_bar_badge'         => true,
			'dashboard_widget'        => true,
			'email_alerts'            => $tier_level >= self::TIERS['solo_guard'],
			'email_digest'            => $tier_level >= self::TIERS['solo_guard'],
			'webhook_alerts'          => $tier_level >= self::TIERS['agency_lite'],

			// Reporting & Export.
			'findings_list'           => true,
			'finding_filters'         => true,
			'printable_summary'       => $tier_level >= self::TIERS['solo_guard'],
			'csv_export'              => $tier_level >= self::TIERS['agency_lite'],
			'json_export'             => $tier_level >= self::TIERS['agency_lite'],
			'rest_api'                => $tier_level >= self::TIERS['agency_lite'],
			'white_label_reports'     => $tier_level >= self::TIERS['enterprise'],

			// Performance & Configuration.
			'auto_throttling'         => true,
			'safe_mode'               => true,
			'batch_size_tuning'       => $tier_level >= self::TIERS['solo_guard'],
			'setup_wizard'            => true,
			'diagnostics_mode'        => true,

			// Beta & Priority.
			'beta_channel'            => $tier_level >= self::TIERS['agency_pro'],
			'priority_support'        => $tier_level >= self::TIERS['agency_pro'],
		);
	}

	/**
	 * Get limits for a specific tier.
	 *
	 * Per PRD Section 5.3 - Feature Matrix.
	 *
	 * @since 1.0.0
	 * @param string $tier Tier slug.
	 * @return array Tier-specific limits.
	 */
	public static function get_tier_limits( $tier = null ) {
		if ( null === $tier ) {
			$tier = self::get_tier();
		}

		switch ( $tier ) {
			case 'enterprise':
				return array(
					'visible_findings'        => -1, // Unlimited.
					'manual_scans_per_day'    => -1,
					'allowlist_entries'       => -1,
					'notification_recipients' => -1,
					'scan_history_days'       => -1, // Custom.
					'rule_pack_delay_days'    => 0,
				);

			case 'agency_pro':
				return array(
					'visible_findings'        => -1,
					'manual_scans_per_day'    => -1,
					'allowlist_entries'       => -1,
					'notification_recipients' => 10,
					'scan_history_days'       => 365,
					'rule_pack_delay_days'    => 0,
				);

			case 'agency_lite':
				return array(
					'visible_findings'        => -1,
					'manual_scans_per_day'    => -1,
					'allowlist_entries'       => -1,
					'notification_recipients' => 5,
					'scan_history_days'       => 90,
					'rule_pack_delay_days'    => 0,
				);

			case 'solo_guard':
				return array(
					'visible_findings'        => -1,
					'manual_scans_per_day'    => -1,
					'allowlist_entries'       => -1,
					'notification_recipients' => 1,
					'scan_history_days'       => 90,
					'rule_pack_delay_days'    => 0,
				);

			case 'free':
			default:
				return array(
					'visible_findings'        => 10,
					'manual_scans_per_day'    => 1,
					'allowlist_entries'       => 5,
					'notification_recipients' => 0,
					'scan_history_days'       => 7,
					'rule_pack_delay_days'    => 14,
				);
		}
	}

	/**
	 * Check if a specific feature is available.
	 *
	 * @since 1.0.0
	 * @param string $feature Feature key.
	 * @return bool True if feature is available.
	 */
	public static function can( $feature ) {
		$features = self::get_tier_features();
		return isset( $features[ $feature ] ) && $features[ $feature ];
	}

	/**
	 * Get a specific limit value.
	 *
	 * @since 1.0.0
	 * @param string $limit Limit key.
	 * @return int Limit value (-1 for unlimited).
	 */
	public static function get_limit( $limit ) {
		$limits = self::get_tier_limits();
		return isset( $limits[ $limit ] ) ? $limits[ $limit ] : 0;
	}

	/**
	 * Check if within a usage limit.
	 *
	 * @since 1.0.0
	 * @param string $limit_key Limit key.
	 * @param int    $current_usage Current usage count.
	 * @return bool True if within limit.
	 */
	public static function within_limit( $limit_key, $current_usage ) {
		$limit = self::get_limit( $limit_key );
		
		// -1 means unlimited.
		if ( -1 === $limit ) {
			return true;
		}

		return $current_usage < $limit;
	}

	/**
	 * Check if user can run a manual scan today.
	 *
	 * @since 1.0.0
	 * @return bool True if scan is allowed.
	 */
	public static function can_run_manual_scan() {
		$limit = self::get_limit( 'manual_scans_per_day' );
		
		// Unlimited scans for paid users.
		if ( -1 === $limit ) {
			return true;
		}

		// Check today's scan count.
		$today = gmdate( 'Y-m-d' );
		$scan_count = get_transient( 'content_guard_pro_manual_scans_' . $today );
		
		if ( false === $scan_count ) {
			return true;
		}

		return absint( $scan_count ) < $limit;
	}

	/**
	 * Increment manual scan count for today.
	 *
	 * @since 1.0.0
	 */
	public static function increment_manual_scan_count() {
		$today = gmdate( 'Y-m-d' );
		$transient_key = 'content_guard_pro_manual_scans_' . $today;
		$scan_count = get_transient( $transient_key );
		
		if ( false === $scan_count ) {
			$scan_count = 0;
		}

		// Increment and store (expires at midnight).
		$seconds_until_midnight = strtotime( 'tomorrow' ) - time();
		set_transient( $transient_key, absint( $scan_count ) + 1, $seconds_until_midnight );
	}

	/**
	 * Get the number of visible findings based on tier.
	 *
	 * @since 1.0.0
	 * @param int $total_findings Total number of findings.
	 * @return int Number of findings to show.
	 */
	public static function get_visible_findings_count( $total_findings ) {
		$limit = self::get_limit( 'visible_findings' );
		
		// -1 means unlimited.
		if ( -1 === $limit ) {
			return $total_findings;
		}

		return min( $total_findings, $limit );
	}

	/**
	 * Check if there are hidden findings due to tier limit.
	 *
	 * @since 1.0.0
	 * @param int $total_findings Total number of findings.
	 * @return int Number of hidden findings (0 if none).
	 */
	public static function get_hidden_findings_count( $total_findings ) {
		$visible = self::get_visible_findings_count( $total_findings );
		return max( 0, $total_findings - $visible );
	}

	/**
	 * Daily license check via cron.
	 *
	 * @since 1.0.0
	 */
	public static function daily_license_check() {
		$license_key = self::get_license_key();
		
		if ( empty( $license_key ) ) {
			return;
		}

		// Force refresh license data.
		self::get_license_data( true );
	}

	/**
	 * AJAX handler for license validation.
	 *
	 * @since 1.0.0
	 */
	public static function ajax_validate_license() {
		// Verify nonce.
		check_ajax_referer( 'cgp_license_nonce', 'nonce' );

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'content-guard-pro' ) ) );
		}

		// Get license key from request.
		$license_key = isset( $_POST['license_key'] ) ? sanitize_text_field( wp_unslash( $_POST['license_key'] ) ) : '';

		if ( empty( $license_key ) ) {
			wp_send_json_error( array( 'message' => __( 'Please enter a license key', 'content-guard-pro' ) ) );
		}

		// Validate with API.
		$result = CGP_API_Client::validate_license( $license_key, true );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		if ( ! $result['valid'] ) {
			wp_send_json_error( array( 
				'message' => isset( $result['message'] ) ? $result['message'] : __( 'Invalid license key', 'content-guard-pro' ),
				'error_code' => isset( $result['error'] ) ? $result['error'] : 'invalid',
			) );
		}

		// Save license key to settings.
		$settings = get_option( 'content_guard_pro_settings', array() );
		$settings['license_key'] = $license_key;
		update_option( 'content_guard_pro_settings', $settings );

		// Store license data.
		update_option( self::OPTION_LICENSE_DATA, $result );

		// Cache in transient.
		$cache_duration = isset( $result['hints']['cache_seconds'] ) 
			? absint( $result['hints']['cache_seconds'] ) 
			: DAY_IN_SECONDS;
		set_transient( self::TRANSIENT_LICENSE_STATUS, $result, $cache_duration );

		// Clear static cache.
		self::$license_cache = null;

		// Re-evaluate scheduled scans now that license tier changed.
		if ( class_exists( 'CGP_Scheduler' ) ) {
			CGP_Scheduler::reschedule_daily_scan();
		}

		wp_send_json_success( array(
			'message' => __( 'License activated successfully!', 'content-guard-pro' ),
			'tier'    => self::get_tier_name( $result['license']['tier'] ),
			'data'    => $result,
		) );
	}

	/**
	 * AJAX handler for license deactivation.
	 *
	 * @since 1.0.0
	 */
	public static function ajax_deactivate_license() {
		// Verify nonce.
		check_ajax_referer( 'cgp_license_nonce', 'nonce' );

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'content-guard-pro' ) ) );
		}

		$license_key = self::get_license_key();

		if ( ! empty( $license_key ) ) {
			// Deactivate via API.
			CGP_API_Client::deactivate_license( $license_key );
		}

		// Clear license data.
		$settings = get_option( 'content_guard_pro_settings', array() );
		$settings['license_key'] = '';
		update_option( 'content_guard_pro_settings', $settings );

		// Clear stored license data.
		delete_option( self::OPTION_LICENSE_DATA );
		delete_transient( self::TRANSIENT_LICENSE_STATUS );

		// Clear static cache.
		self::$license_cache = null;

		// Stop any scheduled scans now that we're back to Free.
		if ( class_exists( 'CGP_Scheduler' ) ) {
			CGP_Scheduler::cancel_all_scans();
			CGP_Scheduler::cancel_daily_scan();
		}

		wp_send_json_success( array(
			'message' => __( 'License deactivated successfully', 'content-guard-pro' ),
		) );
	}

	/**
	 * AJAX handler for API health check.
	 *
	 * @since 1.0.0
	 */
	public static function ajax_check_api_health() {
		// Verify nonce.
		check_ajax_referer( 'cgp_license_nonce', 'nonce' );

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'content-guard-pro' ) ) );
		}

		$health = CGP_API_Client::health_check();

		if ( $health['healthy'] ) {
			wp_send_json_success( $health );
		} else {
			wp_send_json_error( $health );
		}
	}

	/**
	 * Get license status for display.
	 *
	 * @since 1.0.0
	 * @return array Status data for UI rendering.
	 */
	public static function get_status_for_display() {
		$license_data = self::get_license_data();
		$tier = self::get_tier();
		$is_paid = self::is_paid();

		$status = array(
			'tier'          => $tier,
			'tier_name'     => self::get_tier_name(),
			'is_paid'       => $is_paid,
			'is_valid'      => isset( $license_data['valid'] ) ? $license_data['valid'] : false,
			'status'        => isset( $license_data['license']['status'] ) ? $license_data['license']['status'] : 'active',
			'expires_at'    => isset( $license_data['license']['expires_at'] ) ? $license_data['license']['expires_at'] : null,
			'features'      => self::get_tier_features(),
			'limits'        => self::get_tier_limits(),
			'license_key'   => self::get_license_key() ? self::mask_license_key( self::get_license_key() ) : '',
			'upgrade_url'   => 'https://contentguardpro.com/pricing',
		);

		// Add activation info for paid licenses.
		if ( $is_paid && isset( $license_data['activation'] ) ) {
			$status['activation'] = $license_data['activation'];
		}

		return $status;
	}

	/**
	 * Mask a license key for display.
	 *
	 * @since 1.0.0
	 * @param string $license_key Full license key.
	 * @return string Masked license key (e.g., CGP-XXXX-****-****-XXXX).
	 */
	private static function mask_license_key( $license_key ) {
		if ( strlen( $license_key ) < 20 ) {
			return str_repeat( '*', strlen( $license_key ) );
		}

		// Format: CGP-XXXX-XXXX-XXXX-XXXX
		// Show: CGP-XXXX-****-****-XXXX
		$parts = explode( '-', $license_key );
		if ( count( $parts ) === 5 ) {
			$parts[2] = '****';
			$parts[3] = '****';
			return implode( '-', $parts );
		}

		// Fallback: show first 8 and last 4.
		return substr( $license_key, 0, 8 ) . '****' . substr( $license_key, -4 );
	}

	/**
	 * Get upgrade prompt message based on feature.
	 *
	 * @since 1.0.0
	 * @param string $feature Feature that requires upgrade.
	 * @return string Upgrade prompt message.
	 */
	public static function get_upgrade_prompt( $feature ) {
		$prompts = array(
			'standard_scan'    => __( 'Standard scans are available in Solo Guard and higher plans.', 'content-guard-pro' ),
			'quarantine'       => __( 'Quarantine functionality is available in Solo Guard and higher plans.', 'content-guard-pro' ),
			'scheduled_scans'  => __( 'Scheduled scans are available in Solo Guard and higher plans.', 'content-guard-pro' ),
			'email_alerts'     => __( 'Email alerts are available in Solo Guard and higher plans.', 'content-guard-pro' ),
			'csv_export'       => __( 'CSV/JSON export is available in Agency Lite and higher plans.', 'content-guard-pro' ),
			'rest_api'         => __( 'REST API access is available in Agency Lite and higher plans.', 'content-guard-pro' ),
			'webhook_alerts'   => __( 'Webhook notifications are available in Agency Lite and higher plans.', 'content-guard-pro' ),
			'findings_limit'   => __( 'Free plan shows up to 10 findings. Upgrade to see all findings.', 'content-guard-pro' ),
			'scan_limit'       => __( 'Free plan allows 1 manual scan per day. Upgrade for unlimited scans.', 'content-guard-pro' ),
			'reputation_apis'  => __( 'Reputation API checks (Safe Browsing, PhishTank) require Solo Guard or higher.', 'content-guard-pro' ),
		);

		return isset( $prompts[ $feature ] ) 
			? $prompts[ $feature ] 
			: __( 'This feature requires a paid plan.', 'content-guard-pro' );
	}
}

// Initialize license manager.
add_action( 'init', array( 'CGP_License_Manager', 'init' ) );

