<?php
/**
 * Findings List Table
 *
 * Extends WP_List_Table to display security findings in a structured table
 * with sorting, filtering, and bulk actions.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Load WP_List_Table if not already loaded.
if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Class CGP_Findings_List_Table
 *
 * Displays findings from content_guard_pro_findings table with sorting, filtering,
 * and bulk action capabilities.
 *
 * @since 1.0.0
 */
class CGP_Findings_List_Table extends WP_List_Table {

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		parent::__construct(
			array(
				'singular' => 'finding',
				'plural'   => 'findings',
				'ajax'     => false,
			)
		);
	}

	/**
	 * Get a list of columns.
	 *
	 * @since 1.0.0
	 * @return array Column names and labels.
	 */
	public function get_columns() {
		return array(
			'cb'            => '<input type="checkbox" />',
			'severity'      => __( 'Severity', 'content-guard-pro' ),
			'object_type'   => __( 'Content Type', 'content-guard-pro' ),
			'object_id'     => __( 'Location', 'content-guard-pro' ),
			'rule_id'       => __( 'Rule', 'content-guard-pro' ),
			'confidence'    => __( 'Confidence', 'content-guard-pro' ),
			'matched_excerpt' => __( 'Excerpt', 'content-guard-pro' ),
			'last_seen'     => __( 'Last Seen', 'content-guard-pro' ),
			'status'        => __( 'Status', 'content-guard-pro' ),
			'actions'       => __( 'Actions', 'content-guard-pro' ),
		);
	}

	/**
	 * Get a list of sortable columns.
	 *
	 * @since 1.0.0
	 * @return array Sortable columns.
	 */
	public function get_sortable_columns() {
		return array(
			'severity'    => array( 'severity', false ),
			'object_type' => array( 'object_type', false ),
			'confidence'  => array( 'confidence', true ), // true = already sorted.
			'last_seen'   => array( 'last_seen', false ),
			'status'      => array( 'status', false ),
		);
	}

	/**
	 * Get bulk actions available for the table.
	 *
	 * Implements bulk actions from PRD Section 3.3.
	 *
	 * @since 1.0.0
	 * @return array Bulk actions.
	 */
	public function get_bulk_actions() {
		$actions = array();
		
		// Only allow quarantine bulk action if license supports it.
		if ( class_exists( 'CGP_License_Manager' ) && CGP_License_Manager::can( 'quarantine' ) ) {
			$actions['quarantine'] = __( 'Quarantine', 'content-guard-pro' );
		}
		
		$actions['ignore'] = __( 'Ignore', 'content-guard-pro' );
		$actions['delete'] = __( 'Delete', 'content-guard-pro' );
		
		return $actions;
	}

	/**
	 * Get status filter views.
	 *
	 * Displays links to filter findings by status (All, Open, Quarantined, etc.).
	 *
	 * @since 1.0.0
	 * @return array Status views with counts.
	 */
	protected function get_views() {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';
		$blog_id    = get_current_blog_id();
		$base_url   = admin_url( 'admin.php?page=content-guard-pro-findings' );
		
		// Get current status filter.
		// Default to 'open' if empty, to match prepare_items default.
		$current_status = ! empty( $_REQUEST['status'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['status'] ) ) : 'open'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		
		// Get counts for each status.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$status_counts = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT status, COUNT(*) as count FROM `{$table_name}` WHERE blog_id = %d GROUP BY status",
				$blog_id
			),
			ARRAY_A
		);
		
		// Convert to associative array.
		$counts = array(
			'open'        => 0,
			'quarantined' => 0,
			'ignored'     => 0,
			'resolved'    => 0,
			'deleted'     => 0,
		);
		
		$total = 0;
		foreach ( $status_counts as $row ) {
			if ( isset( $counts[ $row['status'] ] ) ) {
				$counts[ $row['status'] ] = absint( $row['count'] );
				$total += absint( $row['count'] );
			}
		}
		
		// Build views array.
		$views = array();
		
		// All.
		$views['all'] = sprintf(
			'<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
			esc_url( add_query_arg( 'status', 'all', $base_url ) ),
			'all' === $current_status ? 'current' : '',
			__( 'All', 'content-guard-pro' ),
			$total
		);
		
		// Open.
		$views['open'] = sprintf(
			'<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
			esc_url( add_query_arg( 'status', 'open', $base_url ) ),
			'open' === $current_status ? 'current' : '',
			__( 'Open', 'content-guard-pro' ),
			$counts['open']
		);
		
		// Quarantined.
		$views['quarantined'] = sprintf(
			'<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
			esc_url( add_query_arg( 'status', 'quarantined', $base_url ) ),
			'quarantined' === $current_status ? 'current' : '',
			__( 'Quarantined', 'content-guard-pro' ),
			$counts['quarantined']
		);
		
		// Ignored.
		$views['ignored'] = sprintf(
			'<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
			esc_url( add_query_arg( 'status', 'ignored', $base_url ) ),
			'ignored' === $current_status ? 'current' : '',
			__( 'Ignored', 'content-guard-pro' ),
			$counts['ignored']
		);
		
		// Resolved.
		$views['resolved'] = sprintf(
			'<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
			esc_url( add_query_arg( 'status', 'resolved', $base_url ) ),
			'resolved' === $current_status ? 'current' : '',
			__( 'Resolved', 'content-guard-pro' ),
			$counts['resolved']
		);
		
		// Deleted.
		$views['deleted'] = sprintf(
			'<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
			esc_url( add_query_arg( 'status', 'deleted', $base_url ) ),
			'deleted' === $current_status ? 'current' : '',
			__( 'Deleted', 'content-guard-pro' ),
			$counts['deleted']
		);
		
		return $views;
	}

	/**
	 * Render the checkbox column.
	 *
	 * @since 1.0.0
	 * @param array $item Row data.
	 * @return string Checkbox HTML.
	 */
	public function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="finding[]" value="%s" />',
			esc_attr( $item['id'] )
		);
	}

	/**
	 * Render the severity column with color coding.
	 *
	 * @since 1.0.0
	 * @param array $item Row data.
	 * @return string Severity HTML.
	 */
	public function column_severity( $item ) {
		$severity = esc_html( ucfirst( $item['severity'] ) );
		
		// Add CSS class for color coding.
		$class = 'content-guard-pro-severity-' . esc_attr( $item['severity'] );
		
		return sprintf(
			'<span class="%s"><strong>%s</strong></span>',
			$class,
			$severity
		);
	}

	/**
	 * Render the object type column.
	 *
	 * @since 1.0.0
	 * @param array $item Row data.
	 * @return string Object type HTML.
	 */
	public function column_object_type( $item ) {
		return sprintf(
			'<span class="content-guard-pro-object-type">%s</span>',
			esc_html( ucfirst( $item['object_type'] ) )
		);
	}

	/**
	 * Render the location column with title and edit link.
	 *
	 * Displays post/page title with ID for better UX, similar to Quarantine page.
	 *
	 * @since 1.0.0
	 * @param array $item Row data.
	 * @return string Location HTML with link.
	 */
	public function column_object_id( $item ) {
		$object_id = absint( $item['object_id'] );
		
		// Generate edit link and title based on object type.
		if ( 'post' === $item['object_type'] ) {
			$edit_link = get_edit_post_link( $object_id );
			$post      = get_post( $object_id );
			$title     = $post ? get_the_title( $post ) : '';
			
			// Truncate long titles for display.
			if ( $title && strlen( $title ) > 40 ) {
				$title = substr( $title, 0, 37 ) . '...';
			}
			
			if ( $edit_link && $title ) {
				return sprintf(
					'<a href="%s" target="_blank">%s <span class="content-guard-pro-object-id">(ID: %d)</span></a>',
					esc_url( $edit_link ),
					esc_html( $title ),
					$object_id
				);
			} elseif ( $edit_link ) {
				// Post exists but has no title.
				return sprintf(
					'<a href="%s" target="_blank">%s <span class="content-guard-pro-object-id">(ID: %d)</span></a>',
					esc_url( $edit_link ),
					esc_html__( 'Untitled', 'content-guard-pro' ),
					$object_id
				);
			}
		}
		
		// Fallback for non-post objects or deleted posts.
		return sprintf( 'ID: %d', $object_id );
	}

	/**
	 * Render the rule ID column.
	 *
	 * @since 1.0.0
	 * @param array $item Row data.
	 * @return string Rule ID HTML.
	 */
	public function column_rule_id( $item ) {
		return sprintf(
			'<code>%s</code>',
			esc_html( $item['rule_id'] )
		);
	}

	/**
	 * Render the confidence column with visual indicator.
	 *
	 * @since 1.0.0
	 * @param array $item Row data.
	 * @return string Confidence HTML with progress bar.
	 */
	public function column_confidence( $item ) {
		$confidence = absint( $item['confidence'] );
		
		// Determine color based on confidence level.
		$color_class = 'content-guard-pro-confidence-low';
		if ( $confidence >= 80 ) {
			$color_class = 'content-guard-pro-confidence-high';
		} elseif ( $confidence >= 50 ) {
			$color_class = 'content-guard-pro-confidence-medium';
		}
		
		return sprintf(
			'<div class="content-guard-pro-confidence-wrapper">
				<span class="content-guard-pro-confidence-value">%d%%</span>
				<div class="content-guard-pro-confidence-bar">
					<div class="content-guard-pro-confidence-fill %s" style="width: %d%%"></div>
				</div>
			</div>',
			$confidence,
			esc_attr( $color_class ),
			$confidence
		);
	}

	/**
	 * Render the matched excerpt column.
	 *
	 * @since 1.0.0
	 * @param array $item Row data.
	 * @return string Excerpt HTML.
	 */
	public function column_matched_excerpt( $item ) {
		// We handle truncation via CSS line-clamp.
		return sprintf(
			'<div class="content-guard-pro-excerpt" title="%s">%s</div>',
			esc_attr( $item['matched_excerpt'] ),
			esc_html( $item['matched_excerpt'] )
		);
	}

	/**
	 * Render the last seen column.
	 *
	 * @since 1.0.0
	 * @param array $item Row data.
	 * @return string Last seen HTML with human-readable time.
	 */
	public function column_last_seen( $item ) {
		$timestamp = strtotime( $item['last_seen'] );
		
		// Check if strtotime() failed.
		if ( false === $timestamp ) {
			return esc_html( $item['last_seen'] );
		}
		
		return sprintf(
			'<abbr title="%s">%s</abbr>',
			esc_attr( $item['last_seen'] ),
			esc_html( human_time_diff( $timestamp, current_time( 'timestamp' ) ) . ' ago' )
		);
	}

	/**
	 * Render the status column with badge.
	 *
	 * @since 1.0.0
	 * @param array $item Row data.
	 * @return string Status HTML.
	 */
	public function column_status( $item ) {
		$status = esc_html( ucfirst( $item['status'] ) );
		$class  = 'content-guard-pro-status-badge content-guard-pro-status-' . esc_attr( $item['status'] );
		
		return sprintf(
			'<span class="%s">%s</span>',
			$class,
			$status
		);
	}

	/**
	 * Render the actions column with row actions.
	 *
	 * @since 1.0.0
	 * @param array $item Row data.
	 * @return string Actions HTML.
	 */
	public function column_actions( $item ) {
		$actions = array();
		
		// View details action.
		$actions['view'] = sprintf(
			'<a href="#" class="content-guard-pro-view-finding" data-finding-id="%d">%s</a>',
			absint( $item['id'] ),
			__( 'View Details', 'content-guard-pro' )
		);
		
		// Edit content action.
		if ( 'post' === $item['object_type'] ) {
			$edit_link = get_edit_post_link( $item['object_id'] );
			if ( $edit_link ) {
				$actions['edit'] = sprintf(
					'<a href="%s" target="_blank">%s</a>',
					esc_url( $edit_link ),
					__( 'Edit Content', 'content-guard-pro' )
				);
			}
		}
		
		// Quarantine action (if not already quarantined).
		if ( 'quarantined' !== $item['status'] ) {
			// Only allow quarantine action if license supports it.
			if ( class_exists( 'CGP_License_Manager' ) && CGP_License_Manager::can( 'quarantine' ) ) {
				$actions['quarantine'] = sprintf(
					'<a href="#" class="content-guard-pro-quarantine-finding" data-finding-id="%d">%s</a>',
					absint( $item['id'] ),
					__( 'Quarantine', 'content-guard-pro' )
				);
			} else {
				// Show disabled action for free users.
				$actions['quarantine'] = sprintf(
					'<span class="content-guard-pro-action-disabled" title="%s">%s</span>',
					esc_attr__( 'Upgrade to quarantine suspicious content.', 'content-guard-pro' ),
					__( 'Quarantine', 'content-guard-pro' )
				);
			}
		} else {
			$actions['unquarantine'] = sprintf(
				'<a href="#" class="content-guard-pro-unquarantine-finding" data-finding-id="%d">%s</a>',
				absint( $item['id'] ),
				__( 'Un-quarantine', 'content-guard-pro' )
			);
		}
		
		// Ignore action (if not already ignored).
		if ( 'ignored' !== $item['status'] ) {
			$actions['ignore'] = sprintf(
				'<a href="#" class="content-guard-pro-ignore-finding" data-finding-id="%d">%s</a>',
				absint( $item['id'] ),
				__( 'Ignore', 'content-guard-pro' )
			);
		}
		
		return $this->row_actions( $actions );
	}

	/**
	 * Default column renderer.
	 *
	 * @since 1.0.0
	 * @param array  $item        Row data.
	 * @param string $column_name Column name.
	 * @return string Column content.
	 */
	public function column_default( $item, $column_name ) {
		return isset( $item[ $column_name ] ) ? esc_html( $item[ $column_name ] ) : '';
	}

	/**
	 * Prepare items for display.
	 *
	 * Fetches data from database and prepares pagination.
	 *
	 * @since 1.0.0
	 */
	public function prepare_items() {
		global $wpdb;
		
		// Register column headers.
		$columns  = $this->get_columns();
		$hidden   = array(); // No hidden columns by default.
		$sortable = $this->get_sortable_columns();
		
		$this->_column_headers = array( $columns, $hidden, $sortable );
		
		// Handle bulk actions.
		$this->process_bulk_action();
		
		// Get table name.
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';
		
		// Get current blog ID for multisite support.
		$blog_id = get_current_blog_id();
		
		// Pagination.
		$per_page     = 20;
		$current_page = $this->get_pagenum();
		$offset       = ( $current_page - 1 ) * $per_page;
		
		// Sorting.
		$orderby = ! empty( $_REQUEST['orderby'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['orderby'] ) ) : 'last_seen'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$order   = ! empty( $_REQUEST['order'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['order'] ) ) : 'desc'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		
		// Validate orderby to prevent SQL injection.
		$allowed_orderby = array( 'severity', 'object_type', 'confidence', 'last_seen', 'status', 'rule_id' );
		if ( ! in_array( $orderby, $allowed_orderby, true ) ) {
			$orderby = 'last_seen';
		}
		
		// Validate order.
		$order = strtoupper( $order );
		if ( ! in_array( $order, array( 'ASC', 'DESC' ), true ) ) {
			$order = 'DESC';
		}
		
		// Build WHERE clause for filtering.
		$where_clauses = array( $wpdb->prepare( 'blog_id = %d', $blog_id ) );
		
		// Filter by status.
		// Default to 'open' if no status is specified (Scenario 3 requirement).
		$status = ! empty( $_REQUEST['status'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['status'] ) ) : 'open'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		
		if ( 'all' !== $status ) {
			$allowed_status = array( 'open', 'quarantined', 'ignored', 'resolved', 'deleted' );
			if ( in_array( $status, $allowed_status, true ) ) {
				$where_clauses[] = $wpdb->prepare( 'status = %s', $status );
			}
		}
		
		// Filter by severity.
		if ( ! empty( $_REQUEST['severity'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$severity = sanitize_text_field( wp_unslash( $_REQUEST['severity'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$allowed_severity = array( 'critical', 'suspicious', 'review' );
			if ( in_array( $severity, $allowed_severity, true ) ) {
				$where_clauses[] = $wpdb->prepare( 'severity = %s', $severity );
			}
		}
		
		// Search functionality.
		$search_term = '';
		$needs_post_join = false;
		if ( ! empty( $_REQUEST['s'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$search_term = sanitize_text_field( wp_unslash( $_REQUEST['s'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$search = '%' . $wpdb->esc_like( $search_term ) . '%';
			
			// Build search conditions for direct table fields.
			$search_conditions = array(
				$wpdb->prepare( "`{$table_name}`.rule_id LIKE %s", $search ),
				$wpdb->prepare( "`{$table_name}`.matched_excerpt LIKE %s", $search ),
				$wpdb->prepare( "`{$table_name}`.object_type LIKE %s", $search ),
				$wpdb->prepare( "`{$table_name}`.severity LIKE %s", $search ),
				$wpdb->prepare( "`{$table_name}`.status LIKE %s", $search ),
				$wpdb->prepare( "`{$table_name}`.field LIKE %s", $search ),
			);
			
			// Check if search term is numeric (for object_id search).
			if ( is_numeric( $search_term ) ) {
				$search_conditions[] = $wpdb->prepare( "`{$table_name}`.object_id = %d", absint( $search_term ) );
			}
			
			// Always try to search post titles if search term is not purely numeric.
			// This requires a JOIN with wp_posts table.
			if ( ! is_numeric( $search_term ) || strlen( trim( $search_term ) ) > 10 ) {
				$needs_post_join = true;
				$posts_table = $wpdb->posts;
				$post_title_search = '%' . $wpdb->esc_like( $search_term ) . '%';
				$search_conditions[] = $wpdb->prepare( "`{$posts_table}`.post_title LIKE %s", $post_title_search );
			}
			
			$where_clauses[] = '(' . implode( ' OR ', $search_conditions ) . ')';
		}
		
		$where_sql = implode( ' AND ', $where_clauses );
		
		// Build FROM clause with optional JOIN for post title search.
		$from_sql = "`{$table_name}`";
		if ( $needs_post_join ) {
			$posts_table = $wpdb->posts;
			$from_sql = "`{$table_name}` LEFT JOIN `{$posts_table}` ON (`{$table_name}`.object_type = 'post' AND `{$table_name}`.object_id = `{$posts_table}`.ID)";
		}
		
		// Get total items count.
		// Use DISTINCT only when JOINing to avoid counting duplicates.
		$count_distinct = $needs_post_join ? 'DISTINCT ' : '';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared
		$total_items = $wpdb->get_var( "SELECT COUNT({$count_distinct}`{$table_name}`.id) FROM {$from_sql} WHERE {$where_sql}" );
		
		// Apply license tier limit on visible findings.
		$visible_limit = -1;
		if ( class_exists( 'CGP_License_Manager' ) ) {
			$visible_limit = CGP_License_Manager::get_limit( 'visible_findings' );
		}
		
		// If there's a visible limit, cap the total items shown.
		$effective_total = absint( $total_items );
		if ( $visible_limit > 0 && $effective_total > $visible_limit ) {
			$effective_total = $visible_limit;
		}
		
		// Recalculate offset if it exceeds the visible limit.
		if ( $visible_limit > 0 && $offset >= $visible_limit ) {
			// Force to last valid page.
			$offset = max( 0, $visible_limit - $per_page );
		}
		
		// Adjust per_page if it would exceed visible limit.
		$effective_per_page = $per_page;
		if ( $visible_limit > 0 ) {
			$remaining = max( 0, $visible_limit - $offset );
			$effective_per_page = min( $per_page, $remaining );
		}
		
		// Get items for current page.
		// Use DISTINCT to avoid duplicates when JOINing with posts table.
		$distinct_clause = $needs_post_join ? 'DISTINCT ' : '';
		$select_fields = $needs_post_join ? "`{$table_name}`.*" : '*';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared
		$this->items = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT {$distinct_clause}{$select_fields} FROM {$from_sql} WHERE {$where_sql} ORDER BY `{$table_name}`.{$orderby} {$order} LIMIT %d OFFSET %d",
				$effective_per_page,
				$offset
			),
			ARRAY_A
		);
		
		// Set pagination args with effective total (limited by license).
		$this->set_pagination_args(
			array(
				'total_items' => $effective_total,
				'per_page'    => $per_page,
				'total_pages' => ceil( $effective_total / $per_page ),
			)
		);
	}

	/**
	 * Process bulk actions.
	 *
	 * Handles quarantine, ignore, and delete actions from PRD Section 3.3.
	 *
	 * @since 1.0.0
	 */
	public function process_bulk_action() {
		// Check if bulk action was triggered.
		$action = $this->current_action();
		
		if ( ! $action ) {
			return;
		}
		
		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'content-guard-pro' ) );
		}
		
		// Verify nonce.
		if ( ! isset( $_REQUEST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] ) ), 'bulk-findings' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'content-guard-pro' ) );
		}
		
		// Get selected finding IDs.
		$finding_ids = isset( $_REQUEST['finding'] ) ? array_map( 'absint', wp_unslash( $_REQUEST['finding'] ) ) : array();
		
		if ( empty( $finding_ids ) ) {
			return;
		}
		
		// Process action.
		switch ( $action ) {
			case 'quarantine':
				$this->bulk_quarantine( $finding_ids );
				break;
			case 'ignore':
				$this->bulk_ignore( $finding_ids );
				break;
			case 'delete':
				$this->bulk_delete( $finding_ids );
				break;
		}
	}

	/**
	 * Bulk quarantine findings.
	 *
	 * Updates finding status to 'quarantined' in the content_guard_pro_findings table.
	 * Uses CGP_Quarantine class to handle the quarantine logic and audit logging.
	 *
	 * @since 1.0.0
	 * @param array $finding_ids Finding IDs to quarantine.
	 */
	private function bulk_quarantine( $finding_ids ) {
		if ( empty( $finding_ids ) ) {
			return;
		}
		
		// Ensure CGP_Quarantine class is available.
		if ( ! class_exists( 'CGP_Quarantine' ) ) {
			add_settings_error(
				'content_guard_pro_findings',
				'quarantine_error',
				__( 'Quarantine system not available.', 'content-guard-pro' ),
				'error'
			);
			return;
		}

		// Use CGP_Quarantine class to handle quarantine.
		$count = CGP_Quarantine::bulk_quarantine( $finding_ids );
		
		if ( $count > 0 ) {
			add_settings_error(
				'content_guard_pro_findings',
				'findings_quarantined',
				sprintf(
					/* translators: %d: Number of findings quarantined */
					_n( '%d finding quarantined.', '%d findings quarantined.', $count, 'content-guard-pro' ),
					$count
				),
				'success'
			);
		} else {
			add_settings_error(
				'content_guard_pro_findings',
				'quarantine_failed',
				__( 'Failed to quarantine findings. Please try again.', 'content-guard-pro' ),
				'error'
			);
		}
	}

	/**
	 * Bulk ignore findings.
	 *
	 * Updates finding status to 'ignored' in the content_guard_pro_findings table.
	 * Logs the action to the audit log.
	 *
	 * @since 1.0.0
	 * @param array $finding_ids Finding IDs to ignore.
	 */
	private function bulk_ignore( $finding_ids ) {
		global $wpdb;
		
		if ( empty( $finding_ids ) ) {
			return;
		}

		$table_name = $wpdb->prefix . 'content_guard_pro_findings';
		$count = 0;

		// Update each finding to 'ignored' status.
		foreach ( $finding_ids as $finding_id ) {
			$updated = $wpdb->update(
				$table_name,
				array( 'status' => 'ignored' ),
				array(
					'id'      => $finding_id,
					'blog_id' => get_current_blog_id(),
				),
				array( '%s' ),
				array( '%d', '%d' )
			);

			if ( false !== $updated ) {
				$count++;
				
				// Log to audit log.
				$this->log_finding_action( $finding_id, 'ignore' );
				
				// Clear quarantine cache.
				do_action( 'content_guard_pro_finding_status_changed', $finding_id, 'ignored' );
			}
		}
		
		if ( $count > 0 ) {
			add_settings_error(
				'content_guard_pro_findings',
				'findings_ignored',
				sprintf(
					/* translators: %d: Number of findings ignored */
					_n( '%d finding ignored.', '%d findings ignored.', $count, 'content-guard-pro' ),
					$count
				),
				'success'
			);
		} else {
			add_settings_error(
				'content_guard_pro_findings',
				'ignore_failed',
				__( 'Failed to ignore findings. Please try again.', 'content-guard-pro' ),
				'error'
			);
		}
	}

	/**
	 * Bulk delete findings.
	 *
	 * Permanently deletes findings from the content_guard_pro_findings table.
	 * Logs the action to the audit log before deletion.
	 *
	 * @since 1.0.0
	 * @param array $finding_ids Finding IDs to delete.
	 */
	private function bulk_delete( $finding_ids ) {
		global $wpdb;
		
		if ( empty( $finding_ids ) ) {
			return;
		}

		$table_name = $wpdb->prefix . 'content_guard_pro_findings';
		$count = 0;

		// Delete each finding.
		foreach ( $finding_ids as $finding_id ) {
			// Log to audit log before deletion.
			$this->log_finding_action( $finding_id, 'delete' );
			
			// Delete the finding.
			$deleted = $wpdb->delete(
				$table_name,
				array(
					'id'      => $finding_id,
					'blog_id' => get_current_blog_id(),
				),
				array( '%d', '%d' )
			);

			if ( false !== $deleted && $deleted > 0 ) {
				$count++;
				
				// Clear quarantine cache.
				do_action( 'content_guard_pro_finding_status_changed', $finding_id, 'deleted' );
			}
		}
		
		if ( $count > 0 ) {
			add_settings_error(
				'content_guard_pro_findings',
				'findings_deleted',
				sprintf(
					/* translators: %d: Number of findings deleted */
					_n( '%d finding deleted.', '%d findings deleted.', $count, 'content-guard-pro' ),
					$count
				),
				'success'
			);
		} else {
			add_settings_error(
				'content_guard_pro_findings',
				'delete_failed',
				__( 'Failed to delete findings. Please try again.', 'content-guard-pro' ),
				'error'
			);
		}
	}

	/**
	 * Log finding action to audit log.
	 *
	 * Records actions (ignore, delete, etc.) to the content_guard_pro_audit_log table.
	 *
	 * @since 1.0.0
	 * @param int    $finding_id Finding ID.
	 * @param string $action     Action performed (ignore, delete, etc.).
	 */
	private function log_finding_action( $finding_id, $action ) {
		global $wpdb;

		// Get finding details.
		$findings_table = $wpdb->prefix . 'content_guard_pro_findings';
		$finding = $wpdb->get_row(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				"SELECT * FROM `{$findings_table}` WHERE id = %d",
				$finding_id
			),
			ARRAY_A
		);

		if ( ! $finding ) {
			return;
		}

		// Prepare audit log entry.
		$audit_table = $wpdb->prefix . 'content_guard_pro_audit_log';
		
		// Determine old and new values based on action.
		$old_value = $finding['status'];
		$new_value = '';
		
		switch ( $action ) {
			case 'ignore':
				$new_value = 'ignored';
				break;
			case 'delete':
				$new_value = 'deleted';
				break;
			default:
				$new_value = $action;
		}

		// Insert audit log entry.
		$inserted = $wpdb->insert(
			$audit_table,
			array(
				'blog_id'    => get_current_blog_id(),
				'finding_id' => $finding_id,
				'user_id'    => get_current_user_id(),
				'action'     => $action,
				'object_type' => $finding['object_type'],
				'object_id'  => $finding['object_id'],
				'field'      => $finding['field'],
				'old_value'  => $old_value,
				'new_value'  => $new_value,
				'metadata'   => wp_json_encode( array( 'rule_id' => $finding['rule_id'] ) ),
				'ip_address' => $this->get_client_ip(),
				'created_at' => current_time( 'mysql' ),
			),
			array( '%d', '%d', '%d', '%s', '%s', '%d', '%s', '%s', '%s', '%s', '%s', '%s' )
		);

		// Log error if insert failed (optional, for debugging).
		if ( false === $inserted ) {
			// Could log to error_log or WordPress debug log if needed.
			// cgp_log( 'CGP: Failed to insert audit log entry for finding ' . $finding_id );
		}
	}

	/**
	 * Get client IP address.
	 *
	 * Uses REMOTE_ADDR as primary source to prevent IP spoofing.
	 * Falls back to X-Forwarded-For only if REMOTE_ADDR is a known proxy/local IP.
	 *
	 * @since 1.0.0
	 * @return string Client IP address.
	 */
	private function get_client_ip() {
		$ip = '';

		// REMOTE_ADDR is the most reliable source (cannot be spoofed by client).
		if ( ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
			$ip = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
		}

		// Validate the IP address.
		$validated_ip = filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE );

		// If REMOTE_ADDR is private/reserved (indicating proxy), check X-Forwarded-For.
		if ( ! $validated_ip && ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$forwarded_ips = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
			
			// HTTP_X_FORWARDED_FOR can contain multiple IPs (comma-separated).
			// Format: client, proxy1, proxy2
			$ip_list = array_map( 'trim', explode( ',', $forwarded_ips ) );
			
			// Get the first valid public IP.
			foreach ( $ip_list as $forwarded_ip ) {
				$validated = filter_var( $forwarded_ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE );
				if ( $validated ) {
					return $validated;
				}
			}
		}

		// Return validated REMOTE_ADDR or fallback.
		return $validated_ip ? $validated_ip : '0.0.0.0';
	}

	/**
	 * Display message when no items found.
	 *
	 * @since 1.0.0
	 */
	public function no_items() {
		esc_html_e( 'No security findings found. Your site appears clean!', 'content-guard-pro' );
	}
}

