<?php
/**
 * Database Management Class
 *
 * Handles database table creation, updates, and maintenance for Content Guard Pro.
 * Creates custom tables for findings, scan history, and audit logging.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Database
 *
 * Manages all database operations including table creation, schema updates,
 * and data maintenance for the Content Guard Pro plugin.
 *
 * @since 1.0.0
 */
class CGP_Database {

	/**
	 * Database version for schema tracking.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const DB_VERSION = '1.0.0';

	/**
	 * Database version option key.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const DB_VERSION_KEY = 'content_guard_pro_db_version';

	/**
	 * Table names (without prefix).
	 *
	 * @since 1.0.0
	 * @var array
	 */
	private static $tables = array(
		'findings'  => 'content_guard_pro_findings',
		'scans'     => 'content_guard_pro_scans',
		'audit_log' => 'content_guard_pro_audit_log',
	);

	/**
	 * Create or update database tables on plugin activation.
	 *
	 * This method is called during plugin activation and uses dbDelta()
	 * to create or update the required custom tables based on the schema
	 * defined in Technical Appendix A of the PRD.
	 *
	 * @since 1.0.0
	 * @return bool True on success, false on failure.
	 */
	public static function activate() {
		// Require the dbDelta function.
		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		global $wpdb;
		$charset_collate = $wpdb->get_charset_collate();
		$created         = array();

		// Create findings table.
		$created['findings'] = self::create_findings_table( $charset_collate );

		// Create scans table.
		$created['scans'] = self::create_scans_table( $charset_collate );

		// Create audit log table.
		$created['audit_log'] = self::create_audit_log_table( $charset_collate );

		// Update database version.
		update_option( self::DB_VERSION_KEY, self::DB_VERSION );

		// Log table creation for diagnostics.
		self::log_table_creation( $created );

		return ! in_array( false, $created, true );
	}

	/**
	 * Create the findings table.
	 *
	 * Stores detected security findings with metadata, confidence scores,
	 * and status tracking. Schema from PRD Appendix A.
	 *
	 * @since 1.0.0
	 * @param string $charset_collate Database charset and collation.
	 * @return bool True on success, false on failure.
	 */
	private static function create_findings_table( $charset_collate ) {
		global $wpdb;

		$table_name = $wpdb->prefix . self::$tables['findings'];

		// SQL for content_guard_pro_findings table - matches PRD Appendix A exactly.
		// Note: Using CURRENT_TIMESTAMP instead of deprecated '0000-00-00 00:00:00'.
		// Status 'deleted' added to track findings where content has been deleted/trashed.
		$sql = "CREATE TABLE `{$table_name}` (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			blog_id INT UNSIGNED NOT NULL DEFAULT 1,
			object_type ENUM('post', 'postmeta', 'option') NOT NULL,
			object_id BIGINT UNSIGNED NOT NULL,
			field VARCHAR(191) NOT NULL DEFAULT '',
			fingerprint CHAR(64) NOT NULL DEFAULT '',
			rule_id VARCHAR(64) NOT NULL DEFAULT '',
			severity ENUM('critical', 'suspicious', 'review') NOT NULL DEFAULT 'review',
			confidence TINYINT UNSIGNED NOT NULL DEFAULT 0,
			matched_excerpt TEXT,
			first_seen DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			last_seen DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			status ENUM('open', 'quarantined', 'ignored', 'resolved', 'deleted') NOT NULL DEFAULT 'open',
			extra TEXT,
			PRIMARY KEY  (id),
			KEY blog_status_severity (blog_id, status, severity),
			KEY object_lookup (object_type, object_id),
			KEY fingerprint (fingerprint),
			KEY rule_id (rule_id),
			KEY last_seen (last_seen)
		) {$charset_collate};";

		dbDelta( $sql );

		// Verify table creation.
		return self::verify_table_exists( $table_name );
	}

	/**
	 * Create the scans table.
	 *
	 * Stores scan history with performance metrics and completion status.
	 * Schema from PRD Appendix A.
	 *
	 * @since 1.0.0
	 * @param string $charset_collate Database charset and collation.
	 * @return bool True on success, false on failure.
	 */
	private static function create_scans_table( $charset_collate ) {
		global $wpdb;

		$table_name = $wpdb->prefix . self::$tables['scans'];

		// SQL for content_guard_pro_scans table - matches PRD Appendix A exactly.
		// Note: Using CURRENT_TIMESTAMP instead of deprecated '0000-00-00 00:00:00'.
		$sql = "CREATE TABLE `{$table_name}` (
			scan_id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			blog_id INT UNSIGNED NOT NULL DEFAULT 1,
			started_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			finished_at DATETIME,
			mode ENUM('quick', 'standard') NOT NULL DEFAULT 'standard',
			targets VARCHAR(255) DEFAULT NULL,
			status ENUM('pending', 'running', 'paused', 'cancelled', 'completed', 'failed') NOT NULL DEFAULT 'pending',
			totals_checked INT UNSIGNED NOT NULL DEFAULT 0,
			totals_flagged INT UNSIGNED NOT NULL DEFAULT 0,
			avg_query_ms SMALLINT UNSIGNED NOT NULL DEFAULT 0,
			peak_mem_mb SMALLINT UNSIGNED NOT NULL DEFAULT 0,
			throttle_state VARCHAR(32) NOT NULL DEFAULT 'normal',
			progress TEXT,
			errors SMALLINT UNSIGNED NOT NULL DEFAULT 0,
			notes TEXT,
			PRIMARY KEY  (scan_id),
			KEY blog_id (blog_id),
			KEY started_at (started_at),
			KEY mode (mode),
			KEY status (status)
		) {$charset_collate};";

		dbDelta( $sql );

		// Verify table creation.
		return self::verify_table_exists( $table_name );
	}

	/**
	 * Create the audit log table.
	 *
	 * Tracks remediation actions, quarantine operations, and changes
	 * to meta/options as mentioned in PRD section 3.4 and US-009.
	 *
	 * @since 1.0.0
	 * @param string $charset_collate Database charset and collation.
	 * @return bool True on success, false on failure.
	 */
	private static function create_audit_log_table( $charset_collate ) {
		global $wpdb;

		$table_name = $wpdb->prefix . self::$tables['audit_log'];

		// SQL for content_guard_pro_audit_log table - tracks all remediation actions.
		// Note: Using CURRENT_TIMESTAMP instead of deprecated '0000-00-00 00:00:00'.
		$sql = "CREATE TABLE `{$table_name}` (
			log_id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			blog_id INT UNSIGNED NOT NULL DEFAULT 1,
			finding_id BIGINT UNSIGNED DEFAULT NULL,
			user_id BIGINT UNSIGNED NOT NULL,
			action VARCHAR(64) NOT NULL,
			object_type VARCHAR(32) NOT NULL,
			object_id BIGINT UNSIGNED NOT NULL,
			field VARCHAR(191) NOT NULL DEFAULT '',
			old_value LONGTEXT,
			new_value LONGTEXT,
			metadata TEXT,
			ip_address VARCHAR(45) NOT NULL DEFAULT '',
			created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY  (log_id),
			KEY blog_id (blog_id),
			KEY finding_id (finding_id),
			KEY user_id (user_id),
			KEY action (action),
			KEY created_at (created_at),
			KEY object_lookup (object_type, object_id)
		) {$charset_collate};";

		dbDelta( $sql );

		// Verify table creation.
		return self::verify_table_exists( $table_name );
	}

	/**
	 * Verify that a table exists in the database.
	 *
	 * @since 1.0.0
	 * @param string $table_name Full table name with prefix.
	 * @return bool True if table exists, false otherwise.
	 */
	private static function verify_table_exists( $table_name ) {
		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$table_exists = $wpdb->get_var(
			$wpdb->prepare(
				'SHOW TABLES LIKE %s',
				$table_name
			)
		);

		return $table_exists === $table_name;
	}

	/**
	 * Log table creation results for diagnostics.
	 *
	 * @since 1.0.0
	 * @param array $results Array of table creation results.
	 */
	private static function log_table_creation( $results ) {
		$log = array(
			'timestamp' => current_time( 'mysql' ),
			'db_version' => self::DB_VERSION,
			'tables'     => $results,
		);

		// Store creation log as transient for diagnostics page.
		set_transient( 'content_guard_pro_db_creation_log', $log, DAY_IN_SECONDS );

		// Log to error log if in debug mode.
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			cgp_log( 'Content Guard Pro: Database tables creation - ' . wp_json_encode( $results ) );
		}
	}

	/**
	 * Get table name with prefix.
	 *
	 * @since 1.0.0
	 * @param string $table Table identifier (findings, scans, audit_log).
	 * @return string|null Full table name with prefix, or null if not found.
	 */
	public static function get_table_name( $table ) {
		global $wpdb;

		if ( isset( self::$tables[ $table ] ) ) {
			return $wpdb->prefix . self::$tables[ $table ];
		}

		return null;
	}

	/**
	 * Get all table names.
	 *
	 * @since 1.0.0
	 * @return array Associative array of table identifiers to full table names.
	 */
	public static function get_all_table_names() {
		global $wpdb;

		$tables = array();
		foreach ( self::$tables as $key => $table ) {
			$tables[ $key ] = $wpdb->prefix . $table;
		}

		return $tables;
	}

	/**
	 * Check if database tables need updating.
	 *
	 * @since 1.0.0
	 * @return bool True if update needed, false otherwise.
	 */
	public static function needs_update() {
		$installed_version = get_option( self::DB_VERSION_KEY, '0.0.0' );
		return version_compare( $installed_version, self::DB_VERSION, '<' );
	}

	/**
	 * Clean up old findings based on retention policy.
	 *
	 * Implements retention logic from PRD Appendix A:
	 * Purge findings older than retention days, except those with status
	 * in ('quarantined', 'ignored', 'resolved') within the retention period.
	 *
	 * @since 1.0.0
	 * @param int $retention_days Number of days to retain findings (default 90).
	 * @return int|false Number of rows deleted, or false on error.
	 */
	public static function cleanup_old_findings( $retention_days = 90 ) {
		global $wpdb;

		$table_name    = self::get_table_name( 'findings' );
		$retention_days = absint( $retention_days );

		if ( ! $table_name || $retention_days < 1 ) {
			return false;
		}

		// Calculate cutoff date.
		$cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$retention_days} days" ) );

		// Delete old findings that are 'open' status.
		// Keep quarantined, ignored, and resolved findings within retention period.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$deleted = $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM `{$table_name}` WHERE last_seen < %s AND status = %s",
				$cutoff_date,
				'open'
			)
		);

		// Also clean up very old resolved findings (older than retention period).
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$deleted_resolved = $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM `{$table_name}` WHERE last_seen < %s AND status = %s",
				$cutoff_date,
				'resolved'
			)
		);

		if ( false !== $deleted && false !== $deleted_resolved ) {
			$total_deleted = $deleted + $deleted_resolved;

			// Log cleanup action.
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				cgp_log( "Content Guard Pro: Cleaned up {$total_deleted} old findings (retention: {$retention_days} days)" );
			}

			return $total_deleted;
		}

		return false;
	}

	/**
	 * Clean up old scan history.
	 *
	 * Removes scan records older than the specified number of days.
	 *
	 * @since 1.0.0
	 * @param int $retention_days Number of days to retain scan history (default 90).
	 * @return int|false Number of rows deleted, or false on error.
	 */
	public static function cleanup_old_scans( $retention_days = 90 ) {
		global $wpdb;

		$table_name     = self::get_table_name( 'scans' );
		$retention_days = absint( $retention_days );

		if ( ! $table_name || $retention_days < 1 ) {
			return false;
		}

		// Calculate cutoff date.
		$cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$retention_days} days" ) );

		// Delete old scan records.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$deleted = $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM `{$table_name}` WHERE started_at < %s",
				$cutoff_date
			)
		);

		if ( false !== $deleted ) {
			// Log cleanup action.
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				cgp_log( "Content Guard Pro: Cleaned up {$deleted} old scan records (retention: {$retention_days} days)" );
			}

			return $deleted;
		}

		return false;
	}

	/**
	 * Clean up old audit log entries.
	 *
	 * Audit logs are typically kept longer than findings.
	 * Default retention is 1 year.
	 *
	 * @since 1.0.0
	 * @param int $retention_days Number of days to retain audit logs (default 365).
	 * @return int|false Number of rows deleted, or false on error.
	 */
	public static function cleanup_old_audit_logs( $retention_days = 365 ) {
		global $wpdb;

		$table_name     = self::get_table_name( 'audit_log' );
		$retention_days = absint( $retention_days );

		if ( ! $table_name || $retention_days < 1 ) {
			return false;
		}

		// Calculate cutoff date.
		$cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$retention_days} days" ) );

		// Delete old audit log entries.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$deleted = $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM `{$table_name}` WHERE created_at < %s",
				$cutoff_date
			)
		);

		if ( false !== $deleted ) {
			// Log cleanup action.
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				cgp_log( "Content Guard Pro: Cleaned up {$deleted} old audit log entries (retention: {$retention_days} days)" );
			}

			return $deleted;
		}

		return false;
	}

	/**
	 * Drop all plugin tables (used during uninstall).
	 *
	 * WARNING: This permanently deletes all plugin data.
	 *
	 * @since 1.0.0
	 * @return bool True on success, false on failure.
	 */
	public static function drop_tables() {
		global $wpdb;

		$success = true;

		foreach ( self::$tables as $table ) {
			$table_name = $wpdb->prefix . $table;
			
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
			$result = $wpdb->query( "DROP TABLE IF EXISTS `{$table_name}`" );
			
			if ( false === $result ) {
				$success = false;
			}
		}

		// Remove database version option.
		delete_option( self::DB_VERSION_KEY );

		return $success;
	}

	/**
	 * Get database statistics.
	 *
	 * Returns counts and sizes for diagnostics.
	 *
	 * @since 1.0.0
	 * @return array Database statistics.
	 */
	public static function get_stats() {
		global $wpdb;

		$stats = array(
			'db_version' => get_option( self::DB_VERSION_KEY, 'Not installed' ),
			'tables'     => array(),
		);

		foreach ( self::$tables as $key => $table ) {
			$table_name = $wpdb->prefix . $table;

			// Check if table exists.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$exists = $wpdb->get_var(
				$wpdb->prepare(
					'SHOW TABLES LIKE %s',
					$table_name
				)
			);

			if ( $exists ) {
				// Get row count.
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$count = $wpdb->get_var( "SELECT COUNT(*) FROM `{$table_name}`" );

				// Get table size.
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$size = $wpdb->get_var(
					$wpdb->prepare(
						"SELECT ROUND((DATA_LENGTH + INDEX_LENGTH) / 1024 / 1024, 2)
						FROM information_schema.TABLES
						WHERE TABLE_SCHEMA = %s
						AND TABLE_NAME = %s",
						DB_NAME,
						$table_name
					)
				);

				$stats['tables'][ $key ] = array(
					'name'   => $table_name,
					'exists' => true,
					'rows'   => absint( $count ),
					'size_mb' => floatval( $size ),
				);
			} else {
				$stats['tables'][ $key ] = array(
					'name'   => $table_name,
					'exists' => false,
					'rows'   => 0,
					'size_mb' => 0,
				);
			}
		}

		return $stats;
	}

	/**
	 * Run all cleanup operations based on settings.
	 *
	 * @since 1.0.0
	 * @return array Results of cleanup operations.
	 */
	public static function run_cleanup() {
		$settings = get_option( 'content_guard_pro_settings', array() );
		$retention_days = isset( $settings['retention_days'] ) ? absint( $settings['retention_days'] ) : 90;

		$results = array(
			'findings_deleted'  => self::cleanup_old_findings( $retention_days ),
			'scans_deleted'     => self::cleanup_old_scans( $retention_days ),
			'audit_logs_deleted' => self::cleanup_old_audit_logs( 365 ), // Keep audit logs longer.
		);

		return $results;
	}
}

