<?php
/**
 * Dashboard Widget Class
 *
 * Registers and displays a WordPress dashboard widget showing
 * Content Guard Pro security status and findings summary.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Dashboard_Widget
 *
 * Displays Content Guard Pro widget on WordPress dashboard showing:
 * - Last scan date/time
 * - Count of findings by severity (Critical, Suspicious, Review)
 * - Quick action buttons (Run Scan, Review Findings)
 *
 * Per PRD Section 3.5: Dashboard widget (status, counts, quick actions).
 *
 * @since 1.0.0
 */
class CGP_Dashboard_Widget {

	/**
	 * Widget ID.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const WIDGET_ID = 'content_guard_pro_dashboard_widget';

	/**
	 * Initialize the dashboard widget.
	 *
	 * @since 1.0.0
	 */
	public static function init() {
		$instance = new self();
		$instance->register_hooks();
	}

	/**
	 * Register WordPress hooks.
	 *
	 * @since 1.0.0
	 */
	private function register_hooks() {
		// Register dashboard widget.
		add_action( 'wp_dashboard_setup', array( $this, 'register_dashboard_widget' ) );

		// Enqueue widget styles.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_widget_styles' ) );

		// Clear cache when findings or scans change.
		add_action( 'content_guard_pro_finding_saved', array( __CLASS__, 'clear_cache' ) );
		add_action( 'content_guard_pro_finding_status_changed', array( __CLASS__, 'clear_cache' ) );
	}

	/**
	 * Register the dashboard widget.
	 *
	 * @since 1.0.0
	 */
	public function register_dashboard_widget() {
		// Only show to users with manage_options capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Add emoji outside of translatable string for better compatibility.
		$widget_title = '🛡️ ' . __( 'Content Guard Pro', 'content-guard-pro' );

		wp_add_dashboard_widget(
			self::WIDGET_ID,
			$widget_title,
			array( $this, 'render_widget' ),
			null,
			null,
			'normal',
			'high'
		);
	}

	/**
	 * Render the dashboard widget content.
	 *
	 * @since 1.0.0
	 */
	public function render_widget() {
		// Get last scan info.
		$last_scan = $this->get_last_scan();

		// Get findings counts.
		$counts = $this->get_findings_counts();

		// Check if scan is running.
		$scan_running = $this->is_scan_running();

		?>
		<div class="content-guard-pro-dashboard-widget">
			<!-- Last Scan Status -->
			<div class="content-guard-pro-widget-section content-guard-pro-last-scan-section">
				<?php if ( $last_scan ) : ?>
					<p class="content-guard-pro-last-scan">
						<span class="dashicons dashicons-backup"></span>
						<strong><?php esc_html_e( 'Last Scan:', 'content-guard-pro' ); ?></strong>
						<?php echo esc_html( human_time_diff( strtotime( $last_scan->started_at ), current_time( 'timestamp' ) ) ); ?>
						<?php esc_html_e( 'ago', 'content-guard-pro' ); ?>
					</p>
					<p class="content-guard-pro-scan-details">
						<?php
						printf(
							/* translators: 1: items checked, 2: items flagged */
							esc_html__( 'Checked %1$d items, flagged %2$d issues', 'content-guard-pro' ),
							absint( $last_scan->totals_checked ),
							absint( $last_scan->totals_flagged )
						);
						?>
					</p>
				<?php else : ?>
					<p class="content-guard-pro-no-scan">
						<span class="dashicons dashicons-info"></span>
						<?php esc_html_e( 'No scans yet. Run your first scan to get started.', 'content-guard-pro' ); ?>
					</p>
				<?php endif; ?>
			</div>

			<!-- Findings Summary -->
			<div class="content-guard-pro-widget-section content-guard-pro-findings-section">
				<h3><?php esc_html_e( 'Security Findings', 'content-guard-pro' ); ?></h3>
				
				<div class="content-guard-pro-findings-grid">
					<!-- Critical Findings -->
					<div class="content-guard-pro-finding-box content-guard-pro-critical">
						<span class="content-guard-pro-finding-icon dashicons dashicons-warning"></span>
						<div class="content-guard-pro-finding-content">
							<span class="content-guard-pro-finding-count"><?php echo absint( $counts['critical'] ); ?></span>
							<span class="content-guard-pro-finding-label"><?php esc_html_e( 'Critical', 'content-guard-pro' ); ?></span>
						</div>
					</div>

					<!-- Suspicious Findings -->
					<div class="content-guard-pro-finding-box content-guard-pro-suspicious">
						<span class="content-guard-pro-finding-icon dashicons dashicons-flag"></span>
						<div class="content-guard-pro-finding-content">
							<span class="content-guard-pro-finding-count"><?php echo absint( $counts['suspicious'] ); ?></span>
							<span class="content-guard-pro-finding-label"><?php esc_html_e( 'Suspicious', 'content-guard-pro' ); ?></span>
						</div>
					</div>

					<!-- Review Findings -->
					<div class="content-guard-pro-finding-box content-guard-pro-review">
						<span class="content-guard-pro-finding-icon dashicons dashicons-visibility"></span>
						<div class="content-guard-pro-finding-content">
							<span class="content-guard-pro-finding-count"><?php echo absint( $counts['review'] ); ?></span>
							<span class="content-guard-pro-finding-label"><?php esc_html_e( 'Review', 'content-guard-pro' ); ?></span>
						</div>
					</div>
				</div>

				<?php if ( $counts['total'] > 0 ) : ?>
					<p class="content-guard-pro-findings-total">
						<?php
						printf(
							/* translators: %d: total findings */
							esc_html( _n(
								'%d total finding requires attention',
								'%d total findings require attention',
								$counts['total'],
								'content-guard-pro'
							) ),
							absint( $counts['total'] )
						);
						?>
					</p>
				<?php elseif ( $last_scan ) : ?>
					<!-- Show "clean" message only if a scan has been completed -->
					<p class="content-guard-pro-no-findings">
						<span class="dashicons dashicons-yes-alt"></span>
						<?php esc_html_e( 'No security issues detected. Site looks clean!', 'content-guard-pro' ); ?>
					</p>
				<?php else : ?>
					<!-- Show different message when no scans have been completed yet -->
					<p class="content-guard-pro-no-scan-yet">
						<span class="dashicons dashicons-info"></span>
						<?php esc_html_e( 'No scans completed yet. Run your first scan to check for security issues.', 'content-guard-pro' ); ?>
					</p>
				<?php endif; ?>
			</div>

			<!-- Quick Actions -->
			<div class="content-guard-pro-widget-section content-guard-pro-actions-section">
				<?php if ( $scan_running ) : ?>
					<p class="content-guard-pro-scan-running">
						<span class="spinner is-active" style="float: none; margin: 0 5px 0 0;"></span>
						<?php esc_html_e( 'Scan in progress...', 'content-guard-pro' ); ?>
					</p>
				<?php endif; ?>

			<div class="content-guard-pro-action-buttons">
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-scans' ) ); ?>" class="button button-primary">
					<span class="dashicons dashicons-update"></span>
					<?php esc_html_e( 'Run Scan', 'content-guard-pro' ); ?>
				</a>

				<?php if ( $counts['total'] > 0 ) : ?>
					<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-findings' ) ); ?>" class="button">
						<span class="dashicons dashicons-search"></span>
						<?php esc_html_e( 'Review Findings', 'content-guard-pro' ); ?>
					</a>
				<?php endif; ?>

				<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-settings' ) ); ?>" class="button">
					<span class="dashicons dashicons-admin-settings"></span>
					<?php esc_html_e( 'Settings', 'content-guard-pro' ); ?>
				</a>
			</div>
			</div>

			<!-- Status Indicator -->
			<?php if ( $counts['critical'] > 0 ) : ?>
				<div class="content-guard-pro-widget-footer content-guard-pro-status-critical">
					<span class="dashicons dashicons-warning"></span>
					<?php esc_html_e( 'Critical issues require immediate attention!', 'content-guard-pro' ); ?>
				</div>
			<?php elseif ( $counts['suspicious'] > 0 ) : ?>
				<div class="content-guard-pro-widget-footer content-guard-pro-status-warning">
					<span class="dashicons dashicons-flag"></span>
					<?php esc_html_e( 'Suspicious content detected. Review recommended.', 'content-guard-pro' ); ?>
				</div>
			<?php elseif ( $last_scan ) : ?>
				<!-- Show "all clear" message only if a scan has been completed -->
				<div class="content-guard-pro-widget-footer content-guard-pro-status-ok">
					<span class="dashicons dashicons-yes"></span>
					<?php esc_html_e( 'All systems operational. Keep up the good work!', 'content-guard-pro' ); ?>
				</div>
			<?php else : ?>
				<!-- Show different message when no scans have been completed yet -->
				<div class="content-guard-pro-widget-footer content-guard-pro-status-info">
					<span class="dashicons dashicons-info"></span>
					<?php esc_html_e( 'Ready to scan. Click "Run Scan" to get started.', 'content-guard-pro' ); ?>
				</div>
			<?php endif; ?>
		</div>
		<?php
	}

	/**
	 * Get last scan information.
	 *
	 * @since 1.0.0
	 * @return object|null Last scan record or null.
	 */
	private function get_last_scan() {
		global $wpdb;

		// Cache for 5 minutes.
		$cache_key = 'content_guard_pro_dashboard_last_scan';
		$last_scan = wp_cache_get( $cache_key );

		if ( false === $last_scan ) {
			$table_name = $wpdb->prefix . 'content_guard_pro_scans';
			
			// Select only needed columns instead of *.
			// Note: Primary key column is 'scan_id' per database schema.
			$last_scan = $wpdb->get_row(
				"SELECT scan_id, started_at, finished_at, totals_checked, totals_flagged 
				FROM `{$table_name}` 
				WHERE finished_at IS NOT NULL 
				ORDER BY started_at DESC 
				LIMIT 1"
			);

			wp_cache_set( $cache_key, $last_scan ? $last_scan : 'none', '', 300 );
		}

		return ( 'none' === $last_scan ) ? null : $last_scan;
	}

	/**
	 * Get findings counts by severity.
	 *
	 * @since 1.0.0
	 * @return array Counts array with keys: critical, suspicious, review, total.
	 */
	private function get_findings_counts() {
		global $wpdb;

		// Cache for 5 minutes.
		$cache_key = 'content_guard_pro_dashboard_counts';
		$counts    = wp_cache_get( $cache_key );

		if ( false === $counts ) {
			$table_name = $wpdb->prefix . 'content_guard_pro_findings';
			
			$results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT severity, COUNT(*) as count 
					FROM `{$table_name}` 
					WHERE status = %s 
					GROUP BY severity",
					'open'
				)
			);

			$counts = array(
				'critical'   => 0,
				'suspicious' => 0,
				'review'     => 0,
				'total'      => 0,
			);

			// Only accept valid severity levels to prevent unexpected keys.
			$valid_severities = array( 'critical', 'suspicious', 'review' );

			foreach ( $results as $row ) {
				if ( in_array( $row->severity, $valid_severities, true ) ) {
					$counts[ $row->severity ] = absint( $row->count );
					$counts['total']         += absint( $row->count );
				}
			}

			wp_cache_set( $cache_key, $counts, '', 300 );
		}

		return $counts;
	}

	/**
	 * Check if a scan is currently running.
	 *
	 * @since 1.0.0
	 * @return bool True if scan running, false otherwise.
	 */
	private function is_scan_running() {
		global $wpdb;

		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		
		// Calculate timestamp for 1 hour ago using WordPress functions.
		$one_hour_ago = gmdate( 'Y-m-d H:i:s', strtotime( '-1 hour' ) );

		$running_scan = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$table_name}` 
				WHERE finished_at IS NULL 
				AND started_at > %s",
				$one_hour_ago
			)
		);

		return absint( $running_scan ) > 0;
	}

	/**
	 * Enqueue widget styles.
	 *
	 * @since 1.0.0
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_widget_styles( $hook ) {
		// Only load on dashboard.
		if ( 'index.php' !== $hook ) {
			return;
		}

		// Ensure dashboard style is enqueued before adding inline styles.
		if ( ! wp_style_is( 'dashboard', 'enqueued' ) && ! wp_style_is( 'dashboard', 'registered' ) ) {
			wp_enqueue_style( 'dashboard' );
		}

		// Add inline styles for the widget.
		wp_add_inline_style( 'dashboard', $this->get_widget_css() );
	}

	/**
	 * Get widget CSS.
	 *
	 * @since 1.0.0
	 * @return string CSS styles.
	 */
	private function get_widget_css() {
		return '
			/* Content Guard Pro Dashboard Widget */
			.content-guard-pro-dashboard-widget {
				font-size: 13px;
			}
			
			.content-guard-pro-widget-section {
				margin-bottom: 15px;
				padding-bottom: 15px;
				border-bottom: 1px solid #dcdcde;
			}
			
			.content-guard-pro-widget-section:last-of-type {
				border-bottom: none;
				margin-bottom: 0;
				padding-bottom: 0;
			}
			
			/* Last Scan Section */
			.content-guard-pro-last-scan-section .dashicons {
				color: #2271b1;
				margin-right: 5px;
			}
			
			.content-guard-pro-last-scan,
			.content-guard-pro-scan-details {
				margin: 5px 0;
			}
			
			.content-guard-pro-no-scan {
				color: #646970;
				font-style: italic;
			}
			
			.content-guard-pro-no-scan .dashicons {
				color: #646970;
			}
			
			/* Findings Section */
			.content-guard-pro-findings-section h3 {
				margin: 0 0 10px 0;
				font-size: 14px;
				font-weight: 600;
			}
			
			.content-guard-pro-findings-grid {
				display: grid;
				grid-template-columns: repeat(3, 1fr);
				gap: 10px;
				margin-bottom: 10px;
			}
			
			.content-guard-pro-finding-box {
				display: flex;
				align-items: center;
				padding: 10px;
				border-radius: 4px;
				background: #f6f7f7;
			}
			
			.content-guard-pro-finding-box.content-guard-pro-critical {
				background: #fef7f7;
				border-left: 3px solid #d63638;
			}
			
			.content-guard-pro-finding-box.content-guard-pro-suspicious {
				background: #fef9f4;
				border-left: 3px solid #dba617;
			}
			
			.content-guard-pro-finding-box.content-guard-pro-review {
				background: #f7f9fc;
				border-left: 3px solid #2271b1;
			}
			
			.content-guard-pro-finding-icon {
				font-size: 24px;
				margin-right: 10px;
			}
			
			.content-guard-pro-critical .content-guard-pro-finding-icon {
				color: #d63638;
			}
			
			.content-guard-pro-suspicious .content-guard-pro-finding-icon {
				color: #dba617;
			}
			
			.content-guard-pro-review .content-guard-pro-finding-icon {
				color: #2271b1;
			}
			
			.content-guard-pro-finding-content {
				display: flex;
				flex-direction: column;
			}
			
			.content-guard-pro-finding-count {
				font-size: 20px;
				font-weight: 700;
				line-height: 1;
			}
			
			.content-guard-pro-finding-label {
				font-size: 11px;
				text-transform: uppercase;
				color: #646970;
				margin-top: 2px;
			}
			
			.content-guard-pro-findings-total {
				text-align: center;
				color: #646970;
				margin: 10px 0 0 0;
			}
			
			.content-guard-pro-no-findings {
				text-align: center;
				color: #00a32a;
				font-weight: 500;
				margin: 10px 0 0 0;
			}
			
			.content-guard-pro-no-findings .dashicons {
				color: #00a32a;
			}
			
			.content-guard-pro-no-scan-yet {
				text-align: center;
				color: #646970;
				font-weight: 500;
				margin: 10px 0 0 0;
			}
			
			.content-guard-pro-no-scan-yet .dashicons {
				color: #646970;
			}
			
			/* Actions Section */
			.content-guard-pro-scan-running {
				text-align: center;
				color: #2271b1;
				font-weight: 500;
				margin: 0 0 10px 0;
			}
			
			.content-guard-pro-action-buttons {
				display: flex;
				gap: 8px;
				flex-wrap: wrap;
			}
			
			.content-guard-pro-action-buttons .button {
				flex: 1;
				min-width: 100px;
				text-align: center;
				display: inline-flex;
				align-items: center;
				justify-content: center;
				gap: 5px;
			}
			
			.content-guard-pro-action-buttons .button .dashicons {
				font-size: 16px;
				width: 16px;
				height: 16px;
			}
			
			/* Widget Footer */
			.content-guard-pro-widget-footer {
				margin-top: 15px;
				padding: 10px;
				border-radius: 4px;
				text-align: center;
				font-weight: 500;
			}
			
			.content-guard-pro-widget-footer .dashicons {
				margin-right: 5px;
			}
			
			.content-guard-pro-status-critical {
				background: #fef7f7;
				color: #d63638;
				border: 1px solid #f0c0c0;
			}
			
			.content-guard-pro-status-critical .dashicons {
				color: #d63638;
			}
			
			.content-guard-pro-status-warning {
				background: #fef9f4;
				color: #996800;
				border: 1px solid #f0e0c0;
			}
			
			.content-guard-pro-status-warning .dashicons {
				color: #dba617;
			}
			
			.content-guard-pro-status-ok {
				background: #f0f6fc;
				color: #007017;
				border: 1px solid #c0e0c0;
			}
			
			.content-guard-pro-status-ok .dashicons {
				color: #00a32a;
			}
			
			.content-guard-pro-status-info {
				background: #f0f6fc;
				color: #2271b1;
				border: 1px solid #c0d0e0;
			}
			
			.content-guard-pro-status-info .dashicons {
				color: #2271b1;
			}
			
			/* Responsive adjustments */
			@media screen and (max-width: 782px) {
				.content-guard-pro-findings-grid {
					grid-template-columns: 1fr;
				}
				
				.content-guard-pro-action-buttons {
					flex-direction: column;
				}
				
				.content-guard-pro-action-buttons .button {
					width: 100%;
				}
			}
		';
	}

	/**
	 * Clear dashboard widget cache.
	 *
	 * Called when findings or scans change.
	 *
	 * @since 1.0.0
	 */
	public static function clear_cache() {
		wp_cache_delete( 'content_guard_pro_dashboard_last_scan' );
		wp_cache_delete( 'content_guard_pro_dashboard_counts' );
	}
}

