<?php
/**
 * API Client Class
 *
 * Handles all communication with the Content Guard Pro API server.
 * Provides methods for license validation, rule pack downloads, reputation checks,
 * and telemetry reporting.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_API_Client
 *
 * Central API client for communicating with the Content Guard Pro server.
 * Implements caching, rate limiting, and error handling.
 *
 * @since 1.0.0
 */
class CGP_API_Client {

	/**
	 * Production API base URL.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const API_URL_PRODUCTION = 'https://api.contentguardpro.com/api/v1';

	/**
	 * Test/Development API base URL.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const API_URL_TEST = 'http://cgp-api.test/api/v1';

	/**
	 * API request timeout in seconds.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const REQUEST_TIMEOUT = 15;

	/**
	 * Cache duration for license validation (24 hours).
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const LICENSE_CACHE_DURATION = DAY_IN_SECONDS;

	/**
	 * Cache duration for rule packs (7 days).
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const RULE_PACK_CACHE_DURATION = WEEK_IN_SECONDS;

	/**
	 * User agent string for API requests.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	private static $user_agent = null;

	/**
	 * Get the API base URL.
	 *
	 * Returns the production API URL by default.
	 * Can be overridden via CONTENT_GUARD_PRO_API_URL constant in wp-config.php.
	 *
	 * @since 1.0.0
	 * @return string API base URL.
	 */
	public static function get_api_url() {
		// Check for explicit override in wp-config.php.
		// This allows the plugin developer to use a test/staging API by adding:
		// define( 'CONTENT_GUARD_PRO_API_URL', 'http://cgp-api.test/api/v1' );
		if ( defined( 'CONTENT_GUARD_PRO_API_URL' ) ) {
			return rtrim( CONTENT_GUARD_PRO_API_URL, '/' );
		}

		// Always use production URL for all users.
		// This ensures the plugin works correctly on local development environments.
		return self::API_URL_PRODUCTION;
	}

	/**
	 * Get user agent string for API requests.
	 *
	 * @since 1.0.0
	 * @return string User agent string.
	 */
	private static function get_user_agent() {
		if ( null === self::$user_agent ) {
			global $wp_version;
			self::$user_agent = sprintf(
				'ContentGuardPro/%s WordPress/%s PHP/%s',
				CONTENT_GUARD_PRO_VERSION,
				$wp_version,
				PHP_VERSION
			);
		}
		return self::$user_agent;
	}

	/**
	 * Make an API request.
	 *
	 * @since 1.0.0
	 * @param string $endpoint API endpoint (e.g., '/license/validate').
	 * @param string $method   HTTP method ('GET', 'POST', 'PUT', 'DELETE').
	 * @param array  $data     Request data (will be JSON encoded for POST/PUT).
	 * @param array  $headers  Additional headers.
	 * @return array|WP_Error Response array with 'success', 'data', 'code' keys, or WP_Error.
	 */
	public static function request( $endpoint, $method = 'GET', $data = array(), $headers = array() ) {
		$url = self::get_api_url() . $endpoint;

		// Build request arguments.
		$args = array(
			'method'     => strtoupper( $method ),
			'timeout'    => self::REQUEST_TIMEOUT,
			'user-agent' => self::get_user_agent(),
			'headers'    => array_merge(
				array(
					'Content-Type' => 'application/json',
					'Accept'       => 'application/json',
				),
				$headers
			),
		);

		// Add body for POST/PUT requests.
		if ( in_array( $method, array( 'POST', 'PUT' ), true ) && ! empty( $data ) ) {
			$args['body'] = wp_json_encode( $data );
		}

		// Add query string for GET requests with data.
		if ( 'GET' === strtoupper( $method ) && ! empty( $data ) ) {
			$url = add_query_arg( $data, $url );
		}

		// Log request in debug mode.
		self::log_debug( sprintf( 'API Request: %s %s', $method, $url ) );

		// Make the request.
		$response = wp_remote_request( $url, $args );

		// Handle WP_Error.
		if ( is_wp_error( $response ) ) {
			self::log_error( 'API request failed: ' . $response->get_error_message() );
			return $response;
		}

		// Get response code and body.
		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );

		// Parse JSON response.
		$response_data = json_decode( $response_body, true );

		// Check for JSON decode errors.
		if ( null === $response_data && json_last_error() !== JSON_ERROR_NONE ) {
			self::log_error( 'API JSON decode error: ' . json_last_error_msg() );
			return new WP_Error( 'json_decode_error', __( 'Invalid API response', 'content-guard-pro' ) );
		}

		// Log response in debug mode.
		self::log_debug( sprintf( 'API Response: %d', $response_code ) );

		return array(
			'success' => $response_code >= 200 && $response_code < 300,
			'code'    => $response_code,
			'data'    => $response_data,
			'headers' => wp_remote_retrieve_headers( $response ),
		);
	}

	/**
	 * Check API health/connectivity.
	 *
	 * @since 1.0.0
	 * @return array Health check result with 'healthy' boolean and 'message' string.
	 */
	public static function health_check() {
		$response = self::request( '/health', 'GET' );

		if ( is_wp_error( $response ) ) {
			return array(
				'healthy' => false,
				'message' => $response->get_error_message(),
			);
		}

		$api_data = isset( $response['data'] ) ? $response['data'] : array();
		$version  = isset( $api_data['version'] ) ? $api_data['version'] : null;

		return array(
			'healthy' => $response['success'],
			'message' => $response['success'] ? __( 'API is reachable', 'content-guard-pro' ) : __( 'API returned an error', 'content-guard-pro' ),
			'version' => $version,
			'data'    => $api_data,
		);
	}

	/**
	 * Get admin email address.
	 *
	 * Returns the current logged-in user's email if available,
	 * otherwise falls back to the site admin email.
	 *
	 * @since 1.0.0
	 * @return string Admin email address.
	 */
	private static function get_admin_email() {
		// Try to get current logged-in user's email.
		$current_user = wp_get_current_user();
		if ( $current_user && $current_user->exists() && ! empty( $current_user->user_email ) ) {
			return $current_user->user_email;
		}

		// Fall back to site admin email.
		return get_option( 'admin_email', '' );
	}

	/**
	 * Validate a license key.
	 *
	 * @since 1.0.0
	 * @param string $license_key License key to validate.
	 * @param bool   $force_refresh Force API call, bypassing cache.
	 * @return array Validation result with license details.
	 */
	public static function validate_license( $license_key, $force_refresh = false ) {
		if ( empty( $license_key ) ) {
			return array(
				'valid'   => false,
				'error'   => 'no_license_key',
				'message' => __( 'No license key provided', 'content-guard-pro' ),
			);
		}

		// Check cache first (unless force refresh).
		$cache_key = 'cgp_license_' . md5( $license_key );
		if ( ! $force_refresh ) {
			$cached = get_transient( $cache_key );
			if ( false !== $cached ) {
				self::log_debug( 'License validation: cache hit' );
				return $cached;
			}
		}

		// Prepare request data per Server-API-PRD.
		$data = array(
			'license_key'    => $license_key,
			'site_url'       => home_url(),
			'site_name'      => get_bloginfo( 'name' ),
			'plugin_version' => CONTENT_GUARD_PRO_VERSION,
			'wp_version'     => get_bloginfo( 'version' ),
			'php_version'    => PHP_VERSION,
			'admin_email'    => self::get_admin_email(),
		);

		$response = self::request( '/license/validate', 'POST', $data );

		if ( is_wp_error( $response ) ) {
			// Don't cache errors - allow retry.
			return array(
				'valid'   => false,
				'error'   => 'api_error',
				'message' => $response->get_error_message(),
			);
		}

		// Build result from API response.
		$result = array(
			'valid'   => isset( $response['data']['valid'] ) ? (bool) $response['data']['valid'] : false,
			'license' => isset( $response['data']['license'] ) ? $response['data']['license'] : array(),
			'features' => isset( $response['data']['features'] ) ? $response['data']['features'] : array(),
			'activation' => isset( $response['data']['activation'] ) ? $response['data']['activation'] : array(),
			'hints'   => isset( $response['data']['hints'] ) ? $response['data']['hints'] : array(),
		);

		// Add error info if invalid.
		if ( ! $result['valid'] ) {
			$result['error'] = isset( $response['data']['error_code'] ) ? $response['data']['error_code'] : 'unknown';
			$result['message'] = isset( $response['data']['error'] ) ? $response['data']['error'] : __( 'License validation failed', 'content-guard-pro' );
			$result['actions'] = isset( $response['data']['actions'] ) ? $response['data']['actions'] : array();
		}

		// Cache the result.
		$cache_duration = isset( $result['hints']['cache_seconds'] ) 
			? absint( $result['hints']['cache_seconds'] ) 
			: self::LICENSE_CACHE_DURATION;
		set_transient( $cache_key, $result, $cache_duration );

		self::log_debug( sprintf( 'License validation: %s', $result['valid'] ? 'valid' : 'invalid' ) );

		return $result;
	}

	/**
	 * Get license information.
	 *
	 * @since 1.0.0
	 * @param string $license_key License key.
	 * @return array|WP_Error License information or error.
	 */
	public static function get_license_info( $license_key ) {
		if ( empty( $license_key ) ) {
			return new WP_Error( 'no_license_key', __( 'No license key provided', 'content-guard-pro' ) );
		}

		$response = self::request( '/license/info', 'GET', array( 'license_key' => $license_key ) );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		if ( ! $response['success'] ) {
			return new WP_Error( 
				'api_error', 
				isset( $response['data']['error']['message'] ) ? $response['data']['error']['message'] : __( 'Failed to get license info', 'content-guard-pro' )
			);
		}

		return $response['data'];
	}

	/**
	 * Deactivate a site from a license.
	 *
	 * @since 1.0.0
	 * @param string $license_key License key.
	 * @return bool True on success, false on failure.
	 */
	public static function deactivate_license( $license_key ) {
		if ( empty( $license_key ) ) {
			return false;
		}

		$data = array(
			'license_key' => $license_key,
			'site_url'    => home_url(),
		);

		$response = self::request( '/license/deactivate', 'POST', $data );

		if ( is_wp_error( $response ) || ! $response['success'] ) {
			return false;
		}

		// Clear license cache.
		delete_transient( 'cgp_license_' . md5( $license_key ) );

		return true;
	}

	/**
	 * Check for latest rule pack.
	 *
	 * @since 1.0.0
	 * @param string $current_version Current rule pack version.
	 * @param string $license_key     License key (optional, for paid features).
	 * @return array Rule pack metadata.
	 */
	public static function check_rule_pack_update( $current_version = '', $license_key = '' ) {
		$data = array(
			'current_version' => $current_version,
			'channel'         => 'stable',
		);

		if ( ! empty( $license_key ) ) {
			$data['license_key'] = $license_key;
		}

		$response = self::request( '/rules/latest', 'GET', $data );

		if ( is_wp_error( $response ) ) {
			return array(
				'update_available' => false,
				'error'            => $response->get_error_message(),
			);
		}

		// Check if API returned an error
		if ( ! $response['success'] && isset( $response['data'] ) ) {
			$error_info = $response['data'];
			
			// Handle Laravel validation error format
			if ( isset( $error_info['message'] ) && isset( $error_info['errors'] ) ) {
				$error_messages = array( $error_info['message'] );
				// Add validation error details
				foreach ( $error_info['errors'] as $field => $messages ) {
					$error_messages[] = $field . ': ' . implode( ', ', (array) $messages );
				}
				return array(
					'update_available' => false,
					'error'            => implode( '; ', $error_messages ),
					'data'             => $error_info,
				);
			}
			
			// Handle generic error format
			if ( isset( $error_info['error'] ) ) {
				return array(
					'update_available' => false,
					'error'            => $error_info['error'],
					'data'             => $error_info,
				);
			}
			
			// Fallback error message
			return array(
				'update_available' => false,
				'error'            => sprintf( 
					/* translators: %d: HTTP status code returned by the API */
					__( 'API returned error (HTTP %d)', 'content-guard-pro' ),
					$response['code']
				),
				'data'             => $error_info,
			);
		}

		return isset( $response['data'] ) ? $response['data'] : array( 'update_available' => false );
	}

	/**
	 * Download a rule pack.
	 *
	 * @since 1.0.0
	 * @param string $version     Rule pack version to download.
	 * @param string $license_key License key (optional).
	 * @return array|WP_Error Rule pack data or error.
	 */
	public static function download_rule_pack( $version, $license_key = '' ) {
		$data = array();
		if ( ! empty( $license_key ) ) {
			$data['license_key'] = $license_key;
		}

		$response = self::request( '/rules/download/' . $version, 'GET', $data );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		if ( ! $response['success'] ) {
			return new WP_Error( 
				'download_failed', 
				isset( $response['data']['error']['message'] ) ? $response['data']['error']['message'] : __( 'Failed to download rule pack', 'content-guard-pro' )
			);
		}

		return $response['data'];
	}

	/**
	 * Submit a false positive report.
	 *
	 * @since 1.0.0
	 * @param array  $report_data Report data.
	 * @param string $license_key License key (optional, for priority).
	 * @return array|WP_Error Report result or error.
	 */
	public static function report_false_positive( $report_data, $license_key = '' ) {
		$data = array(
			'site_hash'       => hash( 'sha256', home_url() ),
			'rule_id'         => isset( $report_data['rule_id'] ) ? $report_data['rule_id'] : '',
			'rule_version'    => isset( $report_data['rule_version'] ) ? $report_data['rule_version'] : '',
			'matched_excerpt' => isset( $report_data['matched_excerpt'] ) ? $report_data['matched_excerpt'] : '',
			'content_context' => isset( $report_data['content_context'] ) ? $report_data['content_context'] : '',
			'object_type'     => isset( $report_data['object_type'] ) ? $report_data['object_type'] : 'post',
			'confidence_score'=> isset( $report_data['confidence'] ) ? $report_data['confidence'] : null,
			'reason'          => isset( $report_data['reason'] ) ? $report_data['reason'] : '',
			'category'        => isset( $report_data['category'] ) ? $report_data['category'] : 'other',
		);

		if ( ! empty( $license_key ) ) {
			$data['license_key'] = $license_key;
		}

		$response = self::request( '/reports/false-positive', 'POST', $data );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		return $response['data'];
	}

	/**
	 * Check URL reputation via proxy API.
	 *
	 * @since 1.0.0
	 * @param array  $urls        URLs to check.
	 * @param string $license_key License key (required for this feature).
	 * @return array Reputation results.
	 */
	public static function check_reputation( $urls, $license_key ) {
		if ( empty( $license_key ) ) {
			return array(
				'error' => 'license_required',
				'message' => __( 'Reputation checks require a valid license', 'content-guard-pro' ),
			);
		}

		$data = array(
			'license_key' => $license_key,
			'urls'        => $urls,
		);

		$response = self::request( '/reputation/check', 'POST', $data );

		if ( is_wp_error( $response ) ) {
			return array(
				'error'   => 'api_error',
				'message' => $response->get_error_message(),
			);
		}

		return isset( $response['data']['results'] ) ? $response['data']['results'] : array();
	}

	/**
	 * Get community allowlist.
	 *
	 * @since 1.0.0
	 * @param string $category Optional category filter.
	 * @return array Allowlist entries.
	 */
	public static function get_community_allowlist( $category = '' ) {
		$cache_key = 'cgp_community_allowlist_' . ( $category ?: 'all' );
		
		// Check cache first.
		$cached = get_transient( $cache_key );
		if ( false !== $cached ) {
			return $cached;
		}

		$data = array( 'format' => 'compact' );
		if ( ! empty( $category ) ) {
			$data['category'] = $category;
		}

		$response = self::request( '/allowlists/community', 'GET', $data );

		if ( is_wp_error( $response ) || ! $response['success'] ) {
			return array();
		}

		$result = isset( $response['data'] ) ? $response['data'] : array();

		// Cache for 1 hour.
		set_transient( $cache_key, $result, HOUR_IN_SECONDS );

		return $result;
	}

	/**
	 * Register a free tier activation.
	 *
	 * Registers a site activation for free tier users. This allows tracking
	 * of free tier installations for statistics and messaging purposes.
	 *
	 * @since 1.0.0
	 * @return array|WP_Error Registration result or error.
	 */
	public static function register_free_tier_activation() {
		$data = array(
			'site_url'       => home_url(),
			'site_name'      => get_bloginfo( 'name' ),
			'plugin_version' => CONTENT_GUARD_PRO_VERSION,
			'wp_version'     => get_bloginfo( 'version' ),
			'php_version'    => PHP_VERSION,
			'admin_email'    => self::get_admin_email(),
		);

		// Use /license/validate endpoint with empty license_key for free tier.
		// Server should recognize empty license_key as free tier activation.
		$data['license_key'] = '';

		$response = self::request( '/license/validate', 'POST', $data );

		if ( is_wp_error( $response ) ) {
			self::log_error( 'Free tier activation registration failed: ' . $response->get_error_message() );
			return $response;
		}

		// Log successful registration.
		if ( $response['success'] ) {
			self::log_debug( 'Free tier activation registered successfully' );
		} else {
			self::log_error( 'Free tier activation registration returned error: ' . wp_json_encode( $response['data'] ) );
		}

		return $response;
	}

	/**
	 * Submit telemetry data (opt-in).
	 *
	 * @since 1.0.0
	 * @param array  $scan_data   Scan telemetry data.
	 * @param string $license_key License key (optional).
	 * @return bool True on success, false on failure.
	 */
	public static function submit_telemetry( $scan_data, $license_key = '' ) {
		// Check if telemetry is enabled.
		$settings = get_option( 'content_guard_pro_settings', array() );
		if ( empty( $settings['telemetry_enabled'] ) ) {
			return false;
		}

		$data = array(
			'site_hash'    => hash( 'sha256', home_url() ),
			'scan_mode'    => isset( $scan_data['mode'] ) ? $scan_data['mode'] : 'standard',
			'trigger_type' => isset( $scan_data['trigger'] ) ? $scan_data['trigger'] : 'manual',
			'items_scanned' => isset( $scan_data['items_scanned'] ) ? $scan_data['items_scanned'] : 0,
			'findings_count' => isset( $scan_data['findings_count'] ) ? $scan_data['findings_count'] : 0,
			'findings_by_severity' => isset( $scan_data['findings_by_severity'] ) ? $scan_data['findings_by_severity'] : array(),
			'duration_seconds' => isset( $scan_data['duration'] ) ? $scan_data['duration'] : 0,
			'plugin_version' => CONTENT_GUARD_PRO_VERSION,
			'wp_version'     => get_bloginfo( 'version' ),
			'php_version'    => PHP_VERSION,
		);

		if ( ! empty( $license_key ) ) {
			$data['license_key'] = $license_key;
		}

		// Non-blocking request.
		$response = self::request( '/telemetry/scan', 'POST', $data );

		return ! is_wp_error( $response ) && $response['success'];
	}

	/**
	 * Log debug message.
	 *
	 * @since 1.0.0
	 * @param string $message Debug message.
	 */
	private static function log_debug( $message ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			cgp_log( '[Content Guard Pro - API] ' . $message );
		}
	}

	/**
	 * Log error message.
	 *
	 * @since 1.0.0
	 * @param string $message Error message.
	 */
	private static function log_error( $message ) {
		cgp_log( '[Content Guard Pro - API ERROR] ' . $message );
	}
}

