<?php
/**
 * Admin Interface Class
 *
 * Handles all admin menu pages, settings pages, and admin UI for Content Guard Pro.
 * Creates the admin menu structure as defined in PRD Appendix G.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Admin
 *
 * Manages all admin interface components including menu registration,
 * page rendering, and admin-specific functionality.
 *
 * Uses singleton pattern to prevent garbage collection issues with
 * hook callbacks in PHP 8.x and certain hosting environments.
 *
 * @since 1.0.0
 */
class CGP_Admin {

	/**
	 * Singleton instance.
	 *
	 * @since 1.0.2
	 * @var CGP_Admin|null
	 */
	private static $instance = null;

	/**
	 * Parent menu slug.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const PARENT_SLUG = 'content-guard-pro';

	/**
	 * Menu capability required.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const MENU_CAPABILITY = 'manage_options';

	/**
	 * View capability for findings.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const VIEW_CAPABILITY = 'edit_posts';

	/**
	 * Plugin icon (Dashicons).
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const MENU_ICON = 'dashicons-shield-alt';

	/**
	 * Menu position.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const MENU_POSITION = 58;

	/**
	 * Cache for critical findings count.
	 *
	 * @since 1.0.2
	 * @var int|null
	 */
	private $critical_count_cache = null;

	/**
	 * Get singleton instance.
	 *
	 * @since 1.0.2
	 * @return CGP_Admin
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Initialize the admin class.
	 *
	 * Uses singleton pattern to ensure the instance persists and hook
	 * callbacks remain valid throughout the request lifecycle.
	 *
	 * @since 1.0.0
	 */
	public static function init() {
		$instance = self::get_instance();
		$instance->load_page_classes();
		$instance->register_hooks();
	}

	/**
	 * Load admin page classes.
	 *
	 * Note: Setup wizard is loaded by dashboard page when needed (via URL parameter).
	 * This avoids hidden menu registration issues in PHP 8.x environments.
	 *
	 * @since 1.0.0
	 */
	private function load_page_classes() {
		$pages = array(
			'dashboard',
			'scans',
			'findings',
			'patterns',
			'quarantine',
			'reports',
			'settings',
			'diagnostics',
			'help',
			// Note: setup-wizard is loaded on-demand by dashboard when setup=1 param is present.
		);

		foreach ( $pages as $page ) {
			$file = CONTENT_GUARD_PRO_PATH . 'admin/pages/class-cgp-admin-' . $page . '.php';
			if ( file_exists( $file ) ) {
				require_once $file;
			}
		}
	}

	/**
	 * Register WordPress hooks for admin functionality.
	 *
	 * @since 1.0.0
	 */
	public function register_hooks() {
		// Register admin menu.
		add_action( 'admin_menu', array( $this, 'register_admin_menu' ) );

		// Process scan actions early (before any output).
		add_action( 'admin_init', array( $this, 'process_scan_actions' ) );

		// Process pattern actions early (before any output).
		add_action( 'admin_init', array( 'CGP_Admin_Patterns', 'handle_actions' ) );

		// Enqueue admin styles and scripts.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );

		// Enqueue block editor scripts for on-save scanning (US-037, US-038).
		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_block_editor_assets' ) );

		// Add admin notices.
		add_action( 'admin_notices', array( $this, 'display_admin_notices' ) );

		// Add admin bar menu.
		add_action( 'admin_bar_menu', array( $this, 'add_admin_bar_menu' ), 100 );

		// AJAX handler for dismissing welcome notice.
		add_action( 'wp_ajax_content_guard_pro_dismiss_welcome_notice', array( $this, 'ajax_dismiss_welcome_notice' ) );
		
		// AJAX handler for dismissing critical notice.
		add_action( 'wp_ajax_content_guard_pro_dismiss_critical_notice', array( $this, 'ajax_dismiss_critical_notice' ) );
		
		// AJAX handler for getting finding details.
		add_action( 'wp_ajax_content_guard_pro_get_finding_details', array( $this, 'ajax_get_finding_details' ) );

		// AJAX handler for ignoring a finding.
		add_action( 'wp_ajax_content_guard_pro_ignore_finding', array( $this, 'ajax_ignore_finding' ) );

		// AJAX handler for quarantining a finding.
		add_action( 'wp_ajax_content_guard_pro_quarantine_finding', array( $this, 'ajax_quarantine_finding' ) );

		// AJAX handler for un-quarantining a finding.
		add_action( 'wp_ajax_content_guard_pro_unquarantine_finding', array( $this, 'ajax_unquarantine_finding' ) );

		// AJAX handler for dismissing on-save scan notice.
		add_action( 'wp_ajax_content_guard_pro_dismiss_scan_notice', array( $this, 'ajax_dismiss_scan_notice' ) );
	}

	/**
	 * Register admin menu and submenus.
	 *
	 * Creates the menu structure defined in PRD Appendix G.
	 *
	 * @since 1.0.0
	 */
	public function register_admin_menu() {
		// Add top-level menu: Content Guard Pro.
		add_menu_page(
			__( 'Content Guard Pro', 'content-guard-pro' ),
			__( 'Content Guard Pro', 'content-guard-pro' ),
			self::MENU_CAPABILITY,
			self::PARENT_SLUG,
			array( $this, 'display_dashboard_page' ),
			self::MENU_ICON,
			self::MENU_POSITION
		);

		// Dashboard (default) - Rename the first submenu to avoid duplicate.
		add_submenu_page(
			self::PARENT_SLUG,
			__( 'Dashboard', 'content-guard-pro' ),
			__( 'Dashboard', 'content-guard-pro' ),
			self::MENU_CAPABILITY,
			self::PARENT_SLUG,
			array( $this, 'display_dashboard_page' )
		);

		// Scans - Run/pause/resume; progress; history.
		add_submenu_page(
			self::PARENT_SLUG,
			__( 'Scans', 'content-guard-pro' ),
			__( 'Scans', 'content-guard-pro' ),
			self::MENU_CAPABILITY,
			self::PARENT_SLUG . '-scans',
			array( $this, 'display_scans_page' )
		);

		// Findings - List table with filters, bulk actions, export.
		add_submenu_page(
			self::PARENT_SLUG,
			__( 'Findings', 'content-guard-pro' ),
			__( 'Findings', 'content-guard-pro' ),
			self::VIEW_CAPABILITY,
			self::PARENT_SLUG . '-findings',
			array( $this, 'display_findings_page' )
		);

		// Patterns - View current rule pack version; allow/deny lists; manual update.
		add_submenu_page(
			self::PARENT_SLUG,
			__( 'Patterns', 'content-guard-pro' ),
			__( 'Patterns', 'content-guard-pro' ),
			self::MENU_CAPABILITY,
			self::PARENT_SLUG . '-patterns',
			array( $this, 'display_patterns_page' )
		);

		// Quarantine - All quarantined items with bulk restore/un-quarantine.
		add_submenu_page(
			self::PARENT_SLUG,
			__( 'Quarantine', 'content-guard-pro' ),
			__( 'Quarantine', 'content-guard-pro' ),
			self::MENU_CAPABILITY,
			self::PARENT_SLUG . '-quarantine',
			array( $this, 'display_quarantine_page' )
		);

		// Reports - Scan summaries; CSV/JSON export.
		add_submenu_page(
			self::PARENT_SLUG,
			__( 'Reports', 'content-guard-pro' ),
			__( 'Reports', 'content-guard-pro' ),
			self::MENU_CAPABILITY,
			self::PARENT_SLUG . '-reports',
			array( $this, 'display_reports_page' )
		);

		// Settings - Wizard re-run; scheduling; notifications; performance; integrations.
		add_submenu_page(
			self::PARENT_SLUG,
			__( 'Settings', 'content-guard-pro' ),
			__( 'Settings', 'content-guard-pro' ),
			self::MENU_CAPABILITY,
			self::PARENT_SLUG . '-settings',
			array( $this, 'display_settings_page' )
		);

		// Diagnostics - Environment checks, logs, "Copy diagnostics", test pattern tool.
		add_submenu_page(
			self::PARENT_SLUG,
			__( 'Diagnostics', 'content-guard-pro' ),
			__( 'Diagnostics', 'content-guard-pro' ),
			self::MENU_CAPABILITY,
			self::PARENT_SLUG . '-diagnostics',
			array( $this, 'display_diagnostics_page' )
		);

		// Help / Docs - Links to documentation, FAQ, support.
		add_submenu_page(
			self::PARENT_SLUG,
			__( 'Help & Documentation', 'content-guard-pro' ),
			__( 'Help & Docs', 'content-guard-pro' ),
			self::VIEW_CAPABILITY,
			self::PARENT_SLUG . '-help',
			array( $this, 'display_help_page' )
		);

		// Note: Setup Wizard is integrated into the Dashboard page via ?setup=1 parameter.
		// This avoids hidden menu registration issues in PHP 8.x environments.
		// Access via: admin.php?page=content-guard-pro&setup=1
	}

	/**
	 * Process scan actions (delete, view) early on admin_init.
	 *
	 * Handles scan-related actions before any output to allow proper redirects.
	 * Similar to setup wizard form processing.
	 *
	 * @since 1.0.1
	 */
	public function process_scan_actions() {
		// Only process on scans page.
		if ( ! isset( $_GET['page'] ) || 'content-guard-pro-scans' !== $_GET['page'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			return;
		}

		// Get action.
		$action = isset( $_GET['action'] ) ? sanitize_text_field( wp_unslash( $_GET['action'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		// Handle delete action.
		if ( 'delete' === $action && isset( $_GET['scan_id'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$this->handle_delete_scan();
		}

		// Handle view action.
		if ( 'view' === $action && isset( $_GET['scan_id'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$this->handle_view_scan();
		}
	}

	/**
	 * Handle scan deletion.
	 *
	 * @since 1.0.1
	 */
	private function handle_delete_scan() {
		$scan_id = absint( $_GET['scan_id'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		// Verify nonce.
		$nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'delete_scan_' . $scan_id ) ) {
			wp_die( esc_html__( 'Security check failed. Please try again.', 'content-guard-pro' ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to delete scans.', 'content-guard-pro' ) );
		}

		// Delete the scan record.
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		$deleted = $wpdb->delete(
			$table_name,
			array(
				'scan_id' => $scan_id,
				'blog_id' => get_current_blog_id(),
			),
			array( '%d', '%d' )
		);

		// Set message based on result.
		if ( false === $deleted ) {
			$message = 'delete_failed';
		} elseif ( $deleted > 0 ) {
			$message = 'deleted';
		} else {
			$message = 'not_found';
		}

		// Redirect with message.
		wp_safe_redirect(
			add_query_arg(
				'message',
				$message,
				admin_url( 'admin.php?page=content-guard-pro-scans' )
			)
		);
		exit;
	}

	/**
	 * Handle view scan details.
	 *
	 * Validates scan_id and allows page to load normally.
	 * The actual modal is opened via JavaScript when action=view is present in URL.
	 *
	 * @since 1.0.1
	 */
	private function handle_view_scan() {
		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to view scan details.', 'content-guard-pro' ) );
		}

		// Validate scan_id.
		$scan_id = isset( $_GET['scan_id'] ) ? absint( $_GET['scan_id'] ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( ! $scan_id ) {
			wp_safe_redirect( admin_url( 'admin.php?page=content-guard-pro-scans' ) );
			exit;
		}

		// Don't redirect - let the page load normally.
		// JavaScript will detect action=view in URL and open the modal.
		// The URL will be cleaned by JavaScript after modal opens to prevent re-opening on refresh.
	}

	/**
	 * Display the Dashboard page.
	 *
	 * Delegates to CGP_Admin_Dashboard class.
	 *
	 * @since 1.0.0
	 */
	public function display_dashboard_page() {
		CGP_Admin_Dashboard::display();
	}

	/**
	 * Display the Scans page.
	 *
	 * Delegates to CGP_Admin_Scans class.
	 *
	 * @since 1.0.0
	 */
	public function display_scans_page() {
		CGP_Admin_Scans::display();
	}

	/**
	 * Display the Findings page.
	 *
	 * Delegates to CGP_Admin_Findings class.
	 *
	 * @since 1.0.0
	 */
	public function display_findings_page() {
		CGP_Admin_Findings::display();
	}

	/**
	 * Display the Patterns page.
	 *
	 * Delegates to CGP_Admin_Patterns class.
	 *
	 * @since 1.0.0
	 */
	public function display_patterns_page() {
		CGP_Admin_Patterns::display();
	}

	/**
	 * Display the Quarantine page.
	 *
	 * Delegates to CGP_Admin_Quarantine class.
	 *
	 * @since 1.0.0
	 */
	public function display_quarantine_page() {
		CGP_Admin_Quarantine::display();
	}

	/**
	 * Display the Reports page.
	 *
	 * Delegates to CGP_Admin_Reports class.
	 *
	 * @since 1.0.0
	 */
	public function display_reports_page() {
		CGP_Admin_Reports::display();
	}

	/**
	 * Display the Settings page.
	 *
	 * Delegates to CGP_Admin_Settings class.
	 *
	 * @since 1.0.0
	 */
	public function display_settings_page() {
		CGP_Admin_Settings::display();
	}

	/**
	 * Display the Diagnostics page.
	 *
	 * Delegates to CGP_Admin_Diagnostics class.
	 *
	 * @since 1.0.0
	 */
	public function display_diagnostics_page() {
		CGP_Admin_Diagnostics::display();
	}

	/**
	 * Display the Help & Documentation page.
	 *
	 * Delegates to CGP_Admin_Help class.
	 *
	 * @since 1.0.0
	 */
	public function display_help_page() {
		CGP_Admin_Help::display();
	}

	/**
	 * Enqueue admin styles and scripts.
	 *
	 * @since 1.0.0
	 * @param string $hook The current admin page hook.
	 */
	public function enqueue_admin_assets( $hook ) {
		$should_enqueue = false;

		// Check if it's a plugin page.
		if ( strpos( $hook, self::PARENT_SLUG ) !== false || strpos( $hook, 'content-guard-pro-' ) !== false ) {
			$should_enqueue = true;
		}

		// If not a plugin page, check if we need to show the critical notice.
		if ( ! $should_enqueue ) {
			$critical_count = $this->get_critical_findings_count();
			
			if ( $critical_count > 0 && current_user_can( self::VIEW_CAPABILITY ) ) {
				$dismissed = get_user_meta( get_current_user_id(), 'content_guard_pro_dismissed_critical_notice', true );
				
				if ( ! $dismissed ) {
					$should_enqueue = true;
				} else {
					$last_dismissed_count = get_user_meta( get_current_user_id(), 'content_guard_pro_last_dismissed_critical_count', true );
					if ( $critical_count > absint( $last_dismissed_count ) ) {
						$should_enqueue = true;
					}
				}
			}
		}

		if ( ! $should_enqueue ) {
			return;
		}

		// Enqueue WordPress media scripts for modal functionality.
		wp_enqueue_media();
		add_thickbox(); // Add ThickBox support for modals.

		// Enqueue admin CSS (to be created).
		wp_enqueue_style(
			'content-guard-pro-admin',
			CONTENT_GUARD_PRO_URL . 'assets/css/admin.css',
			array(),
			CONTENT_GUARD_PRO_VERSION
		);

		// Enqueue admin JS (to be created).
		wp_enqueue_script(
			'content-guard-pro-admin',
			CONTENT_GUARD_PRO_URL . 'assets/js/admin.js',
			array( 'jquery', 'media-editor' ),
			CONTENT_GUARD_PRO_VERSION,
			true
		);

		// Enqueue scan progress JS for real-time updates on Scans page.
		if ( isset( $_GET['page'] ) && 'content-guard-pro-scans' === $_GET['page'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			wp_enqueue_script(
				'content-guard-pro-scan-progress',
				CONTENT_GUARD_PRO_URL . 'assets/js/scan-progress.js',
				array( 'jquery' ),
				CONTENT_GUARD_PRO_VERSION,
				true
			);

			// Localize scan progress script with REST API data.
			wp_localize_script(
				'content-guard-pro-scan-progress',
				'contentGuardProScanProgress',
				array(
					'apiUrl' => rest_url( 'content-guard-pro/v1' ),
					'nonce'  => wp_create_nonce( 'wp_rest' ),
					'debug'  => defined( 'WP_DEBUG' ) && WP_DEBUG,
					'i18n'   => array(
						'errorLoading'     => __( 'Error loading scan progress. Please refresh the page.', 'content-guard-pro' ),
						'modeQuick'        => __( 'Quick Mode', 'content-guard-pro' ),
						'modeStandard'     => __( 'Standard Mode', 'content-guard-pro' ),
						'statusPending'    => __( 'Pending', 'content-guard-pro' ),
						'statusRunning'    => __( 'Running', 'content-guard-pro' ),
						'statusPaused'     => __( 'Paused', 'content-guard-pro' ),
						'statusCancelled'  => __( 'Cancelled', 'content-guard-pro' ),
						'statusCompleted'  => __( 'Completed', 'content-guard-pro' ),
						'statusFailed'     => __( 'Failed', 'content-guard-pro' ),
						'justNow'          => __( 'Just now', 'content-guard-pro' ),
						'running'          => __( 'Running...', 'content-guard-pro' ),
					),
				)
			);
		}

		// Localize script with data.
		wp_localize_script(
			'content-guard-pro-admin',
			'contentGuardProAdmin',
			array(
				'ajaxUrl'    => admin_url( 'admin-ajax.php' ),
				'nonce'      => wp_create_nonce( 'content-guard-pro-admin-nonce' ),
				'pluginUrl'  => CONTENT_GUARD_PRO_URL,
				'strings'    => array(
					'confirmQuarantine' => __( 'Are you sure you want to quarantine this item?', 'content-guard-pro' ),
					'confirmIgnore'     => __( 'Are you sure you want to ignore this finding?', 'content-guard-pro' ),
					'confirmDelete'     => __( 'Are you sure you want to delete this item?', 'content-guard-pro' ),
				),
			)
		);
	}

	/**
	 * Enqueue block editor assets for on-save scanning.
	 *
	 * Implements US-037 and US-038 from PRD:
	 * - Scan triggers on post save/publish
	 * - Warning displayed if issues found
	 * - Results shown in editor interface
	 *
	 * @since 1.0.0
	 */
	public function enqueue_block_editor_assets() {
		// Check if user can edit posts.
		if ( ! current_user_can( 'edit_posts' ) ) {
			return;
		}

		// Get settings to check if real-time scanning is enabled.
		$settings = get_option( 'content_guard_pro_settings', array() );
		$scan_enabled = isset( $settings['realtime_scan_enabled'] ) ? (bool) $settings['realtime_scan_enabled'] : true;
		
		// Check license capability: Full site on-save scanning requires paid tier (Solo Guard+).
		// Free tier only gets single post scanning (manual scans work, auto-scan doesn't).
		// Default to false (free tier) if license manager is not available.
		if ( class_exists( 'CGP_License_Manager' ) ) {
			$can_on_save = CGP_License_Manager::can( 'on_save_scanning' );
			if ( ! $can_on_save ) {
				$scan_enabled = false; // Override setting if license doesn't support it.
			}
			// Debug logging (remove in production).
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( sprintf( 
					'Content Guard Pro [Editor Assets]: scan_enabled=%s, can_on_save=%s, tier=%s', 
					$scan_enabled ? 'true' : 'false',
					$can_on_save ? 'true' : 'false',
					CGP_License_Manager::get_tier()
				) );
			}
		} else {
			// License manager not available - default to free tier (no auto-scan).
			$scan_enabled = false;
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'Content Guard Pro [Editor Assets]: License manager not available, defaulting scan_enabled=false' );
			}
		}

		// Enqueue editor scan panel script.
		wp_enqueue_script(
			'content-guard-pro-editor-panel',
			CONTENT_GUARD_PRO_URL . 'assets/js/editor-scan-panel.js',
			array(
				'wp-plugins',
				'wp-edit-post',
				'wp-element',
				'wp-components',
				'wp-data',
				'wp-i18n',
				'wp-api-fetch',
			),
			CONTENT_GUARD_PRO_VERSION,
			true
		);

		// Localize script with data.
		wp_localize_script(
			'content-guard-pro-editor-panel',
			'contentGuardProEditor',
			array(
				'apiUrl'      => rest_url( 'content-guard-pro/v1' ),
				'nonce'       => wp_create_nonce( 'wp_rest' ),
				'scanEnabled' => (bool) $scan_enabled, // Explicitly cast to boolean for JavaScript.
				'findingsUrl' => admin_url( 'admin.php?page=' . self::PARENT_SLUG . '-findings' ),
				'i18n'        => array(
					'scanning'         => __( 'Scanning...', 'content-guard-pro' ),
					'scanContent'      => __( 'Scan Content', 'content-guard-pro' ),
					'noIssues'         => __( 'No security issues detected.', 'content-guard-pro' ),
					'criticalIssues'   => __( 'Critical security issues found!', 'content-guard-pro' ),
					'issuesFound'      => __( 'Security issues detected', 'content-guard-pro' ),
					'reviewBefore'     => __( 'Please review before publishing.', 'content-guard-pro' ),
					'canStillPublish'  => __( 'You can still publish.', 'content-guard-pro' ),
					'viewFindings'     => __( 'View All Findings', 'content-guard-pro' ),
					'rescan'           => __( 'Re-scan', 'content-guard-pro' ),
					'confidence'       => __( 'confidence', 'content-guard-pro' ),
					'critical'         => __( 'Critical', 'content-guard-pro' ),
					'suspicious'       => __( 'Suspicious', 'content-guard-pro' ),
					'review'           => __( 'Review', 'content-guard-pro' ),
					'disabled'         => __( 'Real-time scanning is disabled in settings.', 'content-guard-pro' ),
				),
			)
		);

		// Add inline styles for the editor panel.
		wp_add_inline_style( 'wp-edit-post', '
			.cgp-scan-notice {
				margin-bottom: 12px !important;
			}
			.cgp-post-status-info {
				border-top: 1px solid #e0e0e0;
				padding-top: 8px;
				margin-top: 8px;
			}
		' );
	}

	/**
	 * Display admin notices.
	 *
	 * Shows important alerts and notifications in the admin area.
	 *
	 * @since 1.0.0
	 */
	public function display_admin_notices() {
		// Show first-time setup notice.
		// Only show if:
		// 1. Not manually dismissed by user
		// 2. AND no scans have been run yet (UX: auto-hide after first scan)
		$dismissed = get_user_meta( get_current_user_id(), 'content_guard_pro_dismissed_welcome_notice', true );
		$has_run_scan = $this->has_run_any_scan();
		
		if ( get_option( 'content_guard_pro_first_activation' ) && ! $dismissed && ! $has_run_scan ) {
			?>
			<div class="notice notice-info is-dismissible" data-dismissible="content-guard-pro-welcome-notice" data-nonce="<?php echo esc_attr( wp_create_nonce( 'content-guard-pro-dismiss-notice' ) ); ?>">
				<p>
					<?php
					printf(
						wp_kses(
							/* translators: %s: Settings page URL */
							__( 'Welcome to Content Guard Pro! <a href="%s">Run your first scan</a> to get started.', 'content-guard-pro' ),
							array(
								'a' => array(
									'href' => array(),
								),
							)
						),
						esc_url( admin_url( 'admin.php?page=' . self::PARENT_SLUG . '-scans' ) )
					);
					?>
				</p>
			</div>
			<?php
		}

		// Show critical findings notice.
		$this->show_critical_findings_notice();
		
		// Show scan completion notice.
		$this->show_scan_completion_notice();

		// Show on-save scan result notice for classic editor.
		$this->show_on_save_scan_notice();
	}
	
	/**
	 * Show critical findings notice.
	 *
	 * Displays an admin notice when critical findings are detected.
	 * Per PRD Section 3.5 - Admin notices for new Critical findings.
	 *
	 * @since 1.0.0
	 */
	private function show_critical_findings_notice() {
		// Check if admin notices are enabled in settings.
		$settings = get_option( 'content_guard_pro_settings', array() );
		if ( empty( $settings['notifications_admin'] ) ) {
			return;
		}

		// Get critical findings count.
		$critical_count = $this->get_critical_findings_count();
		
		if ( $critical_count === 0 ) {
			return;
		}
		
		// Check if user has capability to view findings.
		if ( ! current_user_can( self::VIEW_CAPABILITY ) ) {
			return;
		}
		
		// Check if notice was dismissed.
		$dismissed = get_user_meta( get_current_user_id(), 'content_guard_pro_dismissed_critical_notice', true );
		if ( $dismissed ) {
			// Check if there are new findings since dismissal.
			$last_dismissed_count = get_user_meta( get_current_user_id(), 'content_guard_pro_last_dismissed_critical_count', true );
			if ( $critical_count <= absint( $last_dismissed_count ) ) {
				return;
			}
		}
		
		?>
		<div class="notice notice-error is-dismissible" data-dismissible="content-guard-pro-critical-findings" data-critical-count="<?php echo absint( $critical_count ); ?>" data-nonce="<?php echo esc_attr( wp_create_nonce( 'content-guard-pro-dismiss-notice' ) ); ?>">
			<p>
				<strong><?php esc_html_e( 'Content Guard Pro Alert:', 'content-guard-pro' ); ?></strong>
				<?php
				printf(
					/* translators: %d: Number of critical findings */
					esc_html( _n(
						'%d critical security issue detected in your content!',
						'%d critical security issues detected in your content!',
						$critical_count,
						'content-guard-pro'
					) ),
					absint( $critical_count )
				);
				?>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=' . self::PARENT_SLUG . '-findings&severity=critical' ) ); ?>" class="button button-primary" style="margin-left: 10px;">
					<?php esc_html_e( 'Review Now', 'content-guard-pro' ); ?>
				</a>
			</p>
		</div>
		<?php
	}
	
	/**
	 * Show scan completion notice.
	 *
	 * Displays an admin notice when a scan completes.
	 * Per PRD Section 3.5 - Admin notices for scan completion.
	 *
	 * @since 1.0.0
	 */
	private function show_scan_completion_notice() {
		// Check for scan completion transient.
		$completed_scan = get_transient( 'content_guard_pro_scan_completed' );
		
		if ( ! $completed_scan ) {
			return;
		}
		
		// Delete the transient so we only show once.
		delete_transient( 'content_guard_pro_scan_completed' );
		
		// Check user capability.
		if ( ! current_user_can( self::VIEW_CAPABILITY ) ) {
			return;
		}
		
		$findings_count = isset( $completed_scan['findings'] ) ? absint( $completed_scan['findings'] ) : 0;
		$critical_count = isset( $completed_scan['critical'] ) ? absint( $completed_scan['critical'] ) : 0;
		
		// Determine notice type based on findings.
		$notice_type = 'success';
		if ( $critical_count > 0 ) {
			$notice_type = 'error';
		} elseif ( $findings_count > 0 ) {
			$notice_type = 'warning';
		}
		
		?>
		<div class="notice notice-<?php echo esc_attr( $notice_type ); ?> is-dismissible">
			<p>
				<strong><?php esc_html_e( 'Scan Complete!', 'content-guard-pro' ); ?></strong>
				<?php
				if ( $findings_count === 0 ) {
					esc_html_e( 'No issues found. Your content looks clean!', 'content-guard-pro' );
				} else {
					printf(
						/* translators: 1: Total findings, 2: Critical findings */
						esc_html__( 'Found %1$d issue(s), including %2$d critical.', 'content-guard-pro' ),
						absint( $findings_count ),
						absint( $critical_count )
					);
					?>
					<a href="<?php echo esc_url( admin_url( 'admin.php?page=' . self::PARENT_SLUG . '-findings' ) ); ?>" class="button button-small" style="margin-left: 10px;">
						<?php esc_html_e( 'View Findings', 'content-guard-pro' ); ?>
					</a>
					<?php
				}
				?>
			</p>
		</div>
		<?php
	}
	
	/**
	 * Show on-save scan result notice.
	 *
	 * Displays results from the real-time on-save scan.
	 * Per PRD US-037 and US-038 - Results shown in editor interface.
	 * This method shows notices for the classic editor; Gutenberg uses the editor panel.
	 *
	 * @since 1.0.0
	 */
	private function show_on_save_scan_notice() {
		// Only show on post edit screens.
		$screen = get_current_screen();
		if ( ! $screen || 'post' !== $screen->base ) {
			return;
		}

		// Get the current post ID.
		$post_id = isset( $_GET['post'] ) ? absint( $_GET['post'] ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( ! $post_id ) {
			return;
		}

		// Check user capability.
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		// Check for scan result transient.
		$transient_key = 'cgp_scan_result_' . $post_id . '_' . get_current_user_id();
		$scan_result = get_transient( $transient_key );

		if ( ! $scan_result || ! is_array( $scan_result ) ) {
			return;
		}

		// Delete transient to show only once.
		delete_transient( $transient_key );

		// Check if there are issues.
		$has_issues = isset( $scan_result['has_issues'] ) && $scan_result['has_issues'];
		$has_critical = isset( $scan_result['has_critical'] ) && $scan_result['has_critical'];
		$findings = isset( $scan_result['findings'] ) ? $scan_result['findings'] : array();
		$counts = isset( $scan_result['counts'] ) ? $scan_result['counts'] : array();

		// No issues - show success notice.
		if ( ! $has_issues ) {
			?>
			<div class="notice notice-success is-dismissible cgp-scan-result-notice">
				<p>
					<span class="dashicons dashicons-shield" style="color: #00a32a; margin-right: 5px;"></span>
					<strong><?php esc_html_e( 'Content Guard Pro:', 'content-guard-pro' ); ?></strong>
					<?php esc_html_e( 'No security issues detected in this content.', 'content-guard-pro' ); ?>
				</p>
			</div>
			<?php
			return;
		}

		// Issues found - show warning or error notice.
		$notice_type = $has_critical ? 'error' : 'warning';
		$critical_count = isset( $counts['critical'] ) ? absint( $counts['critical'] ) : 0;
		$suspicious_count = isset( $counts['suspicious'] ) ? absint( $counts['suspicious'] ) : 0;
		$review_count = isset( $counts['review'] ) ? absint( $counts['review'] ) : 0;
		$total_count = count( $findings );
		
		?>
		<div class="notice notice-<?php echo esc_attr( $notice_type ); ?> is-dismissible cgp-scan-result-notice" data-post-id="<?php echo absint( $post_id ); ?>">
			<p>
				<span class="dashicons dashicons-<?php echo $has_critical ? 'warning' : 'flag'; ?>" style="color: <?php echo $has_critical ? '#d63638' : '#dba617'; ?>; margin-right: 5px;"></span>
				<strong><?php esc_html_e( 'Content Guard Pro:', 'content-guard-pro' ); ?></strong>
				<?php
				if ( $has_critical ) {
					esc_html_e( 'Critical security issues found in this content!', 'content-guard-pro' );
				} else {
					esc_html_e( 'Security issues detected in this content.', 'content-guard-pro' );
				}
				?>
			</p>
			
			<div style="margin: 10px 0; display: flex; gap: 10px; flex-wrap: wrap;">
				<?php if ( $critical_count > 0 ) : ?>
					<span style="background: #d63638; color: #fff; padding: 2px 8px; border-radius: 3px; font-size: 12px; font-weight: 600;">
						<?php
						/* translators: %d: Number of critical issues */
						printf( esc_html__( '%d Critical', 'content-guard-pro' ), absint( $critical_count ) );
						?>
					</span>
				<?php endif; ?>
				<?php if ( $suspicious_count > 0 ) : ?>
					<span style="background: #dba617; color: #1d2327; padding: 2px 8px; border-radius: 3px; font-size: 12px; font-weight: 600;">
						<?php
						/* translators: %d: Number of suspicious issues */
						printf( esc_html__( '%d Suspicious', 'content-guard-pro' ), absint( $suspicious_count ) );
						?>
					</span>
				<?php endif; ?>
				<?php if ( $review_count > 0 ) : ?>
					<span style="background: #72aee6; color: #1d2327; padding: 2px 8px; border-radius: 3px; font-size: 12px; font-weight: 600;">
						<?php
						/* translators: %d: Number of review issues */
						printf( esc_html__( '%d Review', 'content-guard-pro' ), absint( $review_count ) );
						?>
					</span>
				<?php endif; ?>
			</div>

			<?php if ( ! empty( $findings ) ) : ?>
				<div style="margin-bottom: 10px; max-height: 200px; overflow-y: auto; background: #fff; border: 1px solid #ddd; border-radius: 4px; padding: 10px;">
					<?php foreach ( array_slice( $findings, 0, 5 ) as $finding ) : ?>
						<div style="padding: 8px; margin-bottom: 8px; border-left: 3px solid <?php echo 'critical' === $finding['severity'] ? '#d63638' : ( 'suspicious' === $finding['severity'] ? '#dba617' : '#72aee6' ); ?>; background: #f9f9f9;">
							<div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 4px;">
								<span style="font-weight: 600; text-transform: uppercase; font-size: 11px; color: <?php echo 'critical' === $finding['severity'] ? '#d63638' : ( 'suspicious' === $finding['severity'] ? '#996800' : '#2271b1' ); ?>;">
									<?php echo esc_html( ucfirst( $finding['severity'] ) ); ?>
								</span>
								<span style="font-size: 11px; color: #757575;">
									<?php
									/* translators: %d: Confidence percentage */
									printf( esc_html__( '%d%% confidence', 'content-guard-pro' ), absint( $finding['confidence'] ) );
									?>
								</span>
							</div>
							<div style="font-size: 13px; color: #1d2327;">
								<?php echo esc_html( $finding['description'] ); ?>
							</div>
							<?php if ( ! empty( $finding['matched_excerpt'] ) ) : ?>
								<div style="font-size: 12px; color: #757575; font-family: monospace; margin-top: 4px; word-break: break-word;">
									<?php echo esc_html( $finding['matched_excerpt'] ); ?>
								</div>
							<?php endif; ?>
						</div>
					<?php endforeach; ?>
					
					<?php if ( $total_count > 5 ) : ?>
						<p style="margin: 0; text-align: center; color: #757575; font-size: 12px;">
							<?php
							/* translators: %d: Number of additional issues */
							printf( esc_html__( '... and %d more issue(s)', 'content-guard-pro' ), absint( $total_count - 5 ) );
							?>
						</p>
					<?php endif; ?>
				</div>
			<?php endif; ?>

			<p>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=' . self::PARENT_SLUG . '-findings&object_id=' . $post_id ) ); ?>" class="button button-primary">
					<?php esc_html_e( 'View All Findings', 'content-guard-pro' ); ?>
				</a>
				<span style="margin-left: 10px; color: #757575; font-size: 12px;">
					<?php esc_html_e( 'The post has been saved. You can continue editing or review the findings.', 'content-guard-pro' ); ?>
				</span>
			</p>
		</div>
		<?php
	}

	/**
	 * Check if any scans have been run.
	 *
	 * Used to determine if the welcome notice should be shown.
	 * Once a scan has been initiated, the welcome notice is no longer relevant.
	 *
	 * @since 1.0.0
	 * @return bool True if any scans exist in the database.
	 */
	private function has_run_any_scan() {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		
        // Check if any scan records exist.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $count = $wpdb->get_var(
            "SELECT COUNT(*) FROM `{$table_name}`"
        );
		
		return absint( $count ) > 0;
	}

	/**
	 * Get critical findings count.
	 *
	 * Queries the content_guard_pro_findings table for open critical findings.
	 *
	 * @since 1.0.0
	 * @return int Number of critical findings.
	 */
	private function get_critical_findings_count() {
		if ( null !== $this->critical_count_cache ) {
			return $this->critical_count_cache;
		}

		global $wpdb;
		
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';
		
		// Query for open critical findings.
		$count = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$table_name}` WHERE status = %s AND severity = %s",
				'open',
				'critical'
			)
		);
		
		$this->critical_count_cache = absint( $count );
		return $this->critical_count_cache;
	}

	/**
	 * Add menu to admin bar.
	 *
	 * Per PRD Section 3.5 - Admin bar badge with count.
	 *
	 * @since 1.0.0
	 * @param WP_Admin_Bar $wp_admin_bar Admin bar object.
	 */
	public function add_admin_bar_menu( $wp_admin_bar ) {
		// Check user capability.
		if ( ! current_user_can( self::VIEW_CAPABILITY ) ) {
			return;
		}

		// Check if admin bar badge is enabled in settings.
		$settings = get_option( 'content_guard_pro_settings', array() );
		if ( empty( $settings['notifications_admin'] ) ) {
			return;
		}

		// Get critical findings count for badge.
		$critical_count = $this->get_critical_findings_count();
		
		// Build title with optional badge.
		// Use ab-icon class for proper admin bar positioning (icon set via CSS).
		$title = '<span class="ab-icon"></span>' . __( 'Content Guard', 'content-guard-pro' );
		
		if ( $critical_count > 0 ) {
			// Add badge with count per PRD Section 3.5.
			$title .= sprintf(
				' <span class="content-guard-pro-admin-bar-badge" style="background: #d63638; color: #fff; border-radius: 10px; padding: 2px 6px; font-size: 11px; font-weight: 600; margin-left: 5px;">%d</span>',
				absint( $critical_count )
			);
		}

		// Add parent menu.
		$wp_admin_bar->add_node(
			array(
				'id'    => 'content-guard-pro-admin-bar',
				'title' => $title,
				'href'  => admin_url( 'admin.php?page=' . self::PARENT_SLUG ),
				'meta'  => array(
					'title' => __( 'Content Guard Pro', 'content-guard-pro' ),
				),
			)
		);

		// Add findings submenu with count.
		$findings_title = __( 'View Findings', 'content-guard-pro' );
		if ( $critical_count > 0 ) {
			$findings_title .= sprintf( ' <span style="color: #d63638;">(%d critical)</span>', absint( $critical_count ) );
		}
		
		$wp_admin_bar->add_node(
			array(
				'id'     => 'content-guard-pro-admin-bar-findings',
				'parent' => 'content-guard-pro-admin-bar',
				'title'  => $findings_title,
				'href'   => admin_url( 'admin.php?page=' . self::PARENT_SLUG . '-findings' ),
			)
		);

		// Add scan submenu.
		$wp_admin_bar->add_node(
			array(
				'id'     => 'content-guard-pro-admin-bar-scan',
				'parent' => 'content-guard-pro-admin-bar',
				'title'  => __( 'Run Scan', 'content-guard-pro' ),
				'href'   => admin_url( 'admin.php?page=' . self::PARENT_SLUG . '-scans' ),
			)
		);
	}

	/**
	 * AJAX handler for dismissing welcome notice.
	 *
	 * Stores dismissal state in user meta so the welcome notice doesn't reappear.
	 *
	 * @since 1.0.0
	 */
	public function ajax_dismiss_welcome_notice() {
		// Verify nonce with proper sanitization.
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'content-guard-pro-dismiss-notice' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'content-guard-pro' ) ) );
		}
		
		// Check user capability.
		if ( ! current_user_can( self::VIEW_CAPABILITY ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'content-guard-pro' ) ) );
		}
		
		// Store dismissal in user meta.
		update_user_meta( get_current_user_id(), 'content_guard_pro_dismissed_welcome_notice', true );
		
		wp_send_json_success( array( 'message' => __( 'Notice dismissed.', 'content-guard-pro' ) ) );
	}
	
	/**
	 * AJAX handler for dismissing critical notice.
	 *
	 * Stores dismissal state in user meta so the notice doesn't reappear
	 * until new critical findings are detected.
	 *
	 * @since 1.0.0
	 */
	public function ajax_dismiss_critical_notice() {
		// Verify nonce with proper sanitization.
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'content-guard-pro-dismiss-notice' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'content-guard-pro' ) ) );
		}
		
		// Check user capability.
		if ( ! current_user_can( self::VIEW_CAPABILITY ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'content-guard-pro' ) ) );
		}
		
		// Get the count at time of dismissal with proper sanitization.
		$count = isset( $_POST['count'] ) ? absint( wp_unslash( $_POST['count'] ) ) : 0;
		
		// Store dismissal in user meta.
		update_user_meta( get_current_user_id(), 'content_guard_pro_dismissed_critical_notice', true );
		update_user_meta( get_current_user_id(), 'content_guard_pro_last_dismissed_critical_count', $count );
		
		wp_send_json_success( array( 'message' => __( 'Notice dismissed.', 'content-guard-pro' ) ) );
	}
	
	/**
	 * AJAX handler for dismissing scan notice.
	 *
	 * Dismisses the on-save scan result notice.
	 *
	 * @since 1.0.0
	 */
	public function ajax_dismiss_scan_notice() {
		// Verify nonce with proper sanitization.
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'content-guard-pro-dismiss-notice' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'content-guard-pro' ) ) );
		}
		
		// Check user capability.
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'content-guard-pro' ) ) );
		}
		
		// Get the post ID with proper sanitization.
		$post_id = isset( $_POST['post_id'] ) ? absint( wp_unslash( $_POST['post_id'] ) ) : 0;
		
		if ( $post_id ) {
			// Delete the scan result transient.
			$transient_key = 'cgp_scan_result_' . $post_id . '_' . get_current_user_id();
			delete_transient( $transient_key );
		}
		
		wp_send_json_success( array( 'message' => __( 'Notice dismissed.', 'content-guard-pro' ) ) );
	}

	/**
	 * AJAX handler for getting finding details.
	 *
	 * Returns detailed information about a specific finding for the modal display.
	 * Per PRD Section 3.3 - Finding detail view.
	 *
	 * @since 1.0.0
	 */
	public function ajax_get_finding_details() {
		// Verify nonce.
		check_ajax_referer( 'content-guard-pro-admin-nonce', 'nonce' );
		
		// Check user capability.
		if ( ! current_user_can( self::VIEW_CAPABILITY ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'content-guard-pro' ) ) );
		}
		
		// Get finding ID from POST with proper sanitization.
		$finding_id = isset( $_POST['finding_id'] ) ? absint( wp_unslash( $_POST['finding_id'] ) ) : 0;
		
		if ( ! $finding_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid finding ID.', 'content-guard-pro' ) ) );
		}
		
		// Query the content_guard_pro_findings table for the finding.
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';
		
		$finding = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$table_name}` WHERE id = %d",
				$finding_id
			)
		);
		
		if ( ! $finding ) {
			wp_send_json_error( array( 'message' => __( 'Finding not found.', 'content-guard-pro' ) ) );
		}
		
		// Parse extra data if it's JSON.
		$extra = array();
		if ( ! empty( $finding->extra ) ) {
			$decoded = json_decode( $finding->extra, true );
			if ( is_array( $decoded ) ) {
				$extra = $decoded;
			}
		}
		
		// Build finding data array for response.
		$finding_data = array(
			'id'              => absint( $finding->id ),
			'object_type'     => esc_html( $finding->object_type ),
			'object_id'       => absint( $finding->object_id ),
			'field'           => esc_html( $finding->field ),
			'fingerprint'     => esc_html( $finding->fingerprint ),
			'rule_id'         => esc_html( $finding->rule_id ),
			'severity'        => esc_html( $finding->severity ),
			'confidence'      => absint( $finding->confidence ),
			'matched_excerpt' => esc_html( $finding->matched_excerpt ),
			'first_seen'      => esc_html( $finding->first_seen ),
			'last_seen'       => esc_html( $finding->last_seen ),
			'status'          => esc_html( $finding->status ),
			'extra'           => $extra,
		);
		
		// Add object context (post title, etc.).
		if ( 'post' === $finding->object_type ) {
			$post = get_post( $finding->object_id );
			if ( $post ) {
				$finding_data['object_title'] = get_the_title( $post );
				$finding_data['object_url'] = get_edit_post_link( $post->ID, 'raw' );
				$finding_data['object_status'] = $post->post_status;
			}
		}
		
		wp_send_json_success( $finding_data );
	}

	/**
	 * AJAX handler for ignoring a finding.
	 *
	 * Updates the finding status to 'ignored'.
	 * Optionally reports as false positive to vendor.
	 *
	 * @since 1.0.0
	 */
	public function ajax_ignore_finding() {
		// Verify nonce.
		check_ajax_referer( 'content-guard-pro-admin-nonce', 'nonce' );
		
		// Check user capability.
		if ( ! current_user_can( self::VIEW_CAPABILITY ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'content-guard-pro' ) ) );
		}
		
		// Get finding ID.
		$finding_id = isset( $_POST['finding_id'] ) ? absint( wp_unslash( $_POST['finding_id'] ) ) : 0;
		
		if ( ! $finding_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid finding ID.', 'content-guard-pro' ) ) );
		}

		// Check if false positive reporting is requested.
		$report_fp = isset( $_POST['report_false_positive'] ) && 'true' === $_POST['report_false_positive'];

		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		// Update status to 'ignored'.
		$updated = $wpdb->update(
			$table_name,
			array( 'status' => 'ignored' ),
			array( 'id' => $finding_id ),
			array( '%s' ),
			array( '%d' )
		);

		if ( false === $updated ) {
			wp_send_json_error( array( 'message' => __( 'Failed to update finding status.', 'content-guard-pro' ) ) );
		}

		// Report false positive if requested.
		if ( $report_fp ) {
			$finding = $wpdb->get_row(
				$wpdb->prepare( "SELECT * FROM `{$table_name}` WHERE id = %d", $finding_id ),
				ARRAY_A
			);
			
			if ( $finding ) {
				CGP_Integrations::report_false_positive( $finding );
			}
		}

		wp_send_json_success( array( 'message' => __( 'Finding ignored.', 'content-guard-pro' ) ) );
	}

	/**
	 * AJAX handler for quarantining a finding.
	 *
	 * Updates the finding status to 'quarantined'.
	 *
	 * @since 1.0.0
	 */
	public function ajax_quarantine_finding() {
		// Verify nonce.
		check_ajax_referer( 'content-guard-pro-admin-nonce', 'nonce' );
		
		// Check user capability.
		if ( ! current_user_can( self::VIEW_CAPABILITY ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'content-guard-pro' ) ) );
		}
		
		// Get finding ID.
		$finding_id = isset( $_POST['finding_id'] ) ? absint( wp_unslash( $_POST['finding_id'] ) ) : 0;
		
		if ( ! $finding_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid finding ID.', 'content-guard-pro' ) ) );
		}

		// Quarantine finding.
		if ( class_exists( 'CGP_Quarantine' ) ) {
			if ( CGP_Quarantine::quarantine_finding( $finding_id ) ) {
				wp_send_json_success( array( 'message' => __( 'Finding quarantined successfully.', 'content-guard-pro' ) ) );
			} else {
				wp_send_json_error( array( 'message' => __( 'Failed to quarantine finding.', 'content-guard-pro' ) ) );
			}
		} else {
			wp_send_json_error( array( 'message' => __( 'Quarantine class not found.', 'content-guard-pro' ) ) );
		}
	}

	/**
	 * AJAX handler for un-quarantining a finding.
	 *
	 * Updates the finding status to 'open'.
	 *
	 * @since 1.0.0
	 */
	public function ajax_unquarantine_finding() {
		// Verify nonce.
		check_ajax_referer( 'content-guard-pro-admin-nonce', 'nonce' );
		
		// Check user capability.
		if ( ! current_user_can( self::VIEW_CAPABILITY ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'content-guard-pro' ) ) );
		}
		
		// Get finding ID.
		$finding_id = isset( $_POST['finding_id'] ) ? absint( wp_unslash( $_POST['finding_id'] ) ) : 0;
		
		if ( ! $finding_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid finding ID.', 'content-guard-pro' ) ) );
		}

		// Un-quarantine finding.
		if ( class_exists( 'CGP_Quarantine' ) ) {
			if ( CGP_Quarantine::unquarantine_finding( $finding_id ) ) {
				wp_send_json_success( array( 'message' => __( 'Finding un-quarantined successfully.', 'content-guard-pro' ) ) );
			} else {
				wp_send_json_error( array( 'message' => __( 'Failed to un-quarantine finding.', 'content-guard-pro' ) ) );
			}
		} else {
			wp_send_json_error( array( 'message' => __( 'Quarantine class not found.', 'content-guard-pro' ) ) );
		}
	}
}

