<?php
/**
 * Plugin Name:       Content Guard Pro
 * Plugin URI:        https://wordpress.org/plugins/content-guard-pro
 * Description:       Advanced WordPress security plugin that detects and quarantines malicious content, spam links, and SEO injections stored in the WordPress database.
 * Version:           1.0.2
 * Requires at least: 6.1
 * Requires PHP:      8.0
 * Author:            Content Guard Pro Team
 * Author URI:        https://contentguardpro.com
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       content-guard-pro
 *
 * @package ContentGuardPro
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Current plugin version.
 * Start at version 1.0.0 and use SemVer - https://semver.org
 */
define( 'CONTENT_GUARD_PRO_VERSION', '1.0.2' );

/**
 * Plugin base name.
 */
define( 'CONTENT_GUARD_PRO_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Plugin root directory path.
 */
define( 'CONTENT_GUARD_PRO_PATH', plugin_dir_path( __FILE__ ) );

/**
 * Plugin root directory URL.
 */
define( 'CONTENT_GUARD_PRO_URL', plugin_dir_url( __FILE__ ) );

/**
 * Minimum PHP version required.
 */
define( 'CONTENT_GUARD_PRO_MIN_PHP', '8.0' );

/**
 * Minimum WordPress version required.
 */
define( 'CONTENT_GUARD_PRO_MIN_WP', '6.1' );

/**
 * No-op logger helper to keep debug hooks without writing to PHP error log.
 *
 * @since 1.0.0
 * @param string $message Message to log.
 * @param string $level   Log level (info|warning|error), defaults to info.
 */
if ( ! function_exists( 'cgp_log' ) ) {
	function cgp_log( $message, $level = 'info' ) {
		/**
		 * Hook for custom logging handlers implemented by site owners.
		 *
		 * @param string $message The log message.
		 * @param string $level   Log level indicator.
		 */
		do_action( 'content_guard_pro_log', $message, $level );
	}
}

/**
 * Main Content Guard Pro plugin class.
 *
 * This class serves as the main entry point and orchestrator for the plugin.
 * It handles initialization, dependency loading, and hook registration.
 *
 * @since 1.0.0
 */
final class Content_Guard_Pro {

	/**
	 * The single instance of the class.
	 *
	 * @var Content_Guard_Pro|null
	 */
	private static $instance = null;

	/**
	 * Plugin version.
	 *
	 * @var string
	 */
	public $version = CONTENT_GUARD_PRO_VERSION;

	/**
	 * Main Content_Guard_Pro instance.
	 *
	 * Ensures only one instance of Content_Guard_Pro is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 * @static
	 * @return Content_Guard_Pro - Main instance.
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Content_Guard_Pro Constructor.
	 *
	 * @since 1.0.0
	 */
	private function __construct() {
		$this->load_action_scheduler();
		$this->define_constants();
		$this->check_requirements();
		$this->init_hooks();
	}

	/**
	 * Load Action Scheduler library.
	 *
	 * Action Scheduler is required for background job processing.
	 * This method loads the library if it's not already available.
	 *
	 * @since 1.0.0
	 */
	private function load_action_scheduler() {
		// Check if Action Scheduler is already loaded (by WooCommerce or another plugin).
		if ( class_exists( 'ActionScheduler' ) ) {
			return;
		}

		// Load bundled Action Scheduler if available.
		$action_scheduler_path = CONTENT_GUARD_PRO_PATH . 'vendor/action-scheduler/action-scheduler.php';
		
		if ( file_exists( $action_scheduler_path ) ) {
			require_once $action_scheduler_path;
		} else {
			// Log warning if Action Scheduler is not available.
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				cgp_log( 'Content Guard Pro: Action Scheduler library not found. Please install WooCommerce or bundle Action Scheduler in vendor/action-scheduler/' );
			}
		}
	}

	/**
	 * Define additional plugin constants if needed.
	 *
	 * @since 1.0.0
	 */
	private function define_constants() {
		// Define additional constants here if needed in future.
		if ( ! defined( 'CONTENT_GUARD_PRO_DEBUG' ) ) {
			define( 'CONTENT_GUARD_PRO_DEBUG', false );
		}
	}

	/**
	 * Check if system requirements are met.
	 *
	 * @since 1.0.0
	 */
	private function check_requirements() {
		// Requirements are checked in activation hook, but we can add runtime checks here.
		// For now, we assume activation hook passed.
	}

	/**
	 * Hook into actions and filters.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		// Initialize plugin on plugins_loaded hook.
		add_action( 'plugins_loaded', array( $this, 'run' ), 10 );

		// Load plugin text domain for translations.
		add_action( 'init', array( $this, 'load_textdomain' ) );

		// Add action links to plugin page.
		add_filter( 'plugin_action_links_' . CONTENT_GUARD_PRO_BASENAME, array( $this, 'plugin_action_links' ) );

		// Register activation with API server (for free tier tracking).
		add_action( 'content_guard_pro_register_activation', array( $this, 'register_activation_with_api' ) );
	}

	/**
	 * Initialize and run the plugin.
	 *
	 * This method is called after all plugins are loaded and handles:
	 * - Loading required files
	 * - Initializing core components
	 * - Registering admin and frontend hooks
	 *
	 * @since 1.0.0
	 */
	public function run() {
		// Verify requirements at runtime.
		if ( ! $this->meets_requirements() ) {
			return;
		}

		// Load core dependencies.
		$this->load_dependencies();

		// Initialize core components.
		$this->init_components();

		/**
		 * Fires after Content Guard Pro has been fully initialized.
		 *
		 * @since 1.0.0
		 *
		 * @param Content_Guard_Pro $this The main plugin instance.
		 */
		do_action( 'content_guard_pro_loaded', $this );
	}

	/**
	 * Check if the system meets minimum requirements.
	 *
	 * @since 1.0.0
	 * @return bool True if requirements are met, false otherwise.
	 */
	private function meets_requirements() {
		// Check PHP version.
		if ( version_compare( PHP_VERSION, CONTENT_GUARD_PRO_MIN_PHP, '<' ) ) {
			add_action( 'admin_notices', array( $this, 'php_version_notice' ) );
			return false;
		}

		// Check WordPress version.
		global $wp_version;
		if ( version_compare( $wp_version, CONTENT_GUARD_PRO_MIN_WP, '<' ) ) {
			add_action( 'admin_notices', array( $this, 'wp_version_notice' ) );
			return false;
		}

		return true;
	}

	/**
	 * Load plugin dependencies.
	 *
	 * Include required core files used in admin and on the frontend.
	 *
	 * @since 1.0.0
	 */
	private function load_dependencies() {
		// Load core classes.
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-database.php';
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-integrations.php';
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-detector.php';
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-admin.php';
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-scheduler.php';
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-scanner.php';
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-quarantine.php';
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-realtime-scanner.php';
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-notifications.php';
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-dashboard-widget.php';
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-rest-api.php';
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-api-client.php';
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-license-manager.php';
	}

	/**
	 * Initialize core plugin components.
	 *
	 * @since 1.0.0
	 */
	private function init_components() {
		// Initialize admin interface.
		if ( is_admin() ) {
			CGP_Admin::init();
			
			// Initialize setup wizard hooks (redirect and form processing).
			// The wizard is displayed within the Dashboard page via ?setup=1 parameter.
			$wizard_file = CONTENT_GUARD_PRO_PATH . 'admin/pages/class-cgp-admin-setup-wizard.php';
			if ( file_exists( $wizard_file ) ) {
				require_once $wizard_file;
				CGP_Admin_Setup_Wizard::init();
			}
			
			// Initialize dashboard widget (WordPress dashboard).
			CGP_Dashboard_Widget::init();
		}
		
		// Initialize scheduler (runs on all contexts).
		CGP_Scheduler::init();
		
		// Initialize scanner (runs on all contexts).
		CGP_Scanner::init();
		
		// Initialize quarantine system (runs on frontend and admin).
		CGP_Quarantine::init();
		
		// Initialize real-time scanner (runs on all contexts).
		CGP_Realtime_Scanner::init();
		
		// Initialize notifications system (alerts and webhooks).
		CGP_Notifications::init();
		
		// Initialize REST API (runs on all contexts).
		CGP_REST_API::init();

		// Register cleanup hook for data retention.
		add_action( 'content_guard_pro_cleanup_old_data', array( 'CGP_Database', 'run_cleanup' ) );
	}

	/**
	 * Load plugin text domain for translations.
	 *
	 * @since 1.0.0
	 */
	public function load_textdomain() {
		// No-op: WordPress.org will load translations automatically for the plugin slug.
	}

	/**
	 * Add action links to the plugin page.
	 *
	 * @since 1.0.0
	 * @param array $links Existing plugin action links.
	 * @return array Modified plugin action links.
	 */
	public function plugin_action_links( $links ) {
		$custom_links = array(
			'settings' => sprintf(
				'<a href="%s">%s</a>',
				esc_url( admin_url( 'admin.php?page=content-guard-pro-settings' ) ),
				esc_html__( 'Settings', 'content-guard-pro' )
			),
			'scan'     => sprintf(
				'<a href="%s">%s</a>',
				esc_url( admin_url( 'admin.php?page=content-guard-pro-scan' ) ),
				esc_html__( 'Scan', 'content-guard-pro' )
			),
		);

		return array_merge( $custom_links, $links );
	}

	/**
	 * Display PHP version requirement notice.
	 *
	 * @since 1.0.0
	 */
	public function php_version_notice() {
		?>
		<div class="notice notice-error">
			<p>
				<?php
				printf(
					/* translators: 1: Required PHP version, 2: Current PHP version */
					esc_html__( 'Content Guard Pro requires PHP version %1$s or higher. You are running version %2$s. Please upgrade PHP to use this plugin.', 'content-guard-pro' ),
					esc_html( CONTENT_GUARD_PRO_MIN_PHP ),
					esc_html( PHP_VERSION )
				);
				?>
			</p>
		</div>
		<?php
	}

	/**
	 * Display WordPress version requirement notice.
	 *
	 * @since 1.0.0
	 */
	public function wp_version_notice() {
		?>
		<div class="notice notice-error">
			<p>
				<?php
				global $wp_version;
				printf(
					/* translators: 1: Required WordPress version, 2: Current WordPress version */
					esc_html__( 'Content Guard Pro requires WordPress version %1$s or higher. You are running version %2$s. Please upgrade WordPress to use this plugin.', 'content-guard-pro' ),
					esc_html( CONTENT_GUARD_PRO_MIN_WP ),
					esc_html( $wp_version )
				);
				?>
			</p>
		</div>
		<?php
	}

	/**
	 * Prevent cloning of the instance.
	 *
	 * @since 1.0.0
	 */
	private function __clone() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Cloning is forbidden.', 'content-guard-pro' ), '1.0.0' );
	}

	/**
	 * Prevent unserializing of the instance.
	 *
	 * @since 1.0.0
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Unserializing instances of this class is forbidden.', 'content-guard-pro' ), '1.0.0' );
	}

	/**
	 * Register plugin activation with API server.
	 *
	 * Called asynchronously after plugin activation to register free tier
	 * installations for tracking and statistics. Only registers if no license
	 * key is present (free tier users).
	 *
	 * @since 1.0.0
	 */
	public function register_activation_with_api() {
		// Only register if no license key is present (free tier).
		$settings = get_option( 'content_guard_pro_settings', array() );
		$license_key = isset( $settings['license_key'] ) ? trim( $settings['license_key'] ) : '';

		if ( ! empty( $license_key ) ) {
			// Paid user - activation will be handled via license validation.
			return;
		}

		// Register free tier activation.
		require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-api-client.php';
		$result = CGP_API_Client::register_free_tier_activation();

		// Log result (non-blocking, failures are acceptable).
		if ( is_wp_error( $result ) ) {
			cgp_log( sprintf( 'Failed to register free tier activation: %s', $result->get_error_message() ), 'warning' );
		} else {
			cgp_log( 'Free tier activation registered successfully', 'info' );
		}
	}

	/**
	 * Clear plugin-specific cache entries.
	 *
	 * Clears only cache entries created by this plugin, avoiding
	 * aggressive cache flushing that affects other plugins and caching systems.
	 *
	 * @since 1.0.0
	 */
	public static function clear_plugin_cache() {
		// Clear dashboard widget cache.
		wp_cache_delete( 'content_guard_pro_dashboard_last_scan' );
		wp_cache_delete( 'content_guard_pro_dashboard_counts' );

		// Clear notifications cache.
		wp_cache_delete( 'content_guard_pro_critical_count' );

		// Note: Action Scheduler manages its own cache and doesn't need clearing here.
		// It uses wp_cache_flush_runtime() during queue processing when appropriate.
	}
}

/**
 * Plugin activation hook callback.
 *
 * Runs when the plugin is activated. Handles:
 * - System requirement checks
 * - Database table creation
 * - Default option setup
 * - Version tracking
 *
 * @since 1.0.0
 */
function content_guard_pro_activate() {
	// Check PHP version.
	if ( version_compare( PHP_VERSION, CONTENT_GUARD_PRO_MIN_PHP, '<' ) ) {
		deactivate_plugins( CONTENT_GUARD_PRO_BASENAME );
		wp_die(
			sprintf(
				/* translators: 1: Required PHP version, 2: Current PHP version */
				esc_html__( 'Content Guard Pro requires PHP version %1$s or higher. You are running version %2$s. Please upgrade PHP before activating this plugin.', 'content-guard-pro' ),
				esc_html( CONTENT_GUARD_PRO_MIN_PHP ),
				esc_html( PHP_VERSION )
			),
			esc_html__( 'Plugin Activation Error', 'content-guard-pro' ),
			array( 'back_link' => true )
		);
	}

	// Check WordPress version.
	global $wp_version;
	if ( version_compare( $wp_version, CONTENT_GUARD_PRO_MIN_WP, '<' ) ) {
		deactivate_plugins( CONTENT_GUARD_PRO_BASENAME );
		wp_die(
			sprintf(
				/* translators: 1: Required WordPress version, 2: Current WordPress version */
				esc_html__( 'Content Guard Pro requires WordPress version %1$s or higher. You are running version %2$s. Please upgrade WordPress before activating this plugin.', 'content-guard-pro' ),
				esc_html( CONTENT_GUARD_PRO_MIN_WP ),
				esc_html( $wp_version )
			),
			esc_html__( 'Plugin Activation Error', 'content-guard-pro' ),
			array( 'back_link' => true )
		);
	}

	// Set activation flag for setup wizard redirect.
	// The transient expires after 5 minutes to handle various activation methods
	// (single activation, bulk activation, AJAX activation, etc.).
	set_transient( 'content_guard_pro_activation_redirect', true, 300 );

	// Store plugin version.
	update_option( 'content_guard_pro_version', CONTENT_GUARD_PRO_VERSION );

	// Set default options if not exist.
	if ( false === get_option( 'content_guard_pro_settings' ) ) {
		$default_settings = array(
			'scan_mode'                      => 'standard',
			'batch_size'                     => 100,
			'batch_delay'                    => 2,
			'schedule_enabled'               => false,
			'schedule_time'                  => '03:00',
			'safe_mode'                      => 'auto',
			'retention_days'                 => 90,
			'email_alerts_enabled'           => true,
			'email_recipients'               => get_option( 'admin_email' ),
			'email_alert_mode'               => 'immediate',
			'webhook_enabled'                => false,
			'webhook_url'                    => '',
			'webhook_secret'                 => '',
			'allowlist_domains'              => '',
			'denylist_patterns'              => '',
			'notifications_admin'            => true,
		);
		add_option( 'content_guard_pro_settings', $default_settings );
	}

	// Schedule daily digest cron job (for email notifications).
	if ( ! wp_next_scheduled( 'content_guard_pro_daily_digest' ) ) {
		wp_schedule_event( time(), 'daily', 'content_guard_pro_daily_digest' );
	}

	// Schedule daily cleanup job for data retention.
	if ( ! wp_next_scheduled( 'content_guard_pro_cleanup_old_data' ) ) {
		wp_schedule_event( time(), 'daily', 'content_guard_pro_cleanup_old_data' );
	}

	// Flag first activation for setup wizard tracking.
	// This permanent option is used to track whether the plugin has ever been activated before.
	// Unlike the transient above, this persists and can be used to show first-time user features,
	// onboarding hints, or to prevent showing the setup wizard on subsequent activations.
	if ( false === get_option( 'content_guard_pro_first_activation' ) ) {
		add_option( 'content_guard_pro_first_activation', true );
	}

	// Load database class.
	require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-database.php';

	// Create database tables.
	CGP_Database::activate();

	// Register free tier activation with API server (non-blocking).
	// This allows tracking of free tier installations for statistics and messaging.
	// Load API client class for activation registration.
	require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-api-client.php';
	
	// Schedule activation registration to run asynchronously to avoid blocking activation.
	// Use a transient to ensure it only runs once per activation.
	$activation_registered = get_transient( 'content_guard_pro_activation_registered' );
	if ( false === $activation_registered ) {
		// Schedule async registration (runs on next page load or via wp-cron).
		wp_schedule_single_event( time() + 5, 'content_guard_pro_register_activation' );
		set_transient( 'content_guard_pro_activation_registered', true, HOUR_IN_SECONDS );
	}

	// Clear plugin-specific cache (if any exists from previous activation).
	// Note: No cache exists yet on first activation, but this ensures clean state
	// if plugin is reactivated. We avoid wp_cache_flush() to prevent affecting
	// other plugins and caching systems like LiteSpeed Cache.
	Content_Guard_Pro::clear_plugin_cache();

	/**
	 * Fires after plugin activation is complete.
	 *
	 * @since 1.0.0
	 */
	do_action( 'content_guard_pro_activated' );
}

/**
 * Plugin deactivation hook callback.
 *
 * Runs when the plugin is deactivated. Handles:
 * - Clearing scheduled events
 * - Cleaning up transients
 * - Optionally preserving user data
 *
 * @since 1.0.0
 */
function content_guard_pro_deactivate() {
	// Clear daily digest cron job (for email notifications).
	$digest_timestamp = wp_next_scheduled( 'content_guard_pro_daily_digest' );
	if ( $digest_timestamp ) {
		wp_unschedule_event( $digest_timestamp, 'content_guard_pro_daily_digest' );
	}

	// Clear license check cron job.
	wp_clear_scheduled_hook( 'content_guard_pro_daily_license_check' );

	// Clear activation registration hook.
	wp_clear_scheduled_hook( 'content_guard_pro_register_activation' );

	// Clear any other WP-Cron hooks the plugin registers.
	wp_clear_scheduled_hook( 'content_guard_pro_daily_scan' );
	wp_clear_scheduled_hook( 'content_guard_pro_scheduled_scan' );
	wp_clear_scheduled_hook( 'content_guard_pro_cleanup_old_data' );

	// Clear Action Scheduler queue (if loaded).
	if ( class_exists( 'ActionScheduler' ) && function_exists( 'as_unschedule_all_actions' ) ) {
		// Cancel all grouped actions created by this plugin.
		as_unschedule_all_actions( '', array(), 'content-guard-pro' );
	}

	// Clear transients.
	delete_transient( 'content_guard_pro_activation_redirect' );
	delete_transient( 'content_guard_pro_scan_progress' );

	// Clear plugin-specific cache entries.
	// We use targeted cache clearing instead of wp_cache_flush() to avoid
	// affecting other plugins and caching systems like LiteSpeed Cache.
	Content_Guard_Pro::clear_plugin_cache();

	/**
	 * Fires after plugin deactivation is complete.
	 *
	 * @since 1.0.0
	 */
	do_action( 'content_guard_pro_deactivated' );
}

// Register activation and deactivation hooks.
register_activation_hook( __FILE__, 'content_guard_pro_activate' );
register_deactivation_hook( __FILE__, 'content_guard_pro_deactivate' );

/**
 * Initialize the plugin.
 *
 * @since 1.0.0
 * @return Content_Guard_Pro The main plugin instance.
 */
function content_guard_pro() {
	return Content_Guard_Pro::instance();
}

// Get the plugin running.
content_guard_pro();

