/**
 * Scan Progress - Real-time AJAX Updates
 *
 * Handles real-time updates for active scans and scan history table.
 * Polls the REST API every 3 seconds to update progress information.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

(function( $ ) {
	'use strict';

	// Check if contentGuardProScanProgress is defined
	if ( typeof window.contentGuardProScanProgress === 'undefined' ) {
		console.error( 'CGP Scan Progress: contentGuardProScanProgress object is not defined. Script localization may have failed.' );
		return;
	}

	const contentGuardProScanProgress = window.contentGuardProScanProgress;

	// Configuration
	const POLL_INTERVAL_ACTIVE = 3000; // 3 seconds when scan is active
	const POLL_INTERVAL_IDLE = 10000;  // 10 seconds when no active scan
	const API_BASE = contentGuardProScanProgress.apiUrl;
	const API_NONCE = contentGuardProScanProgress.nonce;

	// State
	let pollTimer = null;
	let currentPollInterval = POLL_INTERVAL_ACTIVE;
	let hasActiveScans = false;
	let isPageVisible = true;

	/**
	 * Initialize the scan progress poller.
	 *
	 * @since 1.0.0
	 */
	function init() {
		// Check if we're on the scans page
		if ( ! $( '#content-guard-pro-active-scans-container' ).length ) {
			return;
		}

		// Start polling immediately
		updateActiveScanWidget();
		startPolling();

		// Handle visibility change (pause when tab is hidden)
		document.addEventListener( 'visibilitychange', handleVisibilityChange );
	}

	/**
	 * Start polling for updates.
	 *
	 * @since 1.0.0
	 */
	function startPolling() {
		// Clear any existing timer
		if ( pollTimer ) {
			clearInterval( pollTimer );
		}

		// Start new timer
		pollTimer = setInterval( function() {
			updateActiveScanWidget();
			updateScanHistoryRows();
		}, currentPollInterval );
	}

	/**
	 * Stop polling for updates.
	 *
	 * @since 1.0.0
	 */
	function stopPolling() {
		if ( pollTimer ) {
			clearInterval( pollTimer );
			pollTimer = null;
		}
	}

	/**
	 * Handle page visibility change.
	 *
	 * Slows down polling when tab is not visible.
	 *
	 * @since 1.0.0
	 */
	function handleVisibilityChange() {
		if ( document.hidden ) {
			// Tab is hidden - slow down polling
			isPageVisible = false;
			currentPollInterval = POLL_INTERVAL_IDLE;
			stopPolling();
			startPolling();
		} else {
			// Tab is visible - speed up polling
			isPageVisible = true;
			currentPollInterval = hasActiveScans ? POLL_INTERVAL_ACTIVE : POLL_INTERVAL_IDLE;
			stopPolling();
			startPolling();

			// Immediate update when tab becomes visible
			updateActiveScanWidget();
			updateScanHistoryRows();
		}
	}

	/**
	 * Update the Active Scans widget.
	 *
	 * Fetches active scan data from REST API and updates the widget.
	 *
	 * @since 1.0.0
	 */
	function updateActiveScanWidget() {
		const $container = $( '#content-guard-pro-active-scans-container' );
		if ( ! $container.length ) {
			return;
		}

		$.ajax({
			url: API_BASE + '/scans/active',
			method: 'GET',
			beforeSend: function( xhr ) {
				xhr.setRequestHeader( 'X-WP-Nonce', API_NONCE );
			},
			success: function( response ) {
				hasActiveScans = response.has_active;

				// Adjust polling interval based on active scans
				const newInterval = ( hasActiveScans && isPageVisible ) ? POLL_INTERVAL_ACTIVE : POLL_INTERVAL_IDLE;
				if ( newInterval !== currentPollInterval ) {
					currentPollInterval = newInterval;
					stopPolling();
					startPolling();
				}

				// Check if we have any scans to display (active or recently completed)
				if ( response.active_scans.length > 0 ) {
					// Separate truly active from completed scans
					const activeScans = response.active_scans.filter( function( scan ) {
						return scan.status === 'pending' || scan.status === 'running';
					});

					// Render active scans widget (only show truly active scans)
					if ( activeScans.length > 0 ) {
						renderActiveScans( activeScans );
						
						// Hide "Scan starting..." notice if it exists
						const $startingNotice = $( '#cgp-scan-starting-notice' );
						if ( $startingNotice.length ) {
							$startingNotice.fadeOut( 300, function() {
								$( this ).remove();
							});
						}
					} else {
						renderNoActiveScans();
					}

					// Update table with all scans (including recently completed ones for final update)
					updateScanHistoryRowsWithData( response.active_scans );
				} else {
					renderNoActiveScans();
				}
			},
			error: function( xhr, status, error ) {
				// Show error in widget
				$container.html(
					'<div class="notice notice-error inline"><p>' +
					contentGuardProScanProgress.i18n.errorLoading +
					'</p></div>'
				);
			}
		});
	}

	/**
	 * Render active scans in the widget.
	 *
	 * @since 1.0.0
	 * @param {Array} scans Array of scan objects.
	 */
	function renderActiveScans( scans ) {
		const $container = $( '#content-guard-pro-active-scans-container' );
		const template = $( '#content-guard-pro-active-scan-template' ).html();

		if ( ! template ) {
			return;
		}

		// Clear container
		$container.empty();

		// Render each scan
		$.each( scans, function( index, scan ) {
			const html = renderScanFromTemplate( template, scan );
			$container.append( html );
		});
	}

	/**
	 * Render "no active scans" message.
	 *
	 * @since 1.0.0
	 */
	function renderNoActiveScans() {
		const $container = $( '#content-guard-pro-active-scans-container' );
		const template = $( '#content-guard-pro-no-active-scans-template' ).html();

		if ( template ) {
			$container.html( template );
		}
	}

	/**
	 * Render a scan from template.
	 *
	 * Replaces template variables with actual values.
	 *
	 * @since 1.0.0
	 * @param {string} template Template HTML.
	 * @param {Object} scan     Scan object.
	 * @return {string}         Rendered HTML.
	 */
	function renderScanFromTemplate( template, scan ) {
		// Prepare data
		const data = {
			scan_id: scan.scan_id,
			mode_label: scan.mode === 'quick' ? contentGuardProScanProgress.i18n.modeQuick : contentGuardProScanProgress.i18n.modeStandard,
			status: scan.status,
			status_label: getStatusLabel( scan.status ),
			progress_percentage: scan.progress_percentage || 0,
			items_checked: formatNumber( scan.items_checked ),
			items_total: formatNumber( scan.items_total ),
			findings: formatNumber( scan.findings ),
			elapsed_display: scan.elapsed_display || contentGuardProScanProgress.i18n.justNow
		};

		// Replace template variables
		let html = template;
		Object.keys( data ).forEach( function( key ) {
			const regex = new RegExp( '{{' + key + '}}', 'g' );
			html = html.replace( regex, data[ key ] );
		});

		return html;
	}

	/**
	 * Get status label text.
	 *
	 * @since 1.0.0
	 * @param {string} status Status code.
	 * @return {string}        Status label.
	 */
	function getStatusLabel( status ) {
		const labels = {
			'pending': contentGuardProScanProgress.i18n.statusPending,
			'running': contentGuardProScanProgress.i18n.statusRunning,
			'paused': contentGuardProScanProgress.i18n.statusPaused,
			'cancelled': contentGuardProScanProgress.i18n.statusCancelled,
			'completed': contentGuardProScanProgress.i18n.statusCompleted,
			'failed': contentGuardProScanProgress.i18n.statusFailed
		};

		return labels[ status ] || status;
	}

	/**
	 * Update Scan History table rows.
	 *
	 * Updates the items_checked and findings columns for active scans.
	 *
	 * @since 1.0.0
	 */
	function updateScanHistoryRows() {
		// Only update if there are active scans
		if ( ! hasActiveScans ) {
			return;
		}

		const $table = $( '.content-guard-pro-scans-list-table' );
		if ( ! $table.length ) {
			return;
		}

		$.ajax({
			url: API_BASE + '/scans/active',
			method: 'GET',
			beforeSend: function( xhr ) {
				xhr.setRequestHeader( 'X-WP-Nonce', API_NONCE );
			},
			success: function( response ) {
				if ( ! response.has_active ) {
					return;
				}

				// Update each active scan row
				updateScanHistoryRowsWithData( response.active_scans );
			},
			error: function( xhr, status, error ) {
				// Error updating table rows - silently fail
			}
		});
	}

	/**
	 * Update Scan History table rows with provided scan data.
	 *
	 * Helper function that updates table rows with cached or provided scan data.
	 * Used both during regular updates and for final update after completion.
	 *
	 * @since 1.0.0
	 * @param {Array} scans Array of scan objects.
	 */
	function updateScanHistoryRowsWithData( scans ) {
		const $table = $( '.content-guard-pro-scans-list-table' );
		if ( ! $table.length ) {
			return;
		}

		// Update each scan row
		$.each( scans, function( index, scan ) {
			const $row = $table.find( 'tr#scan-' + scan.scan_id );
			
			if ( ! $row.length ) {
				return;
			}

			// Update status column
			const $statusCell = $row.find( '.column-status' );
			if ( $statusCell.length ) {
				const statusHtml = '<span class="content-guard-pro-status-badge content-guard-pro-status-' + scan.status + '">' +
					getStatusLabel( scan.status ) +
					'</span>';
				
				// Always ensure badge class is present - check current HTML and update if needed
				const currentHtml = $statusCell.html();
				const $currentSpan = $statusCell.find( 'span' );
				const currentStatus = $currentSpan.length ? $currentSpan.attr( 'class' ) : '';
				const expectedClass = 'content-guard-pro-status-badge content-guard-pro-status-' + scan.status;
				
				// Normalize whitespace for comparison
				const normalizedCurrent = currentHtml.replace( /\s+/g, ' ' ).trim();
				const normalizedNew = statusHtml.replace( /\s+/g, ' ' ).trim();
				
				// Update if HTML doesn't match OR if badge class is missing
				if ( normalizedCurrent !== normalizedNew || ! currentStatus || currentStatus.indexOf( 'content-guard-pro-status-badge' ) === -1 ) {
					$statusCell.html( statusHtml ).addClass( 'content-guard-pro-flash-update' );
					setTimeout( function() {
						$statusCell.removeClass( 'content-guard-pro-flash-update' );
					}, 1000 );
				}
			}

			// Update items checked column
			const $itemsCell = $row.find( '.column-totals_checked' );
			if ( $itemsCell.length ) {
				const itemsText = formatNumber( scan.items_checked );
				if ( $itemsCell.text() !== itemsText ) {
					$itemsCell.text( itemsText ).addClass( 'content-guard-pro-flash-update' );
					setTimeout( function() {
						$itemsCell.removeClass( 'content-guard-pro-flash-update' );
					}, 1000 );
				}
			}

			// Update findings column
			const $findingsCell = $row.find( '.column-totals_flagged' );
			if ( $findingsCell.length ) {
				const findingsText = formatNumber( scan.findings );
				if ( $findingsCell.text() !== findingsText ) {
					$findingsCell.text( findingsText ).addClass( 'content-guard-pro-flash-update' );
					setTimeout( function() {
						$findingsCell.removeClass( 'content-guard-pro-flash-update' );
					}, 1000 );
				}
			}

			// Update duration column
			const $durationCell = $row.find( '.column-duration' );
			if ( $durationCell.length ) {
				let durationHtml;
				
				// If scan has finished, show the actual duration
				if ( scan.finished_at && scan.elapsed_display ) {
					durationHtml = scan.elapsed_display;
				} else {
					// Still running - show "Running..." status
					durationHtml = '<span class="content-guard-pro-status-running">' + 
						( contentGuardProScanProgress.i18n.running || 'Running...' ) +
						'</span>';
				}
				
				// Only update if content changed
				if ( $durationCell.html() !== durationHtml ) {
					$durationCell.html( durationHtml ).addClass( 'content-guard-pro-flash-update' );
					setTimeout( function() {
						$durationCell.removeClass( 'content-guard-pro-flash-update' );
					}, 1000 );
				}
			}
		});
	}

	/**
	 * Format number with thousands separator.
	 *
	 * @since 1.0.0
	 * @param {number} num Number to format.
	 * @return {string}     Formatted number.
	 */
	function formatNumber( num ) {
		if ( typeof num === 'undefined' || num === null ) {
			return '0';
		}
		return num.toString().replace( /\B(?=(\d{3})+(?!\d))/g, ',' );
	}

	// Initialize when document is ready
	$( document ).ready( init );

})( jQuery );

