/**
 * Content Guard Pro - Gutenberg Editor Scan Panel
 *
 * Provides real-time security scanning feedback in the block editor.
 * Implements US-037 and US-038 from PRD:
 * - Scan triggers on post save/publish
 * - Warning displayed if issues found
 * - Publishing can proceed with acknowledgment
 * - Fail-open: save always succeeds
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

( function( wp ) {
	'use strict';

	// Bail if required WordPress packages are not available.
	if ( ! wp || ! wp.plugins || ! wp.element || ! wp.components || ! wp.data ) {
		console.warn( 'Content Guard Pro: Required WordPress packages not available.' );
		return;
	}

	// Use wp.editor (WP 6.6+) with fallback to wp.editPost (WP 6.1-6.5) for backwards compatibility.
	var editorPackage = wp.editor || wp.editPost;
	if ( ! editorPackage ) {
		console.warn( 'Content Guard Pro: Editor package not available.' );
		return;
	}

	var registerPlugin = wp.plugins.registerPlugin;
	var PluginPrePublishPanel = editorPackage.PluginPrePublishPanel;
	var PluginPostStatusInfo = editorPackage.PluginPostStatusInfo;
	var PluginSidebar = editorPackage.PluginSidebar;
	var PluginSidebarMoreMenuItem = editorPackage.PluginSidebarMoreMenuItem;
	var createElement = wp.element.createElement;
	var Fragment = wp.element.Fragment;
	var useState = wp.element.useState;
	var useEffect = wp.element.useEffect;
	var useCallback = wp.element.useCallback;
	var useSelect = wp.data.useSelect;
	var useDispatch = wp.data.useDispatch;
	var Button = wp.components.Button;
	var Spinner = wp.components.Spinner;
	var Notice = wp.components.Notice;
	var Icon = wp.components.Icon;
	var PanelBody = wp.components.PanelBody;
	var PanelRow = wp.components.PanelRow;
	var __ = wp.i18n.__;

	// Get localized data.
	var cgpData = window.contentGuardProEditor || {};
	// Use relative path for wp.apiFetch - it automatically prepends the REST base URL.
	var apiBasePath = '/content-guard-pro/v1';
	var nonce = cgpData.nonce || '';
	// Default to false (free tier) if scanEnabled is not explicitly set to true.
	// This ensures free tier users don't get auto-scanning.
	// Note: WordPress wp_localize_script converts boolean false to empty string '',
	// so we check for explicit true to handle all cases correctly.
	var scanEnabled = cgpData.scanEnabled === true;
	var findingsUrl = cgpData.findingsUrl || '';
	
	// Plugin branding prefix for notifications.
	var pluginName = 'Content Guard Pro';

	/**
	 * Show a prominent warning notice with link.
	 */
	function showWarningNotice( message, hasLink ) {
		if ( ! wp.data || ! wp.data.dispatch ) {
			return;
		}
		
		var fullMessage = pluginName + ': ' + message;
		if ( hasLink && findingsUrl ) {
			fullMessage += ' ';
		}
		
		wp.data.dispatch( 'core/notices' ).createWarningNotice( fullMessage, {
			id: 'cgp-scan-warning',
			isDismissible: true,
			actions: hasLink && findingsUrl ? [
				{
					label: __( 'View Findings', 'content-guard-pro' ),
					url: findingsUrl
				}
			] : []
		} );
	}

	/**
	 * Show an error notice.
	 */
	function showErrorNotice( message ) {
		if ( ! wp.data || ! wp.data.dispatch ) {
			return;
		}
		
		wp.data.dispatch( 'core/notices' ).createErrorNotice( pluginName + ': ' + message, {
			id: 'cgp-scan-error',
			isDismissible: true,
			actions: findingsUrl ? [
				{
					label: __( 'View Findings', 'content-guard-pro' ),
					url: findingsUrl
				}
			] : []
		} );
	}

	/**
	 * Show a success notice.
	 */
	function showSuccessNotice( message ) {
		if ( ! wp.data || ! wp.data.dispatch ) {
			return;
		}
		
		wp.data.dispatch( 'core/notices' ).createSuccessNotice( pluginName + ': ' + message, {
			id: 'cgp-scan-success',
			isDismissible: true
		} );
	}

	/**
	 * Severity badge component.
	 */
	function SeverityBadge( props ) {
		var severity = props.severity;
		var count = props.count;

		var colors = {
			critical: { bg: '#d63638', text: '#fff' },
			suspicious: { bg: '#dba617', text: '#1d2327' },
			review: { bg: '#72aee6', text: '#1d2327' }
		};

		var labels = {
			critical: __( 'Critical', 'content-guard-pro' ),
			suspicious: __( 'Suspicious', 'content-guard-pro' ),
			review: __( 'Review', 'content-guard-pro' )
		};

		var color = colors[ severity ] || colors.review;
		var label = labels[ severity ] || severity;

		return createElement(
			'span',
			{
				style: {
					display: 'inline-flex',
					alignItems: 'center',
					gap: '4px',
					backgroundColor: color.bg,
					color: color.text,
					padding: '2px 8px',
					borderRadius: '3px',
					fontSize: '11px',
					fontWeight: '600',
					textTransform: 'uppercase',
					marginRight: '8px'
				}
			},
			label,
			count !== undefined && createElement(
				'span',
				{
					style: {
						backgroundColor: 'rgba(255,255,255,0.3)',
						padding: '0 4px',
						borderRadius: '2px',
						marginLeft: '4px'
					}
				},
				count
			)
		);
	}

	/**
	 * Finding item component.
	 */
	function FindingItem( props ) {
		var finding = props.finding;

		return createElement(
			'div',
			{
				style: {
					padding: '10px 12px',
					marginBottom: '8px',
					backgroundColor: '#fff',
					border: '1px solid #ddd',
					borderLeftWidth: '4px',
					borderLeftColor: finding.severity === 'critical' ? '#d63638' :
						finding.severity === 'suspicious' ? '#dba617' : '#72aee6',
					borderRadius: '2px'
				}
			},
			createElement(
				'div',
				{
					style: {
						display: 'flex',
						alignItems: 'center',
						marginBottom: '6px'
					}
				},
				createElement( SeverityBadge, { severity: finding.severity } ),
				createElement(
					'span',
					{
						style: {
							color: '#757575',
							fontSize: '11px'
						}
					},
					finding.confidence + '%',
					' ',
					__( 'confidence', 'content-guard-pro' )
				)
			),
			createElement(
				'div',
				{
					style: {
						fontSize: '13px',
						color: '#1d2327',
						marginBottom: '4px'
					}
				},
				finding.description || finding.rule_id
			),
			finding.matched_excerpt && createElement(
				'div',
				{
					style: {
						fontSize: '12px',
						color: '#757575',
						fontFamily: 'monospace',
						backgroundColor: '#f6f7f7',
						padding: '6px 8px',
						borderRadius: '2px',
						wordBreak: 'break-word',
						marginTop: '6px'
					}
				},
				finding.matched_excerpt
			),
			finding.field && finding.field !== 'post_content' && createElement(
				'div',
				{
					style: {
						fontSize: '11px',
						color: '#757575',
						marginTop: '4px'
					}
				},
				__( 'Field:', 'content-guard-pro' ),
				' ',
				finding.field
			)
		);
	}

	/**
	 * Main scan results component.
	 */
	function ScanResults( props ) {
		var scanResult = props.scanResult;
		var isScanning = props.isScanning;
		var onScan = props.onScan;
		var existingFindings = props.existingFindings;

		// Combine scan results with existing findings.
		var hasExistingIssues = existingFindings && existingFindings.has_issues;
		var hasExistingCritical = existingFindings && existingFindings.has_critical;
		var existingTotal = existingFindings ? existingFindings.total : 0;
		var hasScanIssues = scanResult && scanResult.has_issues;
		var hasScanCritical = scanResult && scanResult.has_critical;

		// Determine effective result.
		// Priority: If a scan was performed, use scan result. Otherwise, show existing findings.
		var effectiveResult = null;
		var showCleanWithPendingResolution = false;
		
		if ( scanResult ) {
			// Scan was performed - use scan result (it reflects CURRENT content).
			effectiveResult = scanResult;
			// If scan is clean but there are existing findings in DB, flag for special message.
			if ( ! scanResult.has_issues && hasExistingIssues ) {
				showCleanWithPendingResolution = true;
			}
		} else if ( hasExistingIssues ) {
			// No scan performed yet, but there are existing findings in database.
			effectiveResult = {
				has_issues: true,
				has_critical: hasExistingCritical,
				findings: existingFindings.findings || [],
				counts: existingFindings.counts || {},
				total: existingFindings.total || 0,
				isExisting: true // Flag to show appropriate message.
			};
		}

		// Show disabled message only when no existing findings and auto-scan is off.
		if ( ! scanEnabled && ! effectiveResult ) {
			return createElement(
				Fragment,
				null,
				createElement(
					'div',
					{
						style: {
							padding: '12px',
							backgroundColor: '#f6f7f7',
							borderRadius: '4px',
							color: '#757575',
							fontSize: '13px',
							marginBottom: '12px'
						}
					},
					createElement( Icon, { icon: 'info', style: { marginRight: '8px' } } ),
					__( 'Auto-scan on save is disabled in settings.', 'content-guard-pro' )
				),
				createElement(
					'div',
					{
						style: {
							padding: '12px',
							textAlign: 'center'
						}
					},
					createElement(
						Button,
						{
							variant: 'secondary',
							onClick: function() { onScan( true ); }, // Manual scan - always works.
							icon: 'search'
						},
						__( 'Scan Content Now', 'content-guard-pro' )
					)
				)
			);
		}

		// Scanning state.
		if ( isScanning ) {
			return createElement(
				'div',
				{
					style: {
						display: 'flex',
						alignItems: 'center',
						padding: '12px',
						backgroundColor: '#f0f6fc',
						borderRadius: '4px'
					}
				},
				createElement( Spinner, null ),
				createElement(
					'span',
					{ style: { marginLeft: '8px' } },
					__( 'Scanning content for security issues...', 'content-guard-pro' )
				)
			);
		}

		// No scan result or existing findings yet.
		if ( ! effectiveResult ) {
			return createElement(
				'div',
				{
					style: {
						padding: '12px',
						textAlign: 'center'
					}
				},
				createElement(
					Button,
					{
						variant: 'secondary',
						onClick: function() { onScan( true ); }, // Manual scan - always works.
						icon: 'search'
					},
					__( 'Scan Content Now', 'content-guard-pro' )
				)
			);
		}

		// Has results (from scan or existing findings).
		var hasIssues = effectiveResult.has_issues;
		var hasCritical = effectiveResult.has_critical;
		var findings = effectiveResult.findings || [];
		var counts = effectiveResult.counts || {};
		var isExisting = effectiveResult.isExisting;

		// No issues found in current content.
		if ( ! hasIssues ) {
			// Check if there are existing findings that will be resolved on save.
			if ( showCleanWithPendingResolution ) {
				return createElement(
					'div',
					{
						style: {
							padding: '12px',
							backgroundColor: '#fff8e5',
							borderRadius: '4px',
							borderLeft: '4px solid #dba617'
						}
					},
					createElement(
						'div',
						{
							style: {
								display: 'flex',
								alignItems: 'center',
								color: '#1e1e1e',
								fontWeight: '600',
								marginBottom: '8px'
							}
						},
						createElement( Icon, { icon: 'yes-alt', style: { marginRight: '8px', color: '#00a32a' } } ),
						__( 'Current content is clean!', 'content-guard-pro' )
					),
					createElement(
						'p',
						{
							style: {
								margin: '0',
								fontSize: '13px',
								color: '#50575e'
							}
						},
						existingTotal + ' ' +
						( existingTotal === 1 ? __( 'existing issue', 'content-guard-pro' ) : __( 'existing issues', 'content-guard-pro' ) ) +
						' ' + __( 'will be resolved when you save.', 'content-guard-pro' )
					)
				);
			}
			
			// Truly no issues anywhere.
			return createElement(
				'div',
				{
					style: {
						padding: '12px',
						backgroundColor: '#edfaef',
						borderRadius: '4px',
						color: '#00a32a',
						display: 'flex',
						alignItems: 'center'
					}
				},
				createElement( Icon, { icon: 'yes-alt', style: { marginRight: '8px' } } ),
				__( 'No security issues detected.', 'content-guard-pro' )
			);
		}

		// Issues found.
		return createElement(
			Fragment,
			null,
			// Summary notice.
			createElement(
				Notice,
				{
					status: hasCritical ? 'error' : 'warning',
					isDismissible: false,
					className: 'cgp-scan-notice'
				},
				createElement(
					'div',
					{
						style: {
							display: 'flex',
							flexWrap: 'wrap',
							gap: '8px',
							alignItems: 'center'
						}
					},
					createElement(
						'strong',
						null,
						isExisting ?
							( hasCritical ?
								__( 'Existing critical security issues!', 'content-guard-pro' ) :
								__( 'Existing security issues', 'content-guard-pro' ) ) :
							( hasCritical ?
								__( 'Critical security issues found!', 'content-guard-pro' ) :
								__( 'Security issues detected', 'content-guard-pro' ) )
					),
					createElement(
						'div',
						{
							style: {
								display: 'flex',
								gap: '4px',
								marginLeft: 'auto'
							}
						},
						counts.critical > 0 && createElement( SeverityBadge, { severity: 'critical', count: counts.critical } ),
						counts.suspicious > 0 && createElement( SeverityBadge, { severity: 'suspicious', count: counts.suspicious } ),
						counts.review > 0 && createElement( SeverityBadge, { severity: 'review', count: counts.review } )
					)
				),
				createElement(
					'p',
					{
						style: {
							margin: '8px 0 0 0',
							fontSize: '13px'
						}
					},
					isExisting ?
						__( 'This post has previously detected security issues. Click Re-scan to check again.', 'content-guard-pro' ) :
						( hasCritical ?
							__( 'This content contains potentially malicious code. Please review before publishing.', 'content-guard-pro' ) :
							__( 'This content has items that may need review. You can still publish.', 'content-guard-pro' ) )
				)
			),

			// Findings list.
			createElement(
				'div',
				{
					style: {
						marginTop: '16px',
						maxHeight: '300px',
						overflowY: 'auto'
					}
				},
				findings.slice( 0, 10 ).map( function( finding, index ) {
					return createElement( FindingItem, {
						key: index,
						finding: finding
					} );
				} ),
				findings.length > 10 && createElement(
					'div',
					{
						style: {
							padding: '12px',
							textAlign: 'center',
							color: '#757575',
							fontSize: '13px'
						}
					},
					__( 'and', 'content-guard-pro' ),
					' ',
					findings.length - 10,
					' ',
					__( 'more...', 'content-guard-pro' )
				)
			),

			// Actions.
			createElement(
				'div',
				{
					style: {
						marginTop: '16px',
						display: 'flex',
						gap: '8px',
						flexWrap: 'wrap'
					}
				},
				createElement(
					Button,
					{
						variant: 'secondary',
						onClick: function() { onScan( true ); }, // Manual scan - always works.
						icon: 'update'
					},
					__( 'Re-scan', 'content-guard-pro' )
				),
				findingsUrl && createElement(
					Button,
					{
						variant: 'link',
						href: findingsUrl,
						target: '_blank'
					},
					__( 'View All Findings', 'content-guard-pro' )
				)
			)
		);
	}

	/**
	 * Pre-publish panel component.
	 * Shows scan results before publishing.
	 */
	function ContentGuardPrePublishPanel() {
		var _useState = useState( null );
		var scanResult = _useState[0];
		var setScanResult = _useState[1];

		var _useState2 = useState( false );
		var isScanning = _useState2[0];
		var setIsScanning = _useState2[1];

		var _useState3 = useState( false );
		var hasScanned = _useState3[0];
		var setHasScanned = _useState3[1];

		var _useState4 = useState( null );
		var existingFindings = _useState4[0];
		var setExistingFindings = _useState4[1];

		// Get post data.
		var postData = useSelect( function( select ) {
			var editor = select( 'core/editor' );
			return {
				postId: editor.getCurrentPostId(),
				postContent: editor.getEditedPostContent(),
				postTitle: editor.getEditedPostAttribute( 'title' ),
				isPublishing: editor.isPublishingPost(),
				isSaving: editor.isSavingPost()
			};
		}, [] );

		// Fetch existing findings on load.
		useEffect( function() {
			if ( postData.postId && ! existingFindings ) {
				wp.apiFetch( {
					path: apiBasePath + '/post-findings/' + postData.postId,
					method: 'GET'
			} ).then( function( response ) {
				setExistingFindings( response );
			} ).catch( function( error ) {
				// Failed to load existing findings - continue without them.
			} );
			}
		}, [ postData.postId ] );

		// Perform scan.
		// @param {boolean} manual - Whether this is a manual scan (user clicked button) vs auto-scan.
		var performScan = useCallback( function( manual ) {
			if ( ! postData.postId || isScanning ) {
				return;
			}

			setIsScanning( true );

			wp.apiFetch( {
				path: apiBasePath + '/scan-post',
				method: 'POST',
				data: {
					post_id: postData.postId,
					content: postData.postContent,
					manual: manual === true
				}
			} ).then( function( response ) {
				setScanResult( response );
				setHasScanned( true );
				setIsScanning( false );
				
				// Show prominent notice based on results.
				if ( response.has_critical ) {
					showErrorNotice( __( 'Critical security issues found!', 'content-guard-pro' ) + ' (' + response.total + ' ' + __( 'issues', 'content-guard-pro' ) + ')' );
				} else if ( response.has_issues ) {
					// Show issues but also mention if some were auto-resolved.
					var issueMessage = __( 'Security issues detected.', 'content-guard-pro' ) + ' (' + response.total + ' ' + __( 'issues', 'content-guard-pro' ) + ')';
					if ( response.auto_resolved && response.auto_resolved > 0 ) {
						issueMessage += ' ' + response.auto_resolved + ' ' + __( 'resolved.', 'content-guard-pro' );
					}
					showWarningNotice( issueMessage, true );
				} else if ( ! response.has_issues && response.auto_resolved && response.auto_resolved > 0 ) {
					// Findings were auto-resolved.
					showSuccessNotice( __( 'Content is clean!', 'content-guard-pro' ) + ' ' + response.auto_resolved + ' ' + __( 'issues resolved.', 'content-guard-pro' ) );
				}

				// Always refresh existing findings after scan to get latest status.
				setExistingFindings( null );
			} ).catch( function( error ) {
				setIsScanning( false );
				// Fail-open: don't block on errors.
				setScanResult( {
					success: false,
					has_issues: false,
					findings: [],
					error: error.message || __( 'Scan failed', 'content-guard-pro' )
				} );
			} );
		}, [ postData.postId, postData.postContent, isScanning ] );

		// Auto-scan when panel opens (first time only, if enabled).
		useEffect( function() {
			if ( ! hasScanned && scanEnabled && postData.postId ) {
				performScan( false ); // Auto-scan, not manual.
			}
		}, [ hasScanned, postData.postId ] );

		// Determine panel title and icon based on scan result or existing findings.
		var panelTitle = __( 'Content Security Scan', 'content-guard-pro' );
		var panelIcon = 'shield';
		var hasIssues = ( scanResult && scanResult.has_issues ) || ( existingFindings && existingFindings.has_issues );
		var hasCritical = ( scanResult && scanResult.has_critical ) || ( existingFindings && existingFindings.has_critical );

		if ( hasCritical ) {
			panelIcon = 'warning';
		} else if ( hasIssues ) {
			panelIcon = 'flag';
		} else if ( scanResult && ! scanResult.has_issues ) {
			panelIcon = 'yes-alt';
		}

		return createElement(
			PluginPrePublishPanel,
			{
				title: panelTitle,
				icon: panelIcon,
				initialOpen: hasIssues
			},
			createElement( ScanResults, {
				scanResult: scanResult,
				isScanning: isScanning,
				onScan: performScan,
				existingFindings: existingFindings
			} )
		);
	}

	/**
	 * Post status info component.
	 * Shows scan status in the post status panel.
	 */
	function ContentGuardPostStatusInfo() {
		var _useState = useState( null );
		var scanResult = _useState[0];
		var setScanResult = _useState[1];

		var _useState2 = useState( false );
		var isScanning = _useState2[0];
		var setIsScanning = _useState2[1];

		var _useState3 = useState( false );
		var hasInitialScanned = _useState3[0];
		var setHasInitialScanned = _useState3[1];

		var _useState4 = useState( false );
		var wasSaving = _useState4[0];
		var setWasSaving = _useState4[1];

		var _useState5 = useState( null );
		var existingFindings = _useState5[0];
		var setExistingFindings = _useState5[1];

		var _useState6 = useState( false );
		var existingFindingsLoaded = _useState6[0];
		var setExistingFindingsLoaded = _useState6[1];

		// Get post data.
		var postData = useSelect( function( select ) {
			var editor = select( 'core/editor' );
			return {
				postId: editor.getCurrentPostId(),
				postContent: editor.getEditedPostContent(),
				hasEdits: editor.hasChangedContent(),
				isSavingPost: editor.isSavingPost(),
				isAutosavingPost: editor.isAutosavingPost()
			};
		}, [] );

		// Helper function to refresh existing findings.
		var refreshExistingFindings = useCallback( function( showGuidanceIfNeeded ) {
			if ( ! postData.postId ) {
				return;
			}
			
			wp.apiFetch( {
				path: apiBasePath + '/post-findings/' + postData.postId,
				method: 'GET'
			} ).then( function( response ) {
				setExistingFindings( response );
				setExistingFindingsLoaded( true );
				
				// If this refresh is after a save on Free tier, show guidance notice if there are still issues.
				if ( showGuidanceIfNeeded && ! scanEnabled && response.has_issues ) {
					setTimeout( function() {
						showWarningNotice( 
							__( 'Post saved. Click "Scan" under CONTENT GUARD PRO section (usually the right side of the screen) to verify if security issues are resolved.', 'content-guard-pro' ),
							false // Don't show link to findings
						);
					}, 500 ); // Small delay for better UX (let save success message show first).
				}
				// Otherwise, show notice if there are quarantined items.
				else if ( response.has_quarantined ) {
					showWarningNotice( __( 'This post has quarantined security issues.', 'content-guard-pro' ), true );
				}
				// Show notice for existing open findings (important when auto-scan is disabled).
				else if ( response.has_issues && ! showGuidanceIfNeeded ) {
					if ( response.has_critical ) {
						showErrorNotice( __( 'This post has existing critical security issues!', 'content-guard-pro' ) + ' (' + response.total + ' ' + __( 'issues', 'content-guard-pro' ) + ')' );
					} else {
						showWarningNotice( __( 'This post has existing security findings.', 'content-guard-pro' ) + ' (' + response.total + ' ' + __( 'issues', 'content-guard-pro' ) + ')', true );
					}
				}
			} ).catch( function( error ) {
				setExistingFindingsLoaded( true ); // Mark as loaded even on error.
			} );
		}, [ postData.postId, scanEnabled ] );

		// Fetch existing findings (including quarantined) on load.
		useEffect( function() {
			if ( postData.postId && ! existingFindingsLoaded ) {
				refreshExistingFindings( false ); // No guidance notice on initial load.
			}
		}, [ postData.postId, existingFindingsLoaded, refreshExistingFindings ] );

		// Perform scan.
		// @param {boolean} manual - Whether this is a manual scan (user clicked button) vs auto-scan.
		var performScan = useCallback( function( manual ) {
			if ( ! postData.postId || isScanning ) {
				return;
			}

			setIsScanning( true );

			wp.apiFetch( {
				path: apiBasePath + '/scan-post',
				method: 'POST',
				data: {
					post_id: postData.postId,
					content: postData.postContent,
					manual: manual === true
				}
			} ).then( function( response ) {
				setScanResult( response );
				setIsScanning( false );
				
				// Show prominent notice based on results.
				if ( response.has_critical ) {
					showErrorNotice( __( 'Critical security issues detected!', 'content-guard-pro' ) + ' (' + response.total + ' ' + __( 'issues', 'content-guard-pro' ) + ')' );
				} else if ( response.has_issues ) {
					// Show issues but also mention if some were auto-resolved.
					var issueMessage = __( 'Security issues detected.', 'content-guard-pro' ) + ' (' + response.total + ' ' + __( 'issues', 'content-guard-pro' ) + ')';
					if ( response.auto_resolved && response.auto_resolved > 0 ) {
						issueMessage += ' ' + response.auto_resolved + ' ' + __( 'resolved.', 'content-guard-pro' );
					}
					showWarningNotice( issueMessage, true );
				} else if ( ! response.has_issues && manual ) {
					// No issues found in current content.
					if ( response.auto_resolved && response.auto_resolved > 0 ) {
						// Findings were auto-resolved because content now matches saved clean content.
						showSuccessNotice( __( 'Content is clean!', 'content-guard-pro' ) + ' ' + response.auto_resolved + ' ' + __( 'issues resolved.', 'content-guard-pro' ) );
					} else if ( existingFindingsLoaded && existingFindings && existingFindings.has_issues ) {
						// Current content is clean but there are existing findings in database.
						// Content doesn't match saved content, so user needs to save to auto-resolve.
						showWarningNotice( __( 'Current content appears clean. Save to resolve existing issues.', 'content-guard-pro' ), true );
					} else {
						// Truly no issues anywhere.
						showSuccessNotice( __( 'No issues found in current content.', 'content-guard-pro' ) );
					}
				}

				// Refresh existing findings after scan to get latest status.
				// Use the refresh function to ensure proper state update.
				refreshExistingFindings( false ); // No guidance notice after manual scan.
			} ).catch( function( error ) {
				setIsScanning( false );
			} );
		}, [ postData.postId, postData.postContent, isScanning, existingFindings, existingFindingsLoaded, refreshExistingFindings ] );

		// Auto-scan on initial load (only if no existing findings - they're the source of truth).
		useEffect( function() {
			if ( ! hasInitialScanned && scanEnabled && postData.postId && postData.postContent && existingFindingsLoaded ) {
				setHasInitialScanned( true );
				// Skip auto-scan if there are existing findings - they already tell us the status.
				// Auto-scan is for detecting NEW issues, not re-validating existing ones.
				if ( existingFindings && existingFindings.has_issues ) {
					return;
				}
				// Delay slightly to let editor fully load.
				setTimeout( function() {
					performScan( false ); // Auto-scan, not manual.
				}, 500 );
			}
		}, [ hasInitialScanned, postData.postId, postData.postContent, existingFindingsLoaded, existingFindings ] );

		// Auto-scan after save completes (only if auto-scan is enabled).
		useEffect( function() {
			if ( postData.isSavingPost && ! postData.isAutosavingPost ) {
				setWasSaving( true );
			} else if ( wasSaving && ! postData.isSavingPost ) {
				// Save just completed.
				setWasSaving( false );
				
				// IMPORTANT: Always refresh existing findings after save completes.
				// This ensures we see the updated status after server-side auto-resolve.
				// Add a small delay to ensure server-side auto-resolve has completed.
				setTimeout( function() {
					// Pass true to show guidance notice if needed (Free tier with existing issues).
					refreshExistingFindings( true );
				}, 1000 ); // 1 second delay to allow server-side auto-resolve to complete.
				
				// Only auto-scan if the setting is enabled.
				if ( scanEnabled ) {
					setTimeout( function() {
						performScan( false ); // Auto-scan, not manual.
					}, 1500 ); // Slightly longer delay to ensure findings refresh first.
				}
			}
		}, [ postData.isSavingPost, postData.isAutosavingPost, wasSaving, refreshExistingFindings, scanEnabled ] );

		// Reset scan when content changes.
		useEffect( function() {
			if ( postData.hasEdits && scanResult ) {
				// Mark as stale but don't clear.
				setScanResult( function( prev ) {
					if ( prev ) {
						return Object.assign( {}, prev, { stale: true } );
					}
					return prev;
				} );
			}
		}, [ postData.hasEdits ] );

		// Determine status based on existing findings and scan results.
		var statusText = __( 'Not scanned', 'content-guard-pro' );
		var statusColor = '#757575';
		var statusIcon = 'shield';
		var hasQuarantined = existingFindings && existingFindings.has_quarantined;
		var hasExistingIssues = existingFindings && existingFindings.has_issues;
		var hasExistingCritical = existingFindings && existingFindings.has_critical;
		var existingCounts = existingFindings ? existingFindings.counts : null;

		// Combine scan result issues with existing findings - either source of issues should be shown.
		var hasScanIssues = scanResult && scanResult.has_issues;
		var hasScanCritical = scanResult && scanResult.has_critical;
		var hasScanStale = scanResult && scanResult.stale;
		
		// Effective issue status: issues from scan OR existing findings in database.
		var effectiveHasIssues = hasScanIssues || hasExistingIssues;
		var effectiveHasCritical = hasScanCritical || hasExistingCritical;
		
		// Calculate effective total (prefer scan result if available, otherwise existing).
		var effectiveTotal = hasScanIssues ? scanResult.total : ( hasExistingIssues ? existingFindings.total : 0 );

		if ( hasQuarantined ) {
			statusText = __( 'Quarantined', 'content-guard-pro' );
			statusColor = '#b32d2e';
			statusIcon = 'lock';
		} else if ( isScanning ) {
			statusText = __( 'Scanning...', 'content-guard-pro' );
			statusColor = '#2271b1';
		} else if ( hasScanStale ) {
			statusText = __( 'Content changed - rescan', 'content-guard-pro' );
			statusColor = '#dba617';
			statusIcon = 'update';
		} else if ( effectiveHasCritical ) {
			// Critical issues from scan or existing findings.
			statusText = __( 'Critical issues', 'content-guard-pro' );
			statusColor = '#d63638';
			statusIcon = 'warning';
		} else if ( effectiveHasIssues ) {
			// Issues from scan or existing findings.
			statusText = effectiveTotal + ' ' + __( 'issues', 'content-guard-pro' );
			statusColor = '#dba617';
			statusIcon = 'flag';
		} else if ( scanResult ) {
			// Scan completed with no issues AND no existing findings.
			statusText = __( 'No issues', 'content-guard-pro' );
			statusColor = '#00a32a';
			statusIcon = 'yes-alt';
		} else if ( ! scanEnabled ) {
			// No scan result, no existing issues, auto-scan disabled.
			statusText = __( 'Auto-scan disabled', 'content-guard-pro' );
			statusColor = '#757575';
			statusIcon = 'shield';
		}

		return createElement(
			PluginPostStatusInfo,
			{
				className: 'cgp-post-status-info'
			},
			// Header with plugin name.
			createElement(
				'div',
				{
					style: {
						display: 'flex',
						flexDirection: 'column',
						width: '100%',
						gap: '8px'
					}
				},
				createElement(
					'div',
					{
						style: {
							display: 'flex',
							alignItems: 'center',
							gap: '6px',
							fontSize: '11px',
							color: '#757575',
							textTransform: 'uppercase',
							letterSpacing: '0.5px',
							fontWeight: '500'
						}
					},
					createElement( Icon, { icon: 'shield', size: 14 } ),
					pluginName
				),
				createElement(
					'div',
					{
						style: {
							display: 'flex',
							alignItems: 'center',
							justifyContent: 'space-between',
							width: '100%'
						}
					},
					createElement(
						'div',
						{
							style: {
								display: 'flex',
								alignItems: 'center',
								gap: '8px',
								color: statusColor
							}
						},
						createElement( Icon, { icon: statusIcon, size: 20 } ),
					createElement(
						'span',
						{
							style: {
								fontSize: '13px',
								fontWeight: effectiveHasIssues || hasQuarantined ? '600' : 'normal'
							}
						},
						statusText
					)
					),
					createElement(
						Button,
						{
							variant: 'link',
							onClick: function( e ) {
								e.preventDefault();
								performScan( true ); // Manual scan - always works regardless of settings.
							},
							disabled: isScanning,
							style: {
								textDecoration: 'none',
								fontSize: '12px'
							}
						},
						isScanning ? createElement( Spinner, null ) : __( 'Scan', 'content-guard-pro' )
					)
				),
				// Show link to findings if there are issues (from scan or existing findings).
				( effectiveHasIssues || hasQuarantined ) && findingsUrl && createElement(
					'a',
					{
						href: findingsUrl + '&object_id=' + postData.postId,
						target: '_blank',
						style: {
							fontSize: '12px',
							color: '#2271b1',
							textDecoration: 'none'
						}
					},
					__( 'View in Findings →', 'content-guard-pro' )
				)
			)
		);
	}

	/**
	 * Sidebar panel component.
	 * Provides detailed scan information in a sidebar.
	 */
	function ContentGuardSidebar() {
		var _useState = useState( null );
		var scanResult = _useState[0];
		var setScanResult = _useState[1];

		var _useState2 = useState( false );
		var isScanning = _useState2[0];
		var setIsScanning = _useState2[1];

		var _useState3 = useState( null );
		var existingFindings = _useState3[0];
		var setExistingFindings = _useState3[1];

		var _useState4 = useState( false );
		var existingFindingsLoaded = _useState4[0];
		var setExistingFindingsLoaded = _useState4[1];

		// Get post data.
		var postData = useSelect( function( select ) {
			var editor = select( 'core/editor' );
			return {
				postId: editor.getCurrentPostId(),
				postContent: editor.getEditedPostContent()
			};
		}, [] );

		// Fetch existing findings on load.
		useEffect( function() {
			if ( postData.postId && ! existingFindingsLoaded ) {
				wp.apiFetch( {
					path: apiBasePath + '/post-findings/' + postData.postId,
					method: 'GET'
			} ).then( function( response ) {
				setExistingFindings( response );
				setExistingFindingsLoaded( true );
			} ).catch( function( error ) {
				setExistingFindingsLoaded( true );
			} );
			}
		}, [ postData.postId, existingFindingsLoaded ] );

		// Perform scan (always manual from sidebar - user clicks the button).
		var performScan = useCallback( function() {
			if ( ! postData.postId || isScanning ) {
				return;
			}

			setIsScanning( true );

			wp.apiFetch( {
				path: apiBasePath + '/scan-post',
				method: 'POST',
				data: {
					post_id: postData.postId,
					content: postData.postContent,
					manual: true // Always manual from sidebar - works regardless of settings.
				}
			} ).then( function( response ) {
				setScanResult( response );
				setIsScanning( false );

				// Show notice based on results.
				if ( response.has_critical ) {
					showErrorNotice( __( 'Critical security issues detected!', 'content-guard-pro' ) + ' (' + response.total + ' ' + __( 'issues', 'content-guard-pro' ) + ')' );
				} else if ( response.has_issues ) {
					// Show issues but also mention if some were auto-resolved.
					var issueMessage = __( 'Security issues detected.', 'content-guard-pro' ) + ' (' + response.total + ' ' + __( 'issues', 'content-guard-pro' ) + ')';
					if ( response.auto_resolved && response.auto_resolved > 0 ) {
						issueMessage += ' ' + response.auto_resolved + ' ' + __( 'resolved.', 'content-guard-pro' );
					}
					showWarningNotice( issueMessage, true );
				} else if ( ! response.has_issues ) {
					// No issues found in current content.
					if ( response.auto_resolved && response.auto_resolved > 0 ) {
						// Findings were auto-resolved because content now matches saved clean content.
						showSuccessNotice( __( 'Content is clean!', 'content-guard-pro' ) + ' ' + response.auto_resolved + ' ' + __( 'issues resolved.', 'content-guard-pro' ) );
					} else if ( existingFindingsLoaded && existingFindings && existingFindings.has_issues ) {
						// Current content is clean but there are existing findings in database.
						// Content doesn't match saved content, so user needs to save to auto-resolve.
						showWarningNotice( __( 'Current content appears clean. Save to resolve existing issues.', 'content-guard-pro' ), true );
					} else {
						// Truly no issues anywhere.
						showSuccessNotice( __( 'No issues found in current content.', 'content-guard-pro' ) );
					}
				}

				// Refresh existing findings after scan to get latest status.
				setExistingFindings( null );
				setExistingFindingsLoaded( false );
			} ).catch( function( error ) {
				setIsScanning( false );
			} );
		}, [ postData.postId, postData.postContent, isScanning, existingFindings, existingFindingsLoaded ] );

		return createElement(
			Fragment,
			null,
			createElement(
				PluginSidebarMoreMenuItem,
				{
					target: 'content-guard-pro-sidebar',
					icon: 'shield'
				},
				__( 'Content Guard Pro', 'content-guard-pro' )
			),
			createElement(
				PluginSidebar,
				{
					name: 'content-guard-pro-sidebar',
					title: __( 'Content Guard Pro', 'content-guard-pro' ),
					icon: 'shield'
				},
				createElement(
					PanelBody,
					{
						title: __( 'Security Scan', 'content-guard-pro' ),
						initialOpen: true
					},
					createElement( ScanResults, {
						scanResult: scanResult,
						isScanning: isScanning,
						onScan: performScan,
						existingFindings: existingFindings
					} )
				),
				createElement(
					PanelBody,
					{
						title: __( 'Quick Actions', 'content-guard-pro' ),
						initialOpen: false
					},
					createElement(
						PanelRow,
						null,
						createElement(
							Button,
							{
								variant: 'secondary',
								href: findingsUrl,
								target: '_blank',
								icon: 'external'
							},
							__( 'View All Findings', 'content-guard-pro' )
						)
					)
				)
			)
		);
	}

	/**
	 * Main plugin component that combines all panels.
	 */
	function ContentGuardProPlugin() {
		return createElement(
			Fragment,
			null,
			createElement( ContentGuardPrePublishPanel, null ),
			createElement( ContentGuardPostStatusInfo, null ),
			createElement( ContentGuardSidebar, null )
		);
	}

	// Register the plugin.
	registerPlugin( 'content-guard-pro', {
		render: ContentGuardProPlugin,
		icon: 'shield'
	} );

} )( window.wp );

