/**
 * Content Guard Pro - Admin JavaScript
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

(function($) {
	'use strict';

	/**
	 * Initialize admin functionality.
	 */
	$(document).ready(function() {
		// Notice dismissal handlers
		initNoticeDismissHandlers();
		
		// Findings page actions
		initFindingsActions();
		
		// Export handlers
		initExportHandlers();
	});

	/**
	 * Initialize notice dismissal handlers.
	 * 
	 * Handles dismissal of admin notices including welcome notice and critical findings notice.
	 */
	function initNoticeDismissHandlers() {
		// Handle welcome notice dismissal
		$(document).on('click', '.notice[data-dismissible="content-guard-pro-welcome-notice"] .notice-dismiss', function() {
			var $notice = $(this).closest('.notice');
			var nonce = $notice.data('nonce');
			
			$.post(ajaxurl, {
				action: 'content_guard_pro_dismiss_welcome_notice',
				nonce: nonce
			});
		});
		
		// Handle critical findings notice dismissal
		$(document).on('click', '.notice[data-dismissible="content-guard-pro-critical-findings"] .notice-dismiss', function() {
			var $notice = $(this).closest('.notice');
			var nonce = $notice.data('nonce');
			var count = $notice.data('critical-count');
			
			$.post(ajaxurl, {
				action: 'content_guard_pro_dismiss_critical_notice',
				nonce: nonce,
				count: count
			});
		});
	}

	/**
	 * Initialize findings page actions.
	 */
	function initFindingsActions() {
		// View finding details
		$('.content-guard-pro-view-finding').on('click', function(e) {
			e.preventDefault();
			var findingId = $(this).data('finding-id');
			
			// Open finding details modal
			openFindingDetailsModal(findingId);
		});

		// Quarantine finding
		$('.content-guard-pro-quarantine-finding').on('click', function(e) {
			e.preventDefault();
			var findingId = $(this).data('finding-id');
			var $clickedElement = $(this);
			
			if (confirm(contentGuardProAdmin.strings.confirmQuarantine)) {
				quarantineFinding(findingId, $clickedElement);
			}
		});

		// Un-quarantine finding
		$('.content-guard-pro-unquarantine-finding').on('click', function(e) {
			e.preventDefault();
			var findingId = $(this).data('finding-id');
			var $clickedElement = $(this);
			
			if (confirm('Are you sure you want to restore this finding?')) {
				unquarantineFinding(findingId, $clickedElement);
			}
		});

		// Ignore finding
		$('.content-guard-pro-ignore-finding').on('click', function(e) {
			e.preventDefault();
			var findingId = $(this).data('finding-id');
			var $clickedElement = $(this);
			
			if (confirm(contentGuardProAdmin.strings.confirmIgnore)) {
				ignoreFinding(findingId, $clickedElement);
			}
		});
	}

	/**
	 * Quarantine a finding via AJAX.
	 *
	 * @param {number} findingId Finding ID to quarantine.
	 * @param {jQuery} $element The clicked element.
	 */
	function quarantineFinding(findingId, $element) {
		var $row = $element.closest('tr');
		
		$.ajax({
			url: contentGuardProAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'content_guard_pro_quarantine_finding',
				finding_id: findingId,
				nonce: contentGuardProAdmin.nonce
			},
			beforeSend: function() {
				// Add loading visual state
				$row.addClass('is-loading');
				$element.prop('disabled', true);
				
				// Add spinner if not already present
				if ($row.find('.spinner').length === 0) {
					$row.find('td:first').prepend('<span class="spinner is-active" style="float: left; margin: 0 10px 0 0;"></span>');
				}
			},
			success: function(response) {
				// Remove loading state
				$row.removeClass('is-loading');
				$element.prop('disabled', false);
				$row.find('.spinner').remove();
				
				if (response.success) {
					// Reload page to show updated status
					location.reload();
				} else {
					alert('Error: ' + (response.data.message || 'Failed to quarantine finding'));
				}
			},
			error: function() {
				// Remove loading state
				$row.removeClass('is-loading');
				$element.prop('disabled', false);
				$row.find('.spinner').remove();
				
				alert('An error occurred while quarantining the finding.');
			}
		});
	}

	/**
	 * Un-quarantine a finding via AJAX.
	 *
	 * @param {number} findingId Finding ID to un-quarantine.
	 * @param {jQuery} $element The clicked element.
	 */
	function unquarantineFinding(findingId, $element) {
		var $row = $element.closest('tr');
		
		$.ajax({
			url: contentGuardProAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'content_guard_pro_unquarantine_finding',
				finding_id: findingId,
				nonce: contentGuardProAdmin.nonce
			},
			beforeSend: function() {
				// Add loading visual state
				$row.addClass('is-loading');
				$element.prop('disabled', true);
				
				// Add spinner if not already present
				if ($row.find('.spinner').length === 0) {
					$row.find('td:first').prepend('<span class="spinner is-active" style="float: left; margin: 0 10px 0 0;"></span>');
				}
			},
			success: function(response) {
				// Remove loading state
				$row.removeClass('is-loading');
				$element.prop('disabled', false);
				$row.find('.spinner').remove();
				
				if (response.success) {
					location.reload();
				} else {
					alert('Error: ' + (response.data.message || 'Failed to restore finding'));
				}
			},
			error: function() {
				// Remove loading state
				$row.removeClass('is-loading');
				$element.prop('disabled', false);
				$row.find('.spinner').remove();
				
				alert('An error occurred while restoring the finding.');
			}
		});
	}

	/**
	 * Ignore a finding via AJAX.
	 *
	 * @param {number} findingId Finding ID to ignore.
	 * @param {jQuery} $element The clicked element.
	 */
	function ignoreFinding(findingId, $element) {
		var $row = $element.closest('tr');
		
		$.ajax({
			url: contentGuardProAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'content_guard_pro_ignore_finding',
				finding_id: findingId,
				nonce: contentGuardProAdmin.nonce
			},
			beforeSend: function() {
				// Add loading visual state
				$row.addClass('is-loading');
				$element.prop('disabled', true);
				
				// Add spinner if not already present
				if ($row.find('.spinner').length === 0) {
					$row.find('td:first').prepend('<span class="spinner is-active" style="float: left; margin: 0 10px 0 0;"></span>');
				}
			},
			success: function(response) {
				// Remove loading state
				$row.removeClass('is-loading');
				$element.prop('disabled', false);
				$row.find('.spinner').remove();
				
				if (response.success) {
					location.reload();
				} else {
					alert('Error: ' + (response.data.message || 'Failed to ignore finding'));
				}
			},
			error: function() {
				// Remove loading state
				$row.removeClass('is-loading');
				$element.prop('disabled', false);
				$row.find('.spinner').remove();
				
				alert('An error occurred while ignoring the finding.');
			}
		});
	}

	/**
	 * Initialize export handlers.
	 * 
	 * Per PRD Section 3.7 - CSV and JSON export functionality.
	 */
	function initExportHandlers() {
		// Export as CSV
		$('#content-guard-pro-export-csv').on('click', function(e) {
			e.preventDefault();
			
			// Add export parameter to current URL and reload.
			// PHP-side logic will catch this parameter and trigger download.
			var currentUrl = window.location.href;
			var exportUrl = addQueryParameter(currentUrl, 'export', 'csv');
			
			window.location.href = exportUrl;
		});

		// Export as JSON
		$('#content-guard-pro-export-json').on('click', function(e) {
			e.preventDefault();
			
			// Add export parameter to current URL and reload.
			// PHP-side logic will catch this parameter and trigger download.
			var currentUrl = window.location.href;
			var exportUrl = addQueryParameter(currentUrl, 'export', 'json');
			
			window.location.href = exportUrl;
		});
	}

	/**
	 * Open finding details modal.
	 * 
	 * Uses WordPress ThickBox for modal display.
	 * Per PRD Section 3.3 - Finding detail view.
	 *
	 * @param {number} findingId Finding ID to display.
	 */
	function openFindingDetailsModal(findingId) {
		// Create a container for the modal content if it doesn't exist
		var $modalContainer = $('#cgp-finding-details-modal');
		if ($modalContainer.length === 0) {
			$modalContainer = $('<div id="cgp-finding-details-modal" style="display:none;"></div>');
			$('body').append($modalContainer);
		}

		// Set initial loading content
		var loadingHtml = '<div class="content-guard-pro-finding-details-modal" style="padding: 20px;">' +
			'<h2>Finding Details #' + findingId + '</h2>' +
			'<div class="content-guard-pro-finding-content">' +
			'<p style="font-size: 14px; color: #666;"><span class="spinner is-active" style="float: none; margin: 0 10px 0 0;"></span>Loading details...</p>' +
			'</div>' +
			'</div>';
		
		$modalContainer.html(loadingHtml);

		// Open ThickBox
		tb_show('Finding Details #' + findingId, '#TB_inline?width=600&height=550&inlineId=cgp-finding-details-modal');

		// Load actual finding details via AJAX
		$.ajax({
			url: contentGuardProAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'content_guard_pro_get_finding_details',
				finding_id: findingId,
				nonce: contentGuardProAdmin.nonce
			},
			success: function(response) {
				if (response.success && response.data) {
					// Format and display the finding details
					var finding = response.data;
					var content = buildFindingDetailsHTML(finding);
					
					// Update the content inside ThickBox
					// Note: ThickBox moves the content, so we target #TB_ajaxContent
					$('#TB_ajaxContent').html(content);
				} else {
					$('#TB_ajaxContent').html('<div class="content-guard-pro-finding-details-modal" style="padding: 20px;">' +
						'<h2>Finding Details #' + findingId + '</h2>' +
						'<div class="notice notice-error">' +
						'<p>Error: ' + (response.data.message || 'Failed to load finding details') + '</p>' +
						'</div>' +
						'</div>');
				}
			},
			error: function() {
				$('#TB_ajaxContent').html('<div class="content-guard-pro-finding-details-modal" style="padding: 20px;">' +
					'<h2>Finding Details #' + findingId + '</h2>' +
					'<div class="notice notice-error">' +
					'<p>An error occurred while loading the finding details.</p>' +
					'</div>' +
					'</div>');
			}
		});
	}
	
	/**
	 * Build HTML for finding details display.
	 *
	 * @param {Object} finding Finding data from AJAX response.
	 * @return {string} HTML content for modal.
	 */
	function buildFindingDetailsHTML(finding) {
		// Determine severity badge color
		var severityColors = {
			'critical': '#d63638',
			'suspicious': '#dba617',
			'review': '#72aee6'
		};
		var severityColor = severityColors[finding.severity] || '#666';
		
		// Build HTML content
		var html = '<div class="content-guard-pro-finding-details-modal" style="padding: 20px; font-family: -apple-system, BlinkMacSystemFont, \'Segoe UI\', Roboto, Oxygen-Sans, Ubuntu, Cantarell, \'Helvetica Neue\', sans-serif;">';
		html += '<h2 style="margin-top: 0;">Finding Details #' + finding.id + '</h2>';
		
		// Severity badge
		html += '<div style="margin-bottom: 20px;">';
		html += '<span style="display: inline-block; background: ' + severityColor + '; color: #fff; padding: 5px 12px; border-radius: 3px; font-size: 12px; font-weight: 600; text-transform: uppercase;">' + finding.severity + '</span>';
		html += '<span style="margin-left: 10px; color: #666; font-size: 13px;">Confidence: ' + finding.confidence + '%</span>';
		html += '<span style="margin-left: 10px; color: #666; font-size: 13px;">Status: ' + finding.status + '</span>';
		html += '</div>';
		
		// Location details
		html += '<div style="background: #f6f7f7; padding: 15px; border-left: 4px solid ' + severityColor + '; margin-bottom: 20px;">';
		html += '<h3 style="margin-top: 0; font-size: 14px; color: #333;">Location</h3>';
		html += '<table style="width: 100%; font-size: 13px;">';
		html += '<tr><td style="padding: 5px 10px 5px 0; color: #666; width: 120px;"><strong>Object Type:</strong></td><td style="padding: 5px 0;">' + finding.object_type + '</td></tr>';
		html += '<tr><td style="padding: 5px 10px 5px 0; color: #666;"><strong>Object ID:</strong></td><td style="padding: 5px 0;">' + finding.object_id + '</td></tr>';
		html += '<tr><td style="padding: 5px 10px 5px 0; color: #666;"><strong>Field:</strong></td><td style="padding: 5px 0;"><code style="background: #fff; padding: 2px 6px; border-radius: 3px;">' + finding.field + '</code></td></tr>';
		
		// Add object title and link if available
		if (finding.object_title) {
			html += '<tr><td style="padding: 5px 10px 5px 0; color: #666;"><strong>Post Title:</strong></td><td style="padding: 5px 0;">' + finding.object_title + '</td></tr>';
			if (finding.object_url) {
				html += '<tr><td style="padding: 5px 10px 5px 0; color: #666;"><strong>Edit Post:</strong></td><td style="padding: 5px 0;"><a href="' + finding.object_url + '" target="_blank" style="color: #2271b1;">Open in editor →</a></td></tr>';
			}
		}
		
		html += '</table>';
		html += '</div>';
		
		// Detection details
		html += '<div style="margin-bottom: 20px;">';
		html += '<h3 style="font-size: 14px; color: #333;">Detection Details</h3>';
		html += '<table style="width: 100%; font-size: 13px;">';
		html += '<tr><td style="padding: 5px 10px 5px 0; color: #666; width: 120px; vertical-align: top;"><strong>Rule ID:</strong></td><td style="padding: 5px 0;"><code style="background: #f6f7f7; padding: 2px 6px; border-radius: 3px;">' + finding.rule_id + '</code></td></tr>';
		html += '<tr><td style="padding: 5px 10px 5px 0; color: #666; vertical-align: top;"><strong>Matched Content:</strong></td><td style="padding: 5px 0;"><div style="background: #fff; border: 1px solid #ddd; padding: 10px; border-radius: 3px; font-family: monospace; font-size: 12px; max-height: 150px; overflow-y: auto; word-wrap: break-word;">' + escapeHtml(finding.matched_excerpt) + '</div></td></tr>';
		html += '</table>';
		html += '</div>';
		
		// Timestamps
		html += '<div style="margin-bottom: 20px;">';
		html += '<h3 style="font-size: 14px; color: #333;">Timeline</h3>';
		html += '<table style="width: 100%; font-size: 13px;">';
		html += '<tr><td style="padding: 5px 10px 5px 0; color: #666; width: 120px;"><strong>First Seen:</strong></td><td style="padding: 5px 0;">' + finding.first_seen + '</td></tr>';
		html += '<tr><td style="padding: 5px 10px 5px 0; color: #666;"><strong>Last Seen:</strong></td><td style="padding: 5px 0;">' + finding.last_seen + '</td></tr>';
		html += '</table>';
		html += '</div>';
		
		// Fingerprint (collapsed by default)
		html += '<div style="margin-bottom: 20px;">';
		html += '<details style="font-size: 13px;">';
		html += '<summary style="cursor: pointer; color: #2271b1; font-weight: 600;">Technical Details</summary>';
		html += '<div style="margin-top: 10px; padding: 10px; background: #f6f7f7; border-radius: 3px;">';
		html += '<p style="margin: 0 0 10px 0;"><strong>Fingerprint:</strong></p>';
		html += '<code style="display: block; background: #fff; padding: 8px; border-radius: 3px; word-break: break-all; font-size: 11px;">' + finding.fingerprint + '</code>';
		
		// Extra metadata if available
		if (finding.extra && Object.keys(finding.extra).length > 0) {
			html += '<p style="margin: 15px 0 5px 0;"><strong>Additional Metadata:</strong></p>';
			html += '<pre style="background: #fff; padding: 10px; border-radius: 3px; font-size: 11px; overflow-x: auto;">' + JSON.stringify(finding.extra, null, 2) + '</pre>';
		}
		
		html += '</div>';
		html += '</details>';
		html += '</div>';
		
		html += '</div>';
		
		return html;
	}
	
	/**
	 * Escape HTML for safe display.
	 *
	 * @param {string} text Text to escape.
	 * @return {string} Escaped text.
	 */
	function escapeHtml(text) {
		var map = {
			'&': '&amp;',
			'<': '&lt;',
			'>': '&gt;',
			'"': '&quot;',
			"'": '&#039;'
		};
		return text.replace(/[&<>"']/g, function(m) { return map[m]; });
	}

	/**
	 * Add or update a query parameter in a URL.
	 *
	 * @param {string} url      The URL to modify.
	 * @param {string} param    The parameter name.
	 * @param {string} value    The parameter value.
	 * @return {string} The modified URL.
	 */
	function addQueryParameter(url, param, value) {
		var urlObj = new URL(url);
		urlObj.searchParams.set(param, value);
		return urlObj.toString();
	}

})(jQuery);

