<?php
/**
 * Admin Setup Wizard
 *
 * Handles the initial setup wizard for new installations.
 * Guides users through essential configuration steps.
 *
 * This wizard is displayed within the Dashboard page when accessed with
 * the ?setup=1 parameter. This approach avoids hidden menu registration
 * issues that can occur in PHP 8.x environments with garbage collection.
 *
 * Access via: admin.php?page=content-guard-pro&setup=1
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Admin_Setup_Wizard
 *
 * Multi-step setup wizard showing:
 * - Step 1: Welcome & Overview
 * - Step 2: Scan Mode Selection
 * - Step 3: Alert Configuration
 * - Step 4: Schedule Setup
 * - Step 5: Complete & First Scan
 *
 * Per PRD Section 3.9: Setup wizard on first run (Scan Mode, Alerts, Schedule).
 * Skippable with sensible defaults.
 *
 * The wizard is now integrated into the Dashboard page (accessed via ?setup=1)
 * instead of having its own hidden menu page. This resolves PHP 8.x garbage
 * collection issues that could cause "Sorry, you are not allowed" errors.
 *
 * @since 1.0.0
 */
class CGP_Admin_Setup_Wizard {

	/**
	 * Current step in wizard.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	private $step = 1;

	/**
	 * Total number of steps.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	private $total_steps = 5;

	/**
	 * Initialize the setup wizard hooks.
	 *
	 * Sets up redirect logic and form processing.
	 * Called early in WordPress lifecycle.
	 *
	 * @since 1.0.0
	 */
	public static function init() {
		// Redirect to setup wizard on first activation.
		// Using admin_init ensures the Dashboard page is registered before redirect.
		add_action( 'admin_init', array( __CLASS__, 'setup_redirect' ), 5 );
		
		// Process wizard form submissions early (before any output).
		add_action( 'admin_init', array( __CLASS__, 'process_wizard_form' ), 10 );
	}

	/**
	 * Get the setup wizard URL.
	 *
	 * @since 1.0.2
	 * @param int $step Optional. Step number.
	 * @return string Setup wizard URL.
	 */
	public static function get_wizard_url( $step = 1 ) {
		return add_query_arg(
			array(
				'page'  => 'content-guard-pro',
				'setup' => '1',
				'step'  => absint( $step ),
			),
			admin_url( 'admin.php' )
		);
	}

	/**
	 * Redirect to setup wizard if needed.
	 *
	 * Called on admin_init hook to redirect new users to the setup wizard.
	 * The wizard is now part of the Dashboard page (accessed via ?setup=1).
	 *
	 * @since 1.0.0
	 */
	public static function setup_redirect() {
		// Check if headers were already sent (can't redirect if they were).
		if ( headers_sent() ) {
			return;
		}

		// Check if we should redirect to setup wizard.
		if ( ! get_transient( 'content_guard_pro_activation_redirect' ) ) {
			return;
		}

		// Don't redirect if doing AJAX, CLI, REST API, or cron.
		if ( wp_doing_ajax() || ( defined( 'WP_CLI' ) && WP_CLI ) || ( defined( 'REST_REQUEST' ) && REST_REQUEST ) || wp_doing_cron() ) {
			delete_transient( 'content_guard_pro_activation_redirect' );
			return;
		}

		// Don't redirect if user can't manage options.
		if ( ! current_user_can( 'manage_options' ) ) {
			delete_transient( 'content_guard_pro_activation_redirect' );
			return;
		}

		// Don't redirect if setup was already completed.
		if ( get_option( 'content_guard_pro_setup_completed', false ) ) {
			delete_transient( 'content_guard_pro_activation_redirect' );
			return;
		}

		// Don't redirect if this is a network admin or user admin page.
		if ( is_network_admin() || is_user_admin() ) {
			delete_transient( 'content_guard_pro_activation_redirect' );
			return;
		}

		// Don't redirect if we're already on the setup wizard (Dashboard with setup=1).
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( isset( $_GET['page'] ) && 'content-guard-pro' === $_GET['page'] && isset( $_GET['setup'] ) ) {
			delete_transient( 'content_guard_pro_activation_redirect' );
			return;
		}

		// Check if we're in a safe context to redirect.
		global $pagenow;
		
		// Don't redirect from certain pages that might break workflows.
		$unsafe_pages = array( 'post.php', 'post-new.php', 'profile.php', 'user-edit.php', 'options.php' );
		
		if ( in_array( $pagenow, $unsafe_pages, true ) ) {
			// Not a safe page to redirect from, keep the transient for next page load.
			return;
		}

		// Delete the transient so we don't redirect again.
		delete_transient( 'content_guard_pro_activation_redirect' );

		// Redirect to setup wizard (now part of Dashboard page).
		// This URL always works because the Dashboard page is always registered.
		wp_safe_redirect( self::get_wizard_url( 1 ) );
		exit;
	}

	/**
	 * Process wizard form submissions early.
	 *
	 * Handles form processing during admin_init (before any output)
	 * to prevent "headers already sent" errors.
	 *
	 * @since 1.0.0
	 */
	public static function process_wizard_form() {
		// Only process on the wizard page (Dashboard with setup=1).
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce checked below for POST actions
		if ( ! isset( $_GET['page'] ) || 'content-guard-pro' !== $_GET['page'] ) {
			return;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just checking URL parameter presence
		if ( ! isset( $_GET['setup'] ) || '1' !== $_GET['setup'] ) {
			return;
		}

		// Only process if form was submitted.
		if ( ! isset( $_POST['content_guard_pro_wizard_action'] ) ) {
			return;
		}

		// Check user capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Verify nonce.
		if ( ! isset( $_POST['content_guard_pro_wizard_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['content_guard_pro_wizard_nonce'] ) ), 'content_guard_pro_wizard_step' ) ) {
			wp_die( esc_html__( 'Security check failed', 'content-guard-pro' ) );
		}

		$wizard = new self();
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Step parameter is validated
		$wizard->step = isset( $_GET['step'] ) ? absint( wp_unslash( $_GET['step'] ) ) : 1;
		$wizard->step = max( 1, min( $wizard->total_steps, $wizard->step ) );
		$wizard->process_step();
	}

	/**
	 * Display the setup wizard.
	 *
	 * @since 1.0.0
	 */
	public static function display() {
		// Check user capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'content-guard-pro' ) );
		}

		$wizard = new self();
		$wizard->handle_step();
	}

	/**
	 * Handle current step.
	 *
	 * @since 1.0.0
	 */
	private function handle_step() {
		// Get current step from request.
		$this->step = isset( $_GET['step'] ) ? absint( wp_unslash( $_GET['step'] ) ) : 1;
		$this->step = max( 1, min( $this->total_steps, $this->step ) );

		// Display step.
		$this->display_header();

		switch ( $this->step ) {
			case 1:
				$this->display_step_welcome();
				break;
			case 2:
				$this->display_step_scan_mode();
				break;
			case 3:
				$this->display_step_alerts();
				break;
			case 4:
				$this->display_step_schedule();
				break;
			case 5:
				$this->display_step_complete();
				break;
		}

		$this->display_footer();
	}

	/**
	 * Process step submission.
	 *
	 * @since 1.0.0
	 */
	private function process_step() {
		// Verify nonce.
		if ( ! isset( $_POST['content_guard_pro_wizard_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['content_guard_pro_wizard_nonce'] ) ), 'content_guard_pro_wizard_step' ) ) {
			wp_die( esc_html__( 'Security check failed', 'content-guard-pro' ) );
		}

		$action = sanitize_text_field( wp_unslash( $_POST['content_guard_pro_wizard_action'] ) );

		// Handle skip.
		if ( 'skip' === $action ) {
			$this->complete_setup();
			return;
		}

		// Get current settings.
		$settings = get_option( 'content_guard_pro_settings', array() );

		// Process based on current step.
		switch ( $this->step ) {
			case 2:
				// Save scan mode.
				if ( isset( $_POST['scan_mode'] ) ) {
					$settings['scan_mode'] = sanitize_text_field( wp_unslash( $_POST['scan_mode'] ) );
				}
				if ( isset( $_POST['realtime_scan_enabled'] ) ) {
					$settings['realtime_scan_enabled'] = true;
				} else {
					$settings['realtime_scan_enabled'] = false;
				}
				break;

			case 3:
				// Save alert settings.
				$settings['email_alerts_enabled'] = ! empty( $_POST['email_alerts_enabled'] );
				if ( isset( $_POST['email_recipients'] ) ) {
					$settings['email_recipients'] = sanitize_textarea_field( wp_unslash( $_POST['email_recipients'] ) );
				}
				if ( isset( $_POST['email_alert_mode'] ) ) {
					$settings['email_alert_mode'] = sanitize_text_field( wp_unslash( $_POST['email_alert_mode'] ) );
				}
				$settings['notifications_admin'] = ! empty( $_POST['notifications_admin'] );
				break;

			case 4:
				// Save schedule settings.
				$settings['schedule_enabled'] = ! empty( $_POST['schedule_enabled'] );
				if ( isset( $_POST['schedule_frequency'] ) ) {
					$settings['schedule_frequency'] = sanitize_text_field( wp_unslash( $_POST['schedule_frequency'] ) );
				}
				if ( isset( $_POST['schedule_time'] ) ) {
					$settings['schedule_time'] = sanitize_text_field( wp_unslash( $_POST['schedule_time'] ) );
				}
				break;

			case 5:
				// Mark setup as complete.
				$this->complete_setup();
				return;
		}

		// Save settings.
		update_option( 'content_guard_pro_settings', $settings );

		// Redirect to next step (using new dashboard-based URL).
		$next_step = $this->step + 1;
		wp_safe_redirect( self::get_wizard_url( $next_step ) );
		exit;
	}

	/**
	 * Complete setup and redirect to dashboard.
	 *
	 * @since 1.0.0
	 */
	private function complete_setup() {
		// Mark setup as completed.
		update_option( 'content_guard_pro_setup_completed', true );

		// Trigger first scan if requested.
		if ( ! empty( $_POST['run_first_scan'] ) ) {
			// IMPORTANT: Create scan record IMMEDIATELY so it appears in Scan History.
			// Load scanner class to create scan record.
			if ( ! class_exists( 'CGP_Scanner' ) ) {
				require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-scanner.php';
			}
			
			global $wpdb;
			$table_name = $wpdb->prefix . 'content_guard_pro_scans';
			
			// Determine targets based on scan mode (always quick for setup wizard).
			$scan_mode = 'quick';
			$targets = 'posts';
			
			// Create scan record with 'pending' status.
			$wpdb->insert(
				$table_name,
				array(
					'mode'           => $scan_mode,
					'targets'        => $targets,
					'started_at'     => current_time( 'mysql' ),
					'finished_at'    => null,
					'status'         => 'pending',
					'throttle_state' => 'normal',
					'totals_checked' => 0,
					'totals_flagged' => 0,
					'avg_query_ms'   => 0,
					'peak_mem_mb'    => 0,
					'errors'         => 0,
					'notes'          => wp_json_encode( array(
						'initiated_by' => get_current_user_id(),
						'scan_type'    => 'setup_wizard',
						'source'       => 'setup_wizard',
					) ),
				),
				array( '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%d', '%d', '%d', '%d', '%s' )
			);
			
			$scan_id = $wpdb->insert_id;
			
			if ( $scan_id ) {
				// Schedule the scan with the scan_id.
				if ( class_exists( 'CGP_Scheduler' ) ) {
					CGP_Scheduler::schedule_scan( $scan_mode, array(
						'scan_id'      => $scan_id,
						'source'       => 'setup_wizard',
						'scan_type'    => 'setup_wizard',
						'initiated_by' => get_current_user_id(),
					) );
				}
			}

			// Redirect to scans page.
			wp_safe_redirect( admin_url( 'admin.php?page=content-guard-pro-scans&scan_initiated=1' ) );
		} else {
			// Redirect to main dashboard.
			wp_safe_redirect( admin_url( 'admin.php?page=content-guard-pro' ) );
		}
		exit;
	}

	/**
	 * Display wizard header.
	 *
	 * @since 1.0.0
	 */
	private function display_header() {
		?>
		<div class="wrap content-guard-pro-setup-wizard">
			<h1><?php esc_html_e( 'Content Guard Pro Setup', 'content-guard-pro' ); ?></h1>

			<!-- Progress Bar -->
			<div class="content-guard-pro-wizard-progress">
				<div class="content-guard-pro-wizard-progress-bar">
					<div class="content-guard-pro-wizard-progress-fill" style="width: <?php echo esc_attr( ( $this->step / $this->total_steps ) * 100 ); ?>%;"></div>
				</div>
				<div class="content-guard-pro-progress-text">
					<?php
					printf(
						/* translators: 1: current step, 2: total steps */
						esc_html__( 'Step %1$d of %2$d', 'content-guard-pro' ),
						absint( $this->step ),
						absint( $this->total_steps )
					);
					?>
				</div>
			</div>

			<div class="content-guard-pro-wizard-content">
		<?php
	}

	/**
	 * Check if a feature is available in the current license tier.
	 *
	 * @since 1.0.0
	 * @param string $feature Feature identifier.
	 * @return bool True if available, false otherwise.
	 */
	private function is_feature_available( $feature ) {
		if ( class_exists( 'CGP_License_Manager' ) ) {
			return CGP_License_Manager::can( $feature );
		}
		// Default to free tier if license manager not available.
		return in_array( $feature, array( 'quick_scan', 'manual_scan', 'admin_notices', 'admin_bar', 'dashboard_widget' ), true );
	}

	/**
	 * Get the current license tier display name.
	 *
	 * @since 1.0.0
	 * @return string Tier name (Free, Solo Guard, Agency, etc.).
	 */
	private function get_license_tier_name() {
		if ( class_exists( 'CGP_License_Manager' ) ) {
			return CGP_License_Manager::get_tier_name();
		}
		return 'Free';
	}

	/**
	 * Display wizard footer.
	 *
	 * @since 1.0.0
	 */
	private function display_footer() {
		?>
			</div><!-- .content-guard-pro-wizard-content -->
		</div><!-- .content-guard-pro-setup-wizard -->

		<style>
		.content-guard-pro-setup-wizard {
			max-width: 800px;
			margin: 40px auto;
		}
		.content-guard-pro-wizard-progress {
			margin: 30px 0;
		}
		.content-guard-pro-wizard-progress-bar {
			height: 10px;
			background: #e0e0e0;
			border-radius: 5px;
			overflow: hidden;
		}
		.content-guard-pro-wizard-progress-fill {
			height: 100%;
			background: #2271b1;
			transition: width 0.3s ease;
		}
		.content-guard-pro-progress-text {
			text-align: center;
			margin-top: 10px;
			color: #646970;
			font-size: 14px;
		}
		.content-guard-pro-wizard-content {
			background: #fff;
			border: 1px solid #ccd0d4;
			border-radius: 4px;
			padding: 40px;
			box-shadow: 0 1px 1px rgba(0,0,0,.04);
		}
		.content-guard-pro-wizard-step {
			min-height: 400px;
		}
		.content-guard-pro-wizard-step h2 {
			margin-top: 0;
			color: #1d2327;
		}
		.content-guard-pro-step-icon {
			text-align: center;
			font-size: 60px;
			margin-bottom: 20px;
		}
		.content-guard-pro-option-card {
			display: block;
			box-sizing: border-box;
			border-radius: 4px;
			padding: 20px;
			margin-bottom: 15px;
			cursor: pointer;
			transition: all 0.2s ease;
			border: 2px solid #dcdcde;
		}
		.content-guard-pro-option-card.locked {
			background: #f6f7f7;
			border-color: #c3c4c7;
			cursor: not-allowed;
			opacity: 0.7;
		}
		.content-guard-pro-option-header {
			display: flex;
			align-items: center;
			margin-bottom: 10px;
			gap: 10px;
		}
		.content-guard-pro-option-header input[type="radio"],
		.content-guard-pro-option-header input[type="checkbox"] {
			margin-right: 0;
			margin-top: 0;
		}
		.content-guard-pro-option-header h3 {
			margin: 0;
			flex: 1;
		}
		.content-guard-pro-option-card p {
			margin: 10px 0 0 28px;
			color: #646970;
		}
		.cgp-tier-badge {
			display: inline-block;
			padding: 2px 8px;
			border-radius: 3px;
			font-size: 11px;
			font-weight: 600;
			text-transform: uppercase;
			margin-left: 8px;
		}
		.cgp-tier-badge.free {
			background: #00a32a;
			color: #fff;
		}
		.cgp-tier-badge.solo {
			background: #2271b1;
			color: #fff;
		}
		.cgp-tier-badge.agency {
			background: #8c5ee8;
			color: #fff;
		}
		.cgp-premium-feature {
			position: relative;
			padding: 15px;
			background: #f6f7f7;
			border: 2px solid #c3c4c7;
			border-radius: 4px;
			margin: 15px 0;
		}
		.cgp-premium-feature.locked::before {
			content: "🔒";
			position: absolute;
			top: 15px;
			right: 15px;
			font-size: 20px;
			opacity: 0.5;
		}
		.cgp-premium-feature label {
			opacity: 0.6;
			cursor: not-allowed;
		}
		.cgp-premium-feature input,
		.cgp-premium-feature select,
		.cgp-premium-feature textarea {
			opacity: 0.5;
			cursor: not-allowed;
		}
		.cgp-upgrade-prompt {
			margin-top: 10px;
			padding: 10px;
			background: #fff;
			border-left: 4px solid #2271b1;
			font-size: 13px;
		}
		.cgp-upgrade-prompt .button-upgrade {
			margin-left: 10px;
			background: #2271b1;
			color: #fff;
			border-color: #2271b1;
			text-decoration: none;
			padding: 4px 12px;
			border-radius: 3px;
			font-size: 12px;
			display: inline-block;
		}
		.cgp-upgrade-prompt .button-upgrade:hover {
			background: #135e96;
			border-color: #135e96;
		}
		.cgp-feature-matrix {
			margin: 20px 0;
			border: 1px solid #dcdcde;
			border-radius: 4px;
			overflow: hidden;
		}
		.cgp-feature-matrix table {
			width: 100%;
			border-collapse: collapse;
		}
		.cgp-feature-matrix th {
			background: #f6f7f7;
			padding: 12px;
			text-align: left;
			font-weight: 600;
			border-bottom: 2px solid #dcdcde;
		}
		.cgp-feature-matrix td {
			padding: 10px 12px;
			border-bottom: 1px solid #f0f0f1;
		}
		.cgp-feature-matrix tr:last-child td {
			border-bottom: none;
		}
		.cgp-feature-matrix .check {
			color: #00a32a;
			font-weight: bold;
		}
		.cgp-feature-matrix .dash {
			color: #c3c4c7;
		}
		.content-guard-pro-wizard-actions {
			margin-top: 30px;
			padding-top: 20px;
			border-top: 1px solid #dcdcde;
			display: flex;
			justify-content: space-between;
			align-items: center;
		}
		.content-guard-pro-wizard-actions .button-primary {
			font-size: 16px;
			padding: 8px 24px;
			height: auto;
		}
		.content-guard-pro-feature-list {
			list-style: none;
			padding: 0;
		}
		.content-guard-pro-feature-list li {
			padding: 10px 0;
			padding-left: 30px;
			position: relative;
		}
		.content-guard-pro-feature-list li:before {
			content: "\2713"; /* Check mark */
			position: absolute;
			left: 0;
			color: #00a32a;
			font-weight: bold;
			font-size: 18px;
		}
		</style>
		<?php
	}

	/**
	 * Display Step 1: Welcome.
	 *
	 * @since 1.0.0
	 */
	private function display_step_welcome() {
		$current_tier = $this->get_license_tier_name();
		?>
		<div class="content-guard-pro-wizard-step">
			<div class="content-guard-pro-step-icon">🛡️</div>
			
			<h2><?php esc_html_e( 'Welcome to Content Guard Pro!', 'content-guard-pro' ); ?></h2>
			
			<p class="description" style="font-size: 16px;">
				<?php esc_html_e( 'Thank you for installing Content Guard Pro! This quick setup wizard will help you configure the essential settings to protect your WordPress site from malicious content, spam links, and SEO injections.', 'content-guard-pro' ); ?>
			</p>

			<p style="margin: 20px 0; padding: 12px; background: #f0f6fc; border-left: 4px solid #2271b1; font-size: 14px;">
				<strong><?php
				/* translators: %s: Current plan tier name */
				printf( esc_html__( 'Current Plan: %s', 'content-guard-pro' ), esc_html( $current_tier ) ); ?></strong>
				<?php if ( 'Free' === $current_tier ) : ?>
					<br><?php esc_html_e( 'Upgrade to unlock advanced features like quarantine, scheduled scans, and email alerts.', 'content-guard-pro' ); ?>
				<?php endif; ?>
			</p>

			<h3><?php esc_html_e( 'Feature Comparison', 'content-guard-pro' ); ?></h3>
			<div class="cgp-feature-matrix">
				<table>
					<thead>
						<tr>
							<th><?php esc_html_e( 'Feature', 'content-guard-pro' ); ?></th>
							<th style="text-align: center;"><?php esc_html_e( 'Free', 'content-guard-pro' ); ?></th>
							<th style="text-align: center;"><?php esc_html_e( 'Solo Guard', 'content-guard-pro' ); ?></th>
							<th style="text-align: center;"><?php esc_html_e( 'Agency', 'content-guard-pro' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<tr>
							<td><?php esc_html_e( 'Quick Scan', 'content-guard-pro' ); ?></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
						</tr>
						<tr>
							<td><?php esc_html_e( 'Single Post Scanning', 'content-guard-pro' ); ?></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
						</tr>
						<tr>
							<td><?php esc_html_e( 'Standard Scan (Full Database)', 'content-guard-pro' ); ?></td>
							<td style="text-align: center;"><span class="dash">—</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
						</tr>
						<tr>
							<td><?php esc_html_e( 'Quarantine Threats', 'content-guard-pro' ); ?></td>
							<td style="text-align: center;"><span class="dash">—</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
						</tr>
						<tr>
							<td><?php esc_html_e( 'Scheduled Scans', 'content-guard-pro' ); ?></td>
							<td style="text-align: center;"><span class="dash">—</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
						</tr>
						<tr>
							<td><?php esc_html_e( 'Email Alerts', 'content-guard-pro' ); ?></td>
							<td style="text-align: center;"><span class="dash">—</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
						</tr>
						<tr>
							<td><?php esc_html_e( 'CSV/JSON Export & API', 'content-guard-pro' ); ?></td>
							<td style="text-align: center;"><span class="dash">—</span></td>
							<td style="text-align: center;"><span class="dash">—</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
						</tr>
						<tr>
							<td><?php esc_html_e( 'Webhook Notifications', 'content-guard-pro' ); ?></td>
							<td style="text-align: center;"><span class="dash">—</span></td>
							<td style="text-align: center;"><span class="dash">—</span></td>
							<td style="text-align: center;"><span class="check">✓</span></td>
						</tr>
					</tbody>
				</table>
			</div>

			<p style="margin-top: 20px; text-align: center;">
				<a href="https://contentguardpro.com/pricing/" target="_blank" rel="noopener noreferrer" class="button">
					<?php esc_html_e( 'View Full Pricing →', 'content-guard-pro' ); ?>
				</a>
			</p>

			<p style="margin-top: 30px; padding: 15px; background: #f0f6fc; border-left: 4px solid #2271b1;">
				<strong><?php esc_html_e( 'Setup takes about 2 minutes', 'content-guard-pro' ); ?></strong><br>
				<?php esc_html_e( 'During setup, you\'ll see all available features. Premium features will be clearly marked and can be configured when you upgrade.', 'content-guard-pro' ); ?>
			</p>

			<form method="post">
				<?php wp_nonce_field( 'content_guard_pro_wizard_step', 'content_guard_pro_wizard_nonce' ); ?>
				<input type="hidden" name="content_guard_pro_wizard_action" value="next" />
				
				<div class="content-guard-pro-wizard-actions">
					<button type="submit" name="content_guard_pro_wizard_action" value="skip" class="button">
						<?php esc_html_e( 'Skip Setup', 'content-guard-pro' ); ?>
					</button>
					<button type="submit" class="button button-primary button-hero">
						<?php esc_html_e( 'Get Started →', 'content-guard-pro' ); ?>
					</button>
				</div>
			</form>
		</div>
		<?php
	}

	/**
	 * Display Step 2: Scan Mode.
	 *
	 * @since 1.0.0
	 */
	private function display_step_scan_mode() {
		$settings = get_option( 'content_guard_pro_settings', array() );
		$scan_mode = isset( $settings['scan_mode'] ) ? $settings['scan_mode'] : 'standard';
		$realtime = isset( $settings['realtime_scan_enabled'] ) ? $settings['realtime_scan_enabled'] : true;
		
		$has_standard_scan = $this->is_feature_available( 'standard_scan' );
		$has_realtime_scan = $this->is_feature_available( 'realtime_scan' );
		
		// Free tier defaults to quick
		if ( ! $has_standard_scan ) {
			$scan_mode = 'quick';
		}
		?>
		<div class="content-guard-pro-wizard-step">
			<h2><?php esc_html_e( 'Choose Your Scan Mode', 'content-guard-pro' ); ?></h2>
			
			<p class="description">
				<?php esc_html_e( 'Select how thoroughly Content Guard Pro should scan your site. You can change this later in Settings.', 'content-guard-pro' ); ?>
			</p>

			<form method="post">
				<?php wp_nonce_field( 'content_guard_pro_wizard_step', 'content_guard_pro_wizard_nonce' ); ?>
				<input type="hidden" name="content_guard_pro_wizard_action" value="next" />

				<div style="margin: 30px 0;">
					<!-- Quick Scan - Always Available -->
					<label class="content-guard-pro-option-card" onclick="this.querySelector('input').checked = true;">
						<div class="content-guard-pro-option-header">
							<input type="radio" name="scan_mode" value="quick" <?php checked( $scan_mode, 'quick' ); ?> />
							<h3>⚡ <?php esc_html_e( 'Quick Scan', 'content-guard-pro' ); ?></h3>
							<span class="cgp-tier-badge free"><?php esc_html_e( 'FREE', 'content-guard-pro' ); ?></span>
						</div>
						<p><?php esc_html_e( 'Scans posts and pages only. Faster and good for basic protection.', 'content-guard-pro' ); ?></p>
						<p><strong><?php esc_html_e( 'Best for:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Blogs, business sites, portfolios', 'content-guard-pro' ); ?></p>
					</label>

					<!-- Standard Scan - Solo Guard+ -->
					<?php if ( $has_standard_scan ) : ?>
						<label class="content-guard-pro-option-card" onclick="this.querySelector('input').checked = true;">
							<div class="content-guard-pro-option-header">
								<input type="radio" name="scan_mode" value="standard" <?php checked( $scan_mode, 'standard' ); ?> />
								<h3>🔍 <?php esc_html_e( 'Standard Scan (Recommended)', 'content-guard-pro' ); ?></h3>
								<span class="cgp-tier-badge solo"><?php esc_html_e( 'SOLO GUARD', 'content-guard-pro' ); ?></span>
							</div>
							<p><?php esc_html_e( 'Scans posts, pages, custom fields, and widgets. More thorough protection.', 'content-guard-pro' ); ?></p>
							<p><strong><?php esc_html_e( 'Best for:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'E-commerce, membership sites, large sites', 'content-guard-pro' ); ?></p>
						</label>
					<?php else : ?>
						<div class="content-guard-pro-option-card locked">
							<div class="content-guard-pro-option-header">
								<input type="radio" name="scan_mode" value="standard" disabled />
								<h3>🔍 <?php esc_html_e( 'Standard Scan (Recommended)', 'content-guard-pro' ); ?></h3>
								<span class="cgp-tier-badge solo"><?php esc_html_e( 'SOLO GUARD', 'content-guard-pro' ); ?></span>
							</div>
							<p><?php esc_html_e( 'Scans posts, pages, custom fields, and widgets. More thorough protection.', 'content-guard-pro' ); ?></p>
							<p><strong><?php esc_html_e( 'Best for:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'E-commerce, membership sites, large sites', 'content-guard-pro' ); ?></p>
							<div class="cgp-upgrade-prompt">
								<?php esc_html_e( 'Unlock full database scanning with Solo Guard for comprehensive protection.', 'content-guard-pro' ); ?>
								<a href="https://contentguardpro.com/pricing/" target="_blank" rel="noopener noreferrer" class="button-upgrade">
									<?php esc_html_e( 'Upgrade to Solo Guard', 'content-guard-pro' ); ?>
								</a>
							</div>
						</div>
					<?php endif; ?>
				</div>

				<!-- Real-time Protection - Solo Guard+ -->
				<div style="margin: 30px 0;">
					<h3>
						<?php esc_html_e( 'Real-time Protection', 'content-guard-pro' ); ?>
						<?php if ( ! $has_realtime_scan ) : ?>
							<span class="cgp-tier-badge solo"><?php esc_html_e( 'SOLO GUARD', 'content-guard-pro' ); ?></span>
						<?php endif; ?>
					</h3>
					
					<?php if ( $has_realtime_scan ) : ?>
						<label>
							<input type="checkbox" name="realtime_scan_enabled" value="1" <?php checked( $realtime, true ); ?> />
							<?php esc_html_e( 'Scan content automatically when posts are saved or published', 'content-guard-pro' ); ?>
						</label>
						<p class="description" style="margin-left: 25px;">
							<?php esc_html_e( 'Recommended: Catches threats immediately without waiting for scheduled scans.', 'content-guard-pro' ); ?>
						</p>
					<?php else : ?>
						<div class="cgp-premium-feature locked">
							<label>
								<input type="checkbox" name="realtime_scan_enabled" value="1" disabled />
								<?php esc_html_e( 'Scan content automatically when posts are saved or published', 'content-guard-pro' ); ?>
							</label>
							<p class="description" style="margin-left: 25px;">
								<?php esc_html_e( 'Catches threats immediately without waiting for scheduled scans.', 'content-guard-pro' ); ?>
							</p>
							<div class="cgp-upgrade-prompt">
								<?php esc_html_e( 'Real-time scanning available in Solo Guard. Automatically scans content as you save.', 'content-guard-pro' ); ?>
								<a href="https://contentguardpro.com/pricing/" target="_blank" rel="noopener noreferrer" class="button-upgrade">
									<?php esc_html_e( 'Upgrade Now', 'content-guard-pro' ); ?>
								</a>
							</div>
						</div>
					<?php endif; ?>
				</div>

				<div class="content-guard-pro-wizard-actions">
					<a href="<?php echo esc_url( self::get_wizard_url( 1 ) ); ?>" class="button">
						<?php esc_html_e( '← Back', 'content-guard-pro' ); ?>
					</a>
					<button type="submit" class="button button-primary button-hero">
						<?php esc_html_e( 'Continue →', 'content-guard-pro' ); ?>
					</button>
				</div>
			</form>
		</div>
		<?php
	}

	/**
	 * Display Step 3: Alerts.
	 *
	 * @since 1.0.0
	 */
	private function display_step_alerts() {
		$settings = get_option( 'content_guard_pro_settings', array() );
		$email_enabled = isset( $settings['email_alerts_enabled'] ) ? $settings['email_alerts_enabled'] : true;
		$email_recipients = isset( $settings['email_recipients'] ) ? $settings['email_recipients'] : get_option( 'admin_email' );
		$email_mode = isset( $settings['email_alert_mode'] ) ? $settings['email_alert_mode'] : 'immediate';
		$admin_notices = isset( $settings['notifications_admin'] ) ? $settings['notifications_admin'] : true;
		$webhook_enabled = isset( $settings['webhook_enabled'] ) ? $settings['webhook_enabled'] : false;
		$webhook_url = isset( $settings['webhook_url'] ) ? $settings['webhook_url'] : '';
		
		$has_email_alerts = $this->is_feature_available( 'email_alerts' );
		$has_webhooks = $this->is_feature_available( 'webhook_notifications' );
		?>
		<div class="content-guard-pro-wizard-step">
			<h2><?php esc_html_e( 'Configure Alerts', 'content-guard-pro' ); ?></h2>
			
			<p class="description">
				<?php esc_html_e( 'Choose how you want to be notified when critical security issues are found.', 'content-guard-pro' ); ?>
			</p>

			<form method="post">
				<?php wp_nonce_field( 'content_guard_pro_wizard_step', 'content_guard_pro_wizard_nonce' ); ?>
				<input type="hidden" name="content_guard_pro_wizard_action" value="next" />

				<!-- Admin Notices - Always Free -->
				<div style="margin: 30px 0;">
					<h3>
						<?php esc_html_e( 'Admin Notifications', 'content-guard-pro' ); ?>
						<span class="cgp-tier-badge free"><?php esc_html_e( 'FREE', 'content-guard-pro' ); ?></span>
					</h3>
					
					<p>
						<label>
							<input type="checkbox" name="notifications_admin" value="1" <?php checked( $admin_notices, true ); ?> />
							<?php esc_html_e( 'Show admin notices in WordPress dashboard for critical findings', 'content-guard-pro' ); ?>
						</label>
					</p>
					<p class="description" style="margin-left: 25px;">
						<?php esc_html_e( 'Displays alerts in your WordPress admin area when threats are detected.', 'content-guard-pro' ); ?>
					</p>
				</div>

				<!-- Email Alerts - Solo Guard+ -->
				<div style="margin: 30px 0;">
					<h3>
						<?php esc_html_e( 'Email Alerts', 'content-guard-pro' ); ?>
						<span class="cgp-tier-badge solo"><?php esc_html_e( 'SOLO GUARD', 'content-guard-pro' ); ?></span>
					</h3>
					
					<?php if ( $has_email_alerts ) : ?>
						<p>
							<label>
								<input type="checkbox" name="email_alerts_enabled" value="1" <?php checked( $email_enabled, true ); ?> />
								<strong><?php esc_html_e( 'Send email alerts for critical findings', 'content-guard-pro' ); ?></strong>
							</label>
						</p>

						<p>
							<label>
								<?php esc_html_e( 'Email Recipients:', 'content-guard-pro' ); ?><br>
								<textarea name="email_recipients" rows="3" class="large-text" placeholder="<?php esc_attr_e( 'admin@example.com', 'content-guard-pro' ); ?>"><?php echo esc_textarea( $email_recipients ); ?></textarea>
							</label>
							<span class="description"><?php esc_html_e( 'One email address per line', 'content-guard-pro' ); ?></span>
						</p>

						<p>
							<label>
								<?php esc_html_e( 'Alert Mode:', 'content-guard-pro' ); ?><br>
								<select name="email_alert_mode" class="regular-text">
									<option value="immediate" <?php selected( $email_mode, 'immediate' ); ?>><?php esc_html_e( 'Immediate (send email for each finding)', 'content-guard-pro' ); ?></option>
									<option value="digest" <?php selected( $email_mode, 'digest' ); ?>><?php esc_html_e( 'Daily Digest (one email per day)', 'content-guard-pro' ); ?></option>
								</select>
							</label>
						</p>
					<?php else : ?>
						<div class="cgp-premium-feature locked">
							<p>
								<label>
									<input type="checkbox" disabled />
									<strong><?php esc_html_e( 'Send email alerts for critical findings', 'content-guard-pro' ); ?></strong>
								</label>
							</p>
							<p class="description" style="margin-left: 25px;">
								<?php esc_html_e( 'Get instant email notifications whenever critical security threats are detected on your site.', 'content-guard-pro' ); ?>
							</p>
							<div class="cgp-upgrade-prompt">
								<?php esc_html_e( 'Email alerts help you respond quickly to threats. Available in Solo Guard.', 'content-guard-pro' ); ?>
								<a href="https://contentguardpro.com/pricing/" target="_blank" rel="noopener noreferrer" class="button-upgrade">
									<?php esc_html_e( 'Upgrade to Solo Guard', 'content-guard-pro' ); ?>
								</a>
							</div>
						</div>
					<?php endif; ?>
				</div>

				<!-- Webhook Notifications - Agency+ -->
				<div style="margin: 30px 0;">
					<h3>
						<?php esc_html_e( 'Webhook Notifications', 'content-guard-pro' ); ?>
						<span class="cgp-tier-badge agency"><?php esc_html_e( 'AGENCY', 'content-guard-pro' ); ?></span>
					</h3>
					
					<?php if ( $has_webhooks ) : ?>
						<p>
							<label>
								<input type="checkbox" name="webhook_enabled" value="1" <?php checked( $webhook_enabled, true ); ?> />
								<strong><?php esc_html_e( 'Send webhook notifications for critical findings', 'content-guard-pro' ); ?></strong>
							</label>
						</p>

						<p>
							<label>
								<?php esc_html_e( 'Webhook URL:', 'content-guard-pro' ); ?><br>
								<input type="url" name="webhook_url" value="<?php echo esc_attr( $webhook_url ); ?>" class="large-text" placeholder="https://your-monitoring-tool.com/webhook" />
							</label>
							<span class="description"><?php esc_html_e( 'Integrate with Slack, Discord, PagerDuty, or custom monitoring tools', 'content-guard-pro' ); ?></span>
						</p>
					<?php else : ?>
						<div class="cgp-premium-feature locked">
							<p>
								<label>
									<input type="checkbox" disabled />
									<strong><?php esc_html_e( 'Send webhook notifications for critical findings', 'content-guard-pro' ); ?></strong>
								</label>
							</p>
							<p class="description" style="margin-left: 25px;">
								<?php esc_html_e( 'Integrate with Slack, Discord, PagerDuty, or custom monitoring tools to receive real-time alerts.', 'content-guard-pro' ); ?>
							</p>
							<div class="cgp-upgrade-prompt">
								<?php esc_html_e( 'Webhooks are perfect for agencies managing multiple sites. Available in Agency plans.', 'content-guard-pro' ); ?>
								<a href="https://contentguardpro.com/pricing/" target="_blank" rel="noopener noreferrer" class="button-upgrade">
									<?php esc_html_e( 'Upgrade to Agency', 'content-guard-pro' ); ?>
								</a>
							</div>
						</div>
					<?php endif; ?>
				</div>

				<div class="content-guard-pro-wizard-actions">
					<a href="<?php echo esc_url( self::get_wizard_url( 2 ) ); ?>" class="button">
						<?php esc_html_e( '← Back', 'content-guard-pro' ); ?>
					</a>
					<button type="submit" class="button button-primary button-hero">
						<?php esc_html_e( 'Continue →', 'content-guard-pro' ); ?>
					</button>
				</div>
			</form>
		</div>
		<?php
	}

	/**
	 * Display Step 4: Schedule.
	 *
	 * @since 1.0.0
	 */
	private function display_step_schedule() {
		$settings = get_option( 'content_guard_pro_settings', array() );
		$schedule_enabled = isset( $settings['schedule_enabled'] ) ? $settings['schedule_enabled'] : false;
		$schedule_frequency = isset( $settings['schedule_frequency'] ) ? $settings['schedule_frequency'] : 'daily';
		$schedule_time = isset( $settings['schedule_time'] ) ? $settings['schedule_time'] : '03:00';
		
		$has_scheduled_scans = $this->is_feature_available( 'scheduled_scans' );
		?>
		<div class="content-guard-pro-wizard-step">
			<h2>
				<?php esc_html_e( 'Schedule Automatic Scans', 'content-guard-pro' ); ?>
				<span class="cgp-tier-badge solo"><?php esc_html_e( 'SOLO GUARD', 'content-guard-pro' ); ?></span>
			</h2>
			
			<p class="description">
				<?php esc_html_e( 'Set up automatic scans to continuously monitor your site for threats.', 'content-guard-pro' ); ?>
			</p>

			<form method="post">
				<?php wp_nonce_field( 'content_guard_pro_wizard_step', 'content_guard_pro_wizard_nonce' ); ?>
				<input type="hidden" name="content_guard_pro_wizard_action" value="next" />

				<?php if ( $has_scheduled_scans ) : ?>
					<div style="margin: 30px 0;">
						<p>
							<label>
								<input type="checkbox" name="schedule_enabled" value="1" <?php checked( $schedule_enabled, true ); ?> />
								<strong><?php esc_html_e( 'Enable automatic scheduled scans', 'content-guard-pro' ); ?></strong>
							</label>
						</p>

						<p style="margin-left: 25px;">
							<label>
								<?php esc_html_e( 'Scan Frequency:', 'content-guard-pro' ); ?><br>
								<select name="schedule_frequency" class="regular-text">
									<option value="daily" <?php selected( $schedule_frequency, 'daily' ); ?>><?php esc_html_e( 'Daily', 'content-guard-pro' ); ?></option>
									<option value="twicedaily" <?php selected( $schedule_frequency, 'twicedaily' ); ?>><?php esc_html_e( 'Twice Daily', 'content-guard-pro' ); ?></option>
									<option value="weekly" <?php selected( $schedule_frequency, 'weekly' ); ?>><?php esc_html_e( 'Weekly', 'content-guard-pro' ); ?></option>
								</select>
							</label>
						</p>

						<p style="margin-left: 25px;">
							<label>
								<?php esc_html_e( 'Scan Time:', 'content-guard-pro' ); ?><br>
								<input type="time" name="schedule_time" value="<?php echo esc_attr( $schedule_time ); ?>" class="regular-text" />
							</label>
							<span class="description"><?php esc_html_e( 'Recommended: 03:00 (3 AM) for minimal impact', 'content-guard-pro' ); ?></span>
						</p>
					</div>

					<div style="padding: 15px; background: #f0f6fc; border-left: 4px solid #2271b1; margin: 30px 0;">
						<strong><?php esc_html_e( 'Recommendation:', 'content-guard-pro' ); ?></strong><br>
						<?php esc_html_e( 'Enable daily scans at night (3 AM) for continuous protection without impacting site performance during peak hours.', 'content-guard-pro' ); ?>
					</div>
				<?php else : ?>
					<div class="cgp-premium-feature locked" style="margin: 30px 0;">
						<p>
							<label>
								<input type="checkbox" disabled />
								<strong><?php esc_html_e( 'Enable automatic scheduled scans', 'content-guard-pro' ); ?></strong>
							</label>
						</p>

						<p class="description" style="margin-left: 25px;">
							<?php esc_html_e( 'Automatically scan your site daily, twice daily, or weekly. Runs in the background without impacting site performance.', 'content-guard-pro' ); ?>
						</p>

						<div style="margin: 20px 0 20px 25px; padding: 15px; background: #fff; border: 1px solid #dcdcde; border-radius: 4px;">
							<h4 style="margin-top: 0;"><?php esc_html_e( 'Why Scheduled Scans?', 'content-guard-pro' ); ?></h4>
							<ul style="margin: 10px 0;">
								<li><?php esc_html_e( '✓ Continuous protection - detect threats before they cause damage', 'content-guard-pro' ); ?></li>
								<li><?php esc_html_e( '✓ Automated monitoring - no manual intervention required', 'content-guard-pro' ); ?></li>
								<li><?php esc_html_e( '✓ Catch backdoor injections that bypass real-time scanning', 'content-guard-pro' ); ?></li>
								<li><?php esc_html_e( '✓ Peace of mind - your site is always monitored', 'content-guard-pro' ); ?></li>
							</ul>
						</div>

						<div class="cgp-upgrade-prompt">
							<?php esc_html_e( 'Scheduled scans provide continuous protection for your site. Available in Solo Guard.', 'content-guard-pro' ); ?>
							<a href="https://contentguardpro.com/pricing/" target="_blank" rel="noopener noreferrer" class="button-upgrade">
								<?php esc_html_e( 'Upgrade to Solo Guard - $69/year', 'content-guard-pro' ); ?>
							</a>
						</div>
					</div>

					<div style="padding: 15px; background: #fffbcc; border-left: 4px solid #dba617; margin: 30px 0;">
						<strong><?php esc_html_e( 'Free Plan:', 'content-guard-pro' ); ?></strong><br>
						<?php esc_html_e( 'You can run manual scans once per day. Upgrade to Solo Guard for automatic daily scans and real-time protection.', 'content-guard-pro' ); ?>
					</div>
				<?php endif; ?>

				<div class="content-guard-pro-wizard-actions">
					<a href="<?php echo esc_url( self::get_wizard_url( 3 ) ); ?>" class="button">
						<?php esc_html_e( '← Back', 'content-guard-pro' ); ?>
					</a>
					<button type="submit" class="button button-primary button-hero">
						<?php esc_html_e( 'Continue →', 'content-guard-pro' ); ?>
					</button>
				</div>
			</form>
		</div>
		<?php
	}

	/**
	 * Display Step 5: Complete.
	 *
	 * @since 1.0.0
	 */
	private function display_step_complete() {
		$settings = get_option( 'content_guard_pro_settings', array() );
		$current_tier = $this->get_license_tier_name();
		$is_free = ( 'Free' === $current_tier );
		
		// Ensure scan_mode has a default value for display.
		$scan_mode = isset( $settings['scan_mode'] ) ? $settings['scan_mode'] : 'quick';
		
		// Count what's enabled
		$enabled_features = array();
		if ( ! empty( $settings['realtime_scan_enabled'] ) ) {
			$enabled_features[] = __( 'Real-time Protection', 'content-guard-pro' );
		}
		if ( ! empty( $settings['email_alerts_enabled'] ) ) {
			$enabled_features[] = __( 'Email Alerts', 'content-guard-pro' );
		}
		if ( ! empty( $settings['schedule_enabled'] ) ) {
			$enabled_features[] = __( 'Scheduled Scans', 'content-guard-pro' );
		}
		?>
		<div class="content-guard-pro-wizard-step">
			<div class="content-guard-pro-step-icon">🎉</div>
			
			<h2><?php esc_html_e( 'Setup Complete!', 'content-guard-pro' ); ?></h2>
			
			<p style="font-size: 16px;">
				<?php esc_html_e( 'Content Guard Pro is now configured and ready to protect your WordPress site.', 'content-guard-pro' ); ?>
			</p>

			<div style="margin: 30px 0; padding: 20px; background: #f6f7f7; border-radius: 4px;">
				<h3><?php esc_html_e( 'Your Configuration:', 'content-guard-pro' ); ?></h3>
				<ul style="list-style: none; padding: 0;">
					<li style="padding: 8px 0;">
						<strong><?php esc_html_e( 'Plan:', 'content-guard-pro' ); ?></strong>
						<?php echo esc_html( $current_tier ); ?>
					</li>
					<li style="padding: 8px 0;">
						<strong><?php esc_html_e( 'Scan Mode:', 'content-guard-pro' ); ?></strong>
						<?php echo esc_html( ucfirst( $scan_mode ) ); ?>
						<?php if ( 'quick' === $scan_mode && $is_free ) : ?>
							<span style="color: #646970; font-size: 12px;">(<?php esc_html_e( 'Standard scan available in Solo Guard', 'content-guard-pro' ); ?>)</span>
						<?php endif; ?>
					</li>
					<li style="padding: 8px 0;">
						<strong><?php esc_html_e( 'Admin Notifications:', 'content-guard-pro' ); ?></strong>
						<?php echo ! empty( $settings['notifications_admin'] ) ? esc_html__( 'Enabled', 'content-guard-pro' ) : esc_html__( 'Disabled', 'content-guard-pro' ); ?>
					</li>
					<?php if ( ! empty( $enabled_features ) ) : ?>
						<li style="padding: 8px 0;">
							<strong><?php esc_html_e( 'Premium Features:', 'content-guard-pro' ); ?></strong>
							<?php echo esc_html( implode( ', ', $enabled_features ) ); ?>
						</li>
					<?php endif; ?>
				</ul>
			</div>

			<?php if ( $is_free ) : ?>
				<div style="margin: 30px 0; padding: 20px; background: #fff; border: 2px solid #2271b1; border-radius: 4px;">
					<h3 style="margin-top: 0; color: #2271b1;"><?php esc_html_e( '🚀 Unlock More Protection with Solo Guard', 'content-guard-pro' ); ?></h3>
					<p><?php esc_html_e( 'Upgrade to get comprehensive protection for just $69/year:', 'content-guard-pro' ); ?></p>
					<ul style="margin: 15px 0;">
						<li style="margin: 8px 0;">✓ <strong><?php esc_html_e( 'Standard Scan', 'content-guard-pro' ); ?></strong> - <?php esc_html_e( 'Full database scanning (posts, pages, custom fields, widgets)', 'content-guard-pro' ); ?></li>
						<li style="margin: 8px 0;">✓ <strong><?php esc_html_e( 'Quarantine Threats', 'content-guard-pro' ); ?></strong> - <?php esc_html_e( 'Automatically neutralize malicious content', 'content-guard-pro' ); ?></li>
						<li style="margin: 8px 0;">✓ <strong><?php esc_html_e( 'Scheduled Scans', 'content-guard-pro' ); ?></strong> - <?php esc_html_e( 'Automatic daily/weekly scanning', 'content-guard-pro' ); ?></li>
						<li style="margin: 8px 0;">✓ <strong><?php esc_html_e( 'Real-time Protection', 'content-guard-pro' ); ?></strong> - <?php esc_html_e( 'Scan content as you save it', 'content-guard-pro' ); ?></li>
						<li style="margin: 8px 0;">✓ <strong><?php esc_html_e( 'Email Alerts', 'content-guard-pro' ); ?></strong> - <?php esc_html_e( 'Instant notifications for critical threats', 'content-guard-pro' ); ?></li>
						<li style="margin: 8px 0;">✓ <strong><?php esc_html_e( 'Google Safe Browsing & PhishTank', 'content-guard-pro' ); ?></strong> - <?php esc_html_e( 'Real-time reputation checks', 'content-guard-pro' ); ?></li>
						<li style="margin: 8px 0;">✓ <strong><?php esc_html_e( 'All Findings Visible', 'content-guard-pro' ); ?></strong> - <?php esc_html_e( 'No 10-finding limit', 'content-guard-pro' ); ?></li>
						<li style="margin: 8px 0;">✓ <strong><?php esc_html_e( 'Priority Email Support', 'content-guard-pro' ); ?></strong> - <?php esc_html_e( '48-hour response time', 'content-guard-pro' ); ?></li>
					</ul>
					<p style="text-align: center; margin: 20px 0;">
						<a href="https://contentguardpro.com/pricing/" target="_blank" rel="noopener noreferrer" class="button button-primary button-large">
							<?php esc_html_e( 'Upgrade to Solo Guard - $69/year', 'content-guard-pro' ); ?>
						</a>
					</p>
				</div>
			<?php endif; ?>

			<form method="post">
				<?php wp_nonce_field( 'content_guard_pro_wizard_step', 'content_guard_pro_wizard_nonce' ); ?>
				<input type="hidden" name="content_guard_pro_wizard_action" value="complete" />

				<div style="margin: 30px 0;">
					<h3><?php esc_html_e( 'Run Your First Scan?', 'content-guard-pro' ); ?></h3>
					
					<p>
						<label>
							<input type="checkbox" name="run_first_scan" value="1" checked />
							<strong><?php esc_html_e( 'Run a quick scan now to check for existing threats', 'content-guard-pro' ); ?></strong>
						</label>
					</p>

					<p class="description" style="margin-left: 25px;">
						<?php esc_html_e( 'Recommended: This will scan your content immediately and show you any existing security issues.', 'content-guard-pro' ); ?>
					</p>
				</div>

				<div style="padding: 15px; background: #f0f6fc; border-left: 4px solid #2271b1;">
					<strong><?php esc_html_e( 'Next Steps:', 'content-guard-pro' ); ?></strong>
					<ul>
						<li><?php esc_html_e( 'Review your first scan results', 'content-guard-pro' ); ?></li>
						<?php if ( $is_free ) : ?>
							<li><?php esc_html_e( 'Upgrade to unlock quarantine and scheduled scans', 'content-guard-pro' ); ?></li>
						<?php endif; ?>
						<li><?php esc_html_e( 'Configure advanced settings if needed', 'content-guard-pro' ); ?></li>
						<li><?php esc_html_e( 'Add trusted domains to the allowlist', 'content-guard-pro' ); ?></li>
					</ul>
				</div>

				<div class="content-guard-pro-wizard-actions">
					<a href="<?php echo esc_url( self::get_wizard_url( 4 ) ); ?>" class="button">
						<?php esc_html_e( '← Back', 'content-guard-pro' ); ?>
					</a>
					<button type="submit" class="button button-primary button-hero">
						<?php esc_html_e( 'Complete Setup →', 'content-guard-pro' ); ?>
					</button>
				</div>
			</form>
		</div>
		<?php
	}
}

// Note: CGP_Admin_Setup_Wizard::init() is called from the main plugin file
// (content-guard-pro.php) to ensure proper initialization timing.

