<?php
/**
 * Admin Settings Page
 *
 * Handles the settings page display with configuration options
 * for scheduling, notifications, performance, and integrations.
 * Uses WordPress Settings API for proper form handling.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Admin_Settings
 *
 * Renders the settings page using WordPress Settings API.
 * Per PRD Section 3.9: Configuration and Settings.
 *
 * Sections:
 * - General: API keys, license
 * - Scheduling: Daily/weekly, on-save toggle
 * - Notifications: Email, webhook, severity thresholds
 * - Performance: Batch size, delay, Safe Mode
 *
 * @since 1.0.0
 */
class CGP_Admin_Settings {

	/**
	 * Settings option name.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const OPTION_NAME = 'content_guard_pro_settings';

	/**
	 * Convenience feature gate helper.
	 *
	 * @param string $feature Feature key from CGP_License_Manager.
	 * @return bool
	 */
	private static function is_feature_available( $feature ) {
		if ( class_exists( 'CGP_License_Manager' ) ) {
			return CGP_License_Manager::can( $feature );
		}
		return true;
	}

	/**
	 * Initialize settings.
	 *
	 * @since 1.0.0
	 */
	public static function init() {
		add_action( 'admin_init', array( __CLASS__, 'register_settings' ) );
	}

	/**
	 * Register settings, sections, and fields.
	 *
	 * @since 1.0.0
	 */
	public static function register_settings() {
		// Register the main settings option.
		register_setting(
			'content_guard_pro_settings_group',
			self::OPTION_NAME,
			array(
				'type'              => 'array',
				'sanitize_callback' => array( __CLASS__, 'sanitize_settings' ),
			)
		);

		// =====================================================
		// GENERAL TAB - Data Settings
		// =====================================================
		add_settings_section(
			'content_guard_pro_data_section',
			__( 'Data Settings', 'content-guard-pro' ),
			array( __CLASS__, 'render_data_section' ),
			'content-guard-pro-settings-general'
		);

		add_settings_field(
			'preserve_data',
			__( 'Preserve Data on Uninstall', 'content-guard-pro' ),
			array( __CLASS__, 'render_checkbox_field' ),
			'content-guard-pro-settings-general',
			'content_guard_pro_data_section',
			array(
				'label_for'   => 'preserve_data',
				'field_name'  => 'preserve_data',
				'description' => __( 'Keep findings and settings when plugin is deleted', 'content-guard-pro' ),
			)
		);

		// =====================================================
		// SCHEDULING TAB
		// =====================================================
		add_settings_section(
			'content_guard_pro_scheduling_section',
			__( 'Scheduled Scans', 'content-guard-pro' ),
			array( __CLASS__, 'render_scheduling_section' ),
			'content-guard-pro-settings-scheduling'
		);

		add_settings_field(
			'schedule_enabled',
			__( 'Enable Scheduled Scans', 'content-guard-pro' ),
			array( __CLASS__, 'render_checkbox_field' ),
			'content-guard-pro-settings-scheduling',
			'content_guard_pro_scheduling_section',
			array(
				'label_for'   => 'schedule_enabled',
				'field_name'  => 'schedule_enabled',
				'description' => __( 'Run automatic scans on a schedule', 'content-guard-pro' ),
				'disabled'    => ! self::is_feature_available( 'scheduled_scans' ),
				'upgrade_message' => __( 'Upgrade to enable scheduled scans.', 'content-guard-pro' ),
			)
		);

		add_settings_field(
			'schedule_frequency',
			__( 'Schedule Frequency', 'content-guard-pro' ),
			array( __CLASS__, 'render_select_field' ),
			'content-guard-pro-settings-scheduling',
			'content_guard_pro_scheduling_section',
			array(
				'label_for'   => 'schedule_frequency',
				'field_name'  => 'schedule_frequency',
				'options'     => array(
					'daily'      => __( 'Daily', 'content-guard-pro' ),
					'weekly'     => __( 'Weekly', 'content-guard-pro' ),
					'twicedaily' => __( 'Twice Daily', 'content-guard-pro' ),
				),
				'description' => __( 'How often to run scheduled scans', 'content-guard-pro' ),
				'disabled'    => ! self::is_feature_available( 'scheduled_scans' ),
				'upgrade_message' => __( 'Upgrade to enable scheduled scans.', 'content-guard-pro' ),
			)
		);

		add_settings_field(
			'schedule_time',
			__( 'Schedule Time', 'content-guard-pro' ),
			array( __CLASS__, 'render_text_field' ),
			'content-guard-pro-settings-scheduling',
			'content_guard_pro_scheduling_section',
			array(
				'label_for'   => 'schedule_time',
				'field_name'  => 'schedule_time',
				'type'        => 'time',
				'description' => __( 'Time to run daily scans (24-hour format, e.g., 03:00)', 'content-guard-pro' ),
				'class'       => 'regular-text',
				'disabled'    => ! self::is_feature_available( 'scheduled_scans' ),
				'upgrade_message' => __( 'Upgrade to enable scheduled scans.', 'content-guard-pro' ),
			)
		);

		// Real-time Scanning Section.
		add_settings_section(
			'content_guard_pro_realtime_section',
			__( 'Real-time Scanning', 'content-guard-pro' ),
			array( __CLASS__, 'render_realtime_section' ),
			'content-guard-pro-settings-scheduling'
		);

		add_settings_field(
			'realtime_scan_enabled',
			__( 'Enable On-Save Scanning', 'content-guard-pro' ),
			array( __CLASS__, 'render_checkbox_field' ),
			'content-guard-pro-settings-scheduling',
			'content_guard_pro_realtime_section',
			array(
				'label_for'   => 'realtime_scan_enabled',
				'field_name'  => 'realtime_scan_enabled',
				'description' => __( 'Scan content automatically when posts are saved or published', 'content-guard-pro' ),
				'disabled'    => ! self::is_feature_available( 'on_save_scanning' ),
				'upgrade_message' => __( 'Upgrade to enable on-save scanning.', 'content-guard-pro' ),
			)
		);

		// Scan Defaults Section.
		add_settings_section(
			'content_guard_pro_scan_defaults_section',
			__( 'Scan Defaults', 'content-guard-pro' ),
			array( __CLASS__, 'render_scan_defaults_section' ),
			'content-guard-pro-settings-scheduling'
		);

		add_settings_field(
			'scan_mode',
			__( 'Default Scan Mode', 'content-guard-pro' ),
			array( __CLASS__, 'render_select_field' ),
			'content-guard-pro-settings-scheduling',
			'content_guard_pro_scan_defaults_section',
			array(
				'label_for'   => 'scan_mode',
				'field_name'  => 'scan_mode',
				'options'     => array(
					'quick'    => __( 'Quick (Posts only)', 'content-guard-pro' ),
					'standard' => __( 'Standard (Posts + Meta + Options)', 'content-guard-pro' ),
				),
				'description' => __( 'Default scan thoroughness', 'content-guard-pro' ),
				'disabled'    => ! self::is_feature_available( 'scheduled_scans' ),
				'upgrade_message' => __( 'Upgrade to customize scheduled scan defaults.', 'content-guard-pro' ),
			)
		);

		// =====================================================
		// NOTIFICATIONS TAB
		// =====================================================
		add_settings_section(
			'content_guard_pro_email_section',
			__( 'Email Alerts', 'content-guard-pro' ),
			array( __CLASS__, 'render_email_section' ),
			'content-guard-pro-settings-notifications'
		);

		add_settings_field(
			'email_alerts_enabled',
			__( 'Enable Email Alerts', 'content-guard-pro' ),
			array( __CLASS__, 'render_checkbox_field' ),
			'content-guard-pro-settings-notifications',
			'content_guard_pro_email_section',
			array(
				'label_for'   => 'email_alerts_enabled',
				'field_name'  => 'email_alerts_enabled',
				'description' => __( 'Send email notifications for critical findings', 'content-guard-pro' ),
				'disabled'    => ! self::is_feature_available( 'email_alerts' ),
				'upgrade_message' => __( 'Upgrade to enable email alerts.', 'content-guard-pro' ),
			)
		);

		add_settings_field(
			'email_recipients',
			__( 'Email Recipients', 'content-guard-pro' ),
			array( __CLASS__, 'render_textarea_field' ),
			'content-guard-pro-settings-notifications',
			'content_guard_pro_email_section',
			array(
				'label_for'   => 'email_recipients',
				'field_name'  => 'email_recipients',
				'description' => __( 'Email addresses to notify (one per line)', 'content-guard-pro' ),
				'rows'        => 4,
				'disabled'    => ! self::is_feature_available( 'email_alerts' ),
				'upgrade_message' => __( 'Upgrade to configure email recipients.', 'content-guard-pro' ),
			)
		);

		add_settings_field(
			'email_alert_mode',
			__( 'Email Alert Mode', 'content-guard-pro' ),
			array( __CLASS__, 'render_select_field' ),
			'content-guard-pro-settings-notifications',
			'content_guard_pro_email_section',
			array(
				'label_for'   => 'email_alert_mode',
				'field_name'  => 'email_alert_mode',
				'options'     => array(
					'immediate' => __( 'Immediate (each finding)', 'content-guard-pro' ),
					'digest'    => __( 'Daily Digest', 'content-guard-pro' ),
				),
				'description' => __( 'When to send email alerts', 'content-guard-pro' ),
				'disabled'    => ! self::is_feature_available( 'email_alerts' ),
				'upgrade_message' => __( 'Upgrade to configure email alerts.', 'content-guard-pro' ),
			)
		);

		add_settings_field(
			'email_severity_threshold',
			__( 'Email Severity Threshold', 'content-guard-pro' ),
			array( __CLASS__, 'render_select_field' ),
			'content-guard-pro-settings-notifications',
			'content_guard_pro_email_section',
			array(
				'label_for'   => 'email_severity_threshold',
				'field_name'  => 'email_severity_threshold',
				'options'     => array(
					'critical'   => __( 'Critical only', 'content-guard-pro' ),
					'suspicious' => __( 'Suspicious and above', 'content-guard-pro' ),
					'review'     => __( 'All findings', 'content-guard-pro' ),
				),
				'description' => __( 'Minimum severity to trigger email alerts', 'content-guard-pro' ),
				'disabled'    => ! self::is_feature_available( 'email_alerts' ),
				'upgrade_message' => __( 'Upgrade to configure email alerts.', 'content-guard-pro' ),
			)
		);

		// Webhook Section.
		add_settings_section(
			'content_guard_pro_webhook_section',
			__( 'Webhook Notifications', 'content-guard-pro' ),
			array( __CLASS__, 'render_webhook_section' ),
			'content-guard-pro-settings-notifications'
		);

		add_settings_field(
			'webhook_enabled',
			__( 'Enable Webhook', 'content-guard-pro' ),
			array( __CLASS__, 'render_checkbox_field' ),
			'content-guard-pro-settings-notifications',
			'content_guard_pro_webhook_section',
			array(
				'label_for'   => 'webhook_enabled',
				'field_name'  => 'webhook_enabled',
				'description' => __( 'Send webhook notifications for findings', 'content-guard-pro' ),
				'disabled'    => ! self::is_feature_available( 'webhook_alerts' ),
				'upgrade_message' => __( 'Upgrade (Agency Lite+) to enable webhooks.', 'content-guard-pro' ),
			)
		);

		add_settings_field(
			'webhook_url',
			__( 'Webhook URL', 'content-guard-pro' ),
			array( __CLASS__, 'render_text_field' ),
			'content-guard-pro-settings-notifications',
			'content_guard_pro_webhook_section',
			array(
				'label_for'   => 'webhook_url',
				'field_name'  => 'webhook_url',
				'type'        => 'url',
				'description' => __( 'URL to receive webhook POST requests', 'content-guard-pro' ),
				'class'       => 'regular-text',
				'disabled'    => ! self::is_feature_available( 'webhook_alerts' ),
				'upgrade_message' => __( 'Upgrade (Agency Lite+) to enable webhooks.', 'content-guard-pro' ),
			)
		);

		add_settings_field(
			'webhook_secret',
			__( 'Webhook Secret', 'content-guard-pro' ),
			array( __CLASS__, 'render_text_field' ),
			'content-guard-pro-settings-notifications',
			'content_guard_pro_webhook_section',
			array(
				'label_for'   => 'webhook_secret',
				'field_name'  => 'webhook_secret',
				'type'        => 'password',
				'description' => __( 'Secret for HMAC-SHA256 signature verification', 'content-guard-pro' ),
				'class'       => 'regular-text',
				'disabled'    => ! self::is_feature_available( 'webhook_alerts' ),
				'upgrade_message' => __( 'Upgrade (Agency Lite+) to enable webhooks.', 'content-guard-pro' ),
			)
		);

		add_settings_field(
			'webhook_severity_threshold',
			__( 'Webhook Severity Threshold', 'content-guard-pro' ),
			array( __CLASS__, 'render_select_field' ),
			'content-guard-pro-settings-notifications',
			'content_guard_pro_webhook_section',
			array(
				'label_for'   => 'webhook_severity_threshold',
				'field_name'  => 'webhook_severity_threshold',
				'options'     => array(
					'critical'   => __( 'Critical only', 'content-guard-pro' ),
					'suspicious' => __( 'Suspicious and above', 'content-guard-pro' ),
					'review'     => __( 'All findings', 'content-guard-pro' ),
				),
				'description' => __( 'Minimum severity to trigger webhooks', 'content-guard-pro' ),
				'disabled'    => ! self::is_feature_available( 'webhook_alerts' ),
				'upgrade_message' => __( 'Upgrade (Agency Lite+) to enable webhooks.', 'content-guard-pro' ),
			)
		);

		// Admin Notices Section.
		add_settings_section(
			'content_guard_pro_admin_notices_section',
			__( 'Admin Notifications', 'content-guard-pro' ),
			array( __CLASS__, 'render_admin_notices_section' ),
			'content-guard-pro-settings-notifications'
		);

		add_settings_field(
			'notifications_admin',
			__( 'Admin Bar & Notices', 'content-guard-pro' ),
			array( __CLASS__, 'render_checkbox_field' ),
			'content-guard-pro-settings-notifications',
			'content_guard_pro_admin_notices_section',
			array(
				'label_for'   => 'notifications_admin',
				'field_name'  => 'notifications_admin',
				'description' => __( 'Show admin notices and admin bar badge for critical findings', 'content-guard-pro' ),
			)
		);

		// =====================================================
		// PERFORMANCE TAB
		// =====================================================
		add_settings_section(
			'content_guard_pro_batch_section',
			__( 'Batch Processing', 'content-guard-pro' ),
			array( __CLASS__, 'render_batch_section' ),
			'content-guard-pro-settings-performance'
		);

		add_settings_field(
			'batch_size',
			__( 'Batch Size', 'content-guard-pro' ),
			array( __CLASS__, 'render_number_field' ),
			'content-guard-pro-settings-performance',
			'content_guard_pro_batch_section',
			array(
				'label_for'   => 'batch_size',
				'field_name'  => 'batch_size',
				'min'         => 10,
				'max'         => 500,
				'step'        => 10,
				'description' => __( 'Number of items to process per batch (10-500)', 'content-guard-pro' ),
			)
		);

		add_settings_field(
			'batch_delay',
			__( 'Batch Delay', 'content-guard-pro' ),
			array( __CLASS__, 'render_number_field' ),
			'content-guard-pro-settings-performance',
			'content_guard_pro_batch_section',
			array(
				'label_for'   => 'batch_delay',
				'field_name'  => 'batch_delay',
				'min'         => 1,
				'max'         => 60,
				'step'        => 1,
				'description' => __( 'Delay between batches in seconds (1-60)', 'content-guard-pro' ),
			)
		);

		add_settings_field(
			'safe_mode',
			__( 'Safe Mode', 'content-guard-pro' ),
			array( __CLASS__, 'render_select_field' ),
			'content-guard-pro-settings-performance',
			'content_guard_pro_batch_section',
			array(
				'label_for'   => 'safe_mode',
				'field_name'  => 'safe_mode',
				'options'     => array(
					'auto' => __( 'Auto (enable on large sites)', 'content-guard-pro' ),
					'on'   => __( 'Always On', 'content-guard-pro' ),
					'off'  => __( 'Always Off', 'content-guard-pro' ),
				),
				'description' => __( 'Use smaller batches and longer delays for better stability', 'content-guard-pro' ),
			)
		);

		// Data Retention Section.
		add_settings_section(
			'content_guard_pro_retention_section',
			__( 'Data Retention', 'content-guard-pro' ),
			array( __CLASS__, 'render_retention_section' ),
			'content-guard-pro-settings-performance'
		);

		add_settings_field(
			'retention_days',
			__( 'Finding Retention', 'content-guard-pro' ),
			array( __CLASS__, 'render_select_field' ),
			'content-guard-pro-settings-performance',
			'content_guard_pro_retention_section',
			array(
				'label_for'   => 'retention_days',
				'field_name'  => 'retention_days',
				'options'     => array(
					'30'  => __( '30 days', 'content-guard-pro' ),
					'90'  => __( '90 days', 'content-guard-pro' ),
					'180' => __( '180 days', 'content-guard-pro' ),
					'365' => __( '1 year', 'content-guard-pro' ),
				),
				'description' => __( 'How long to keep finding records', 'content-guard-pro' ),
			)
		);
	}

	/**
	 * Display the settings page.
	 *
	 * @since 1.0.0
	 */
	public static function display() {
		// Check user capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'content-guard-pro' ) );
		}

		// Determine current tab.
		$current_tab = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'general';
		$tabs = array(
			'general'       => __( 'General', 'content-guard-pro' ),
			'license'       => __( 'License', 'content-guard-pro' ),
			'scheduling'    => __( 'Scheduling', 'content-guard-pro' ),
			'notifications' => __( 'Notifications', 'content-guard-pro' ),
			'performance'   => __( 'Performance', 'content-guard-pro' ),
		);

		?>
		<div class="wrap content-guard-pro-settings">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

			<?php settings_errors( 'content_guard_pro_settings' ); ?>

			<nav class="nav-tab-wrapper">
				<?php foreach ( $tabs as $tab_key => $tab_label ) : ?>
					<a href="<?php echo esc_url( add_query_arg( 'tab', $tab_key, admin_url( 'admin.php?page=content-guard-pro-settings' ) ) ); ?>" 
					   class="nav-tab <?php echo $current_tab === $tab_key ? 'nav-tab-active' : ''; ?>">
						<?php echo esc_html( $tab_label ); ?>
					</a>
				<?php endforeach; ?>
			</nav>

			<div class="cgp-settings-tab-content">
				<?php
				switch ( $current_tab ) {
					case 'license':
						self::render_license_tab();
						break;
					default:
						self::render_settings_form( $current_tab );
						break;
				}
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Render the settings form for a specific tab.
	 *
	 * @since 1.0.0
	 * @param string $tab Current tab slug.
	 */
	private static function render_settings_form( $tab ) {
		?>
		<form method="post" action="options.php">
			<?php
			settings_fields( 'content_guard_pro_settings_group' );

			// Show only the relevant section based on tab.
			switch ( $tab ) {
				case 'scheduling':
					do_settings_sections( 'content-guard-pro-settings-scheduling' );
					break;
				case 'notifications':
					do_settings_sections( 'content-guard-pro-settings-notifications' );
					break;
				case 'performance':
					do_settings_sections( 'content-guard-pro-settings-performance' );
					break;
				case 'general':
				default:
					do_settings_sections( 'content-guard-pro-settings-general' );
					break;
			}
			
			submit_button();
			?>
		</form>

		<?php if ( 'general' === $tab ) : ?>
			<hr>
			<h2><?php esc_html_e( 'Setup Wizard', 'content-guard-pro' ); ?></h2>
			<p class="description">
				<?php esc_html_e( 'Re-run the setup wizard to reconfigure your scan mode, alerts, and schedule settings.', 'content-guard-pro' ); ?>
			</p>
			<p>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro&setup=1' ) ); ?>" class="button button-secondary">
					<?php esc_html_e( 'Re-run Setup Wizard', 'content-guard-pro' ); ?>
				</a>
			</p>
			
			<hr>
			<h2><?php esc_html_e( 'Diagnostics', 'content-guard-pro' ); ?></h2>
			<p>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-diagnostics' ) ); ?>" class="button">
					<?php esc_html_e( 'Run Diagnostics', 'content-guard-pro' ); ?>
				</a>
				<a href="#" class="button" id="content-guard-pro-clear-cache">
					<?php esc_html_e( 'Clear Reputation Cache', 'content-guard-pro' ); ?>
				</a>
			</p>

			<script>
			jQuery(document).ready(function($) {
				$('#content-guard-pro-clear-cache').on('click', function(e) {
					e.preventDefault();
					if (!confirm('<?php esc_html_e( 'Clear all reputation cache? This will force fresh API checks on next scan.', 'content-guard-pro' ); ?>')) {
						return;
					}
					
					var button = $(this);
					button.prop('disabled', true).text('<?php esc_html_e( 'Clearing...', 'content-guard-pro' ); ?>');
					
					$.post(ajaxurl, {
						action: 'content_guard_pro_clear_reputation_cache',
						nonce: '<?php echo esc_js( wp_create_nonce( 'content_guard_pro_clear_cache' ) ); ?>'
					}, function(response) {
						alert(response.data);
						button.prop('disabled', false).text('<?php esc_html_e( 'Clear Reputation Cache', 'content-guard-pro' ); ?>');
					});
				});
			});
			</script>
		<?php endif;
	}

	/**
	 * Render the License tab.
	 *
	 * @since 1.0.0
	 */
	private static function render_license_tab() {
		$license_status = class_exists( 'CGP_License_Manager' ) 
			? CGP_License_Manager::get_status_for_display() 
			: array( 'tier' => 'free', 'tier_name' => 'Free', 'is_paid' => false );

		$license_key = $license_status['license_key'] ?? '';
		$tier_name   = $license_status['tier_name'];
		$is_paid     = $license_status['is_paid'];
		$features    = $license_status['features'] ?? array();
		$limits      = $license_status['limits'] ?? array();

		?>
		<div class="cgp-license-tab">
			<table class="form-table cgp-license-table">
				<tr>
					<th scope="row"><?php esc_html_e( 'API Connection Status', 'content-guard-pro' ); ?></th>
					<td>
						<div id="cgp-api-status" class="cgp-api-status">
							<span class="dashicons dashicons-update cgp-spin"></span>
							<span><?php esc_html_e( 'Checking API connection...', 'content-guard-pro' ); ?></span>
						</div>
						<p class="description"><?php esc_html_e( 'Checks connectivity to the licensing service.', 'content-guard-pro' ); ?></p>
						<p><button type="button" class="button" id="cgp-check-api"><?php esc_html_e( 'Check Connection', 'content-guard-pro' ); ?></button></p>
					</td>
				</tr>
				<tr>
					<th scope="row"><?php esc_html_e( 'License Key', 'content-guard-pro' ); ?></th>
					<td>
						<div class="cgp-license-input-group">
							<input type="text"
							       id="cgp-license-key"
							       class="regular-text"
							       value="<?php echo esc_attr( $license_key ); ?>"
							       placeholder="CGP-XXXX-XXXX-XXXX-XXXX"
							       <?php echo $is_paid ? 'readonly' : ''; ?> />
							<?php if ( $is_paid ) : ?>
								<button type="button" class="button" id="cgp-deactivate-license">
									<?php esc_html_e( 'Deactivate', 'content-guard-pro' ); ?>
								</button>
							<?php else : ?>
								<button type="button" class="button button-primary" id="cgp-activate-license">
									<?php esc_html_e( 'Activate License', 'content-guard-pro' ); ?>
								</button>
							<?php endif; ?>
						</div>
						<p class="description">
							<?php if ( $is_paid ) : ?>
								<?php esc_html_e( 'Your license is active. To change licenses, deactivate first.', 'content-guard-pro' ); ?>
							<?php else : ?>
								<?php esc_html_e( 'Enter your license key to unlock premium features.', 'content-guard-pro' ); ?>
								<a href="https://contentguardpro.com/pricing" target="_blank" rel="noopener noreferrer">
									<?php esc_html_e( 'Get a license →', 'content-guard-pro' ); ?>
								</a>
							<?php endif; ?>
						</p>
						<div id="cgp-license-result" class="cgp-license-result"></div>
					</td>
				</tr>
				<tr>
					<th scope="row"><?php esc_html_e( 'Current Plan', 'content-guard-pro' ); ?></th>
					<td>
						<div class="cgp-current-plan">
							<span class="cgp-tier-badge cgp-tier-badge-<?php echo esc_attr( $license_status['tier'] ?? 'free' ); ?>">
								<?php echo esc_html( $tier_name ); ?>
							</span>
							<?php if ( ! $is_paid ) : ?>
								<a href="<?php echo esc_url( $license_status['upgrade_url'] ?? 'https://contentguardpro.com/pricing' ); ?>"
								   class="button button-primary cgp-upgrade-button"
								   target="_blank"
								   rel="noopener noreferrer">
									<?php esc_html_e( 'Upgrade Now', 'content-guard-pro' ); ?>
								</a>
							<?php endif; ?>
						</div>
					</td>
				</tr>
			</table>

			<div class="cgp-license-section">
				<h3><?php esc_html_e( 'Plan Limits', 'content-guard-pro' ); ?></h3>
				<table class="cgp-limits-table">
					<thead>
						<tr>
							<th><?php esc_html_e( 'Limit', 'content-guard-pro' ); ?></th>
							<th><?php esc_html_e( 'Your Plan', 'content-guard-pro' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<tr>
							<td><?php esc_html_e( 'Visible Findings', 'content-guard-pro' ); ?></td>
							<td><?php echo -1 === ( $limits['visible_findings'] ?? 10 ) ? esc_html__( 'Unlimited', 'content-guard-pro' ) : esc_html( $limits['visible_findings'] ?? 10 ); ?></td>
						</tr>
						<tr>
							<td><?php esc_html_e( 'Manual Scans Per Day', 'content-guard-pro' ); ?></td>
							<td><?php echo -1 === ( $limits['manual_scans_per_day'] ?? 1 ) ? esc_html__( 'Unlimited', 'content-guard-pro' ) : esc_html( $limits['manual_scans_per_day'] ?? 1 ); ?></td>
						</tr>
						<tr>
							<td><?php esc_html_e( 'Allowlist Entries', 'content-guard-pro' ); ?></td>
							<td><?php echo -1 === ( $limits['allowlist_entries'] ?? 5 ) ? esc_html__( 'Unlimited', 'content-guard-pro' ) : esc_html( $limits['allowlist_entries'] ?? 5 ); ?></td>
						</tr>
						<tr>
							<td><?php esc_html_e( 'Scan History', 'content-guard-pro' ); ?></td>
							<td>
								<?php 
								$history_days = $limits['scan_history_days'] ?? 7;
								if ( -1 === $history_days ) {
									echo esc_html__( 'Custom', 'content-guard-pro' );
								} else {
									/* translators: %d: number of days of scan history included in the plan */
									echo sprintf( esc_html__( '%d days', 'content-guard-pro' ), absint( $history_days ) );
								}
								?>
							</td>
						</tr>
						<tr>
							<td><?php esc_html_e( 'Rule Pack Delay', 'content-guard-pro' ); ?></td>
							<td>
								<?php 
								$delay = $limits['rule_pack_delay_days'] ?? 14;
								if ( 0 === $delay ) {
									echo esc_html__( 'Immediate', 'content-guard-pro' );
								} else {
									/* translators: %d: number of days delay for rule pack availability */
									echo sprintf( esc_html__( '%d day delay', 'content-guard-pro' ), absint( $delay ) );
								}
								?>
							</td>
						</tr>
					</tbody>
				</table>
			</div>

			<div class="cgp-license-section">
				<h3><?php esc_html_e( 'Feature Availability', 'content-guard-pro' ); ?></h3>
				<div class="cgp-feature-grid">
					<?php
					$feature_labels = array(
						'quick_scan'           => __( 'Quick Scan', 'content-guard-pro' ),
						'single_post_scanning' => __( 'Single Post Scanning', 'content-guard-pro' ),
						'standard_scan'        => __( 'Standard Scan', 'content-guard-pro' ),
						'scheduled_scans'      => __( 'Scheduled Scans', 'content-guard-pro' ),
						'on_save_scanning'     => __( 'Full Site Real-time Scanning', 'content-guard-pro' ),
						'quarantine'           => __( 'Quarantine System', 'content-guard-pro' ),
						'email_alerts'         => __( 'Email Alerts', 'content-guard-pro' ),
						'webhook_alerts'       => __( 'Webhook Notifications', 'content-guard-pro' ),
						'csv_export'           => __( 'CSV/JSON Export', 'content-guard-pro' ),
						'rest_api'             => __( 'REST API Access', 'content-guard-pro' ),
						'reputation_apis'      => __( 'Reputation API Checks', 'content-guard-pro' ),
						'audit_log'            => __( 'Audit Log', 'content-guard-pro' ),
						'priority_support'     => __( 'Priority Support', 'content-guard-pro' ),
					);

					foreach ( $feature_labels as $feature_key => $feature_label ) :
						$is_available = ! empty( $features[ $feature_key ] );
						$item_class = $is_available ? 'cgp-feature-available' : 'cgp-feature-locked';
						$icon = $is_available ? 'yes-alt' : 'lock';
					?>
						<div class="cgp-feature-item <?php echo esc_attr( $item_class ); ?>">
							<span class="dashicons dashicons-<?php echo esc_attr( $icon ); ?>"></span>
							<span><?php echo esc_html( $feature_label ); ?></span>
						</div>
					<?php endforeach; ?>
				</div>
			</div>

			<?php if ( ! $is_paid ) : ?>
			<div class="cgp-license-section">
				<h3><?php esc_html_e( 'Upgrade for Full Access', 'content-guard-pro' ); ?></h3>
				<p><?php esc_html_e( 'Unlock advanced scanning, quarantine, notifications, exports, and integrations.', 'content-guard-pro' ); ?></p>
				<a href="https://contentguardpro.com/pricing" 
				   class="button button-primary cgp-upgrade-button" 
				   target="_blank" 
				   rel="noopener noreferrer">
					<?php esc_html_e( 'View Plans', 'content-guard-pro' ); ?>
				</a>
			</div>
			<?php endif; ?>
		</div>

		<script>
		jQuery(document).ready(function($) {
			var nonce = '<?php echo esc_js( wp_create_nonce( 'cgp_license_nonce' ) ); ?>';

			function cgpFormatMessage(message) {
				if (!message) {
					return '<?php echo esc_js( __( 'An unexpected error occurred.', 'content-guard-pro' ) ); ?>';
				}

				if (typeof message === 'string') {
					return message;
				}

				if ($.isArray(message)) {
					return message.join(', ');
				}

				if (typeof message === 'object') {
					// Laravel-style error object { code, message } or arbitrary key=>message.
					if (message.message) {
						return (message.code ? message.code + ': ' : '') + message.message;
					}
					return Object.values(message).join(', ');
				}

				return String(message);
			}

			// Check API health on load.
			checkApiHealth();

			// Check API button.
			$('#cgp-check-api').on('click', function() {
				checkApiHealth();
			});

			function checkApiHealth() {
				$('#cgp-api-status')
					.removeClass('cgp-status-healthy cgp-status-unhealthy')
					.html('<span class="dashicons dashicons-update cgp-spin"></span> <?php esc_html_e( 'Checking...', 'content-guard-pro' ); ?>');

				$.post(ajaxurl, {
					action: 'cgp_check_api_health',
					nonce: nonce
				}, function(response) {
					var version = '';
					if (response.data) {
						version = response.data.version || (response.data.data && response.data.data.version) || '';
					}

					if (response.success && response.data.healthy) {
						$('#cgp-api-status')
							.addClass('cgp-status-healthy')
							.html('<span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e( 'API Connected', 'content-guard-pro' ); ?>' + (version ? ' (v' + version + ')' : ''));
					} else {
						$('#cgp-api-status')
							.addClass('cgp-status-unhealthy')
							.html('<span class="dashicons dashicons-warning"></span> <?php esc_html_e( 'API Unavailable', 'content-guard-pro' ); ?>');
					}
				}).fail(function() {
					$('#cgp-api-status')
						.addClass('cgp-status-unhealthy')
						.html('<span class="dashicons dashicons-warning"></span> <?php esc_html_e( 'Connection Error', 'content-guard-pro' ); ?>');
				});
			}

			// Activate license.
			$('#cgp-activate-license').on('click', function() {
				var button = $(this);
				var licenseKey = $('#cgp-license-key').val().trim();

				if (!licenseKey) {
					$('#cgp-license-result').html('<div class="notice notice-error"><p><?php esc_html_e( 'Please enter a license key', 'content-guard-pro' ); ?></p></div>');
					return;
				}

				button.prop('disabled', true).text('<?php esc_html_e( 'Activating...', 'content-guard-pro' ); ?>');

				$.post(ajaxurl, {
					action: 'cgp_validate_license',
					nonce: nonce,
					license_key: licenseKey
				}, function(response) {
					if (response.success) {
						$('#cgp-license-result').html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
						setTimeout(function() {
							location.reload();
						}, 1500);
					} else {
						var errorMessage = cgpFormatMessage(response.data && response.data.message ? response.data.message : response.data);
						$('#cgp-license-result').html('<div class="notice notice-error"><p>' + errorMessage + '</p></div>');
						button.prop('disabled', false).text('<?php esc_html_e( 'Activate License', 'content-guard-pro' ); ?>');
					}
				}).fail(function() {
					$('#cgp-license-result').html('<div class="notice notice-error"><p><?php esc_html_e( 'Network error. Please try again.', 'content-guard-pro' ); ?></p></div>');
					button.prop('disabled', false).text('<?php esc_html_e( 'Activate License', 'content-guard-pro' ); ?>');
				});
			});

			// Deactivate license.
			$('#cgp-deactivate-license').on('click', function() {
				if (!confirm('<?php esc_html_e( 'Are you sure you want to deactivate this license?', 'content-guard-pro' ); ?>')) {
					return;
				}

				var button = $(this);
				button.prop('disabled', true).text('<?php esc_html_e( 'Deactivating...', 'content-guard-pro' ); ?>');

				$.post(ajaxurl, {
					action: 'cgp_deactivate_license',
					nonce: nonce
				}, function(response) {
					if (response.success) {
						$('#cgp-license-result').html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
						setTimeout(function() {
							location.reload();
						}, 1500);
					} else {
						$('#cgp-license-result').html('<div class="notice notice-error"><p>' + response.data.message + '</p></div>');
						button.prop('disabled', false).text('<?php esc_html_e( 'Deactivate', 'content-guard-pro' ); ?>');
					}
				});
			});

			// Add spin animation
			$('<style>.cgp-spin { animation: cgp-spin 1s linear infinite; } @keyframes cgp-spin { from { transform: rotate(0deg); } to { transform: rotate(360deg); } }</style>').appendTo('head');
		});
		</script>
		<?php
	}

	/**
	 * Render data section description.
	 *
	 * @since 1.0.0
	 */
	public static function render_data_section() {
		echo '<p>' . esc_html__( 'Configure data handling and plugin uninstallation behavior.', 'content-guard-pro' ) . '</p>';
	}

	/**
	 * Render scheduling section description.
	 *
	 * @since 1.0.0
	 */
	public static function render_scheduling_section() {
		echo '<p>' . esc_html__( 'Configure automatic scheduled scans. Requires Solo Guard or higher plan.', 'content-guard-pro' ) . '</p>';
	}

	/**
	 * Render realtime section description.
	 *
	 * @since 1.0.0
	 */
	public static function render_realtime_section() {
		echo '<p>' . esc_html__( 'Automatically scan content when posts are saved or updated.', 'content-guard-pro' ) . '</p>';
	}

	/**
	 * Render scan defaults section description.
	 *
	 * @since 1.0.0
	 */
	public static function render_scan_defaults_section() {
		echo '<p>' . esc_html__( 'Default settings for manual and scheduled scans.', 'content-guard-pro' ) . '</p>';
	}

	/**
	 * Render email section description.
	 *
	 * @since 1.0.0
	 */
	public static function render_email_section() {
		echo '<p>' . esc_html__( 'Configure email notifications for scan findings. Requires Solo Guard or higher plan.', 'content-guard-pro' ) . '</p>';
	}

	/**
	 * Render webhook section description.
	 *
	 * @since 1.0.0
	 */
	public static function render_webhook_section() {
		echo '<p>' . esc_html__( 'Send findings to external services via webhooks. Requires Agency Lite or higher plan.', 'content-guard-pro' ) . '</p>';
	}

	/**
	 * Render admin notices section description.
	 *
	 * @since 1.0.0
	 */
	public static function render_admin_notices_section() {
		echo '<p>' . esc_html__( 'Configure how findings are displayed in the WordPress admin area.', 'content-guard-pro' ) . '</p>';
	}

	/**
	 * Render notifications section description (legacy).
	 *
	 * @since 1.0.0
	 */
	public static function render_notifications_section() {
		echo '<p>' . esc_html__( 'Configure email alerts, webhooks, and notification preferences.', 'content-guard-pro' ) . '</p>';
	}

	/**
	 * Render batch processing section description.
	 *
	 * @since 1.0.0
	 */
	public static function render_batch_section() {
		echo '<p>' . esc_html__( 'Configure how the scanner processes content in batches to avoid server overload.', 'content-guard-pro' ) . '</p>';
	}

	/**
	 * Render data retention section description.
	 *
	 * @since 1.0.0
	 */
	public static function render_retention_section() {
		echo '<p>' . esc_html__( 'Configure how long to keep scan findings and history.', 'content-guard-pro' ) . '</p>';
	}

	/**
	 * Render performance section description (legacy).
	 *
	 * @since 1.0.0
	 */
	public static function render_performance_section() {
		echo '<p>' . esc_html__( 'Configure scan performance, batch processing, and data retention.', 'content-guard-pro' ) . '</p>';
	}

	/**
	 * Render checkbox field.
	 *
	 * @since 1.0.0
	 * @param array $args Field arguments.
	 */
	public static function render_checkbox_field( $args ) {
		$options    = get_option( self::OPTION_NAME, array() );
		$field_name = $args['field_name'];
		$value      = isset( $options[ $field_name ] ) ? $options[ $field_name ] : false;
		$checked    = checked( $value, true, false );
		$disabled   = ! empty( $args['disabled'] ) ? 'disabled' : '';

		printf(
			'<label><input type="checkbox" id="%1$s" name="%2$s[%3$s]" value="1" %4$s %6$s /> %5$s</label>',
			esc_attr( $args['label_for'] ),
			esc_attr( self::OPTION_NAME ),
			esc_attr( $field_name ),
			esc_attr( $checked ),
			esc_html( $args['description'] ),
			esc_attr( $disabled )
		);

		if ( ! empty( $args['upgrade_message'] ) && ! empty( $args['disabled'] ) ) {
			printf(
				'<p class="description cgp-upgrade-note">%s</p>',
				esc_html( $args['upgrade_message'] )
			);
		}
	}

	/**
	 * Render text/password/url field.
	 *
	 * @since 1.0.0
	 * @param array $args Field arguments.
	 */
	public static function render_text_field( $args ) {
		$options    = get_option( self::OPTION_NAME, array() );
		$field_name = $args['field_name'];
		$value      = isset( $options[ $field_name ] ) ? $options[ $field_name ] : '';
		$type       = isset( $args['type'] ) ? $args['type'] : 'text';
		$class      = isset( $args['class'] ) ? $args['class'] : 'regular-text';
		$disabled   = ! empty( $args['disabled'] ) ? 'disabled' : '';

		printf(
			'<input type="%1$s" id="%2$s" name="%3$s[%4$s]" value="%5$s" class="%6$s" %7$s />',
			esc_attr( $type ),
			esc_attr( $args['label_for'] ),
			esc_attr( self::OPTION_NAME ),
			esc_attr( $field_name ),
			esc_attr( $value ),
			esc_attr( $class ),
			esc_attr( $disabled )
		);

		if ( ! empty( $args['description'] ) ) {
			printf( '<p class="description">%s</p>', esc_html( $args['description'] ) );
		}

		if ( ! empty( $args['upgrade_message'] ) && ! empty( $args['disabled'] ) ) {
			printf(
				'<p class="description cgp-upgrade-note">%s</p>',
				esc_html( $args['upgrade_message'] )
			);
		}
	}

	/**
	 * Render textarea field.
	 *
	 * @since 1.0.0
	 * @param array $args Field arguments.
	 */
	public static function render_textarea_field( $args ) {
		$options    = get_option( self::OPTION_NAME, array() );
		$field_name = $args['field_name'];
		$value      = isset( $options[ $field_name ] ) ? $options[ $field_name ] : '';
		$rows       = isset( $args['rows'] ) ? $args['rows'] : 4;
		$disabled   = ! empty( $args['disabled'] ) ? 'disabled' : '';

		printf(
			'<textarea id="%1$s" name="%2$s[%3$s]" rows="%4$s" class="large-text" %6$s>%5$s</textarea>',
			esc_attr( $args['label_for'] ),
			esc_attr( self::OPTION_NAME ),
			esc_attr( $field_name ),
			esc_attr( $rows ),
			esc_textarea( $value ),
			esc_attr( $disabled )
		);

		if ( ! empty( $args['description'] ) ) {
			printf( '<p class="description">%s</p>', esc_html( $args['description'] ) );
		}

		if ( ! empty( $args['upgrade_message'] ) && ! empty( $args['disabled'] ) ) {
			printf(
				'<p class="description cgp-upgrade-note">%s</p>',
				esc_html( $args['upgrade_message'] )
			);
		}
	}

	/**
	 * Render select field.
	 *
	 * @since 1.0.0
	 * @param array $args Field arguments.
	 */
	public static function render_select_field( $args ) {
		$options    = get_option( self::OPTION_NAME, array() );
		$field_name = $args['field_name'];
		$value      = isset( $options[ $field_name ] ) ? $options[ $field_name ] : '';
		$disabled   = ! empty( $args['disabled'] ) ? 'disabled' : '';

		printf(
			'<select id="%1$s" name="%2$s[%3$s]" %4$s>',
			esc_attr( $args['label_for'] ),
			esc_attr( self::OPTION_NAME ),
			esc_attr( $field_name ),
			esc_attr( $disabled )
		);

		foreach ( $args['options'] as $option_value => $option_label ) {
			printf(
				'<option value="%1$s" %2$s>%3$s</option>',
				esc_attr( $option_value ),
				selected( $value, $option_value, false ),
				esc_html( $option_label )
			);
		}

		echo '</select>';

		if ( ! empty( $args['description'] ) ) {
			printf( '<p class="description">%s</p>', esc_html( $args['description'] ) );
		}

		if ( ! empty( $args['upgrade_message'] ) && ! empty( $args['disabled'] ) ) {
			printf(
				'<p class="description cgp-upgrade-note">%s</p>',
				esc_html( $args['upgrade_message'] )
			);
		}
	}

	/**
	 * Render number field.
	 *
	 * @since 1.0.0
	 * @param array $args Field arguments.
	 */
	public static function render_number_field( $args ) {
		$options    = get_option( self::OPTION_NAME, array() );
		$field_name = $args['field_name'];
		$value      = isset( $options[ $field_name ] ) ? $options[ $field_name ] : '';
		$min        = isset( $args['min'] ) ? $args['min'] : '';
		$max        = isset( $args['max'] ) ? $args['max'] : '';
		$step       = isset( $args['step'] ) ? $args['step'] : '1';
		$disabled   = ! empty( $args['disabled'] ) ? 'disabled' : '';

		// Get default value for display.
		$default_value = '';
		if ( 'batch_size' === $field_name ) {
			$default_value = 100; // CGP_Scanner::DEFAULT_BATCH_SIZE
		} elseif ( 'batch_delay' === $field_name ) {
			$default_value = 2; // CGP_Scanner::DEFAULT_BATCH_DELAY
		}

		printf(
			'<input type="number" id="%1$s" name="%2$s[%3$s]" value="%4$s" min="%5$s" max="%6$s" step="%7$s" class="small-text" %8$s placeholder="%9$s" />',
			esc_attr( $args['label_for'] ),
			esc_attr( self::OPTION_NAME ),
			esc_attr( $field_name ),
			esc_attr( $value ),
			esc_attr( $min ),
			esc_attr( $max ),
			esc_attr( $step ),
			esc_attr( $disabled ),
			esc_attr( $default_value )
		);

		if ( ! empty( $args['description'] ) ) {
			$description = esc_html( $args['description'] );
			if ( ! empty( $default_value ) && empty( $value ) ) {
				$description .= sprintf(
					' <strong>%s</strong>',
					/* translators: %d: default value */
					sprintf( esc_html__( '(Default: %d)', 'content-guard-pro' ), $default_value )
				);
			}
			printf( '<p class="description">%s</p>', wp_kses_post( $description ) );
		}

		if ( ! empty( $args['upgrade_message'] ) && ! empty( $args['disabled'] ) ) {
			printf(
				'<p class="description cgp-upgrade-note">%s</p>',
				esc_html( $args['upgrade_message'] )
			);
		}
	}

	/**
	 * Sanitize settings.
	 *
	 * @since 1.0.0
	 * @param array $input Settings input.
	 * @return array Sanitized settings.
	 */
	public static function sanitize_settings( $input ) {
		$sanitized = array();

		// Checkboxes.
		$checkboxes = array(
			'schedule_enabled',
			'realtime_scan_enabled',
			'email_alerts_enabled',
			'webhook_enabled',
			'notifications_admin',
			'preserve_data',
		);

		foreach ( $checkboxes as $checkbox ) {
			$sanitized[ $checkbox ] = ! empty( $input[ $checkbox ] );
		}

		// Text fields.
		$text_fields = array(
			'license_key',
			'schedule_time',
			'email_recipients',
			'webhook_secret',
		);

		foreach ( $text_fields as $field ) {
			if ( isset( $input[ $field ] ) ) {
				$sanitized[ $field ] = sanitize_text_field( $input[ $field ] );
			}
		}

	// Textarea fields (preserve line breaks).
	$textarea_fields = array(
		'allowlist_domains',
		'denylist_patterns',
	);

	foreach ( $textarea_fields as $field ) {
		if ( isset( $input[ $field ] ) ) {
			// Handle both string and array values (arrays from defaults, strings from form input).
			if ( is_array( $input[ $field ] ) ) {
				// If it's an array, convert to string (one item per line).
				$sanitized[ $field ] = implode( "\n", array_map( 'sanitize_text_field', $input[ $field ] ) );
			} else {
				// Preserve line breaks, just remove slashes and normalize line endings.
				$value = wp_unslash( $input[ $field ] );
				$value = str_replace( array( "\r\n", "\r" ), "\n", $value );
				$sanitized[ $field ] = trim( $value );
			}
		}
	}

		// URL field.
		if ( isset( $input['webhook_url'] ) ) {
			$sanitized['webhook_url'] = esc_url_raw( $input['webhook_url'] );
		}

		// Select fields.
		$select_fields = array(
			'schedule_frequency'        => array( 'daily', 'weekly', 'twicedaily' ),
			'scan_mode'                 => array( 'quick', 'standard' ),
			'email_alert_mode'          => array( 'immediate', 'digest' ),
			'email_severity_threshold'  => array( 'critical', 'suspicious', 'review' ),
			'webhook_severity_threshold'=> array( 'critical', 'suspicious', 'review' ),
			'safe_mode'                 => array( 'auto', 'on', 'off' ),
			'retention_days'            => array( '30', '90', '180', '365' ),
		);

		foreach ( $select_fields as $field => $allowed_values ) {
			if ( isset( $input[ $field ] ) && in_array( $input[ $field ], $allowed_values, true ) ) {
				$sanitized[ $field ] = $input[ $field ];
			}
		}

		// Number fields.
		if ( isset( $input['batch_size'] ) ) {
			$sanitized['batch_size'] = absint( $input['batch_size'] );
			$sanitized['batch_size'] = max( 10, min( 500, $sanitized['batch_size'] ) );
		}

		if ( isset( $input['batch_delay'] ) ) {
			$sanitized['batch_delay'] = absint( $input['batch_delay'] );
			$sanitized['batch_delay'] = max( 1, min( 60, $sanitized['batch_delay'] ) );
		}

		// Validate webhook URL if enabled.
		if ( ! empty( $sanitized['webhook_enabled'] ) && empty( $sanitized['webhook_url'] ) ) {
			add_settings_error(
				'content_guard_pro_settings',
				'missing_webhook_url',
				__( 'Webhook is enabled but no URL is provided.', 'content-guard-pro' ),
				'warning'
			);
		}

		// Validate email recipients if enabled.
		if ( ! empty( $sanitized['email_alerts_enabled'] ) && empty( $sanitized['email_recipients'] ) ) {
			add_settings_error(
				'content_guard_pro_settings',
				'missing_email_recipients',
				__( 'Email alerts are enabled but no recipients are specified.', 'content-guard-pro' ),
				'warning'
			);
		}

		// Success message.
		add_settings_error(
			'content_guard_pro_settings',
			'settings_updated',
			__( 'Settings saved successfully.', 'content-guard-pro' ),
			'success'
		);

		return $sanitized;
	}

	/**
	 * AJAX handler to clear reputation cache.
	 *
	 * @since 1.0.0
	 */
	public static function ajax_clear_reputation_cache() {
		check_ajax_referer( 'content_guard_pro_clear_cache', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Permission denied', 'content-guard-pro' ) );
		}

		$count = CGP_Integrations::clear_reputation_cache();

		wp_send_json_success(
			sprintf(
				/* translators: %d: number of cache entries cleared */
				__( 'Cleared %d reputation cache entries.', 'content-guard-pro' ),
				$count
			)
		);
	}
}

// Initialize settings.
CGP_Admin_Settings::init();

// Register AJAX handlers.
add_action( 'wp_ajax_content_guard_pro_clear_reputation_cache', array( 'CGP_Admin_Settings', 'ajax_clear_reputation_cache' ) );
