<?php
/**
 * Admin Scans Page
 *
 * Handles the scans page display with scan controls, progress tracking,
 * and scan history.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Admin_Scans
 *
 * Renders the scans page showing:
 * - Scan controls (run/pause/resume)
 * - Real-time progress
 * - Scan history
 *
 * @since 1.0.0
 */
class CGP_Admin_Scans {

	/**
	 * Display the scans page.
	 *
	 * Shows scan controls (run/pause/resume), progress, and history.
	 *
	 * @since 1.0.0
	 */
	public static function display() {
		// Check user capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'content-guard-pro' ) );
		}

		// Handle form submissions.
		self::handle_actions();

		// Display messages from redirects.
		self::display_redirect_messages();

		// Handle scan_initiated parameter.
		$scan_initiated = isset( $_GET['scan_initiated'] ) && '1' === $_GET['scan_initiated']; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		// Get active scan status.
		$active_scan = CGP_Scanner::get_active_scan();
		$has_active_scan = ! empty( $active_scan );
		
		// Check for pending scans if scan was just initiated.
		$has_pending_scan = false;
		if ( $scan_initiated && ! $has_active_scan ) {
			// Check for recently created pending scans (within last 30 seconds).
			global $wpdb;
			$table_name = $wpdb->prefix . 'content_guard_pro_scans';
			$pending_scan = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT scan_id FROM `{$table_name}`
					WHERE finished_at IS NULL
					  AND status = %s
					  AND started_at > %s
					ORDER BY started_at DESC
					LIMIT 1",
					'pending',
					gmdate( 'Y-m-d H:i:s', time() - 30 )
				)
			);
			$has_pending_scan = ! empty( $pending_scan );
		}

		// Get scheduler stats.
		$scheduler_stats = CGP_Scheduler::get_stats();

		?>
		<div class="wrap content-guard-pro-scans">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<?php self::display_notices(); ?>
			
			<?php if ( $scan_initiated ) : ?>
				<div class="notice notice-success is-dismissible">
					<p>
						<strong><?php esc_html_e( 'Scan initiated successfully!', 'content-guard-pro' ); ?></strong>
						<?php esc_html_e( 'Your scan has been scheduled and will start shortly. You can monitor its progress below.', 'content-guard-pro' ); ?>
					</p>
				</div>
			<?php endif; ?>
			
			<div class="content-guard-pro-scans-content">
				<p><?php esc_html_e( 'Run scans, view progress, and manage scan history.', 'content-guard-pro' ); ?></p>
				
				<!-- Scan Controls -->
				<div class="content-guard-pro-scan-controls">
					<h2><?php esc_html_e( 'Scan Controls', 'content-guard-pro' ); ?></h2>
					
					<?php if ( $has_active_scan ) : ?>
						<div class="notice notice-info inline">
							<p>
								<?php esc_html_e( 'A scan is currently running.', 'content-guard-pro' ); ?>
								<a href="<?php echo esc_url( add_query_arg( 'view_progress', '1' ) ); ?>" class="button">
									<?php esc_html_e( 'View Progress', 'content-guard-pro' ); ?>
								</a>
							</p>
						</div>
					<?php elseif ( $has_pending_scan || $scan_initiated ) : ?>
						<div class="notice notice-info inline" id="cgp-scan-starting-notice">
							<p>
								<span class="spinner is-active" style="float: none; margin: 0 5px 0 0;"></span>
								<strong><?php esc_html_e( 'Scan starting...', 'content-guard-pro' ); ?></strong>
								<?php esc_html_e( 'Your scan is being initialized and will appear in the Active Scans section below shortly.', 'content-guard-pro' ); ?>
							</p>
						</div>
					<?php endif; ?>
					
					<?php
					// Get license info for scan controls.
					$can_standard_scan = true;
					$can_manual_scan = true;
					$is_free_tier = false;
					$is_paid_tier = false;
					
					if ( class_exists( 'CGP_License_Manager' ) ) {
						$can_standard_scan = CGP_License_Manager::can( 'standard_scan' );
						$can_manual_scan = CGP_License_Manager::can_run_manual_scan();
						$is_free_tier = ! CGP_License_Manager::is_paid();
						$is_paid_tier = CGP_License_Manager::is_paid();
					}
					?>
					
					<?php if ( ! $can_manual_scan ) : ?>
						<div class="notice notice-warning inline" style="margin-bottom: 15px;">
							<p>
								<?php if ( $is_free_tier ) : ?>
									<strong><?php esc_html_e( 'Daily scan limit reached.', 'content-guard-pro' ); ?></strong>
									<?php esc_html_e( 'Free plan allows 1 manual scan per day.', 'content-guard-pro' ); ?>
								<?php else : ?>
									<strong><?php esc_html_e( 'Daily scan limit reached.', 'content-guard-pro' ); ?></strong>
									<?php esc_html_e( 'Your current plan limit has been reached for today.', 'content-guard-pro' ); ?>
								<?php endif; ?>
								<a href="https://contentguardpro.com/pricing" target="_blank" rel="noopener noreferrer" class="button button-small">
									<?php esc_html_e( 'Upgrade for Unlimited', 'content-guard-pro' ); ?>
								</a>
							</p>
						</div>
					<?php endif; ?>
					
					<div class="content-guard-pro-scan-actions">
						<!-- Run New Scan -->
						<form method="post" style="display: inline-block; margin-right: 10px;" id="cgp-run-scan-form">
							<?php wp_nonce_field( 'content_guard_pro_run_scan', 'content_guard_pro_run_scan_nonce' ); ?>
							<input type="hidden" name="content_guard_pro_action" value="run_scan" />
							
							<label for="scan_mode">
								<strong><?php esc_html_e( 'Scan Mode:', 'content-guard-pro' ); ?></strong>
							</label>
							<select name="scan_mode" id="scan_mode">
								<option value="quick"><?php esc_html_e( 'Quick (Posts Only)', 'content-guard-pro' ); ?></option>
								<option value="standard" <?php echo $can_standard_scan ? 'selected' : 'disabled'; ?>>
									<?php 
									if ( $can_standard_scan ) {
										esc_html_e( 'Standard (Posts + Meta + Options)', 'content-guard-pro' );
									} else {
										esc_html_e( 'Standard (Posts + Meta + Options) — Upgrade Required', 'content-guard-pro' );
									}
									?>
								</option>
							</select>
							
							<button type="submit" class="button button-primary" id="cgp-run-scan-button" <?php disabled( $has_active_scan || ! $can_manual_scan ); ?>>
								<?php esc_html_e( 'Run New Scan', 'content-guard-pro' ); ?>
							</button>
						</form>
						
						<?php if ( $has_active_scan ) : ?>
							<!-- Cancel Active Scan -->
							<form method="post" style="display: inline-block;">
								<?php wp_nonce_field( 'content_guard_pro_cancel_scan', 'content_guard_pro_cancel_scan_nonce' ); ?>
								<input type="hidden" name="content_guard_pro_action" value="cancel_scan" />
								<button type="submit" class="button" onclick="return confirm('<?php esc_attr_e( 'Are you sure you want to cancel the active scan?', 'content-guard-pro' ); ?>');">
									<?php esc_html_e( 'Cancel Scan', 'content-guard-pro' ); ?>
								</button>
							</form>
						<?php endif; ?>
					</div>
					
					<!-- Scheduler Status -->
					<div class="content-guard-pro-scheduler-status" style="margin-top: 20px;">
						<h3><?php esc_html_e( 'Scheduler Status', 'content-guard-pro' ); ?></h3>
						<ul>
							<li>
								<strong><?php esc_html_e( 'Action Scheduler:', 'content-guard-pro' ); ?></strong>
								<?php if ( $scheduler_stats['action_scheduler_available'] ) : ?>
									<span class="content-guard-pro-status-active"><?php esc_html_e( 'Available', 'content-guard-pro' ); ?></span>
								<?php else : ?>
									<span class="content-guard-pro-status-error"><?php esc_html_e( 'Not Available', 'content-guard-pro' ); ?></span>
								<?php endif; ?>
							</li>
							<li>
								<strong><?php esc_html_e( 'Daily Scan:', 'content-guard-pro' ); ?></strong>
								<?php if ( $scheduler_stats['daily_scan_scheduled'] ) : ?>
									<span class="content-guard-pro-status-active"><?php esc_html_e( 'Enabled', 'content-guard-pro' ); ?></span>
									<?php if ( $scheduler_stats['next_scan_time'] ) : ?>
										<span class="description">
											<?php
											printf(
												/* translators: %s: time until next scan */
												esc_html__( '(Next scan in %s)', 'content-guard-pro' ),
												esc_html( human_time_diff( time(), $scheduler_stats['next_scan_time'] ) )
											);
											?>
										</span>
									<?php endif; ?>
								<?php else : ?>
									<span class="content-guard-pro-status-inactive"><?php esc_html_e( 'Disabled', 'content-guard-pro' ); ?></span>
								<?php endif; ?>
							</li>
						</ul>
					</div>
				</div>
				
				<!-- Active Scans Widget -->
				<?php self::display_active_scans_widget(); ?>
				
				<!-- Scan History -->
				<div class="content-guard-pro-scan-history" style="margin-top: 30px;">
					<h2><?php esc_html_e( 'Scan History', 'content-guard-pro' ); ?></h2>
					<?php
					// Load and display the scans list table.
					if ( ! class_exists( 'CGP_Scans_List_Table' ) ) {
						require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-scans-list-table.php';
					}
					
					$scans_table = new CGP_Scans_List_Table();
					$scans_table->prepare_items();
					$scans_table->display();
					?>
				</div>
			</div>
		</div>
		
		<!-- Scan Details Modal Container -->
		<div id="cgp-scan-details-modal" style="display:none;">
			<div class="content-guard-pro-scan-details-modal" style="padding: 20px;">
				<div class="cgp-scan-details-loading">
					<span class="spinner is-active" style="float: none; margin: 0;"></span>
					<span><?php esc_html_e( 'Loading scan details...', 'content-guard-pro' ); ?></span>
				</div>
				<div class="cgp-scan-details-content" style="display:none;"></div>
			</div>
		</div>

		<!-- Fix #6: Prevent double-submit -->
		<script type="text/javascript">
		(function() {
			var form = document.getElementById('cgp-run-scan-form');
			var button = document.getElementById('cgp-run-scan-button');
			
			if (form && button) {
				form.addEventListener('submit', function(e) {
					// Disable button immediately to prevent double-submit
					if (!button.disabled) {
						button.disabled = true;
						button.textContent = '<?php echo esc_js( __( 'Starting Scan...', 'content-guard-pro' ) ); ?>';
						
						// Re-enable after 5 seconds as a safety fallback
						setTimeout(function() {
							button.disabled = false;
							button.textContent = '<?php echo esc_js( __( 'Run New Scan', 'content-guard-pro' ) ); ?>';
						}, 5000);
					}
				});
			}

			// Handle View Details links and URL parameter.
			function openScanDetailsModal(scanId) {
				if (!scanId) {
					return;
				}

				var $modal = jQuery('#cgp-scan-details-modal');
				var $loading = $modal.find('.cgp-scan-details-loading');
				var $content = $modal.find('.cgp-scan-details-content');

				// Show loading state.
				$loading.show();
				$content.hide().empty();

				// Open ThickBox.
				tb_show(
					'<?php echo esc_js( __( 'Scan Details', 'content-guard-pro' ) ); ?> #' + scanId,
					'#TB_inline?width=700&height=600&inlineId=cgp-scan-details-modal'
				);

				// Fetch scan details via REST API.
				jQuery.ajax({
					url: '<?php echo esc_url( rest_url( 'content-guard-pro/v1/scans/' ) ); ?>' + scanId + '/details',
					method: 'GET',
					beforeSend: function(xhr) {
						xhr.setRequestHeader('X-WP-Nonce', '<?php echo esc_js( wp_create_nonce( 'wp_rest' ) ); ?>');
					},
					success: function(response) {
						$loading.hide();
						
						if (!response || response.code) {
							$content.html(
								'<div class="notice notice-error"><p>' +
								(response.message || '<?php echo esc_js( __( 'Failed to load scan details.', 'content-guard-pro' ) ); ?>') +
								'</p></div>'
							).show();
							return;
						}

						// Build HTML content.
						var html = '<div class="content-guard-pro-scan-details">';
						
						// Status
						var statusClass = 'content-guard-pro-status-' + response.status;
						var statusLabel = response.status.charAt(0).toUpperCase() + response.status.slice(1);
						html += '<div class="cgp-detail-row">';
						html += '<strong><?php echo esc_js( __( 'Status:', 'content-guard-pro' ) ); ?></strong> ';
						html += '<span class="content-guard-pro-status-badge ' + statusClass + '">' + statusLabel + '</span>';
						html += '</div>';

						// Date Started
						if (response.started_at_formatted) {
							html += '<div class="cgp-detail-row">';
							html += '<strong><?php echo esc_js( __( 'Date Started:', 'content-guard-pro' ) ); ?></strong> ';
							html += '<span>' + response.started_at_formatted + '</span>';
							html += '</div>';
						}

						// Duration
						if (response.duration_formatted) {
							html += '<div class="cgp-detail-row">';
							html += '<strong><?php echo esc_js( __( 'Duration:', 'content-guard-pro' ) ); ?></strong> ';
							html += '<span>' + response.duration_formatted + '</span>';
							html += '</div>';
						}

						// Items Checked
						if (response.items_checked !== undefined) {
							html += '<div class="cgp-detail-row">';
							html += '<strong><?php echo esc_js( __( 'Items Checked:', 'content-guard-pro' ) ); ?></strong> ';
							html += '<span>' + response.items_checked.toLocaleString() + '</span>';
							html += '</div>';
						}

						// Findings Summary
						if (response.findings) {
							html += '<div class="cgp-detail-row cgp-findings-summary">';
							html += '<strong><?php echo esc_js( __( 'Findings Summary:', 'content-guard-pro' ) ); ?></strong>';
							
							// Show approximation note if breakdown is approximate.
							if (response.findings.is_approximate) {
								html += '<div class="cgp-approximation-note" style="margin-top: 5px; padding: 8px; background: #fff3cd; border-left: 4px solid #ffc107; color: #856404; font-size: 12px;">';
								html += '<strong><?php echo esc_js( __( 'Note:', 'content-guard-pro' ) ); ?></strong> ';
								html += '<?php echo esc_js( __( 'Findings breakdown is approximate; individual findings may have changed since this scan.', 'content-guard-pro' ) ); ?>';
								html += '</div>';
							}
							
							html += '<div class="cgp-findings-breakdown" style="margin-top: 10px;">';
							
							if (response.findings.total > 0) {
								if (response.findings.critical > 0) {
									html += '<div style="margin: 5px 0;">';
									html += '<span class="content-guard-pro-status-badge content-guard-pro-status-critical"><?php echo esc_js( __( 'Critical', 'content-guard-pro' ) ); ?></span> ';
									html += '<strong>' + response.findings.critical + '</strong>';
									html += '</div>';
								}
								if (response.findings.suspicious > 0) {
									html += '<div style="margin: 5px 0;">';
									html += '<span class="content-guard-pro-status-badge content-guard-pro-status-suspicious"><?php echo esc_js( __( 'Suspicious', 'content-guard-pro' ) ); ?></span> ';
									html += '<strong>' + response.findings.suspicious + '</strong>';
									html += '</div>';
								}
								if (response.findings.review > 0) {
									html += '<div style="margin: 5px 0;">';
									html += '<span class="content-guard-pro-status-badge content-guard-pro-status-review"><?php echo esc_js( __( 'Review', 'content-guard-pro' ) ); ?></span> ';
									html += '<strong>' + response.findings.review + '</strong>';
									html += '</div>';
								}
								html += '<div style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #ddd;">';
								html += '<strong><?php echo esc_js( __( 'Total:', 'content-guard-pro' ) ); ?></strong> ' + response.findings.total;
								html += ' <span style="color: #666; font-size: 11px;">(<?php echo esc_js( __( 'accurate count', 'content-guard-pro' ) ); ?>)</span>';
								html += '</div>';
							} else {
								html += '<p style="margin: 5px 0; color: #46b450;"><?php echo esc_js( __( 'No findings detected.', 'content-guard-pro' ) ); ?></p>';
							}
							
							html += '</div>';
							html += '</div>';
						}

						// Link to Findings
						if (response.findings && response.findings.total > 0) {
							var findingsUrl = '<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-findings&scan_id=' ) ); ?>' + scanId;
							html += '<div class="cgp-detail-row" style="margin-top: 15px; padding-top: 15px; border-top: 1px solid #ddd;">';
							html += '<a href="' + findingsUrl + '" class="button button-primary"><?php echo esc_js( __( 'View All Findings', 'content-guard-pro' ) ); ?></a>';
							html += '</div>';
						}

						html += '</div>';
						
						$content.html(html).show();
					},
					error: function(xhr, status, error) {
						$loading.hide();
						$content.html(
							'<div class="notice notice-error"><p><?php echo esc_js( __( 'Failed to load scan details. Please try again.', 'content-guard-pro' ) ); ?></p></div>'
						).show();
					}
				});
			}

			// Handle clicks on View Details links.
			jQuery(document).on('click', '.cgp-view-scan-details', function(e) {
				e.preventDefault();
				var scanId = jQuery(this).data('scan-id');
				openScanDetailsModal(scanId);
			});

			// Check URL for action=view parameter on page load.
			var urlParams = new URLSearchParams(window.location.search);
			if (urlParams.get('action') === 'view' && urlParams.get('scan_id')) {
				var scanId = parseInt(urlParams.get('scan_id'), 10);
				if (scanId > 0) {
					// Small delay to ensure ThickBox is ready.
					setTimeout(function() {
						openScanDetailsModal(scanId);
						// Clean URL after opening modal to prevent re-opening on refresh.
						var cleanUrl = window.location.pathname + '?page=content-guard-pro-scans';
						window.history.replaceState({}, document.title, cleanUrl);
					}, 100);
				}
			}
		})();
		</script>
		<?php
	}

	/**
	 * Handle page actions.
	 *
	 * Processes form submissions for running scans, canceling scans, etc.
	 *
	 * @since 1.0.0
	 */
	private static function handle_actions() {
		// Run new scan.
		if ( isset( $_POST['content_guard_pro_action'] ) && 'run_scan' === $_POST['content_guard_pro_action'] ) {
			// Verify nonce.
			if ( ! isset( $_POST['content_guard_pro_run_scan_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['content_guard_pro_run_scan_nonce'] ) ), 'content_guard_pro_run_scan' ) ) {
				add_settings_error(
					'content_guard_pro_scans',
					'invalid_nonce',
					__( 'Security check failed. Please try again.', 'content-guard-pro' ),
					'error'
				);
				return;
			}

			// Check permissions.
			if ( ! current_user_can( 'manage_options' ) ) {
				add_settings_error(
					'content_guard_pro_scans',
					'insufficient_permissions',
					__( 'You do not have permission to run scans.', 'content-guard-pro' ),
					'error'
				);
				return;
			}

			// Get scan mode.
			$scan_mode = isset( $_POST['scan_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['scan_mode'] ) ) : 'standard';
			if ( ! in_array( $scan_mode, array( 'quick', 'standard' ), true ) ) {
				$scan_mode = 'standard';
			}

			// License tier checks.
			if ( class_exists( 'CGP_License_Manager' ) ) {
				// Check if user can run a manual scan today (free tier: 1 per day).
				if ( ! CGP_License_Manager::can_run_manual_scan() ) {
					$is_paid = CGP_License_Manager::is_paid();
					add_settings_error(
						'content_guard_pro_scans',
						'scan_limit_reached',
						sprintf(
							$is_paid
								/* translators: %s: Upgrade URL */
								? __( 'Daily scan limit reached for your plan. <a href="%s" target="_blank">Upgrade</a> for more scans.', 'content-guard-pro' )
								/* translators: %s: Upgrade URL */
								: __( 'Daily scan limit reached. Free plan allows 1 manual scan per day. <a href="%s" target="_blank">Upgrade</a> for unlimited scans.', 'content-guard-pro' ),
							esc_url( 'https://contentguardpro.com/pricing' )
						),
						'error'
					);
					return;
				}

				// Check if standard scan mode is available (requires paid plan).
				if ( 'standard' === $scan_mode && ! CGP_License_Manager::can( 'standard_scan' ) ) {
					// Downgrade to quick scan for free users.
					$scan_mode = 'quick';
					add_settings_error(
						'content_guard_pro_scans',
						'scan_mode_downgraded',
						sprintf(
							/* translators: %s: upgrade URL */
							__( 'Standard scan requires a paid plan. Running Quick scan instead. <a href="%s" target="_blank">Upgrade</a> to access Standard scans.', 'content-guard-pro' ),
							esc_url( 'https://contentguardpro.com/pricing' )
						),
						'warning'
					);
				}
			}

			// IMPORTANT: Create scan record IMMEDIATELY so it appears in Scan History.
			// Load scanner class to create scan record.
			if ( ! class_exists( 'CGP_Scanner' ) ) {
				require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-scanner.php';
			}
			
			global $wpdb;
			$table_name = $wpdb->prefix . 'content_guard_pro_scans';
			
			// Determine targets based on scan mode.
			if ( 'quick' === $scan_mode ) {
				$targets = 'posts';
			} else {
				// Standard mode: posts, postmeta, options.
				$targets = 'posts,postmeta,options';
			}
			
			// Create scan record with 'pending' status.
			$wpdb->insert(
				$table_name,
				array(
					'mode'           => $scan_mode,
					'targets'        => $targets,
					'started_at'     => current_time( 'mysql' ),
					'finished_at'    => null,
					'status'         => 'pending',
					'throttle_state' => 'normal',
					'totals_checked' => 0,
					'totals_flagged' => 0,
					'avg_query_ms'   => 0,
					'peak_mem_mb'    => 0,
					'errors'         => 0,
					'notes'          => wp_json_encode( array(
						'initiated_by' => get_current_user_id(),
						'scan_type'    => 'manual',
					) ),
				),
				array( '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%d', '%d', '%d', '%d', '%s' )
			);
			
			$scan_id = $wpdb->insert_id;
			
			if ( ! $scan_id ) {
				add_settings_error(
					'content_guard_pro_scans',
					'scan_create_failed',
					__( 'Failed to create scan record.', 'content-guard-pro' ),
					'error'
				);
				return;
			}

			// Schedule the scan with the scan_id.
			$action_id = CGP_Scheduler::schedule_scan(
				$scan_mode,
				array(
					'scan_id'      => $scan_id,
					'scan_type'    => 'manual',
					'initiated_by' => get_current_user_id(),
				)
			);

			if ( $action_id ) {
				// SUCCESS: Scan was scheduled successfully.
				// Only NOW increment the manual scan counter (Fix #1).
				if ( class_exists( 'CGP_License_Manager' ) ) {
					CGP_License_Manager::increment_manual_scan_count();
				}
				
				add_settings_error(
					'content_guard_pro_scans',
					'scan_scheduled',
					sprintf(
						/* translators: %s: scan mode (Quick or Standard) */
						__( '%s scan has been scheduled and will start shortly.', 'content-guard-pro' ),
						ucfirst( $scan_mode )
					),
					'success'
				);
			} else {
				// FAILURE: Scheduling failed. Clean up the orphaned scan record.
				$wpdb->delete(
					$table_name,
					array( 'scan_id' => $scan_id ),
					array( '%d' )
				);
				
				add_settings_error(
					'content_guard_pro_scans',
					'scan_failed',
					__( 'Failed to schedule scan. Please check that Action Scheduler is available.', 'content-guard-pro' ),
					'error'
				);
			}
		}

		// Cancel active scan.
		if ( isset( $_POST['content_guard_pro_action'] ) && 'cancel_scan' === $_POST['content_guard_pro_action'] ) {
			// Verify nonce.
			if ( ! isset( $_POST['content_guard_pro_cancel_scan_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['content_guard_pro_cancel_scan_nonce'] ) ), 'content_guard_pro_cancel_scan' ) ) {
				add_settings_error(
					'content_guard_pro_scans',
					'invalid_nonce',
					__( 'Security check failed. Please try again.', 'content-guard-pro' ),
					'error'
				);
				return;
			}

			// Check permissions.
			if ( ! current_user_can( 'manage_options' ) ) {
				add_settings_error(
					'content_guard_pro_scans',
					'insufficient_permissions',
					__( 'You do not have permission to cancel scans.', 'content-guard-pro' ),
					'error'
				);
				return;
			}

			// Cancel all pending scans.
			$cancelled = CGP_Scheduler::cancel_all_scans();

			if ( $cancelled ) {
				add_settings_error(
					'content_guard_pro_scans',
					'scan_cancelled',
					__( 'Scan has been cancelled successfully.', 'content-guard-pro' ),
					'success'
				);
			} else {
				add_settings_error(
					'content_guard_pro_scans',
					'cancel_failed',
					__( 'Failed to cancel scan.', 'content-guard-pro' ),
					'error'
				);
			}
		}

		// Delete scan record.
		// Note: Delete and view actions are now handled in CGP_Admin::process_scan_actions()
		// on admin_init to allow proper redirects before any output.
	}

	/**
	 * Display the Active Scans widget.
	 *
	 * Shows real-time progress for currently active scans.
	 * This widget will be updated via AJAX every 3 seconds.
	 *
	 * @since 1.0.0
	 */
	private static function display_active_scans_widget() {
		?>
		<div class="content-guard-pro-active-scans-widget" style="margin-top: 30px;">
			<h2><?php esc_html_e( 'Active Scans', 'content-guard-pro' ); ?></h2>
			
			<!-- This container will be populated via AJAX -->
			<div id="content-guard-pro-active-scans-container" class="content-guard-pro-widget-box">
				<div class="content-guard-pro-loading">
					<span class="spinner is-active" style="float: none; margin: 0;"></span>
					<span><?php esc_html_e( 'Loading active scans...', 'content-guard-pro' ); ?></span>
				</div>
			</div>
		</div>

		<!-- Template for active scan display (will be cloned by JavaScript) -->
		<script type="text/html" id="content-guard-pro-active-scan-template">
			<div class="content-guard-pro-active-scan" data-scan-id="{{scan_id}}">
				<div class="content-guard-pro-scan-header">
					<strong><?php esc_html_e( 'Scan #{{scan_id}}', 'content-guard-pro' ); ?> • {{mode_label}}</strong>
					<span class="content-guard-pro-scan-status content-guard-pro-status-{{status}}">{{status_label}}</span>
				</div>
				
				<div class="content-guard-pro-scan-progress">
					<div class="content-guard-pro-progress-bar-container">
						<div class="content-guard-pro-progress-bar" style="width: {{progress_percentage}}%;">
							<span class="content-guard-pro-progress-text">{{progress_percentage}}%</span>
						</div>
					</div>
				</div>
				
				<div class="content-guard-pro-scan-stats">
					<div class="content-guard-pro-stat">
						<span class="dashicons dashicons-yes-alt"></span>
						<strong><?php esc_html_e( 'Items Scanned:', 'content-guard-pro' ); ?></strong>
						<span class="content-guard-pro-stat-value">{{items_checked}} / {{items_total}}</span>
					</div>
					<div class="content-guard-pro-stat">
						<span class="dashicons dashicons-warning"></span>
						<strong><?php esc_html_e( 'Findings:', 'content-guard-pro' ); ?></strong>
						<span class="content-guard-pro-stat-value content-guard-pro-findings-count">{{findings}}</span>
					</div>
					<div class="content-guard-pro-stat">
						<span class="dashicons dashicons-clock"></span>
						<strong><?php esc_html_e( 'Elapsed:', 'content-guard-pro' ); ?></strong>
						<span class="content-guard-pro-stat-value">{{elapsed_display}}</span>
					</div>
				</div>
				
				<div class="content-guard-pro-scan-meta">
					<small class="content-guard-pro-updated-indicator">
						<?php esc_html_e( 'Updated: Just now', 'content-guard-pro' ); ?>
						<span class="dashicons dashicons-update content-guard-pro-spin"></span>
					</small>
				</div>
			</div>
		</script>

		<!-- Template for "no active scans" message -->
		<script type="text/html" id="content-guard-pro-no-active-scans-template">
			<div class="content-guard-pro-no-active-scans">
				<span class="dashicons dashicons-yes-alt" style="font-size: 48px; opacity: 0.3;"></span>
				<p><?php esc_html_e( 'No active scans at this time.', 'content-guard-pro' ); ?></p>
				<p class="description"><?php esc_html_e( 'Start a new scan using the controls above.', 'content-guard-pro' ); ?></p>
			</div>
		</script>
		<?php
	}

	/**
	 * Display messages from redirect actions.
	 *
	 * Shows success/error messages after delete or other actions.
	 *
	 * @since 1.0.1
	 */
	private static function display_redirect_messages() {
		if ( ! isset( $_GET['message'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			return;
		}

		$message = sanitize_text_field( wp_unslash( $_GET['message'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		switch ( $message ) {
			case 'deleted':
				?>
				<div class="notice notice-success is-dismissible">
					<p><?php esc_html_e( 'Scan record deleted successfully.', 'content-guard-pro' ); ?></p>
				</div>
				<?php
				break;

			case 'delete_failed':
				?>
				<div class="notice notice-error is-dismissible">
					<p><?php esc_html_e( 'Failed to delete scan record.', 'content-guard-pro' ); ?></p>
				</div>
				<?php
				break;

			case 'not_found':
				?>
				<div class="notice notice-error is-dismissible">
					<p><?php esc_html_e( 'Scan record not found.', 'content-guard-pro' ); ?></p>
				</div>
				<?php
				break;
		}
	}

	/**
	 * Display admin notices.
	 *
	 * @since 1.0.0
	 */
	private static function display_notices() {
		settings_errors( 'content_guard_pro_scans' );
	}
}

