<?php
/**
 * Admin Reports Page
 *
 * Displays scan summaries and generates printable reports.
 * Per PRD Section 3.7: Reporting and Export, Scan Summary.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Admin_Reports
 *
 * Renders the reports page showing:
 * - Scan summary overview
 * - Last scan details
 * - Findings by severity
 * - Performance metrics
 * - Printable format
 *
 * @since 1.0.0
 */
class CGP_Admin_Reports {

	/**
	 * Display the reports page.
	 *
	 * @since 1.0.0
	 */
	public static function display() {
		// Check user capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'content-guard-pro' ) );
		}

		// Get scan data.
		$last_scan       = self::get_last_scan();
		$findings_stats  = self::get_findings_stats();
		$scan_history    = self::get_scan_history( 10 );
		$overall_stats   = self::get_overall_stats();

		?>
		<div class="wrap content-guard-pro-reports">
			<h1>
				<?php echo esc_html( get_admin_page_title() ); ?>
				<button onclick="window.print()" class="button button-secondary no-print" style="margin-left: 10px;">
					<span class="dashicons dashicons-printer" style="margin-top: 3px;"></span>
					<?php esc_html_e( 'Print Report', 'content-guard-pro' ); ?>
				</button>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-findings' ) ); ?>" class="button button-secondary no-print">
					<?php esc_html_e( 'View All Findings', 'content-guard-pro' ); ?>
				</a>
			</h1>

			<!-- Report Header -->
			<div class="content-guard-pro-report-header">
				<h2><?php esc_html_e( 'Security Scan Summary', 'content-guard-pro' ); ?></h2>
				<p class="content-guard-pro-report-meta">
					<strong><?php esc_html_e( 'Site:', 'content-guard-pro' ); ?></strong> <?php echo esc_html( get_bloginfo( 'name' ) ); ?> (<?php echo esc_html( home_url() ); ?>)<br>
					<strong><?php esc_html_e( 'Generated:', 'content-guard-pro' ); ?></strong> <?php echo esc_html( wp_date( 'F j, Y g:i a' ) ); ?><br>
					<strong><?php esc_html_e( 'Plugin Version:', 'content-guard-pro' ); ?></strong> <?php echo esc_html( CONTENT_GUARD_PRO_VERSION ); ?>
				</p>
			</div>

			<!-- Executive Summary -->
			<div class="content-guard-pro-report-section">
				<h3><?php esc_html_e( 'Executive Summary', 'content-guard-pro' ); ?></h3>
				
				<div class="content-guard-pro-summary-grid">
					<!-- Total Scans -->
					<div class="content-guard-pro-summary-box">
						<div class="content-guard-pro-summary-icon">
							<span class="dashicons dashicons-backup"></span>
						</div>
						<div class="content-guard-pro-summary-content">
							<div class="content-guard-pro-summary-value"><?php echo absint( $overall_stats['total_scans'] ); ?></div>
							<div class="content-guard-pro-summary-label"><?php esc_html_e( 'Total Scans Run', 'content-guard-pro' ); ?></div>
						</div>
					</div>

					<!-- Total Items Scanned -->
					<div class="content-guard-pro-summary-box">
						<div class="content-guard-pro-summary-icon">
							<span class="dashicons dashicons-search"></span>
						</div>
						<div class="content-guard-pro-summary-content">
							<div class="content-guard-pro-summary-value"><?php echo esc_html( number_format_i18n( $overall_stats['total_items'] ) ); ?></div>
							<div class="content-guard-pro-summary-label"><?php esc_html_e( 'Items Scanned (All Time)', 'content-guard-pro' ); ?></div>
						</div>
					</div>

					<!-- Total Findings -->
					<div class="content-guard-pro-summary-box">
						<div class="content-guard-pro-summary-icon">
							<span class="dashicons dashicons-warning"></span>
						</div>
						<div class="content-guard-pro-summary-content">
							<div class="content-guard-pro-summary-value"><?php echo esc_html( number_format_i18n( $findings_stats['total'] ) ); ?></div>
							<div class="content-guard-pro-summary-label"><?php esc_html_e( 'Open Security Findings', 'content-guard-pro' ); ?></div>
						</div>
					</div>

					<!-- Critical Findings -->
					<div class="content-guard-pro-summary-box content-guard-pro-critical">
						<div class="content-guard-pro-summary-icon">
							<span class="dashicons dashicons-shield"></span>
						</div>
						<div class="content-guard-pro-summary-content">
							<div class="content-guard-pro-summary-value"><?php echo absint( $findings_stats['critical'] ); ?></div>
							<div class="content-guard-pro-summary-label"><?php esc_html_e( 'Critical Issues', 'content-guard-pro' ); ?></div>
						</div>
					</div>
				</div>
			</div>

			<!-- Last Scan Details -->
			<?php if ( $last_scan ) : ?>
			<div class="content-guard-pro-report-section">
				<h3><?php esc_html_e( 'Last Scan Details', 'content-guard-pro' ); ?></h3>
				
				<table class="content-guard-pro-report-table">
					<tbody>
						<tr>
							<th><?php esc_html_e( 'Scan Date & Time', 'content-guard-pro' ); ?></th>
							<td><?php echo esc_html( mysql2date( 'F j, Y g:i a', $last_scan->started_at ) ); ?></td>
						</tr>
						<tr>
							<th><?php esc_html_e( 'Scan Mode', 'content-guard-pro' ); ?></th>
							<td>
								<span class="content-guard-pro-badge content-guard-pro-mode-<?php echo esc_attr( $last_scan->mode ); ?>">
									<?php echo esc_html( ucfirst( $last_scan->mode ) ); ?>
								</span>
							</td>
						</tr>
						<tr>
							<th><?php esc_html_e( 'Total Items Scanned', 'content-guard-pro' ); ?></th>
							<td><?php echo esc_html( number_format_i18n( absint( $last_scan->totals_checked ) ) ); ?></td>
						</tr>
						<tr>
							<th><?php esc_html_e( 'Total Issues Found', 'content-guard-pro' ); ?></th>
							<td><?php echo esc_html( number_format_i18n( absint( $last_scan->totals_flagged ) ) ); ?></td>
						</tr>
						<tr>
							<th><?php esc_html_e( 'Scan Duration', 'content-guard-pro' ); ?></th>
							<td><?php echo esc_html( self::calculate_duration( $last_scan->started_at, $last_scan->finished_at ) ); ?></td>
						</tr>
						<tr>
							<th><?php esc_html_e( 'Average Query Time', 'content-guard-pro' ); ?></th>
							<td><?php echo absint( $last_scan->avg_query_ms ); ?> ms</td>
						</tr>
						<tr>
							<th><?php esc_html_e( 'Peak Memory Usage', 'content-guard-pro' ); ?></th>
							<td><?php echo absint( $last_scan->peak_mem_mb ); ?> MB</td>
						</tr>
						<tr>
							<th><?php esc_html_e( 'Throttle State', 'content-guard-pro' ); ?></th>
							<td><?php echo esc_html( $last_scan->throttle_state ? ucfirst( $last_scan->throttle_state ) : __( 'Normal', 'content-guard-pro' ) ); ?></td>
						</tr>
						<?php if ( $last_scan->errors > 0 ) : ?>
						<tr>
							<th><?php esc_html_e( 'Errors', 'content-guard-pro' ); ?></th>
							<td class="content-guard-pro-error"><?php echo absint( $last_scan->errors ); ?> <?php esc_html_e( 'error(s) occurred', 'content-guard-pro' ); ?></td>
						</tr>
						<?php endif; ?>
					</tbody>
				</table>
			</div>
			<?php else : ?>
			<div class="content-guard-pro-report-section">
				<div class="content-guard-pro-no-data">
					<span class="dashicons dashicons-info"></span>
					<p><?php esc_html_e( 'No scans have been completed yet. Run your first scan to see results here.', 'content-guard-pro' ); ?></p>
					<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-scans' ) ); ?>" class="button button-primary no-print">
						<?php esc_html_e( 'Run First Scan', 'content-guard-pro' ); ?>
					</a>
				</div>
			</div>
			<?php endif; ?>

			<!-- Findings by Severity -->
			<div class="content-guard-pro-report-section">
				<h3><?php esc_html_e( 'Current Findings by Severity', 'content-guard-pro' ); ?></h3>
				
				<div class="content-guard-pro-severity-breakdown">
					<!-- Critical -->
					<div class="content-guard-pro-severity-item content-guard-pro-severity-critical">
						<div class="content-guard-pro-severity-header">
							<span class="dashicons dashicons-warning"></span>
							<h4><?php esc_html_e( 'Critical', 'content-guard-pro' ); ?></h4>
							<span class="content-guard-pro-severity-count"><?php echo absint( $findings_stats['critical'] ); ?></span>
						</div>
						<div class="content-guard-pro-severity-description">
							<?php esc_html_e( 'High-risk threats requiring immediate attention. External scripts, iframes, or confirmed malicious content.', 'content-guard-pro' ); ?>
						</div>
						<div class="content-guard-pro-severity-bar">
							<div class="content-guard-pro-severity-bar-fill content-guard-pro-critical" style="width: <?php echo esc_attr( self::calculate_percentage( $findings_stats['critical'], $findings_stats['total'] ) ); ?>%;"></div>
						</div>
					</div>

					<!-- Suspicious -->
					<div class="content-guard-pro-severity-item content-guard-pro-severity-suspicious">
						<div class="content-guard-pro-severity-header">
							<span class="dashicons dashicons-flag"></span>
							<h4><?php esc_html_e( 'Suspicious', 'content-guard-pro' ); ?></h4>
							<span class="content-guard-pro-severity-count"><?php echo absint( $findings_stats['suspicious'] ); ?></span>
						</div>
						<div class="content-guard-pro-severity-description">
							<?php esc_html_e( 'Potentially harmful content. URL shorteners, hidden elements, or obfuscated code.', 'content-guard-pro' ); ?>
						</div>
						<div class="content-guard-pro-severity-bar">
							<div class="content-guard-pro-severity-bar-fill content-guard-pro-suspicious" style="width: <?php echo esc_attr( self::calculate_percentage( $findings_stats['suspicious'], $findings_stats['total'] ) ); ?>%;"></div>
						</div>
					</div>

					<!-- Review -->
					<div class="content-guard-pro-severity-item content-guard-pro-severity-review">
						<div class="content-guard-pro-severity-header">
							<span class="dashicons dashicons-visibility"></span>
							<h4><?php esc_html_e( 'Review', 'content-guard-pro' ); ?></h4>
							<span class="content-guard-pro-severity-count"><?php echo absint( $findings_stats['review'] ); ?></span>
						</div>
						<div class="content-guard-pro-severity-description">
							<?php esc_html_e( 'Anomalies requiring manual review. Unusual link patterns or content characteristics.', 'content-guard-pro' ); ?>
						</div>
						<div class="content-guard-pro-severity-bar">
							<div class="content-guard-pro-severity-bar-fill content-guard-pro-review" style="width: <?php echo esc_attr( self::calculate_percentage( $findings_stats['review'], $findings_stats['total'] ) ); ?>%;"></div>
						</div>
					</div>
				</div>

				<?php if ( $findings_stats['total'] === 0 ) : ?>
				<div class="content-guard-pro-no-findings">
					<span class="dashicons dashicons-yes-alt"></span>
					<p><?php esc_html_e( 'No open security findings. Your site looks clean!', 'content-guard-pro' ); ?></p>
				</div>
				<?php endif; ?>
			</div>

			<!-- Scan History -->
			<?php if ( ! empty( $scan_history ) ) : ?>
			<div class="content-guard-pro-report-section">
				<h3><?php esc_html_e( 'Recent Scan History', 'content-guard-pro' ); ?></h3>
				
				<table class="content-guard-pro-report-table content-guard-pro-scan-history">
					<thead>
						<tr>
							<th><?php esc_html_e( 'Date & Time', 'content-guard-pro' ); ?></th>
							<th><?php esc_html_e( 'Mode', 'content-guard-pro' ); ?></th>
							<th><?php esc_html_e( 'Items Scanned', 'content-guard-pro' ); ?></th>
							<th><?php esc_html_e( 'Issues Found', 'content-guard-pro' ); ?></th>
							<th><?php esc_html_e( 'Duration', 'content-guard-pro' ); ?></th>
							<th><?php esc_html_e( 'Performance', 'content-guard-pro' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php foreach ( $scan_history as $scan ) : ?>
						<tr>
							<td><?php echo esc_html( mysql2date( 'M j, Y g:i a', $scan->started_at ) ); ?></td>
							<td>
								<span class="content-guard-pro-badge content-guard-pro-mode-<?php echo esc_attr( $scan->mode ); ?>">
									<?php echo esc_html( ucfirst( $scan->mode ) ); ?>
								</span>
							</td>
							<td><?php echo esc_html( number_format_i18n( absint( $scan->totals_checked ) ) ); ?></td>
							<td>
								<?php if ( $scan->totals_flagged > 0 ) : ?>
									<strong><?php echo esc_html( number_format_i18n( absint( $scan->totals_flagged ) ) ); ?></strong>
								<?php else : ?>
									<span style="color: #00a32a;">0</span>
								<?php endif; ?>
							</td>
							<td><?php echo esc_html( self::calculate_duration( $scan->started_at, $scan->finished_at ) ); ?></td>
							<td>
								<span class="content-guard-pro-performance-metric">
									<?php echo absint( $scan->avg_query_ms ); ?>ms / <?php echo absint( $scan->peak_mem_mb ); ?>MB
								</span>
							</td>
						</tr>
						<?php endforeach; ?>
					</tbody>
				</table>
			</div>
			<?php endif; ?>

			<!-- Recommendations -->
			<?php if ( $findings_stats['critical'] > 0 || $findings_stats['suspicious'] > 0 ) : ?>
			<div class="content-guard-pro-report-section no-print">
				<h3><?php esc_html_e( 'Recommended Actions', 'content-guard-pro' ); ?></h3>
				
				<div class="content-guard-pro-recommendations">
					<?php if ( $findings_stats['critical'] > 0 ) : ?>
					<div class="content-guard-pro-recommendation content-guard-pro-critical">
						<span class="dashicons dashicons-warning"></span>
						<div>
							<strong><?php esc_html_e( 'Critical Issues Require Immediate Attention', 'content-guard-pro' ); ?></strong>
							<p>
								<?php
								printf(
									/* translators: %d: number of critical findings */
									esc_html( _n(
										'You have %d critical security finding. Review and quarantine or remove the malicious content immediately.',
										'You have %d critical security findings. Review and quarantine or remove the malicious content immediately.',
										$findings_stats['critical'],
										'content-guard-pro'
									) ),
									absint( $findings_stats['critical'] )
								);
								?>
							</p>
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-findings&severity=critical' ) ); ?>" class="button button-primary">
								<?php esc_html_e( 'Review Critical Findings', 'content-guard-pro' ); ?>
							</a>
						</div>
					</div>
					<?php endif; ?>

					<?php if ( $findings_stats['suspicious'] > 0 ) : ?>
					<div class="content-guard-pro-recommendation content-guard-pro-suspicious">
						<span class="dashicons dashicons-flag"></span>
						<div>
							<strong><?php esc_html_e( 'Suspicious Content Detected', 'content-guard-pro' ); ?></strong>
							<p>
								<?php
								printf(
									/* translators: %d: number of suspicious findings */
									esc_html( _n(
										'%d suspicious finding requires review. Investigate these items to determine if they are legitimate or malicious.',
										'%d suspicious findings require review. Investigate these items to determine if they are legitimate or malicious.',
										$findings_stats['suspicious'],
										'content-guard-pro'
									) ),
									absint( $findings_stats['suspicious'] )
								);
								?>
							</p>
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-findings&severity=suspicious' ) ); ?>" class="button">
								<?php esc_html_e( 'Review Suspicious Findings', 'content-guard-pro' ); ?>
							</a>
						</div>
					</div>
					<?php endif; ?>

					<?php if ( $findings_stats['review'] > 0 ) : ?>
					<div class="content-guard-pro-recommendation content-guard-pro-review">
						<span class="dashicons dashicons-visibility"></span>
						<div>
							<strong><?php esc_html_e( 'Items for Manual Review', 'content-guard-pro' ); ?></strong>
							<p>
								<?php
								printf(
									/* translators: %d: number of review findings */
									esc_html( _n(
										'%d item flagged for manual review. These are typically anomalies that require human judgment.',
										'%d items flagged for manual review. These are typically anomalies that require human judgment.',
										$findings_stats['review'],
										'content-guard-pro'
									) ),
									absint( $findings_stats['review'] )
								);
								?>
							</p>
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-findings&severity=review' ) ); ?>" class="button">
								<?php esc_html_e( 'Review Items', 'content-guard-pro' ); ?>
							</a>
						</div>
					</div>
					<?php endif; ?>
				</div>
			</div>
			<?php endif; ?>

			<!-- Report Footer -->
			<div class="content-guard-pro-report-footer">
				<p>
					<strong><?php esc_html_e( 'Content Guard Pro', 'content-guard-pro' ); ?></strong> - 
					<?php esc_html_e( 'Database-first WordPress security scanning for malicious content, spam links, and SEO injections.', 'content-guard-pro' ); ?>
				</p>
				<p class="content-guard-pro-disclaimer">
					<?php esc_html_e( 'This report is generated automatically. While we strive for accuracy, manual verification is recommended for all findings before taking action.', 'content-guard-pro' ); ?>
				</p>
			</div>
		</div>

		<style>
		/* Print-specific styles */
		@media print {
			.no-print,
			#wpadminbar,
			#adminmenumain,
			.update-nag,
			.notice {
				display: none !important;
			}
			
			.wrap {
				margin: 0 !important;
			}
			
			.content-guard-pro-report-section {
				page-break-inside: avoid;
			}
			
			.content-guard-pro-severity-item {
				break-inside: avoid;
			}
		}

		/* Report Styles */
		.content-guard-pro-reports {
			max-width: 1200px;
		}

		.content-guard-pro-report-header {
			background: #fff;
			border: 1px solid #ccd0d4;
			border-radius: 4px;
			padding: 30px;
			margin-bottom: 20px;
			box-shadow: 0 1px 1px rgba(0,0,0,.04);
		}

		.content-guard-pro-report-header h2 {
			margin-top: 0;
			color: #1d2327;
		}

		.content-guard-pro-report-meta {
			color: #646970;
			line-height: 1.8;
		}

		.content-guard-pro-report-section {
			background: #fff;
			border: 1px solid #ccd0d4;
			border-radius: 4px;
			padding: 25px;
			margin-bottom: 20px;
			box-shadow: 0 1px 1px rgba(0,0,0,.04);
		}

		.content-guard-pro-report-section h3 {
			margin-top: 0;
			padding-bottom: 15px;
			border-bottom: 2px solid #2271b1;
			color: #1d2327;
		}

		/* Summary Grid */
		.content-guard-pro-summary-grid {
			display: grid;
			grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
			gap: 20px;
			margin-top: 20px;
		}

		.content-guard-pro-summary-box {
			display: flex;
			align-items: center;
			padding: 20px;
			background: #f6f7f7;
			border-radius: 4px;
			border-left: 4px solid #2271b1;
		}

		.content-guard-pro-summary-box.content-guard-pro-critical {
			border-left-color: #d63638;
			background: #fef7f7;
		}

		.content-guard-pro-summary-icon {
			font-size: 40px;
			margin-right: 15px;
			color: #2271b1;
		}

		.content-guard-pro-summary-box.content-guard-pro-critical .content-guard-pro-summary-icon {
			color: #d63638;
		}

		.content-guard-pro-summary-value {
			font-size: 32px;
			font-weight: 700;
			line-height: 1;
			color: #1d2327;
		}

		.content-guard-pro-summary-label {
			font-size: 13px;
			color: #646970;
			margin-top: 5px;
		}

		/* Report Table */
		.content-guard-pro-report-table {
			width: 100%;
			border-collapse: collapse;
			margin-top: 20px;
		}

		.content-guard-pro-report-table th,
		.content-guard-pro-report-table td {
			padding: 12px;
			text-align: left;
			border-bottom: 1px solid #dcdcde;
		}

		.content-guard-pro-report-table th {
			background: #f6f7f7;
			font-weight: 600;
			width: 30%;
		}

		.content-guard-pro-report-table tbody th {
			background: #f0f6fc;
		}

		.content-guard-pro-report-table .content-guard-pro-error {
			color: #d63638;
			font-weight: 600;
		}

		/* Badges */
		.content-guard-pro-badge {
			display: inline-block;
			padding: 4px 12px;
			border-radius: 3px;
			font-size: 12px;
			font-weight: 600;
			text-transform: uppercase;
		}

		.content-guard-pro-badge.content-guard-pro-mode-quick {
			background: #f0f6fc;
			color: #2271b1;
		}

		.content-guard-pro-badge.content-guard-pro-mode-standard {
			background: #f0f6fc;
			color: #00a32a;
		}

		/* Severity Breakdown */
		.content-guard-pro-severity-breakdown {
			margin-top: 20px;
		}

		.content-guard-pro-severity-item {
			margin-bottom: 20px;
			padding: 20px;
			border-radius: 4px;
			border: 1px solid #dcdcde;
		}

		.content-guard-pro-severity-header {
			display: flex;
			align-items: center;
			margin-bottom: 10px;
		}

		.content-guard-pro-severity-header .dashicons {
			font-size: 24px;
			margin-right: 10px;
		}

		.content-guard-pro-severity-header h4 {
			margin: 0;
			flex: 1;
		}

		.content-guard-pro-severity-count {
			font-size: 24px;
			font-weight: 700;
		}

		.content-guard-pro-severity-item.content-guard-pro-severity-critical {
			background: #fef7f7;
			border-left: 4px solid #d63638;
		}

		.content-guard-pro-severity-item.content-guard-pro-severity-critical .dashicons,
		.content-guard-pro-severity-item.content-guard-pro-severity-critical .content-guard-pro-severity-count {
			color: #d63638;
		}

		.content-guard-pro-severity-item.content-guard-pro-severity-suspicious {
			background: #fef9f4;
			border-left: 4px solid #dba617;
		}

		.content-guard-pro-severity-item.content-guard-pro-severity-suspicious .dashicons,
		.content-guard-pro-severity-item.content-guard-pro-severity-suspicious .content-guard-pro-severity-count {
			color: #dba617;
		}

		.content-guard-pro-severity-item.content-guard-pro-severity-review {
			background: #f0f6fc;
			border-left: 4px solid #2271b1;
		}

		.content-guard-pro-severity-item.content-guard-pro-severity-review .dashicons,
		.content-guard-pro-severity-item.content-guard-pro-severity-review .content-guard-pro-severity-count {
			color: #2271b1;
		}

		.content-guard-pro-severity-description {
			color: #646970;
			margin-bottom: 10px;
		}

		.content-guard-pro-severity-bar {
			height: 8px;
			background: #dcdcde;
			border-radius: 4px;
			overflow: hidden;
		}

		.content-guard-pro-severity-bar-fill {
			height: 100%;
			transition: width 0.3s ease;
		}

		.content-guard-pro-severity-bar-fill.content-guard-pro-critical {
			background: #d63638;
		}

		.content-guard-pro-severity-bar-fill.content-guard-pro-suspicious {
			background: #dba617;
		}

		.content-guard-pro-severity-bar-fill.content-guard-pro-review {
			background: #2271b1;
		}

		/* No Data */
		.content-guard-pro-no-data,
		.content-guard-pro-no-findings {
			text-align: center;
			padding: 40px 20px;
			color: #646970;
		}

		.content-guard-pro-no-data .dashicons,
		.content-guard-pro-no-findings .dashicons {
			font-size: 48px;
			width: 48px;
			height: 48px;
			opacity: 0.5;
		}

		.content-guard-pro-no-findings .dashicons {
			color: #00a32a;
		}

		/* Recommendations */
		.content-guard-pro-recommendations {
			margin-top: 20px;
		}

		.content-guard-pro-recommendation {
			display: flex;
			padding: 20px;
			margin-bottom: 15px;
			border-radius: 4px;
			border-left: 4px solid #2271b1;
		}

		.content-guard-pro-recommendation.content-guard-pro-critical {
			background: #fef7f7;
			border-left-color: #d63638;
		}

		.content-guard-pro-recommendation.content-guard-pro-suspicious {
			background: #fef9f4;
			border-left-color: #dba617;
		}

		.content-guard-pro-recommendation.content-guard-pro-review {
			background: #f0f6fc;
			border-left-color: #2271b1;
		}

		.content-guard-pro-recommendation .dashicons {
			font-size: 24px;
			margin-right: 15px;
			margin-top: 2px;
		}

		.content-guard-pro-recommendation.content-guard-pro-critical .dashicons {
			color: #d63638;
		}

		.content-guard-pro-recommendation.content-guard-pro-suspicious .dashicons {
			color: #dba617;
		}

		.content-guard-pro-recommendation.content-guard-pro-review .dashicons {
			color: #2271b1;
		}

		.content-guard-pro-recommendation strong {
			display: block;
			margin-bottom: 5px;
		}

		.content-guard-pro-recommendation p {
			margin: 5px 0 15px 0;
			color: #646970;
		}

		/* Report Footer */
		.content-guard-pro-report-footer {
			background: #f6f7f7;
			border: 1px solid #dcdcde;
			border-radius: 4px;
			padding: 20px;
			margin-top: 20px;
			text-align: center;
			color: #646970;
		}

		.content-guard-pro-report-footer p {
			margin: 10px 0;
		}

		.content-guard-pro-disclaimer {
			font-size: 12px;
			font-style: italic;
		}

		/* Performance Metric */
		.content-guard-pro-performance-metric {
			font-family: 'Courier New', monospace;
			font-size: 12px;
		}
		</style>
		<?php
	}

	/**
	 * Get last completed scan.
	 *
	 * @since 1.0.0
	 * @return object|null Scan object or null.
	 */
	private static function get_last_scan() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';

		return $wpdb->get_row(
			$wpdb->prepare(
				"SELECT scan_id, started_at, finished_at, mode, totals_checked, totals_flagged, 
				avg_query_ms, peak_mem_mb, throttle_state, errors
				FROM `{$table_name}`
				WHERE finished_at IS NOT NULL
				ORDER BY started_at DESC
				LIMIT %d",
				1
			)
		);
	}

	/**
	 * Get findings statistics.
	 *
	 * @since 1.0.0
	 * @return array Statistics array.
	 */
	private static function get_findings_stats() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT severity, COUNT(*) as count
				FROM `{$table_name}`
				WHERE status = %s
				GROUP BY severity",
				'open'
			)
		);

		$stats = array(
			'total'      => 0,
			'critical'   => 0,
			'suspicious' => 0,
			'review'     => 0,
		);

		foreach ( $results as $row ) {
			$stats[ $row->severity ] = absint( $row->count );
			$stats['total']         += absint( $row->count );
		}

		return $stats;
	}

	/**
	 * Get scan history.
	 *
	 * @since 1.0.0
	 * @param int $limit Number of scans to retrieve.
	 * @return array Array of scan objects.
	 */
	private static function get_scan_history( $limit = 10 ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';

		return $wpdb->get_results(
			$wpdb->prepare(
				"SELECT scan_id, started_at, finished_at, mode, totals_checked, totals_flagged, 
				avg_query_ms, peak_mem_mb
				FROM `{$table_name}`
				WHERE finished_at IS NOT NULL
				ORDER BY started_at DESC
				LIMIT %d",
				absint( $limit )
			)
		);
	}

	/**
	 * Get overall statistics.
	 *
	 * @since 1.0.0
	 * @return array Statistics array.
	 */
	private static function get_overall_stats() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';

		$stats = array(
			'total_scans' => 0,
			'total_items' => 0,
		);

		// Get total scans.
		$stats['total_scans'] = $wpdb->get_var(
			"SELECT COUNT(*) FROM `{$table_name}`
			WHERE finished_at IS NOT NULL"
		);

		// Get total items scanned.
		$stats['total_items'] = $wpdb->get_var(
			"SELECT SUM(totals_checked) FROM `{$table_name}`
			WHERE finished_at IS NOT NULL"
		);

		// Ensure values are integers.
		$stats['total_scans'] = absint( $stats['total_scans'] );
		$stats['total_items'] = absint( $stats['total_items'] );

		return $stats;
	}

	/**
	 * Calculate scan duration.
	 *
	 * @since 1.0.0
	 * @param string $start Start datetime.
	 * @param string $end   End datetime.
	 * @return string Human-readable duration.
	 */
	private static function calculate_duration( $start, $end ) {
		if ( ! $start || ! $end ) {
			return __( 'N/A', 'content-guard-pro' );
		}

		$start_time = strtotime( $start );
		$end_time   = strtotime( $end );

		// Check if strtotime() failed.
		if ( false === $start_time || false === $end_time ) {
			return __( 'N/A', 'content-guard-pro' );
		}

		$duration = $end_time - $start_time;

		// Check for negative duration (end before start).
		if ( $duration < 0 ) {
			return __( 'N/A', 'content-guard-pro' );
		}

		if ( $duration < 60 ) {
			/* translators: %d: number of seconds */
			return sprintf( _n( '%d second', '%d seconds', $duration, 'content-guard-pro' ), $duration );
		} elseif ( $duration < 3600 ) {
			$minutes = floor( $duration / 60 );
			$seconds = $duration % 60;
			/* translators: 1: minutes, 2: seconds */
			return sprintf( __( '%1$d min %2$d sec', 'content-guard-pro' ), $minutes, $seconds );
		} else {
			$hours   = floor( $duration / 3600 );
			$minutes = floor( ( $duration % 3600 ) / 60 );
			/* translators: 1: hours, 2: minutes */
			return sprintf( __( '%1$d hr %2$d min', 'content-guard-pro' ), $hours, $minutes );
		}
	}

	/**
	 * Calculate percentage.
	 *
	 * @since 1.0.0
	 * @param int $part  Part value.
	 * @param int $total Total value.
	 * @return float Percentage.
	 */
	private static function calculate_percentage( $part, $total ) {
		if ( $total === 0 ) {
			return 0;
		}

		return round( ( $part / $total ) * 100, 1 );
	}
}
