<?php
/**
 * Admin Quarantine Page
 *
 * Handles the quarantine page display showing all quarantined items
 * with bulk restore and un-quarantine options.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Admin_Quarantine
 *
 * Renders the quarantine page showing:
 * - All quarantined items
 * - Bulk restore options
 * - Un-quarantine actions
 *
 * @since 1.0.0
 */
class CGP_Admin_Quarantine {

	/**
	 * Display the quarantine page.
	 *
	 * Shows all quarantined items with bulk restore/un-quarantine options.
	 *
	 * @since 1.0.0
	 */
	public static function display() {
		// Check user capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'content-guard-pro' ) );
		}

		// Handle single row actions (un-quarantine, delete).
		self::handle_actions();

		// Display any admin notices.
		self::display_notices();

		// Load the list table class.
		if ( ! class_exists( 'CGP_Quarantine_List_Table' ) ) {
			require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-quarantine-list-table.php';
		}

		// Create list table instance.
		$list_table = new CGP_Quarantine_List_Table();
		$list_table->prepare_items();

		?>
		<div class="wrap content-guard-pro-quarantine">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<div class="content-guard-pro-quarantine-intro">
				<p class="description">
					<?php esc_html_e( 'Quarantined items are neutralized at render-time but remain in your database. You can un-quarantine them to restore normal display, or delete them permanently.', 'content-guard-pro' ); ?>
				</p>
			</div>

			<form method="post" id="content-guard-pro-quarantine-form">
				<?php
				$list_table->display();
				?>
			</form>
		</div>
		<?php
	}

	/**
	 * Handle single row actions (un-quarantine, delete).
	 *
	 * @since 1.0.0
	 */
	private static function handle_actions() {
		// Check if action is set.
		if ( ! isset( $_GET['action'] ) || ! isset( $_GET['finding_id'] ) ) {
			return;
		}

		$action     = sanitize_text_field( wp_unslash( $_GET['action'] ) );
		$finding_id = absint( $_GET['finding_id'] );

		if ( ! $finding_id ) {
			return;
		}

		// Handle un-quarantine action.
		if ( 'unquarantine' === $action ) {
			// Verify nonce.
			if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'content_guard_pro_unquarantine_' . $finding_id ) ) {
				wp_die( esc_html__( 'Security check failed.', 'content-guard-pro' ) );
			}

			// Un-quarantine via CGP_Quarantine class.
			if ( class_exists( 'CGP_Quarantine' ) ) {
				$result = CGP_Quarantine::unquarantine_finding( $finding_id );

				if ( $result ) {
					set_transient( 'content_guard_pro_admin_notice', array(
						'type'    => 'success',
						'message' => __( 'Finding un-quarantined successfully.', 'content-guard-pro' ),
					), 30 );
				} else {
					set_transient( 'content_guard_pro_admin_notice', array(
						'type'    => 'error',
						'message' => __( 'Failed to un-quarantine finding.', 'content-guard-pro' ),
					), 30 );
				}
			}

			// Redirect to avoid resubmission.
			wp_safe_redirect( admin_url( 'admin.php?page=content-guard-pro-quarantine' ) );
			exit;
		}

		// Handle delete action.
		if ( 'delete' === $action ) {
			// Verify nonce.
			if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'content_guard_pro_delete_' . $finding_id ) ) {
				wp_die( esc_html__( 'Security check failed.', 'content-guard-pro' ) );
			}

			global $wpdb;
			$table_name = $wpdb->prefix . 'content_guard_pro_findings';

			// Get finding details for audit log.
			$finding = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT * FROM `{$table_name}` WHERE id = %d",
					$finding_id
				)
			);

			if ( $finding ) {
				// Log to audit before deletion.
				self::log_finding_action( $finding, 'delete' );

				// Delete the finding.
				$deleted = $wpdb->delete(
					$table_name,
					array( 'id' => $finding_id ),
					array( '%d' )
				);

				if ( false !== $deleted ) {
					// Clear cache.
					if ( class_exists( 'CGP_Quarantine' ) ) {
						CGP_Quarantine::clear_cache();
					}

					set_transient( 'content_guard_pro_admin_notice', array(
						'type'    => 'success',
						'message' => __( 'Finding deleted successfully.', 'content-guard-pro' ),
					), 30 );
				} else {
					set_transient( 'content_guard_pro_admin_notice', array(
						'type'    => 'error',
						'message' => __( 'Failed to delete finding.', 'content-guard-pro' ),
					), 30 );
				}
			}

			// Redirect to avoid resubmission.
			wp_safe_redirect( admin_url( 'admin.php?page=content-guard-pro-quarantine' ) );
			exit;
		}
	}

	/**
	 * Display admin notices.
	 *
	 * @since 1.0.0
	 */
	private static function display_notices() {
		$notice = get_transient( 'content_guard_pro_admin_notice' );

		if ( $notice && is_array( $notice ) ) {
			$type    = isset( $notice['type'] ) ? $notice['type'] : 'info';
			$message = isset( $notice['message'] ) ? $notice['message'] : '';

			if ( $message ) {
				printf(
					'<div class="notice notice-%s is-dismissible"><p>%s</p></div>',
					esc_attr( $type ),
					esc_html( $message )
				);
			}

			delete_transient( 'content_guard_pro_admin_notice' );
		}
	}

	/**
	 * Log finding action to audit log.
	 *
	 * @since 1.0.0
	 * @param object $finding Finding object.
	 * @param string $action  Action performed.
	 */
	private static function log_finding_action( $finding, $action ) {
		global $wpdb;

		// Prepare audit log entry.
		$audit_data = array(
			'blog_id'     => get_current_blog_id(),
			'finding_id'  => $finding->id,
			'user_id'     => get_current_user_id(),
			'action'      => $action,
			'object_type' => $finding->object_type,
			'object_id'   => $finding->object_id,
			'field'       => $finding->field,
			'old_value'   => $finding->status,
			'new_value'   => ( 'delete' === $action ) ? 'deleted' : 'open',
			'metadata'    => wp_json_encode( array(
				'severity'    => $finding->severity,
				'rule_id'     => $finding->rule_id,
				'confidence'  => $finding->confidence,
				'fingerprint' => $finding->fingerprint,
			) ),
			'ip_address'  => self::get_client_ip(),
			'created_at'  => wp_date( 'Y-m-d H:i:s' ),
		);

		// Insert into audit log.
		$result = $wpdb->insert(
			$wpdb->prefix . 'content_guard_pro_audit_log',
			$audit_data,
			array( '%d', '%d', '%d', '%s', '%s', '%d', '%s', '%s', '%s', '%s', '%s', '%s' )
		);

		// Log error if insert failed.
		if ( false === $result ) {
			cgp_log( 'Content Guard Pro: Failed to insert audit log entry for action: ' . $action );
		}
	}

	/**
	 * Get client IP address.
	 *
	 * Prioritizes REMOTE_ADDR (most reliable), and only checks forwarded headers
	 * if REMOTE_ADDR is a private/reserved IP (indicating a proxy).
	 *
	 * @since 1.0.0
	 * @return string IP address.
	 */
	private static function get_client_ip() {
		$ip = '';

		// REMOTE_ADDR is the most reliable source (cannot be spoofed by client).
		if ( ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
			$ip = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
		}

		// Validate the IP address and check if it's public.
		$validated_ip = filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE );

		// If REMOTE_ADDR is private/reserved (indicating proxy), check X-Forwarded-For.
		if ( ! $validated_ip && ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$forwarded_ips = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
			
			// HTTP_X_FORWARDED_FOR can contain multiple IPs (comma-separated).
			// Format: client, proxy1, proxy2
			$ip_list = array_map( 'trim', explode( ',', $forwarded_ips ) );
			
			// Get the first valid public IP.
			foreach ( $ip_list as $forwarded_ip ) {
				$validated = filter_var( $forwarded_ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE );
				if ( $validated ) {
					return $validated;
				}
			}
		}

		// Return validated REMOTE_ADDR or fallback.
		return $validated_ip ? $validated_ip : '0.0.0.0';
	}
}

