<?php
/**
 * Admin Patterns Page
 *
 * Handles the patterns page display with rule pack management,
 * allow/deny lists, and pattern updates.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Admin_Patterns
 *
 * Renders the patterns page showing:
 * - Current rule pack version
 * - Allow/deny list management
 * - Manual pattern updates
 *
 * Per PRD Section 3.9: Allow/Deny lists for domains and regex patterns.
 *
 * @since 1.0.0
 */
class CGP_Admin_Patterns {

	/**
	 * Display the patterns page.
	 *
	 * Shows current rule pack version, allow/deny lists, manual update.
	 *
	 * @since 1.0.0
	 */
	public static function display() {
		// Check user capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'content-guard-pro' ) );
		}

		// Get current lists.
		$settings   = get_option( 'content_guard_pro_settings', array() );
		$allowlist  = isset( $settings['allowlist_domains'] ) ? $settings['allowlist_domains'] : '';
		$denylist   = isset( $settings['denylist_patterns'] ) ? $settings['denylist_patterns'] : '';

		// Get rule pack info.
		$rule_pack_version = get_option( 'content_guard_pro_rule_pack_version', '' );
		
		// If no version or old semantic version format, convert to date format
		if ( empty( $rule_pack_version ) || version_compare( $rule_pack_version, '2000.01.01', '<' ) ) {
			$rule_pack_version = gmdate( 'Y.m.d' );
			update_option( 'content_guard_pro_rule_pack_version', $rule_pack_version );
		}
		
		$last_update = get_option( 'content_guard_pro_rule_pack_last_update', false );

		?>
		<div class="wrap content-guard-pro-patterns">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<p><?php esc_html_e( 'Manage detection patterns, allow/deny lists, and rule pack updates.', 'content-guard-pro' ); ?></p>

			<?php self::display_notices(); ?>

			<!-- Rule Pack Information -->
			<div class="content-guard-pro-rule-pack card">
				<h2><?php esc_html_e( 'Rule Pack Information', 'content-guard-pro' ); ?></h2>
				
				<table class="form-table">
					<tr>
						<th scope="row"><?php esc_html_e( 'Current Version', 'content-guard-pro' ); ?></th>
						<td>
							<strong><?php echo esc_html( $rule_pack_version ); ?></strong>
							<?php if ( $last_update ) : ?>
								<p class="description">
									<?php
									printf(
										/* translators: %s: human-readable time difference */
										esc_html__( 'Last updated: %s ago', 'content-guard-pro' ),
										esc_html( human_time_diff( $last_update, time() ) )
									);
									?>
								</p>
							<?php endif; ?>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php esc_html_e( 'Update Status', 'content-guard-pro' ); ?></th>
						<td>
							<p><?php esc_html_e( 'Rule packs are included with plugin updates.', 'content-guard-pro' ); ?></p>
							<form method="post" style="display: inline;">
								<?php wp_nonce_field( 'content_guard_pro_check_updates', 'content_guard_pro_updates_nonce' ); ?>
								<input type="hidden" name="action" value="check_updates" />
								<button type="submit" class="button">
									<?php esc_html_e( 'Check for Updates', 'content-guard-pro' ); ?>
								</button>
							</form>
						</td>
					</tr>
				</table>
			</div>

			<!-- Allow/Deny Lists -->
			<form method="post" action="">
				<?php wp_nonce_field( 'content_guard_pro_save_lists', 'content_guard_pro_lists_nonce' ); ?>
				<input type="hidden" name="action" value="save_lists" />

				<!-- Allowlist -->
				<div class="content-guard-pro-allowlist card">
					<h2><?php esc_html_e( 'Domain Allowlist', 'content-guard-pro' ); ?></h2>
					
					<p class="description">
						<?php esc_html_e( 'Enter trusted domains (one per line). External resources from these domains will not be flagged as threats.', 'content-guard-pro' ); ?>
					</p>

					<p class="description">
						<strong><?php esc_html_e( 'Examples:', 'content-guard-pro' ); ?></strong><br>
						<code>googletagmanager.com</code><br>
						<code>facebook.net</code><br>
						<code>hotjar.com</code><br>
						<code>mixpanel.com</code>
					</p>

					<p class="description">
						<strong><?php esc_html_e( 'Default Safe Domains:', 'content-guard-pro' ); ?></strong><br>
						<?php echo esc_html( implode( ', ', self::get_default_allowlist() ) ); ?>
					</p>

					<textarea 
						name="allowlist_domains" 
						id="allowlist_domains" 
						rows="15" 
						class="large-text code"
						placeholder="<?php esc_attr_e( 'example.com', 'content-guard-pro' ); ?>"
					><?php echo esc_textarea( $allowlist ); ?></textarea>

					<p class="description">
						<?php
						printf(
							/* translators: %d: number of domains */
							esc_html( _n(
								'%d domain in allowlist',
								'%d domains in allowlist',
								self::count_lines( $allowlist ),
								'content-guard-pro'
							) ),
							absint( self::count_lines( $allowlist ) )
						);
						?>
					</p>

					<p>
						<?php submit_button( __( 'Save Allow/Deny Lists', 'content-guard-pro' ), 'primary', 'submit', false ); ?>
					</p>
				</div>

				<!-- Denylist -->
				<div class="content-guard-pro-denylist card">
					<h2><?php esc_html_e( 'Domain & Pattern Denylist', 'content-guard-pro' ); ?></h2>
					
					<p class="description">
						<?php esc_html_e( 'Enter blocked domains or regex patterns (one per line). Resources matching these patterns will always be flagged as threats.', 'content-guard-pro' ); ?>
					</p>

					<p class="description">
						<strong><?php esc_html_e( 'Domain Examples:', 'content-guard-pro' ); ?></strong><br>
						<code>suspicious-site.com</code><br>
						<code>malware-domain.xyz</code>
					</p>

					<p class="description">
						<strong><?php esc_html_e( 'Regex Pattern Examples:', 'content-guard-pro' ); ?></strong><br>
						<code>/casino-.*\.com/i</code> <em><?php esc_html_e( '(matches casino-*.com)', 'content-guard-pro' ); ?></em><br>
						<code>/pharma\d+\.net/i</code> <em><?php esc_html_e( '(matches pharma123.net)', 'content-guard-pro' ); ?></em>
					</p>

					<textarea 
						name="denylist_patterns" 
						id="denylist_patterns" 
						rows="15" 
						class="large-text code"
						placeholder="<?php esc_attr_e( 'example.com or /pattern/i', 'content-guard-pro' ); ?>"
					><?php echo esc_textarea( $denylist ); ?></textarea>

					<p class="description">
						<?php
						printf(
							/* translators: %d: number of patterns */
							esc_html( _n(
								'%d pattern in denylist',
								'%d patterns in denylist',
								self::count_lines( $denylist ),
								'content-guard-pro'
							) ),
							absint( self::count_lines( $denylist ) )
						);
						?>
					</p>
				</div>

				<?php submit_button( __( 'Save Allow/Deny Lists', 'content-guard-pro' ) ); ?>
			</form>

			<!-- Import/Export -->
			<div class="content-guard-pro-import-export card">
				<h2><?php esc_html_e( 'Import / Export', 'content-guard-pro' ); ?></h2>
				
				<p class="description">
					<?php esc_html_e( 'Export your allow/deny lists to share with other sites, or import lists from a file.', 'content-guard-pro' ); ?>
				</p>

				<p>
					<a href="<?php echo esc_url( self::get_export_url() ); ?>" class="button">
						<?php esc_html_e( 'Export Lists (JSON)', 'content-guard-pro' ); ?>
					</a>
				</p>

				<form method="post" enctype="multipart/form-data">
					<?php wp_nonce_field( 'content_guard_pro_import_lists', 'content_guard_pro_import_nonce' ); ?>
					<input type="hidden" name="action" value="import_lists" />
					
					<p>
						<label for="import_file"><?php esc_html_e( 'Import from JSON file:', 'content-guard-pro' ); ?></label><br>
						<input type="file" name="import_file" id="import_file" accept=".json" />
					</p>

					<p>
						<button type="submit" class="button">
							<?php esc_html_e( 'Import Lists', 'content-guard-pro' ); ?>
						</button>
					</p>
				</form>
			</div>

			<!-- Pattern Testing Tool -->
			<div class="content-guard-pro-pattern-test card">
				<h2><?php esc_html_e( 'Pattern Testing Tool', 'content-guard-pro' ); ?></h2>
				
				<p class="description">
					<?php esc_html_e( 'Test your custom patterns against sample content to verify they work correctly.', 'content-guard-pro' ); ?>
				</p>

				<p>
					<textarea id="test-content" rows="5" class="large-text code" placeholder="<?php esc_attr_e( 'Paste content to test...', 'content-guard-pro' ); ?>"></textarea>
				</p>

				<p>
					<button type="button" class="button" id="content-guard-pro-test-patterns">
						<?php esc_html_e( 'Test Patterns', 'content-guard-pro' ); ?>
					</button>
				</p>

				<div id="test-results" style="display:none; margin-top: 20px;"></div>
			</div>
		</div>

		<style>
		.content-guard-pro-patterns .card {
			background: #fff;
			border: 1px solid #ccd0d4;
			border-radius: 4px;
			padding: 20px;
			margin-bottom: 20px;
			box-shadow: 0 1px 1px rgba(0,0,0,.04);
		}
		.content-guard-pro-patterns .card h2 {
			margin-top: 0;
			padding-bottom: 10px;
			border-bottom: 1px solid #dcdcde;
		}
		.content-guard-pro-patterns textarea.code {
			font-family: Consolas, Monaco, monospace;
			font-size: 13px;
		}
		.content-guard-pro-patterns .description {
			margin-top: 10px;
		}
		</style>

		<script>
		jQuery(document).ready(function($) {
			$('#content-guard-pro-test-patterns').on('click', function(e) {
				e.preventDefault();
				var button = $(this);
				var content = $('#test-content').val();
				
				if (!content) {
					alert('<?php esc_html_e( 'Please enter content to test', 'content-guard-pro' ); ?>');
					return;
				}

						button.prop('disabled', true).text('<?php esc_html_e( 'Testing...', 'content-guard-pro' ); ?>');
						
						$.post(ajaxurl, {
							action: 'content_guard_pro_test_patterns',
							nonce: '<?php echo esc_js( wp_create_nonce( 'content_guard_pro_test_patterns' ) ); ?>',
							content: content
						}, function(response) {
							$('#test-results').html(response.data).show();
							button.prop('disabled', false).text('<?php esc_html_e( 'Test Patterns', 'content-guard-pro' ); ?>');
						});
			});
		});
		</script>
		<?php
	}

	/**
	 * Handle form actions.
	 *
	 * @since 1.0.0
	 */
	public static function handle_actions() {
		if ( ! isset( $_POST['action'] ) ) {
			return;
		}

		$action = sanitize_text_field( wp_unslash( $_POST['action'] ) );

		switch ( $action ) {
			case 'save_lists':
				self::handle_save_lists();
				break;
			case 'import_lists':
				self::handle_import_lists();
				break;
			case 'check_updates':
				self::handle_check_updates();
				break;
		}
	}

	/**
	 * Handle save lists action.
	 *
	 * @since 1.0.0
	 */
	private static function handle_save_lists() {
		// Verify nonce.
		if ( ! isset( $_POST['content_guard_pro_lists_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['content_guard_pro_lists_nonce'] ) ), 'content_guard_pro_save_lists' ) ) {
			set_transient( 'content_guard_pro_admin_notice', array(
				'type'    => 'error',
				'message' => __( 'Security check failed. Please try again.', 'content-guard-pro' ),
			), 30 );
			wp_safe_redirect( admin_url( 'admin.php?page=content-guard-pro-patterns' ) );
			exit;
		}

		// Check capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'content-guard-pro' ) );
		}

		// Get current settings.
		$settings = get_option( 'content_guard_pro_settings', array() );
		if ( ! is_array( $settings ) ) {
			$settings = array();
		}
		
		// Check if the option exists at all
		$option_exists = ( false !== get_option( 'content_guard_pro_settings', false ) );

		// Sanitize and save allowlist.
		$allowlist = '';
		if ( isset( $_POST['allowlist_domains'] ) ) {
			// Simple sanitization - just remove slashes and trim
			$allowlist = trim( wp_unslash( $_POST['allowlist_domains'] ) );
			
			// Normalize line endings to \n
			$allowlist = str_replace( array( "\r\n", "\r" ), "\n", $allowlist );
			
			$settings['allowlist_domains'] = $allowlist;
		}

		// Sanitize and save denylist.
		$denylist = '';
		if ( isset( $_POST['denylist_patterns'] ) ) {
			// Simple sanitization - just remove slashes and trim
			$denylist = trim( wp_unslash( $_POST['denylist_patterns'] ) );
			
			// Normalize line endings to \n
			$denylist = str_replace( array( "\r\n", "\r" ), "\n", $denylist );
			
			$settings['denylist_patterns'] = $denylist;
		}

		// Save settings - use add_option if it doesn't exist, otherwise update_option
		if ( ! $option_exists ) {
			// Option doesn't exist, create it
			$updated = add_option( 'content_guard_pro_settings', $settings, '', 'yes' );
		} else {
			// Option exists, update it
			$updated = update_option( 'content_guard_pro_settings', $settings );
		}
		
		// If update_option returned false, it might mean the value hasn't changed
		// Try to force update by deleting and re-adding
		if ( ! $updated && $option_exists ) {
			delete_option( 'content_guard_pro_settings' );
			add_option( 'content_guard_pro_settings', $settings, '', 'yes' );
		}

		// Clear detector cache.
		delete_transient( 'content_guard_pro_allowlist_pattern' );
		delete_transient( 'content_guard_pro_denylist_patterns' );

		// Set success message - include counts for verification
		$allowlist_lines = self::count_lines( $allowlist );
		$denylist_lines = self::count_lines( $denylist );
		
		set_transient( 'content_guard_pro_admin_notice', array(
			'type'    => 'success',
			'message' => sprintf(
				/* translators: %1$d: number of allowlist domains, %2$d: number of denylist patterns */
				__( 'Allow/Deny lists saved successfully. (%1$d allowlist domains, %2$d denylist patterns)', 'content-guard-pro' ),
				$allowlist_lines,
				$denylist_lines
			),
		), 30 );

		// Redirect to avoid resubmission.
		wp_safe_redirect( admin_url( 'admin.php?page=content-guard-pro-patterns' ) );
		exit;
	}

	/**
	 * Handle import lists action.
	 *
	 * @since 1.0.0
	 */
	private static function handle_import_lists() {
		// Verify nonce.
		if ( ! isset( $_POST['content_guard_pro_import_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['content_guard_pro_import_nonce'] ) ), 'content_guard_pro_import_lists' ) ) {
			return;
		}

		// Check capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Check if file was uploaded.
		if ( empty( $_FILES['import_file']['tmp_name'] ) ) {
			set_transient( 'content_guard_pro_admin_notice', array(
				'type'    => 'error',
				'message' => __( 'No file selected for import.', 'content-guard-pro' ),
			), 30 );
			return;
		}

		// Validate file size (max 1MB).
		$file_size = isset( $_FILES['import_file']['size'] ) ? absint( $_FILES['import_file']['size'] ) : 0;
		if ( $file_size > 1048576 ) {
			set_transient( 'content_guard_pro_admin_notice', array(
				'type'    => 'error',
				'message' => __( 'File too large. Maximum size is 1MB.', 'content-guard-pro' ),
			), 30 );
			return;
		}

		// Handle upload using WordPress API for proper validation.
		require_once ABSPATH . 'wp-admin/includes/file.php';

		$allowed_mimes = array(
			'json' => 'application/json',
			'txt'  => 'text/plain',
		);

		$upload = wp_handle_upload(
			$_FILES['import_file'],
			array(
				'test_form' => false,
				'mimes'     => $allowed_mimes,
			)
		);

		if ( isset( $upload['error'] ) ) {
			set_transient( 'content_guard_pro_admin_notice', array(
				'type'    => 'error',
				'message' => $upload['error'],
			), 30 );
			return;
		}

		$file_path = $upload['file'];

		// Double-check mime/ext to satisfy WP.org review.
		$checked = wp_check_filetype_and_ext( $file_path, wp_basename( $file_path ), $allowed_mimes );
		if ( empty( $checked['ext'] ) || empty( $checked['type'] ) ) {
			wp_delete_file( $file_path );
			set_transient( 'content_guard_pro_admin_notice', array(
				'type'    => 'error',
				'message' => __( 'Invalid file type. Please upload a JSON file.', 'content-guard-pro' ),
			), 30 );
			return;
		}

		// Read file contents after validation.
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
		$file_content = file_get_contents( $file_path );
		
		// Remove the uploaded file after reading.
		wp_delete_file( $file_path );

		if ( false === $file_content ) {
			set_transient( 'content_guard_pro_admin_notice', array(
				'type'    => 'error',
				'message' => __( 'Failed to read file.', 'content-guard-pro' ),
			), 30 );
			return;
		}

		$data = json_decode( $file_content, true );

		if ( json_last_error() !== JSON_ERROR_NONE || ! is_array( $data ) ) {
			set_transient( 'content_guard_pro_admin_notice', array(
				'type'    => 'error',
				'message' => __( 'Invalid JSON file.', 'content-guard-pro' ),
			), 30 );
			return;
		}

		// Get current settings.
		$settings = get_option( 'content_guard_pro_settings', array() );

		// Import allowlist if present.
		if ( isset( $data['allowlist_domains'] ) ) {
			$settings['allowlist_domains'] = sanitize_textarea_field( $data['allowlist_domains'] );
		}

		// Import denylist if present.
		if ( isset( $data['denylist_patterns'] ) ) {
			$settings['denylist_patterns'] = sanitize_textarea_field( $data['denylist_patterns'] );
		}

		// Update settings.
		update_option( 'content_guard_pro_settings', $settings );

		// Clear cache.
		delete_transient( 'content_guard_pro_allowlist_pattern' );
		delete_transient( 'content_guard_pro_denylist_patterns' );

		// Set success message.
		set_transient( 'content_guard_pro_admin_notice', array(
			'type'    => 'success',
			'message' => __( 'Lists imported successfully.', 'content-guard-pro' ),
		), 30 );

		// Redirect.
		wp_safe_redirect( admin_url( 'admin.php?page=content-guard-pro-patterns' ) );
		exit;
	}

	/**
	 * Handle check updates action.
	 *
	 * @since 1.0.0
	 */
	private static function handle_check_updates() {
		// Verify nonce.
		if ( ! isset( $_POST['content_guard_pro_updates_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['content_guard_pro_updates_nonce'] ) ), 'content_guard_pro_check_updates' ) ) {
			return;
		}

		// Check capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Check for updates via API.
		$license_key = get_option( 'content_guard_pro_license_key', '' );
		$current_version = get_option( 'content_guard_pro_rule_pack_version', '' );
		
		// If no version is set, use today's date as default (API expects YYYY.MM.DD format)
		if ( empty( $current_version ) || version_compare( $current_version, '2000.01.01', '<' ) ) {
			$current_version = gmdate( 'Y.m.d' );
			update_option( 'content_guard_pro_rule_pack_version', $current_version );
		}
		
		$update_info = CGP_API_Client::check_rule_pack_update( $current_version, $license_key );

		if ( isset( $update_info['error'] ) ) {
			// Error handling - improved to handle nested arrays
			$error_message = $update_info['error'];
			if ( is_array( $error_message ) ) {
				// Handle nested arrays better
				array_walk_recursive( $error_message, function( &$value ) {
					if ( ! is_string( $value ) ) {
						$value = print_r( $value, true );
					}
				});
				$error_message = implode( ' ', self::flatten_array( $error_message ) );
			}
			
			set_transient( 'content_guard_pro_admin_notice', array(
				'type'    => 'error',
				'message' => $error_message,
			), 30 );
		} elseif ( ! empty( $update_info['update_available'] ) && isset( $update_info['latest']['version'] ) ) {
			// Update available, download it
			$version = $update_info['latest']['version'];
			$download = CGP_API_Client::download_rule_pack( $version, $license_key );
			
			if ( is_wp_error( $download ) ) {
				set_transient( 'content_guard_pro_admin_notice', array(
					'type'    => 'error',
					'message' => $download->get_error_message(),
				), 30 );
			} else {
				// Save rules.
				update_option( 'content_guard_pro_rules', $download );
				update_option( 'content_guard_pro_rule_pack_version', $version );
				update_option( 'content_guard_pro_rule_pack_last_update', time() );
				
				set_transient( 'content_guard_pro_admin_notice', array(
					'type'    => 'success',
					/* translators: %s: rule pack version string */
					'message' => sprintf( __( 'Rule pack updated to version %s.', 'content-guard-pro' ), $version ),
				), 30 );
			}
		} else {
			// No update
			set_transient( 'content_guard_pro_admin_notice', array(
				'type'    => 'info',
				'message' => __( 'You have the latest rule pack version.', 'content-guard-pro' ),
			), 30 );
		}

		// Redirect.
		wp_safe_redirect( admin_url( 'admin.php?page=content-guard-pro-patterns' ) );
		exit;
	}

	/**
	 * Display admin notices.
	 *
	 * @since 1.0.0
	 */
	private static function display_notices() {
		$notice = get_transient( 'content_guard_pro_admin_notice' );

		if ( $notice ) {
			$message = $notice['message'];
			if ( is_array( $message ) ) {
				// Flatten array messages if any
				$message = implode( ' ', array_map( 'esc_html', $message ) );
			} else {
				$message = esc_html( $message );
			}

			printf(
				'<div class="notice notice-%s is-dismissible"><p>%s</p></div>',
				esc_attr( $notice['type'] ),
				esc_html( $message )
			);
			delete_transient( 'content_guard_pro_admin_notice' );
		}
	}

	/**
	 * Get default allowlist domains.
	 *
	 * @since 1.0.0
	 * @return array Default safe domains.
	 */
	private static function get_default_allowlist() {
		return array(
			'youtube.com',
			'youtu.be',
			'vimeo.com',
			'twitter.com',
			'facebook.com',
			'instagram.com',
			'google.com',
			'googleapis.com',
			'gstatic.com',
			'cloudflare.com',
			'cdnjs.cloudflare.com',
			'unpkg.com',
			'jsdelivr.net',
		);
	}

	/**
	 * Count non-empty lines in text.
	 *
	 * @since 1.0.0
	 * @param string $text Text to count.
	 * @return int Line count.
	 */
	private static function count_lines( $text ) {
		if ( empty( $text ) ) {
			return 0;
		}

		$lines = explode( "\n", trim( $text ) );
		$lines = array_filter( $lines, function( $line ) {
			return trim( $line ) !== '';
		});

		return count( $lines );
	}

	/**
	 * Get export URL.
	 *
	 * @since 1.0.0
	 * @return string Export URL.
	 */
	private static function get_export_url() {
		return add_query_arg( array(
			'action' => 'content_guard_pro_export_lists',
			'nonce'  => wp_create_nonce( 'content_guard_pro_export_lists' ),
		), admin_url( 'admin-ajax.php' ) );
	}

	/**
	 * Flatten a multidimensional array for error display.
	 *
	 * @since 1.0.0
	 * @param array $array Array to flatten.
	 * @return array Flattened array.
	 */
	private static function flatten_array( $array ) {
		$result = array();
		array_walk_recursive( $array, function( $value ) use ( &$result ) {
			$result[] = $value;
		});
		return $result;
	}

	/**
	 * AJAX handler for pattern testing.
	 *
	 * @since 1.0.0
	 */
	public static function ajax_test_patterns() {
		check_ajax_referer( 'content_guard_pro_test_patterns', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Permission denied', 'content-guard-pro' ) );
		}

		if ( empty( $_POST['content'] ) ) {
			wp_send_json_error( __( 'No content provided', 'content-guard-pro' ) );
		}

		// Sanitize content - allow HTML but strip dangerous tags.
		$content = wp_kses_post( wp_unslash( $_POST['content'] ) );

		// Run detector on content.
		$detector = new CGP_Detector();
		$findings = $detector->scan_content( $content, 0, 'test', 'test_content' );

		// Build results HTML.
		$output = '<div class="notice notice-info">';
		$output .= '<h3>' . __( 'Pattern Test Results', 'content-guard-pro' ) . '</h3>';

		if ( empty( $findings ) ) {
			$output .= '<p>' . __( 'No issues detected in test content.', 'content-guard-pro' ) . '</p>';
		} else {
			$output .= '<p>' . sprintf(
				/* translators: %d: number of findings */
				_n( '%d issue detected:', '%d issues detected:', count( $findings ), 'content-guard-pro' ),
				count( $findings )
			) . '</p>';

			$output .= '<ul>';
			foreach ( $findings as $finding ) {
				$output .= sprintf(
					'<li><strong>%s</strong> (%s, confidence: %d%%)<br><code>%s</code></li>',
					esc_html( $finding['rule_id'] ),
					esc_html( $finding['severity'] ),
					absint( $finding['confidence'] ),
					esc_html( substr( $finding['matched_text'], 0, 100 ) )
				);
			}
			$output .= '</ul>';
		}

		$output .= '</div>';

		wp_send_json_success( $output );
	}

	/**
	 * AJAX handler for exporting lists.
	 *
	 * @since 1.0.0
	 */
	public static function ajax_export_lists() {
		// Verify nonce.
		if ( ! isset( $_GET['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['nonce'] ) ), 'content_guard_pro_export_lists' ) ) {
			wp_die( esc_html__( 'Security check failed', 'content-guard-pro' ) );
		}

		// Check capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Permission denied', 'content-guard-pro' ) );
		}

		// Get settings.
		$settings = get_option( 'content_guard_pro_settings', array() );

		// Prepare export data.
		$export_data = array(
			'version'           => CONTENT_GUARD_PRO_VERSION,
			'exported_at'       => wp_date( 'Y-m-d H:i:s' ),
			'site_url'          => get_site_url(),
			'allowlist_domains' => isset( $settings['allowlist_domains'] ) ? $settings['allowlist_domains'] : '',
			'denylist_patterns' => isset( $settings['denylist_patterns'] ) ? $settings['denylist_patterns'] : '',
		);

		// Check if headers already sent.
		if ( headers_sent() ) {
			wp_die( esc_html__( 'Cannot export: headers already sent', 'content-guard-pro' ) );
		}

		// Set headers for download.
		header( 'Content-Type: application/json' );
		header( 'Content-Disposition: attachment; filename="content-guard-pro-lists-' . gmdate( 'Y-m-d' ) . '.json"' );
		header( 'Cache-Control: no-cache, must-revalidate' );
		header( 'Expires: 0' );

		// Output JSON.
		echo wp_json_encode( $export_data, JSON_PRETTY_PRINT );
		exit;
	}
}

// Register AJAX handlers.
add_action( 'wp_ajax_content_guard_pro_test_patterns', array( 'CGP_Admin_Patterns', 'ajax_test_patterns' ) );
add_action( 'wp_ajax_content_guard_pro_export_lists', array( 'CGP_Admin_Patterns', 'ajax_export_lists' ) );
