<?php
/**
 * Admin Help & Documentation Page
 *
 * Handles the help page display with documentation links,
 * FAQ, support information, and API examples.
 *
 * Per PRD Appendix G: Help / Docs page.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Admin_Help
 *
 * Renders the help page showing:
 * - Getting Started guide
 * - Documentation links
 * - FAQ
 * - Support contact
 * - API usage examples
 * - System information
 *
 * @since 1.0.0
 */
class CGP_Admin_Help {

	/**
	 * Display the help & documentation page.
	 *
	 * Shows links to documentation, FAQ, support, API examples.
	 *
	 * @since 1.0.0
	 */
	public static function display() {
		// Check user capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'content-guard-pro' ) );
		}

		?>
		<div class="wrap content-guard-pro-help">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<p class="description">
				<?php esc_html_e( 'Comprehensive guides, FAQs, and API documentation to help you get the most out of Content Guard Pro.', 'content-guard-pro' ); ?>
			</p>

			<!-- Quick Links -->
			<div class="content-guard-pro-help-quick-links">
				<a href="#getting-started" class="button button-secondary">
					<span class="dashicons dashicons-flag"></span>
					<?php esc_html_e( 'Getting Started', 'content-guard-pro' ); ?>
				</a>
				<a href="#faq" class="button button-secondary">
					<span class="dashicons dashicons-editor-help"></span>
					<?php esc_html_e( 'FAQ', 'content-guard-pro' ); ?>
				</a>
				<a href="#api" class="button button-secondary">
					<span class="dashicons dashicons-admin-plugins"></span>
					<?php esc_html_e( 'API Usage', 'content-guard-pro' ); ?>
				</a>
				<a href="https://contentguardpro.com/docs" target="_blank" class="button button-primary">
					<span class="dashicons dashicons-external"></span>
					<?php esc_html_e( 'Full Documentation', 'content-guard-pro' ); ?>
				</a>
				<a href="https://contentguardpro.com/support" target="_blank" class="button button-primary">
					<span class="dashicons dashicons-sos"></span>
					<?php esc_html_e( 'Get Support', 'content-guard-pro' ); ?>
				</a>
			</div>

			<!-- Getting Started -->
			<div id="getting-started" class="content-guard-pro-help-section">
				<h2>
					<span class="dashicons dashicons-flag"></span>
					<?php esc_html_e( 'Getting Started', 'content-guard-pro' ); ?>
				</h2>
				
				<div class="content-guard-pro-help-content">
					<h3><?php esc_html_e( 'Welcome to Content Guard Pro!', 'content-guard-pro' ); ?></h3>
					<p><?php esc_html_e( 'Content Guard Pro is a specialized WordPress security plugin that scans your database for malicious content, spam links, and SEO injections. Follow these steps to get started:', 'content-guard-pro' ); ?></p>
					
					<ol class="content-guard-pro-step-list">
						<li>
							<strong><?php esc_html_e( 'Run Your First Scan', 'content-guard-pro' ); ?></strong>
							<p><?php esc_html_e( 'Navigate to Content Guard Pro → Scans and click "Run New Scan". Choose "Quick" for posts only, or "Standard" for a comprehensive scan.', 'content-guard-pro' ); ?></p>
						</li>
						<li>
							<strong><?php esc_html_e( 'Review Findings', 'content-guard-pro' ); ?></strong>
							<p><?php esc_html_e( 'After the scan completes, go to Content Guard Pro → Findings to view detected issues. Findings are categorized by severity: Critical, Suspicious, or Review.', 'content-guard-pro' ); ?></p>
						</li>
						<li>
							<strong><?php esc_html_e( 'Take Action & Auto-Resolve', 'content-guard-pro' ); ?></strong>
							<p><?php esc_html_e( 'For each finding, you can: Quarantine (neutralize without deleting), Edit the content directly, or Ignore if it\'s a false positive. If you fix the content and save the post, open findings will be automatically resolved.', 'content-guard-pro' ); ?></p>
						</li>
						<li>
							<strong><?php esc_html_e( 'Configure Settings', 'content-guard-pro' ); ?></strong>
							<p><?php esc_html_e( 'Go to Content Guard Pro → Settings to enable scheduled scans, configure email alerts, and customize detection patterns.', 'content-guard-pro' ); ?></p>
						</li>
						<li>
							<strong><?php esc_html_e( 'Set Up Automation', 'content-guard-pro' ); ?></strong>
							<p><?php esc_html_e( 'Enable daily scans and real-time on-save scanning to continuously monitor your site for threats.', 'content-guard-pro' ); ?></p>
						</li>
					</ol>

					<div class="content-guard-pro-tip-box">
						<h4><span class="dashicons dashicons-lightbulb"></span> <?php esc_html_e( 'Pro Tip', 'content-guard-pro' ); ?></h4>
						<p><?php esc_html_e( 'Start with a Quick scan to get familiar with the interface. Once you\'re comfortable, run a Standard scan for comprehensive protection.', 'content-guard-pro' ); ?></p>
					</div>
				</div>
			</div>

			<!-- How to Use / FAQ -->
			<div id="faq" class="content-guard-pro-help-section">
				<h2>
					<span class="dashicons dashicons-editor-help"></span>
					<?php esc_html_e( 'Frequently Asked Questions', 'content-guard-pro' ); ?>
				</h2>
				
				<div class="content-guard-pro-help-content content-guard-pro-faq">
					<div class="content-guard-pro-faq-item">
						<h3><?php esc_html_e( 'What does Content Guard Pro scan?', 'content-guard-pro' ); ?></h3>
						<p><?php esc_html_e( 'Content Guard Pro scans your WordPress database, including:', 'content-guard-pro' ); ?></p>
						<ul>
							<li><?php esc_html_e( 'Posts and pages (Quick mode)', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'Custom post types', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'Post metadata (Standard mode)', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'Widgets and options (Standard mode)', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'Elementor page builder data (Standard mode)', 'content-guard-pro' ); ?></li>
						</ul>
						<p><?php esc_html_e( 'Unlike file-based scanners, we focus on database-resident content where attackers often hide malicious code.', 'content-guard-pro' ); ?></p>
					</div>

					<div class="content-guard-pro-faq-item">
						<h3><?php esc_html_e( 'What is the difference between Quick and Standard scans?', 'content-guard-pro' ); ?></h3>
						<p><strong><?php esc_html_e( 'Quick Scan:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Scans only posts and pages. Faster, recommended for regular checks.', 'content-guard-pro' ); ?></p>
						<p><strong><?php esc_html_e( 'Standard Scan:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Comprehensive scan including posts, pages, metadata, and options. More thorough but takes longer.', 'content-guard-pro' ); ?></p>
					</div>

					<div class="content-guard-pro-faq-item">
						<h3><?php esc_html_e( 'How does Auto-Resolution work?', 'content-guard-pro' ); ?></h3>
						<p><?php esc_html_e( 'When you edit a post that has open findings:', 'content-guard-pro' ); ?></p>
						<ol>
							<li><?php esc_html_e( 'Fix the malicious content in the editor.', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'Save the post.', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'Content Guard Pro automatically rescans the saved content.', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'If the issues are gone, the existing findings are automatically marked as "Resolved".', 'content-guard-pro' ); ?></li>
						</ol>
						<p><?php esc_html_e( 'This keeps your findings list clean without manual management.', 'content-guard-pro' ); ?></p>
					</div>

					<div class="content-guard-pro-faq-item">
						<h3><?php esc_html_e( 'What happens to findings when content is deleted?', 'content-guard-pro' ); ?></h3>
						<p><?php esc_html_e( 'If you trash or permanently delete a post that has findings, those findings are automatically marked as "Deleted". They are kept in the database for audit purposes but are separated from active issues. If you restore a post from the trash, the findings reopen automatically.', 'content-guard-pro' ); ?></p>
					</div>

					<div class="content-guard-pro-faq-item">
						<h3><?php esc_html_e( 'What does "Quarantine" do?', 'content-guard-pro' ); ?></h3>
						<p><?php esc_html_e( 'Quarantine is a non-destructive action that neutralizes malicious content at render-time without deleting it from the database. When you quarantine a finding:', 'content-guard-pro' ); ?></p>
						<ul>
							<li><?php esc_html_e( 'Malicious scripts and iframes are stripped from display', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'Suspicious links are converted to plain text', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'The original content remains in the database unchanged', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'You can un-quarantine at any time if it was a false positive', 'content-guard-pro' ); ?></li>
						</ul>
					</div>

					<div class="content-guard-pro-faq-item">
						<h3><?php esc_html_e( 'How do I reduce false positives?', 'content-guard-pro' ); ?></h3>
						<p><?php esc_html_e( 'To minimize false positives:', 'content-guard-pro' ); ?></p>
						<ol>
							<li><?php esc_html_e( 'Add trusted domains to your Allowlist (Content Guard Pro → Patterns)', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'Review and ignore legitimate findings using the "Ignore" action', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'Common services like YouTube, Vimeo, and Google Maps are pre-allowlisted', 'content-guard-pro' ); ?></li>
						</ol>
					</div>

					<div class="content-guard-pro-faq-item">
						<h3><?php esc_html_e( 'Will scanning slow down my site?', 'content-guard-pro' ); ?></h3>
						<p><?php esc_html_e( 'No. Scans run in the background using Action Scheduler and automatically throttle based on server performance. Visitors will not experience any slowdown. The plugin includes:', 'content-guard-pro' ); ?></p>
						<ul>
							<li><?php esc_html_e( 'Automatic batch processing with delays', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'Memory and query time monitoring', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'Safe Mode for large sites (auto-enabled when needed)', 'content-guard-pro' ); ?></li>
						</ul>
					</div>

					<div class="content-guard-pro-faq-item">
						<h3><?php esc_html_e( 'How often should I run scans?', 'content-guard-pro' ); ?></h3>
						<p><?php esc_html_e( 'We recommend:', 'content-guard-pro' ); ?></p>
						<ul>
							<li><strong><?php esc_html_e( 'Daily scheduled scans:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Enable in Settings for continuous monitoring', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'Real-time on-save scans:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Catches threats immediately when content is published', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'Manual scans:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Run after plugin/theme updates or if you suspect an issue', 'content-guard-pro' ); ?></li>
						</ul>
					</div>

					<div class="content-guard-pro-faq-item">
						<h3><?php esc_html_e( 'What types of threats does it detect?', 'content-guard-pro' ); ?></h3>
						<p><?php esc_html_e( 'Content Guard Pro detects a comprehensive range of threats:', 'content-guard-pro' ); ?></p>
						
						<h4><?php esc_html_e( 'Core Detection:', 'content-guard-pro' ); ?></h4>
						<ul>
							<li><strong><?php esc_html_e( 'Malicious scripts:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'External JavaScript and iframes from non-allowlisted domains', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'Hidden content:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Display:none elements with external links', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'SEO spam:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Pharma, casino, essay spam keywords', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'URL shorteners:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Suspicious redirectors like bit.ly, t.co', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'Obfuscated code:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Base64, fromCharCode, eval patterns', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'Link injection:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Anomalous link profiles and high external link ratios', 'content-guard-pro' ); ?></li>
						</ul>

						<h4><?php esc_html_e( 'Enhanced Detection:', 'content-guard-pro' ); ?></h4>
						<ul>
							<li><strong><?php esc_html_e( 'Inline event handlers:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'onclick, onerror, onload, and 30+ other DOM events', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'document.write():', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Dynamic content injection', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'javascript: URIs:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Execution vectors in links', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'JavaScript redirects:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Suspicious window.location manipulations', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'Object/Embed tags:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( '&lt;object&gt;, &lt;embed&gt;, &lt;applet&gt; with external sources', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'Meta refresh:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Redirect spam via meta tags', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'PHP functions:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Dangerous functions (eval, exec, system) and serialized objects', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'Extended CSS cloaking:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'opacity:0, font-size:0, text-indent:-9999px, clip:rect()', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'SVG attacks:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'SVG elements with embedded JavaScript', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'Cryptocurrency miners:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Coinhive, CryptoLoot, and other cryptojacking scripts', 'content-guard-pro' ); ?></li>
							<li><strong><?php esc_html_e( 'Encoded attacks:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'HTML entity-encoded malicious content', 'content-guard-pro' ); ?></li>
						</ul>

						<h4><?php esc_html_e( 'Page Builder Support:', 'content-guard-pro' ); ?></h4>
						<ul>
							<li><strong><?php esc_html_e( 'Elementor:', 'content-guard-pro' ); ?></strong> <?php esc_html_e( 'Deep scanning of all widget data and nested elements', 'content-guard-pro' ); ?></li>
						</ul>
					</div>

					<div class="content-guard-pro-faq-item">
						<h3><?php esc_html_e( 'Can I export scan results?', 'content-guard-pro' ); ?></h3>
						<p><?php esc_html_e( 'Export functionality (CSV and JSON) is planned for a future release. In the meantime, you can:', 'content-guard-pro' ); ?></p>
						<ul>
							<li><?php esc_html_e( 'Use the REST API to retrieve findings programmatically (see API Usage section)', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'Print the findings page for physical records', 'content-guard-pro' ); ?></li>
							<li><?php esc_html_e( 'Access findings data via the admin interface with filtering and sorting', 'content-guard-pro' ); ?></li>
						</ul>
					</div>
				</div>
			</div>

			<!-- API Usage -->
			<div id="api" class="content-guard-pro-help-section">
				<h2>
					<span class="dashicons dashicons-admin-plugins"></span>
					<?php esc_html_e( 'REST API Usage', 'content-guard-pro' ); ?>
				</h2>
				
				<div class="content-guard-pro-help-content">
					<p><?php esc_html_e( 'Content Guard Pro provides a REST API for programmatic access to scan findings and statistics. Perfect for agency dashboards, monitoring tools, and custom integrations.', 'content-guard-pro' ); ?></p>

					<h3><?php esc_html_e( 'Authentication', 'content-guard-pro' ); ?></h3>
					<p><?php esc_html_e( 'API endpoints require authentication using WordPress Application Passwords. To create an application password:', 'content-guard-pro' ); ?></p>
					<ol>
						<li><?php esc_html_e( 'Go to Users → Your Profile', 'content-guard-pro' ); ?></li>
						<li><?php esc_html_e( 'Scroll to "Application Passwords"', 'content-guard-pro' ); ?></li>
						<li><?php esc_html_e( 'Enter a name (e.g., "Content Guard API") and click "Add New Application Password"', 'content-guard-pro' ); ?></li>
						<li><?php esc_html_e( 'Copy the generated password and store it securely', 'content-guard-pro' ); ?></li>
					</ol>

					<h3><?php esc_html_e( 'Base URL', 'content-guard-pro' ); ?></h3>
					<div class="content-guard-pro-code-block">
						<code><?php echo esc_url( rest_url( 'content-guard-pro/v1/' ) ); ?></code>
					</div>

					<h3><?php esc_html_e( 'Available Endpoints', 'content-guard-pro' ); ?></h3>

					<!-- Get All Findings -->
					<div class="content-guard-pro-api-endpoint">
						<h4>GET /content-guard-pro/v1/findings</h4>
						<p><?php esc_html_e( 'Retrieve all findings with optional filtering and pagination.', 'content-guard-pro' ); ?></p>
						
						<h5><?php esc_html_e( 'Query Parameters:', 'content-guard-pro' ); ?></h5>
						<table class="widefat">
							<thead>
								<tr>
									<th><?php esc_html_e( 'Parameter', 'content-guard-pro' ); ?></th>
									<th><?php esc_html_e( 'Type', 'content-guard-pro' ); ?></th>
									<th><?php esc_html_e( 'Description', 'content-guard-pro' ); ?></th>
								</tr>
							</thead>
							<tbody>
								<tr>
									<td><code>status</code></td>
									<td>string</td>
									<td><?php esc_html_e( 'Filter by status: open, quarantined, ignored, resolved, deleted', 'content-guard-pro' ); ?></td>
								</tr>
								<tr>
									<td><code>severity</code></td>
									<td>string</td>
									<td><?php esc_html_e( 'Filter by severity: critical, suspicious, review', 'content-guard-pro' ); ?></td>
								</tr>
								<tr>
									<td><code>object_type</code></td>
									<td>string</td>
									<td><?php esc_html_e( 'Filter by type: post, postmeta, option', 'content-guard-pro' ); ?></td>
								</tr>
								<tr>
									<td><code>object_id</code></td>
									<td>int</td>
									<td><?php esc_html_e( 'Filter by specific post/object ID', 'content-guard-pro' ); ?></td>
								</tr>
								<tr>
									<td><code>page</code></td>
									<td>int</td>
									<td><?php esc_html_e( 'Page number (default: 1)', 'content-guard-pro' ); ?></td>
								</tr>
								<tr>
									<td><code>per_page</code></td>
									<td>int</td>
									<td><?php esc_html_e( 'Items per page (default: 50, max: 100)', 'content-guard-pro' ); ?></td>
								</tr>
								<tr>
									<td><code>orderby</code></td>
									<td>string</td>
									<td><?php esc_html_e( 'Order by: id, severity, confidence, last_seen', 'content-guard-pro' ); ?></td>
								</tr>
								<tr>
									<td><code>order</code></td>
									<td>string</td>
									<td><?php esc_html_e( 'Order direction: asc, desc (default)', 'content-guard-pro' ); ?></td>
								</tr>
							</tbody>
						</table>

						<h5><?php esc_html_e( 'Example Request (cURL):', 'content-guard-pro' ); ?></h5>
						<div class="content-guard-pro-code-block">
							<pre><code>curl -X GET "<?php echo esc_url( rest_url( 'content-guard-pro/v1/findings?status=open&severity=critical' ) ); ?>" \
  -u username:application_password</code></pre>
						</div>

						<h5><?php esc_html_e( 'Example Response:', 'content-guard-pro' ); ?></h5>
						<div class="content-guard-pro-code-block">
							<pre><code>{
  "findings": [
    {
      "id": 42,
      "object_type": "post",
      "object_id": 123,
      "field": "post_content",
      "rule_id": "ext_script_non_allowlist",
      "severity": "critical",
      "confidence": 92,
      "first_seen": "2025-11-12T10:30:00",
      "last_seen": "2025-11-12T10:30:00",
      "status": "open",
      "edit_link": "<?php echo esc_url( admin_url( 'post.php?post=123&action=edit' ) ); ?>"
    }
  ],
  "total": 8,
  "pages": 1,
  "page": 1,
  "per_page": 50
}</code></pre>
						</div>
					</div>

					<!-- Get Single Finding -->
					<div class="content-guard-pro-api-endpoint">
						<h4>GET /content-guard-pro/v1/findings/{id}</h4>
						<p><?php esc_html_e( 'Retrieve a specific finding by ID.', 'content-guard-pro' ); ?></p>
						
						<h5><?php esc_html_e( 'Example Request:', 'content-guard-pro' ); ?></h5>
						<div class="content-guard-pro-code-block">
							<pre><code>curl -X GET "<?php echo esc_url( rest_url( 'content-guard-pro/v1/findings/42' ) ); ?>" \
  -u username:application_password</code></pre>
						</div>
					</div>

					<!-- Get Active Scans -->
					<div class="content-guard-pro-api-endpoint">
						<h4>GET /content-guard-pro/v1/scans/active</h4>
						<p><?php esc_html_e( 'Retrieve progress information for currently active scans.', 'content-guard-pro' ); ?></p>
						
						<h5><?php esc_html_e( 'Example Request:', 'content-guard-pro' ); ?></h5>
						<div class="content-guard-pro-code-block">
							<pre><code>curl -X GET "<?php echo esc_url( rest_url( 'content-guard-pro/v1/scans/active' ) ); ?>" \
  -u username:application_password</code></pre>
						</div>
					</div>

					<!-- Scan Post -->
					<div class="content-guard-pro-api-endpoint">
						<h4>POST /content-guard-pro/v1/scan-post</h4>
						<p><?php esc_html_e( 'Perform a quick synchronous scan on a post.', 'content-guard-pro' ); ?></p>
						
						<h5><?php esc_html_e( 'Parameters:', 'content-guard-pro' ); ?></h5>
						<table class="widefat">
							<thead>
								<tr>
									<th><?php esc_html_e( 'Parameter', 'content-guard-pro' ); ?></th>
									<th><?php esc_html_e( 'Type', 'content-guard-pro' ); ?></th>
									<th><?php esc_html_e( 'Description', 'content-guard-pro' ); ?></th>
								</tr>
							</thead>
							<tbody>
								<tr>
									<td><code>post_id</code></td>
									<td>int</td>
									<td><?php esc_html_e( 'ID of the post to scan', 'content-guard-pro' ); ?></td>
								</tr>
								<tr>
									<td><code>content</code></td>
									<td>string</td>
									<td><?php esc_html_e( 'Optional content to scan (if not saved yet)', 'content-guard-pro' ); ?></td>
								</tr>
								<tr>
									<td><code>manual</code></td>
									<td>boolean</td>
									<td><?php esc_html_e( 'Force scan even if auto-scan is disabled', 'content-guard-pro' ); ?></td>
								</tr>
							</tbody>
						</table>
					</div>

					<!-- Post Findings -->
					<div class="content-guard-pro-api-endpoint">
						<h4>GET /content-guard-pro/v1/post-findings/{post_id}</h4>
						<p><?php esc_html_e( 'Get all open findings for a specific post.', 'content-guard-pro' ); ?></p>
					</div>

					<!-- Get Statistics -->
					<div class="content-guard-pro-api-endpoint">
						<h4>GET /content-guard-pro/v1/stats</h4>
						<p><?php esc_html_e( 'Retrieve summary statistics and last scan information.', 'content-guard-pro' ); ?></p>
						
						<h5><?php esc_html_e( 'Example Request:', 'content-guard-pro' ); ?></h5>
						<div class="content-guard-pro-code-block">
							<pre><code>curl -X GET "<?php echo esc_url( rest_url( 'content-guard-pro/v1/stats' ) ); ?>" \
  -u username:application_password</code></pre>
						</div>

						<h5><?php esc_html_e( 'Example Response:', 'content-guard-pro' ); ?></h5>
						<div class="content-guard-pro-code-block">
							<pre><code>{
  "summary": {
    "total": 10,
    "critical": 5,
    "suspicious": 3,
    "review": 2
  },
  "last_scan": {
    "date": "2025-11-12T10:30:00",
    "mode": "standard",
    "items_scanned": 1247,
    "findings": 10
  }
}</code></pre>
						</div>
					</div>

					<div class="content-guard-pro-tip-box">
						<h4><span class="dashicons dashicons-info"></span> <?php esc_html_e( 'Rate Limiting', 'content-guard-pro' ); ?></h4>
						<p><?php esc_html_e( 'API requests are subject to WordPress REST API rate limits. For production use, implement proper error handling and respect rate limits.', 'content-guard-pro' ); ?></p>
					</div>
				</div>
			</div>

			<!-- Support & Resources -->
			<div class="content-guard-pro-help-section">
				<h2>
					<span class="dashicons dashicons-sos"></span>
					<?php esc_html_e( 'Support & Resources', 'content-guard-pro' ); ?>
				</h2>
				
				<div class="content-guard-pro-help-content">
					<div class="content-guard-pro-support-grid">
						<div class="content-guard-pro-support-box">
							<span class="dashicons dashicons-book"></span>
							<h3><?php esc_html_e( 'Documentation', 'content-guard-pro' ); ?></h3>
							<p><?php esc_html_e( 'Comprehensive guides, tutorials, and feature walkthroughs.', 'content-guard-pro' ); ?></p>
							<a href="https://contentguardpro.com/docs" target="_blank" class="button">
								<?php esc_html_e( 'View Documentation →', 'content-guard-pro' ); ?>
							</a>
						</div>

						<div class="content-guard-pro-support-box">
							<span class="dashicons dashicons-email-alt"></span>
							<h3><?php esc_html_e( 'Email Support', 'content-guard-pro' ); ?></h3>
							<p><?php esc_html_e( 'Get help directly from our support team. Response within 24-48 hours.', 'content-guard-pro' ); ?></p>
							<a href="https://contentguardpro.com/support" target="_blank" class="button button-primary">
								<?php esc_html_e( 'Contact Support →', 'content-guard-pro' ); ?>
							</a>
						</div>

						<div class="content-guard-pro-support-box">
							<span class="dashicons dashicons-chart-line"></span>
							<h3><?php esc_html_e( 'System Status', 'content-guard-pro' ); ?></h3>
							<p><?php esc_html_e( 'Check your plugin configuration and server environment.', 'content-guard-pro' ); ?></p>
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-diagnostics' ) ); ?>" class="button">
								<?php esc_html_e( 'View Diagnostics →', 'content-guard-pro' ); ?>
							</a>
						</div>
					</div>
				</div>
			</div>

			<!-- System Information -->
			<div class="content-guard-pro-help-section">
				<h2>
					<span class="dashicons dashicons-admin-settings"></span>
					<?php esc_html_e( 'System Information', 'content-guard-pro' ); ?>
				</h2>
				
				<div class="content-guard-pro-help-content">
					<table class="widefat striped">
						<tbody>
							<tr>
								<th><?php esc_html_e( 'Plugin Version', 'content-guard-pro' ); ?></th>
								<td><strong><?php echo esc_html( CONTENT_GUARD_PRO_VERSION ); ?></strong></td>
							</tr>
							<tr>
								<th><?php esc_html_e( 'WordPress Version', 'content-guard-pro' ); ?></th>
								<td><?php echo esc_html( get_bloginfo( 'version' ) ); ?></td>
							</tr>
							<tr>
								<th><?php esc_html_e( 'PHP Version', 'content-guard-pro' ); ?></th>
								<td><?php echo esc_html( PHP_VERSION ); ?></td>
							</tr>
							<tr>
								<th><?php esc_html_e( 'Action Scheduler', 'content-guard-pro' ); ?></th>
								<td><?php echo class_exists( 'ActionScheduler' ) ? '<span style="color: #00a32a;">✓ ' . esc_html__( 'Active', 'content-guard-pro' ) . '</span>' : '<span style="color: #d63638;">✗ ' . esc_html__( 'Not Found', 'content-guard-pro' ) . '</span>'; ?></td>
							</tr>
							<tr>
								<th><?php esc_html_e( 'Database Tables', 'content-guard-pro' ); ?></th>
								<td><?php echo self::check_tables_exist() ? '<span style="color: #00a32a;">✓ ' . esc_html__( 'Installed', 'content-guard-pro' ) . '</span>' : '<span style="color: #d63638;">✗ ' . esc_html__( 'Missing', 'content-guard-pro' ) . '</span>'; ?></td>
							</tr>
						</tbody>
					</table>
				</div>
			</div>
		</div>

		<style>
		.content-guard-pro-help {
			max-width: 1200px;
		}

		.content-guard-pro-help-quick-links {
			margin: 20px 0 30px 0;
			padding: 15px;
			background: #f0f6fc;
			border-left: 4px solid #2271b1;
			border-radius: 4px;
		}

		.content-guard-pro-help-quick-links .button {
			margin-right: 10px;
			margin-bottom: 10px;
			vertical-align: middle;
		}

		.content-guard-pro-help-quick-links .dashicons {
			margin-top: 3px;
		}

		.content-guard-pro-help-section {
			background: #fff;
			border: 1px solid #ccd0d4;
			border-radius: 4px;
			padding: 30px;
			margin-bottom: 30px;
			box-shadow: 0 1px 1px rgba(0,0,0,.04);
		}

		.content-guard-pro-help-section h2 {
			margin-top: 0;
			padding-bottom: 15px;
			border-bottom: 2px solid #2271b1;
			color: #1d2327;
		}

		.content-guard-pro-help-section h2 .dashicons {
			color: #2271b1;
			margin-right: 5px;
		}

		.content-guard-pro-help-content {
			margin-top: 20px;
		}

		.content-guard-pro-step-list {
			padding-left: 20px;
		}

		.content-guard-pro-step-list li {
			margin-bottom: 20px;
			line-height: 1.6;
		}

		.content-guard-pro-step-list strong {
			display: block;
			margin-bottom: 5px;
			color: #1d2327;
		}

		.content-guard-pro-tip-box {
			margin: 20px 0;
			padding: 15px 20px;
			background: #f0f6fc;
			border-left: 4px solid #2271b1;
			border-radius: 4px;
		}

		.content-guard-pro-tip-box h4 {
			margin-top: 0;
			color: #1d2327;
		}

		.content-guard-pro-tip-box .dashicons {
			color: #2271b1;
		}

		.content-guard-pro-faq-item {
			margin-bottom: 30px;
			padding-bottom: 30px;
			border-bottom: 1px solid #dcdcde;
		}

		.content-guard-pro-faq-item:last-child {
			border-bottom: none;
		}

		.content-guard-pro-faq-item h3 {
			color: #1d2327;
			margin-top: 0;
		}

		.content-guard-pro-faq-item ul,
		.content-guard-pro-faq-item ol {
			margin-left: 20px;
		}

		.content-guard-pro-api-endpoint {
			margin: 30px 0;
			padding: 20px;
			background: #f6f7f7;
			border-radius: 4px;
		}

		.content-guard-pro-api-endpoint h4 {
			margin-top: 0;
			color: #1d2327;
			font-family: 'Courier New', monospace;
			font-size: 16px;
		}

		.content-guard-pro-api-endpoint h5 {
			margin-top: 20px;
			margin-bottom: 10px;
			color: #1d2327;
		}

		.content-guard-pro-code-block {
			margin: 10px 0;
			padding: 15px;
			background: #1e1e1e;
			color: #d4d4d4;
			border-radius: 4px;
			overflow-x: auto;
		}

		.content-guard-pro-code-block code,
		.content-guard-pro-code-block pre {
			font-family: 'Courier New', Consolas, monospace;
			font-size: 13px;
			line-height: 1.5;
			color: #d4d4d4;
			background: transparent;
			margin: 0;
			padding: 0;
		}

		.content-guard-pro-code-block pre {
			white-space: pre;
			margin: 0;
		}

		.content-guard-pro-api-endpoint table {
			margin: 10px 0;
		}

		.content-guard-pro-api-endpoint table code {
			background: #1e1e1e;
			color: #d4d4d4;
			padding: 2px 6px;
			border-radius: 3px;
			font-size: 12px;
		}

		.content-guard-pro-support-grid {
			display: grid;
			grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
			gap: 20px;
			margin-top: 20px;
		}

		.content-guard-pro-support-box {
			padding: 20px;
			background: #f6f7f7;
			border-radius: 4px;
			text-align: center;
			border: 1px solid #dcdcde;
		}

		.content-guard-pro-support-box .dashicons {
			font-size: 48px;
			width: 48px;
			height: 48px;
			color: #2271b1;
			margin-bottom: 10px;
		}

		.content-guard-pro-support-box h3 {
			margin: 10px 0;
			color: #1d2327;
		}

		.content-guard-pro-support-box p {
			color: #646970;
			margin-bottom: 15px;
		}
		</style>
		<?php
	}

	/**
	 * Check if custom tables exist.
	 *
	 * @since 1.0.0
	 * @return bool True if all tables exist.
	 */
	private static function check_tables_exist() {
		global $wpdb;

		$tables = array(
			$wpdb->prefix . 'content_guard_pro_findings',
			$wpdb->prefix . 'content_guard_pro_scans',
			$wpdb->prefix . 'content_guard_pro_audit_log',
		);

		foreach ( $tables as $table ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) );
			if ( $exists !== $table ) {
				return false;
			}
		}

		return true;
	}
}
