<?php
/**
 * Admin Findings Page
 *
 * Handles the findings page display with list table, filters,
 * bulk actions, and export functionality.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Admin_Findings
 *
 * Renders the findings page showing:
 * - List table with all findings
 * - Filters (status, severity, type)
 * - Bulk actions
 * - Export options
 *
 * @since 1.0.0
 */
class CGP_Admin_Findings {

	/**
	 * Display the findings page.
	 *
	 * Shows list table with filters (status, severity, type), bulk actions, export.
	 *
	 * @since 1.0.0
	 */
	public static function display() {
		// Check user capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'content-guard-pro' ) );
		}

		// Load list table class if not already loaded.
		if ( ! class_exists( 'CGP_Findings_List_Table' ) ) {
			require_once CONTENT_GUARD_PRO_PATH . 'includes/class-cgp-findings-list-table.php';
		}

		// Create list table instance.
		$list_table = new CGP_Findings_List_Table();
		$list_table->prepare_items();

		// Get hidden findings count for free tier notice.
		$hidden_findings = 0;
		$is_free_tier = false;
		$visible_limit = -1;
		$can_export = true;
		
		if ( class_exists( 'CGP_License_Manager' ) ) {
			$is_free_tier = ! CGP_License_Manager::is_paid();
			$visible_limit = CGP_License_Manager::get_limit( 'visible_findings' );
			$can_export = CGP_License_Manager::can( 'csv_export' );
			
			if ( $visible_limit > 0 ) {
				$total_findings = self::get_total_findings_count();
				$hidden_findings = max( 0, $total_findings - $visible_limit );
			}
		}

		?>
		<div class="wrap content-guard-pro-findings">
			<h1 class="wp-heading-inline"><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-scans' ) ); ?>" class="page-title-action">
				<?php esc_html_e( 'Run New Scan', 'content-guard-pro' ); ?>
			</a>
			
			<hr class="wp-header-end">
			
			<?php settings_errors( 'content_guard_pro_findings' ); ?>
			
			<?php if ( $hidden_findings > 0 ) : ?>
				<div class="cgp-free-tier-notice" style="margin-bottom: 20px;">
					<div class="cgp-notice-icon">
						<span class="dashicons dashicons-lock"></span>
					</div>
					<div class="cgp-notice-content">
						<strong>
							<?php 
							printf(
								/* translators: 1: number of hidden findings, 2: visible limit */
								esc_html__( '%1$d additional findings hidden. Free plan shows up to %2$d findings.', 'content-guard-pro' ),
								absint( $hidden_findings ),
								absint( $visible_limit )
							); 
							?>
						</strong>
						<p style="margin: 8px 0 0;">
							<?php esc_html_e( 'Upgrade to see all findings and take action on potential security issues.', 'content-guard-pro' ); ?>
						</p>
						<a href="https://contentguardpro.com/pricing" class="button button-primary" target="_blank" rel="noopener noreferrer" style="margin-top: 10px;">
							<span class="dashicons dashicons-star-filled" style="vertical-align: middle; margin-right: 3px;"></span>
							<?php esc_html_e( 'Upgrade to See All Findings', 'content-guard-pro' ); ?>
						</a>
					</div>
				</div>
			<?php endif; ?>
			
		<div class="content-guard-pro-findings-content">
			<form method="post">
				<input type="hidden" name="page" value="content-guard-pro-findings" />
				<?php
				$list_table->views();
				$list_table->search_box( esc_html__( 'Search findings', 'content-guard-pro' ), 'content-guard-pro-findings' );
				$list_table->display();
				?>
			</form>
		</div>
			
			<div class="content-guard-pro-findings-export">
				<h2><?php esc_html_e( 'Export Findings', 'content-guard-pro' ); ?></h2>
				<?php if ( $can_export ) : ?>
					<p>
						<button type="button" class="button" id="content-guard-pro-export-csv" disabled>
							<?php esc_html_e( 'Export as CSV', 'content-guard-pro' ); ?>
						</button>
						<button type="button" class="button" id="content-guard-pro-export-json" disabled>
							<?php esc_html_e( 'Export as JSON', 'content-guard-pro' ); ?>
						</button>
					</p>
					<p class="description">
						<?php esc_html_e( 'Export functionality coming soon.', 'content-guard-pro' ); ?>
					</p>
				<?php else : ?>
					<p>
						<button type="button" class="button" disabled>
							<?php esc_html_e( 'Export as CSV', 'content-guard-pro' ); ?>
						</button>
						<button type="button" class="button" disabled>
							<?php esc_html_e( 'Export as JSON', 'content-guard-pro' ); ?>
						</button>
					</p>
					<p class="description">
						<span class="dashicons dashicons-lock" style="vertical-align: middle;"></span>
						<?php esc_html_e( 'Export is available in Agency Lite and higher plans.', 'content-guard-pro' ); ?>
						<a href="https://contentguardpro.com/pricing" target="_blank" rel="noopener noreferrer">
							<?php esc_html_e( 'Upgrade →', 'content-guard-pro' ); ?>
						</a>
					</p>
				<?php endif; ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Get total count of findings in the database.
	 *
	 * @since 1.0.0
	 * @return int Total findings count.
	 */
	private static function get_total_findings_count() {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'content_guard_pro_findings';
		$blog_id = get_current_blog_id();
		
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		return (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$table_name}` WHERE blog_id = %d AND status = 'open'",
				$blog_id
			)
		);
	}
}

