<?php
/**
 * Admin Diagnostics Page
 *
 * Handles the diagnostics page display with environment checks,
 * system logs, and debugging tools. Similar to WordPress Site Health.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Admin_Diagnostics
 *
 * Renders the diagnostics page showing:
 * - Server information (PHP, memory, etc.)
 * - WordPress information (version, constants)
 * - Plugin information (version, scan data)
 * - Copy to clipboard functionality
 *
 * Per PRD Section 3.9 & Appendix G: Diagnostics page.
 *
 * @since 1.0.0
 */
class CGP_Admin_Diagnostics {

	/**
	 * Display the diagnostics page.
	 *
	 * Shows environment checks, logs, "Copy diagnostics", test pattern tool.
	 *
	 * @since 1.0.0
	 */
	public static function display() {
		// Check user capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'content-guard-pro' ) );
		}

		// Handle emergency action form submissions.
		self::handle_emergency_actions();

		// Gather diagnostic information.
		$diagnostics = self::gather_diagnostics();

		?>
		<div class="wrap content-guard-pro-diagnostics">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<p class="description">
				<?php esc_html_e( 'System diagnostics and environment information. Use this page to troubleshoot issues or share information with support.', 'content-guard-pro' ); ?>
			</p>

			<!-- Copy to Clipboard Button -->
			<div class="content-guard-pro-diagnostics-actions">
				<button id="content-guard-pro-copy-diagnostics" class="button button-primary">
					<span class="dashicons dashicons-clipboard"></span>
					<?php esc_html_e( 'Copy All Diagnostics to Clipboard', 'content-guard-pro' ); ?>
				</button>
				<span id="content-guard-pro-copy-message" style="margin-left: 10px; color: #00a32a; display: none;">
					✓ <?php esc_html_e( 'Copied to clipboard!', 'content-guard-pro' ); ?>
				</span>
			</div>

			<!-- Server Information -->
			<div class="content-guard-pro-diagnostic-section">
				<h2>
					<span class="dashicons dashicons-admin-tools"></span>
					<?php esc_html_e( 'Server Information', 'content-guard-pro' ); ?>
				</h2>
				
				<table class="widefat striped content-guard-pro-diagnostic-table">
					<tbody>
						<?php foreach ( $diagnostics['server'] as $key => $value ) : ?>
						<tr>
							<th><?php echo esc_html( $key ); ?></th>
							<td><?php echo wp_kses_post( $value['display'] ); ?></td>
						</tr>
						<?php endforeach; ?>
					</tbody>
				</table>
			</div>

			<!-- WordPress Information -->
			<div class="content-guard-pro-diagnostic-section">
				<h2>
					<span class="dashicons dashicons-wordpress-alt"></span>
					<?php esc_html_e( 'WordPress Information', 'content-guard-pro' ); ?>
				</h2>
				
				<table class="widefat striped content-guard-pro-diagnostic-table">
					<tbody>
						<?php foreach ( $diagnostics['wordpress'] as $key => $value ) : ?>
						<tr>
							<th><?php echo esc_html( $key ); ?></th>
							<td><?php echo wp_kses_post( $value['display'] ); ?></td>
						</tr>
						<?php endforeach; ?>
					</tbody>
				</table>
			</div>

			<!-- Plugin Information -->
			<div class="content-guard-pro-diagnostic-section">
				<h2>
					<span class="dashicons dashicons-shield-alt"></span>
					<?php esc_html_e( 'Content Guard Pro Information', 'content-guard-pro' ); ?>
				</h2>
				
				<table class="widefat striped content-guard-pro-diagnostic-table">
					<tbody>
						<?php foreach ( $diagnostics['plugin'] as $key => $value ) : ?>
						<tr>
							<th><?php echo esc_html( $key ); ?></th>
							<td><?php echo wp_kses_post( $value['display'] ); ?></td>
						</tr>
						<?php endforeach; ?>
					</tbody>
				</table>
			</div>

			<!-- Database Information -->
			<div class="content-guard-pro-diagnostic-section">
				<h2>
					<span class="dashicons dashicons-database"></span>
					<?php esc_html_e( 'Database Information', 'content-guard-pro' ); ?>
				</h2>
				
				<table class="widefat striped content-guard-pro-diagnostic-table">
					<tbody>
						<?php foreach ( $diagnostics['database'] as $key => $value ) : ?>
						<tr>
							<th><?php echo esc_html( $key ); ?></th>
							<td><?php echo wp_kses_post( $value['display'] ); ?></td>
						</tr>
						<?php endforeach; ?>
					</tbody>
				</table>
			</div>

			<!-- Action Scheduler Information -->
			<?php if ( class_exists( 'ActionScheduler' ) ) : ?>
			<div class="content-guard-pro-diagnostic-section">
				<h2>
					<span class="dashicons dashicons-clock"></span>
					<?php esc_html_e( 'Action Scheduler', 'content-guard-pro' ); ?>
				</h2>
				
				<?php self::render_action_scheduler_details(); ?>
				
				<table class="widefat striped content-guard-pro-diagnostic-table" style="margin-top: 15px;">
					<tbody>
						<?php foreach ( $diagnostics['scheduler'] as $key => $value ) : ?>
						<tr>
							<th><?php echo esc_html( $key ); ?></th>
							<td><?php echo wp_kses_post( $value['display'] ); ?></td>
						</tr>
						<?php endforeach; ?>
					</tbody>
				</table>
			</div>
			<?php endif; ?>

			<!-- Emergency Controls -->
			<?php if ( class_exists( 'ActionScheduler' ) ) : ?>
			<div class="content-guard-pro-diagnostic-section">
				<h2>
					<span class="dashicons dashicons-sos"></span>
					<?php esc_html_e( 'Emergency Controls', 'content-guard-pro' ); ?>
				</h2>
				<?php self::render_emergency_controls( $diagnostics ); ?>
			</div>
			<?php endif; ?>

			<!-- Raw Text (Hidden, for clipboard) -->
			<textarea id="content-guard-pro-diagnostics-raw" style="position: absolute; left: -9999px;" readonly><?php echo esc_textarea( self::format_diagnostics_text( $diagnostics ) ); ?></textarea>

		</div>

		<style>
		.content-guard-pro-diagnostics {
			max-width: 1200px;
		}

		.content-guard-pro-diagnostics-actions {
			margin: 20px 0;
			padding: 15px;
			background: #f0f6fc;
			border-left: 4px solid #2271b1;
			border-radius: 4px;
		}

		.content-guard-pro-diagnostics-actions .button {
			vertical-align: middle;
		}

		.content-guard-pro-diagnostics-actions .dashicons {
			margin-top: 3px;
		}

		.content-guard-pro-diagnostic-section {
			margin-top: 30px;
		}

		.content-guard-pro-diagnostic-section h2 {
			margin-bottom: 15px;
			padding-bottom: 10px;
			border-bottom: 2px solid #2271b1;
		}

		.content-guard-pro-diagnostic-section h2 .dashicons {
			color: #2271b1;
			margin-right: 5px;
		}

		.content-guard-pro-diagnostic-table th {
			width: 30%;
			font-weight: 600;
		}

		.content-guard-pro-diagnostic-table td {
			font-family: 'Courier New', monospace;
			font-size: 13px;
		}

		.content-guard-pro-status-good {
			color: #00a32a;
			font-weight: 600;
		}

		.content-guard-pro-status-warning {
			color: #dba617;
			font-weight: 600;
		}

		.content-guard-pro-status-critical {
			color: #d63638;
			font-weight: 600;
		}

		.content-guard-pro-diagnostic-badge {
			display: inline-block;
			padding: 2px 8px;
			border-radius: 3px;
			font-size: 11px;
			font-weight: 600;
			text-transform: uppercase;
		}

		.content-guard-pro-diagnostic-badge.good {
			background: #00a32a;
			color: #fff;
		}

		.content-guard-pro-diagnostic-badge.warning {
			background: #dba617;
			color: #fff;
		}

		.content-guard-pro-diagnostic-badge.critical {
			background: #d63638;
			color: #fff;
		}
		</style>

		<script>
		jQuery(document).ready(function($) {
			$('#content-guard-pro-copy-diagnostics').on('click', function(e) {
				e.preventDefault();
				
				var textarea = document.getElementById('content-guard-pro-diagnostics-raw');
				textarea.style.position = 'static';
				textarea.select();
				textarea.setSelectionRange(0, 99999); // For mobile
				
				try {
					document.execCommand('copy');
					textarea.style.position = 'absolute';
					
					// Show success message
					$('#content-guard-pro-copy-message').fadeIn().delay(3000).fadeOut();
				} catch (err) {
					textarea.style.position = 'absolute';
					alert('<?php esc_html_e( 'Failed to copy. Please select and copy manually.', 'content-guard-pro' ); ?>');
				}
			});
		});
		</script>
		<?php
	}

	/**
	 * Gather all diagnostic information.
	 *
	 * @since 1.0.0
	 * @return array Diagnostic data organized by category.
	 */
	private static function gather_diagnostics() {
		global $wpdb;

		$diagnostics = array(
			'server'    => array(),
			'wordpress' => array(),
			'plugin'    => array(),
			'database'  => array(),
			'scheduler' => array(),
		);

		// Server Information.
		$diagnostics['server']['PHP Version'] = array(
			'value'   => PHP_VERSION,
			'display' => PHP_VERSION . self::get_status_badge( version_compare( PHP_VERSION, '8.0', '>=' ), 'good', 'warning' ),
		);

		$diagnostics['server']['PHP Memory Limit'] = array(
			'value'   => ini_get( 'memory_limit' ),
			'display' => ini_get( 'memory_limit' ),
		);

		$diagnostics['server']['PHP Max Execution Time'] = array(
			'value'   => ini_get( 'max_execution_time' ),
			'display' => ini_get( 'max_execution_time' ) . ' ' . __( 'seconds', 'content-guard-pro' ),
		);

		$diagnostics['server']['PHP Post Max Size'] = array(
			'value'   => ini_get( 'post_max_size' ),
			'display' => ini_get( 'post_max_size' ),
		);

		$diagnostics['server']['PHP Upload Max Filesize'] = array(
			'value'   => ini_get( 'upload_max_filesize' ),
			'display' => ini_get( 'upload_max_filesize' ),
		);

		$diagnostics['server']['Server Software'] = array(
			'value'   => isset( $_SERVER['SERVER_SOFTWARE'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_SOFTWARE'] ) ) : 'Unknown',
			'display' => isset( $_SERVER['SERVER_SOFTWARE'] ) ? esc_html( sanitize_text_field( wp_unslash( $_SERVER['SERVER_SOFTWARE'] ) ) ) : 'Unknown',
		);

		$diagnostics['server']['Operating System'] = array(
			'value'   => PHP_OS,
			'display' => PHP_OS,
		);

		$diagnostics['server']['Server Date/Time'] = array(
			'value'   => gmdate( 'Y-m-d H:i:s' ) . ' UTC',
			'display' => date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) ) . ' <span style="color: #646970;">(' . wp_timezone_string() . ')</span>',
		);

		// WordPress Information.
		$diagnostics['wordpress']['WordPress Version'] = array(
			'value'   => get_bloginfo( 'version' ),
			'display' => get_bloginfo( 'version' ) . self::get_status_badge( version_compare( get_bloginfo( 'version' ), '6.1', '>=' ), 'good', 'warning' ),
		);

		$diagnostics['wordpress']['Site URL'] = array(
			'value'   => home_url(),
			'display' => '<a href="' . esc_url( home_url() ) . '" target="_blank">' . esc_html( home_url() ) . '</a>',
		);

		$diagnostics['wordpress']['WordPress Address'] = array(
			'value'   => site_url(),
			'display' => esc_html( site_url() ),
		);

		$diagnostics['wordpress']['WP Memory Limit'] = array(
			'value'   => WP_MEMORY_LIMIT,
			'display' => WP_MEMORY_LIMIT,
		);

		$diagnostics['wordpress']['WP Max Memory Limit'] = array(
			'value'   => WP_MAX_MEMORY_LIMIT,
			'display' => WP_MAX_MEMORY_LIMIT,
		);

		$diagnostics['wordpress']['WP Debug Mode'] = array(
			'value'   => WP_DEBUG ? 'Enabled' : 'Disabled',
			'display' => WP_DEBUG ? '<span class="content-guard-pro-status-warning">Enabled</span>' : '<span class="content-guard-pro-status-good">Disabled</span>',
		);

		$diagnostics['wordpress']['WP Cron'] = array(
			'value'   => defined( 'DISABLE_WP_CRON' ) && DISABLE_WP_CRON ? 'Disabled' : 'Enabled',
			'display' => ( defined( 'DISABLE_WP_CRON' ) && DISABLE_WP_CRON ) ? '<span class="content-guard-pro-status-warning">Disabled</span>' : '<span class="content-guard-pro-status-good">Enabled</span>',
		);

		$diagnostics['wordpress']['Multisite'] = array(
			'value'   => is_multisite() ? 'Yes' : 'No',
			'display' => is_multisite() ? 'Yes' : 'No',
		);

		$diagnostics['wordpress']['Active Theme'] = array(
			'value'   => wp_get_theme()->get( 'Name' ) . ' ' . wp_get_theme()->get( 'Version' ),
			'display' => esc_html( wp_get_theme()->get( 'Name' ) ) . ' <span style="color: #646970;">(v' . esc_html( wp_get_theme()->get( 'Version' ) ) . ')</span>',
		);

		// Plugin Information.
		$diagnostics['plugin']['Plugin Version'] = array(
			'value'   => CONTENT_GUARD_PRO_VERSION,
			'display' => CONTENT_GUARD_PRO_VERSION,
		);

		$diagnostics['plugin']['Plugin Path'] = array(
			'value'   => CONTENT_GUARD_PRO_PATH,
			'display' => '<code>' . esc_html( CONTENT_GUARD_PRO_PATH ) . '</code>',
		);

		// Get last scan info.
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		$last_scan = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT scan_id, started_at, finished_at, status, mode, totals_checked, totals_flagged
				FROM `{$table_name}`
				WHERE finished_at IS NOT NULL
				ORDER BY started_at DESC
				LIMIT %d",
				1
			)
		);

		if ( $last_scan ) {
			// Check if strtotime succeeds for human_time_diff.
			$started_timestamp = strtotime( $last_scan->started_at );
			$time_ago = ( false !== $started_timestamp ) ? human_time_diff( $started_timestamp ) . ' ago' : '';
			
			$diagnostics['plugin']['Last Scan Date'] = array(
				'value'   => $last_scan->started_at,
				'display' => mysql2date( 'F j, Y g:i a', $last_scan->started_at ) . ( $time_ago ? ' <span style="color: #646970;">(' . esc_html( $time_ago ) . ')</span>' : '' ),
			);

			$diagnostics['plugin']['Last Scan Mode'] = array(
				'value'   => $last_scan->mode,
				'display' => ucfirst( $last_scan->mode ),
			);

			$diagnostics['plugin']['Last Scan Duration'] = array(
				'value'   => self::calculate_duration( $last_scan->started_at, $last_scan->finished_at ),
				'display' => self::calculate_duration( $last_scan->started_at, $last_scan->finished_at ),
			);

			$diagnostics['plugin']['Last Scan Items'] = array(
				'value'   => $last_scan->totals_checked,
				'display' => number_format_i18n( absint( $last_scan->totals_checked ) ),
			);

			$diagnostics['plugin']['Last Scan Findings'] = array(
				'value'   => $last_scan->totals_flagged,
				'display' => number_format_i18n( absint( $last_scan->totals_flagged ) ) . ( $last_scan->totals_flagged > 0 ? ' <span class="content-guard-pro-status-warning">⚠</span>' : ' <span class="content-guard-pro-status-good">✓</span>' ),
			);
		} else {
			$diagnostics['plugin']['Last Scan Date'] = array(
				'value'   => 'Never',
				'display' => '<span style="color: #646970;">Never</span>',
			);
		}

		// Get total findings.
		$findings_table = $wpdb->prefix . 'content_guard_pro_findings';
		$findings_count = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$findings_table}`
				WHERE status = %s",
				'open'
			)
		);

		$critical_count = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$findings_table}`
				WHERE status = %s AND severity = %s",
				'open',
				'critical'
			)
		);

		$diagnostics['plugin']['Total Open Findings'] = array(
			'value'   => $findings_count,
			'display' => number_format_i18n( absint( $findings_count ) ) . ( $findings_count > 0 ? ' <span class="content-guard-pro-status-warning">⚠</span>' : ' <span class="content-guard-pro-status-good">✓</span>' ),
		);

		$diagnostics['plugin']['Critical Findings'] = array(
			'value'   => $critical_count,
			'display' => number_format_i18n( absint( $critical_count ) ) . ( $critical_count > 0 ? ' <span class="content-guard-pro-status-critical">⚠</span>' : ' <span class="content-guard-pro-status-good">✓</span>' ),
		);

		// Database Information.
		$diagnostics['database']['Database Version'] = array(
			'value'   => $wpdb->db_version(),
			'display' => $wpdb->db_version(),
		);

		$diagnostics['database']['Database Name'] = array(
			'value'   => DB_NAME,
			'display' => DB_NAME,
		);

		$diagnostics['database']['Database Host'] = array(
			'value'   => DB_HOST,
			'display' => DB_HOST,
		);

		$diagnostics['database']['Database Charset'] = array(
			'value'   => DB_CHARSET,
			'display' => DB_CHARSET,
		);

		$diagnostics['database']['Table Prefix'] = array(
			'value'   => $wpdb->prefix,
			'display' => '<code>' . esc_html( $wpdb->prefix ) . '</code>',
		);

		// Check if custom tables exist.
		$tables_exist = self::check_tables_exist();
		$diagnostics['database']['CGP Tables'] = array(
			'value'   => $tables_exist ? 'Installed' : 'Missing',
			'display' => $tables_exist ? '<span class="content-guard-pro-status-good">Installed ✓</span>' : '<span class="content-guard-pro-status-critical">Missing ✗</span>',
		);

		// Get table sizes.
		if ( $tables_exist ) {
			$findings_count_db = $wpdb->get_var( "SELECT COUNT(*) FROM `{$wpdb->prefix}content_guard_pro_findings`" );
			$scans_count_db    = $wpdb->get_var( "SELECT COUNT(*) FROM `{$wpdb->prefix}content_guard_pro_scans`" );
			$audit_count_db    = $wpdb->get_var( "SELECT COUNT(*) FROM `{$wpdb->prefix}content_guard_pro_audit_log`" );

			$diagnostics['database']['Findings Records'] = array(
				'value'   => $findings_count_db,
				'display' => number_format_i18n( absint( $findings_count_db ) ),
			);

			$diagnostics['database']['Scans Records'] = array(
				'value'   => $scans_count_db,
				'display' => number_format_i18n( absint( $scans_count_db ) ),
			);

			$diagnostics['database']['Audit Log Records'] = array(
				'value'   => $audit_count_db,
				'display' => number_format_i18n( absint( $audit_count_db ) ),
			);
		}

		// Action Scheduler Information.
		if ( class_exists( 'ActionScheduler' ) ) {
			$pending_actions  = self::get_action_scheduler_count( 'pending' );
			$running_actions  = self::get_action_scheduler_count( 'in-progress' );
			$complete_actions = self::get_action_scheduler_count( 'complete' );
			$failed_actions   = self::get_action_scheduler_count( 'failed' );

			$diagnostics['scheduler']['Action Scheduler Version'] = array(
				'value'   => ActionScheduler_Versions::instance()->latest_version(),
				'display' => ActionScheduler_Versions::instance()->latest_version(),
			);

			$diagnostics['scheduler']['Pending Actions'] = array(
				'value'   => $pending_actions,
				'display' => number_format_i18n( absint( $pending_actions ) ),
			);

			$diagnostics['scheduler']['Running Actions'] = array(
				'value'   => $running_actions,
				'display' => number_format_i18n( absint( $running_actions ) ),
			);

			$diagnostics['scheduler']['Completed Actions'] = array(
				'value'   => $complete_actions,
				'display' => number_format_i18n( absint( $complete_actions ) ),
			);

			$diagnostics['scheduler']['Failed Actions'] = array(
				'value'   => $failed_actions,
				'display' => number_format_i18n( absint( $failed_actions ) ) . ( $failed_actions > 0 ? ' <span class="content-guard-pro-status-warning">⚠</span>' : '' ),
			);
		}

		return $diagnostics;
	}

	/**
	 * Format diagnostics as plain text for clipboard.
	 *
	 * @since 1.0.0
	 * @param array $diagnostics Diagnostic data.
	 * @return string Formatted text.
	 */
	private static function format_diagnostics_text( $diagnostics ) {
		$output = "=== Content Guard Pro Diagnostics ===\n";
		$output .= 'Generated: ' . wp_date( 'F j, Y g:i a' ) . "\n";
		$output .= 'Site: ' . home_url() . "\n\n";

		foreach ( $diagnostics as $section => $items ) {
			if ( empty( $items ) ) {
				continue;
			}

			$output .= '=== ' . ucfirst( $section ) . " ===\n";

			foreach ( $items as $key => $value ) {
				$output .= $key . ': ' . $value['value'] . "\n";
			}

			$output .= "\n";
		}

		return $output;
	}

	/**
	 * Check if custom tables exist.
	 *
	 * @since 1.0.0
	 * @return bool True if all tables exist.
	 */
	private static function check_tables_exist() {
		global $wpdb;

		$tables = array(
			$wpdb->prefix . 'content_guard_pro_findings',
			$wpdb->prefix . 'content_guard_pro_scans',
			$wpdb->prefix . 'content_guard_pro_audit_log',
		);

		foreach ( $tables as $table ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) );
			if ( $exists !== $table ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Get Action Scheduler action count by status for Content Guard Pro only.
	 *
	 * @since 1.0.0
	 * @param string $status Action status.
	 * @return int Count of actions.
	 */
	private static function get_action_scheduler_count( $status ) {
		if ( ! class_exists( 'ActionScheduler_Store' ) ) {
			return 0;
		}

		// Validate status against allowed values.
		$valid_statuses = array( 'pending', 'in-progress', 'complete', 'failed', 'canceled' );
		if ( ! in_array( $status, $valid_statuses, true ) ) {
			return 0;
		}

		$store = ActionScheduler_Store::instance();
		
		// Query only Content Guard Pro actions by filtering by group.
		return $store->query_actions(
			array(
				'status'   => $status,
				'group'    => 'content-guard-pro',
				'per_page' => 1,
			),
			'count'
		);
	}

	/**
	 * Get status badge HTML.
	 *
	 * @since 1.0.0
	 * @param bool   $condition Condition to evaluate.
	 * @param string $good_class Class for good status.
	 * @param string $bad_class  Class for bad status.
	 * @return string Badge HTML.
	 */
	private static function get_status_badge( $condition, $good_class = 'good', $bad_class = 'critical' ) {
		if ( $condition ) {
			return ' <span class="content-guard-pro-diagnostic-badge ' . esc_attr( $good_class ) . '">OK</span>';
		} else {
			return ' <span class="content-guard-pro-diagnostic-badge ' . esc_attr( $bad_class ) . '">!</span>';
		}
	}

	/**
	 * Calculate scan duration.
	 *
	 * @since 1.0.0
	 * @param string $start Start datetime.
	 * @param string $end   End datetime.
	 * @return string Human-readable duration.
	 */
	private static function calculate_duration( $start, $end ) {
		if ( ! $start || ! $end ) {
			return __( 'N/A', 'content-guard-pro' );
		}

		$start_time = strtotime( $start );
		$end_time   = strtotime( $end );

		// Check if strtotime succeeded.
		if ( false === $start_time || false === $end_time ) {
			return __( 'Invalid date', 'content-guard-pro' );
		}

		$duration = $end_time - $start_time;

		// Check for negative duration (end before start).
		if ( $duration < 0 ) {
			return __( 'Invalid duration', 'content-guard-pro' );
		}

		if ( $duration < 60 ) {
			/* translators: %d: number of seconds */
			return sprintf( _n( '%d second', '%d seconds', $duration, 'content-guard-pro' ), $duration );
		} elseif ( $duration < 3600 ) {
			$minutes = floor( $duration / 60 );
			$seconds = $duration % 60;
			/* translators: 1: minutes, 2: seconds */
			return sprintf( __( '%1$d min %2$d sec', 'content-guard-pro' ), $minutes, $seconds );
		} else {
			$hours   = floor( $duration / 3600 );
			$minutes = floor( ( $duration % 3600 ) / 60 );
			/* translators: 1: hours, 2: minutes */
			return sprintf( __( '%1$d hr %2$d min', 'content-guard-pro' ), $hours, $minutes );
		}
	}

	/**
	 * Render detailed Action Scheduler status.
	 *
	 * @since 1.0.0
	 */
	private static function render_action_scheduler_details() {
		global $wpdb;

		$actions_table = $wpdb->prefix . 'actionscheduler_actions';
		$groups_table  = $wpdb->prefix . 'actionscheduler_groups';

		// Get group_id for content-guard-pro.
		$group_id = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT group_id FROM `{$groups_table}` WHERE slug = %s",
				'content-guard-pro'
			)
		);

		$stats = array(
			'total'       => 0,
			'pending'     => 0,
			'in-progress' => 0,
			'complete'    => 0,
			'failed'      => 0,
			'canceled'    => 0,
		);

		if ( $group_id ) {
			$results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT status, COUNT(*) as count 
					FROM `{$actions_table}` 
					WHERE group_id = %d 
					GROUP BY status",
					$group_id
				)
			);

			foreach ( $results as $row ) {
				$stats[ $row->status ] = (int) $row->count;
				$stats['total']       += (int) $row->count;
			}
		}

		// Check for excessive pending jobs (potential runaway).
		$is_runaway = $stats['pending'] > 100;

		if ( $is_runaway ) :
			?>
			<div class="notice notice-error inline" style="margin: 0 0 15px 0;">
				<p><strong><?php esc_html_e( '⚠️ WARNING: Excessive pending jobs detected!', 'content-guard-pro' ); ?></strong></p>
				<p><?php esc_html_e( 'You have an unusually high number of pending jobs. This might indicate a runaway batch process.', 'content-guard-pro' ); ?></p>
				<p><?php esc_html_e( 'Use the Emergency Cleanup button in the Emergency Controls section below to cancel all pending jobs.', 'content-guard-pro' ); ?></p>
			</div>
			<?php
		endif;

		?>
		<table class="widefat striped">
			<thead>
				<tr>
					<th style="width: 30%;"><?php esc_html_e( 'Job Status', 'content-guard-pro' ); ?></th>
					<th><?php esc_html_e( 'Count', 'content-guard-pro' ); ?></th>
				</tr>
			</thead>
			<tbody>
				<tr>
					<td><strong><?php esc_html_e( 'Total Jobs', 'content-guard-pro' ); ?></strong></td>
					<td><strong><?php echo esc_html( number_format_i18n( $stats['total'] ) ); ?></strong></td>
				</tr>
				<tr>
					<td><?php esc_html_e( 'Pending', 'content-guard-pro' ); ?></td>
					<td>
						<?php echo esc_html( number_format_i18n( $stats['pending'] ) ); ?>
						<?php if ( $stats['pending'] > 100 ) : ?>
							<span class="content-guard-pro-status-critical"> ⚠️</span>
						<?php endif; ?>
					</td>
				</tr>
				<tr>
					<td><?php esc_html_e( 'In Progress', 'content-guard-pro' ); ?></td>
					<td><?php echo esc_html( number_format_i18n( $stats['in-progress'] ) ); ?></td>
				</tr>
				<tr>
					<td><?php esc_html_e( 'Complete', 'content-guard-pro' ); ?></td>
					<td><?php echo esc_html( number_format_i18n( $stats['complete'] ) ); ?></td>
				</tr>
				<tr>
					<td><?php esc_html_e( 'Failed', 'content-guard-pro' ); ?></td>
					<td>
						<?php echo esc_html( number_format_i18n( $stats['failed'] ) ); ?>
						<?php if ( $stats['failed'] > 0 ) : ?>
							<span class="content-guard-pro-status-warning"> ⚠️</span>
						<?php endif; ?>
					</td>
				</tr>
				<tr>
					<td><?php esc_html_e( 'Canceled', 'content-guard-pro' ); ?></td>
					<td><?php echo esc_html( number_format_i18n( $stats['canceled'] ) ); ?></td>
				</tr>
			</tbody>
		</table>

		<p style="margin-top: 15px;">
			<a href="<?php echo esc_url( admin_url( 'tools.php?page=action-scheduler&s=content-guard-pro' ) ); ?>" class="button">
				<span class="dashicons dashicons-external" style="margin-top: 3px;"></span>
				<?php esc_html_e( 'View in Action Scheduler', 'content-guard-pro' ); ?>
			</a>
		</p>
		<?php
	}

	/**
	 * Render emergency control buttons.
	 *
	 * @since 1.0.0
	 * @param array $diagnostics Diagnostic data with scheduler info.
	 */
	private static function render_emergency_controls( $diagnostics ) {
		global $wpdb;

		// Get counts for disabling buttons.
		$pending_jobs = isset( $diagnostics['scheduler']['Pending Actions']['value'] ) 
			? absint( $diagnostics['scheduler']['Pending Actions']['value'] ) 
			: 0;

		// Count running scans from database.
		$table_name = $wpdb->prefix . 'content_guard_pro_scans';
		$running_scans = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$table_name}` 
				WHERE status IN (%s, %s) 
				AND finished_at IS NULL",
				'running',
				'pending'
			)
		);
		$running_scans = absint( $running_scans );

		?>
		<div style="background: #fff; padding: 20px; border: 1px solid #c3c4c7; border-radius: 4px;">
			<p style="margin-top: 0;">
				<?php esc_html_e( 'Use these controls only if you need to stop all running scans or clear a runaway batch process.', 'content-guard-pro' ); ?>
			</p>

			<?php if ( $pending_jobs > 0 || $running_scans > 0 ) : ?>
			<div style="margin-bottom: 20px; padding: 15px; background: #fff8e5; border-left: 4px solid #dba617;">
				<p style="margin: 0;">
					<strong><?php esc_html_e( '⚠️ Warning:', 'content-guard-pro' ); ?></strong>
					<?php esc_html_e( 'These actions cannot be undone. Use with caution.', 'content-guard-pro' ); ?>
				</p>
			</div>
			<?php else : ?>
			<div style="margin-bottom: 20px; padding: 15px; background: #f0f6fc; border-left: 4px solid #72aee6;">
				<p style="margin: 0;">
					<strong><?php esc_html_e( '✓ All Clear:', 'content-guard-pro' ); ?></strong>
					<?php esc_html_e( 'No pending jobs or running scans detected. Emergency controls are disabled.', 'content-guard-pro' ); ?>
				</p>
			</div>
			<?php endif; ?>

			<form method="post" action="" style="margin-bottom: 20px;" onsubmit="return confirm('<?php echo esc_js( __( 'Are you sure? This will cancel ALL pending scan jobs.', 'content-guard-pro' ) ); ?>');">
				<?php wp_nonce_field( 'content_guard_pro_emergency_cleanup', 'content_guard_pro_cleanup_nonce' ); ?>
				<p>
					<button 
						type="submit" 
						name="content_guard_pro_action" 
						value="cleanup_scheduler" 
						class="button button-secondary button-large"
						<?php echo ( 0 === $pending_jobs ) ? 'disabled="disabled"' : ''; ?>
						style="<?php echo ( 0 === $pending_jobs ) ? 'opacity: 0.5; cursor: not-allowed;' : ''; ?>"
					>
						<span class="dashicons dashicons-trash" style="margin-top: 3px;"></span>
						<?php esc_html_e( 'Cancel All Pending Jobs', 'content-guard-pro' ); ?>
						<?php if ( $pending_jobs > 0 ) : ?>
							<span class="content-guard-pro-button-badge"><?php echo esc_html( number_format_i18n( $pending_jobs ) ); ?></span>
						<?php endif; ?>
					</button>
				</p>
				<p class="description">
					<?php
					if ( 0 === $pending_jobs ) {
						esc_html_e( 'No pending jobs to cancel. Button is disabled.', 'content-guard-pro' );
					} else {
						echo esc_html(
							sprintf(
								/* translators: %d: number of pending jobs */
								_n(
									'Cancels %d pending Action Scheduler job for Content Guard Pro.',
									'Cancels %d pending Action Scheduler jobs for Content Guard Pro.',
									$pending_jobs,
									'content-guard-pro'
								),
								$pending_jobs
							)
						);
					}
					?>
				</p>
			</form>

			<hr style="border: 0; border-top: 1px solid #c3c4c7; margin: 20px 0;">

			<form method="post" action="" onsubmit="return confirm('<?php echo esc_js( __( 'Are you sure? This will mark all running scans as cancelled.', 'content-guard-pro' ) ); ?>');">
				<?php wp_nonce_field( 'content_guard_pro_emergency_stop_scans', 'content_guard_pro_stop_scans_nonce' ); ?>
				<p>
					<button 
						type="submit" 
						name="content_guard_pro_action" 
						value="stop_all_scans" 
						class="button button-secondary button-large"
						<?php echo ( 0 === $running_scans ) ? 'disabled="disabled"' : ''; ?>
						style="<?php echo ( 0 === $running_scans ) ? 'opacity: 0.5; cursor: not-allowed;' : ''; ?>"
					>
						<span class="dashicons dashicons-no" style="margin-top: 3px;"></span>
						<?php esc_html_e( 'Stop All Running Scans', 'content-guard-pro' ); ?>
						<?php if ( $running_scans > 0 ) : ?>
							<span class="content-guard-pro-button-badge"><?php echo esc_html( number_format_i18n( $running_scans ) ); ?></span>
						<?php endif; ?>
					</button>
				</p>
				<p class="description">
					<?php
					if ( 0 === $running_scans ) {
						esc_html_e( 'No running scans to stop. Button is disabled.', 'content-guard-pro' );
					} else {
						echo esc_html(
							sprintf(
								/* translators: %d: number of running scans */
								_n(
									'Stops %d running scan in the database.',
									'Stops %d running scans in the database.',
									$running_scans,
									'content-guard-pro'
								),
								$running_scans
							)
						);
					}
					?>
				</p>
			</form>
		</div>

		<style>
		.content-guard-pro-button-badge {
			display: inline-block;
			background: #d63638;
			color: #fff;
			padding: 2px 8px;
			border-radius: 10px;
			font-size: 11px;
			font-weight: 600;
			margin-left: 8px;
			vertical-align: middle;
		}
		
		button[disabled] .content-guard-pro-button-badge {
			background: #787c82;
		}
		</style>
		<?php
	}

	/**
	 * Handle emergency action form submissions.
	 *
	 * @since 1.0.0
	 */
	private static function handle_emergency_actions() {
		if ( ! isset( $_POST['content_guard_pro_action'] ) ) {
			return;
		}

		$action = sanitize_text_field( wp_unslash( $_POST['content_guard_pro_action'] ) );

		// Cleanup Action Scheduler jobs.
		if ( 'cleanup_scheduler' === $action ) {
			check_admin_referer( 'content_guard_pro_emergency_cleanup', 'content_guard_pro_cleanup_nonce' );

			if ( ! current_user_can( 'manage_options' ) ) {
				wp_die( esc_html__( 'You do not have permission to perform this action.', 'content-guard-pro' ) );
			}

			$cancelled = 0;
			if ( function_exists( 'as_unschedule_all_actions' ) ) {
				$cancelled = as_unschedule_all_actions( '', array(), 'content-guard-pro' );
			}

			add_action(
				'admin_notices',
				function() use ( $cancelled ) {
					?>
					<div class="notice notice-success is-dismissible">
						<p>
							<?php
							echo esc_html(
								sprintf(
									/* translators: %d: number of cancelled jobs */
									__( 'Successfully cancelled %d Action Scheduler jobs.', 'content-guard-pro' ),
									$cancelled
								)
							);
							?>
						</p>
					</div>
					<?php
				}
			);

			return;
		}

		// Stop all running scans.
		if ( 'stop_all_scans' === $action ) {
			check_admin_referer( 'content_guard_pro_emergency_stop_scans', 'content_guard_pro_stop_scans_nonce' );

			if ( ! current_user_can( 'manage_options' ) ) {
				wp_die( esc_html__( 'You do not have permission to perform this action.', 'content-guard-pro' ) );
			}

			global $wpdb;
			$table_name = $wpdb->prefix . 'content_guard_pro_scans';

			$updated = $wpdb->query(
				"UPDATE `{$table_name}` 
				SET status = 'cancelled', finished_at = NOW() 
				WHERE status IN ('running', 'pending') 
				AND finished_at IS NULL"
			);

			add_action(
				'admin_notices',
				function() use ( $updated ) {
					?>
					<div class="notice notice-success is-dismissible">
						<p>
							<?php
							echo esc_html(
								sprintf(
									/* translators: %d: number of stopped scans */
									__( 'Successfully stopped %d running scans.', 'content-guard-pro' ),
									$updated
								)
							);
							?>
						</p>
					</div>
					<?php
				}
			);

			return;
		}
	}
}
