<?php
/**
 * Admin Dashboard Page
 *
 * Handles the main dashboard page display with last scan info,
 * findings summary, and quick actions.
 *
 * Also handles the setup wizard display when accessed with ?setup=1 parameter.
 * This approach avoids hidden menu registration issues in PHP 8.x environments.
 *
 * @package ContentGuardPro
 * @since   1.0.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CGP_Admin_Dashboard
 *
 * Renders the dashboard page showing:
 * - Last scan status
 * - Findings counts by severity
 * - Quick action buttons
 *
 * Also handles setup wizard when ?setup=1 parameter is present.
 *
 * @since 1.0.0
 */
class CGP_Admin_Dashboard {

	/**
	 * Check if we're in setup wizard mode.
	 *
	 * @since 1.0.2
	 * @return bool True if setup parameter is present.
	 */
	public static function is_setup_mode() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just checking URL parameter presence
		return isset( $_GET['setup'] ) && '1' === $_GET['setup'];
	}

	/**
	 * Display the dashboard page.
	 *
	 * Shows last scan, counts by severity, quick actions (Run Scan, Review Findings, Settings).
	 * If ?setup=1 parameter is present, displays the setup wizard instead.
	 *
	 * @since 1.0.0
	 */
	public static function display() {
		// Check user capability.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'content-guard-pro' ) );
		}

		// Check if we're in setup wizard mode.
		if ( self::is_setup_mode() ) {
			self::display_setup_wizard();
			return;
		}

		// Get last scan info.
		$last_scan = self::get_last_scan();

		// Get findings counts.
		$findings_counts = self::get_findings_counts();

		// Get license status.
		$license_status = self::get_license_status();

		?>
		<div class="wrap content-guard-pro-dashboard">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<div class="content-guard-pro-dashboard-content">
				<p><?php esc_html_e( 'Welcome to Content Guard Pro! This dashboard provides an overview of your site\'s security status.', 'content-guard-pro' ); ?></p>
				
				<!-- License Status Banner -->
				<?php self::render_license_banner( $license_status ); ?>
				
				<div class="content-guard-pro-dashboard-widgets">
					<div class="content-guard-pro-widget content-guard-pro-last-scan">
						<h2><?php esc_html_e( 'Last Scan', 'content-guard-pro' ); ?></h2>
						<?php if ( $last_scan ) : ?>
							<p>
								<strong><?php esc_html_e( 'Status:', 'content-guard-pro' ); ?></strong>
								<?php echo esc_html( ucfirst( $last_scan->status ) ); ?>
							</p>
							<p>
								<strong><?php esc_html_e( 'Started:', 'content-guard-pro' ); ?></strong>
								<?php echo esc_html( self::format_datetime( $last_scan->started_at ) ); ?>
							</p>
							<?php if ( ! empty( $last_scan->finished_at ) ) : ?>
								<p>
									<strong><?php esc_html_e( 'Finished:', 'content-guard-pro' ); ?></strong>
									<?php echo esc_html( self::format_datetime( $last_scan->finished_at ) ); ?>
								</p>
							<?php endif; ?>
							<p>
								<strong><?php esc_html_e( 'Items Checked:', 'content-guard-pro' ); ?></strong>
								<?php echo esc_html( number_format_i18n( $last_scan->totals_checked ) ); ?>
							</p>
							<p>
								<strong><?php esc_html_e( 'Findings:', 'content-guard-pro' ); ?></strong>
								<?php echo esc_html( number_format_i18n( $last_scan->totals_flagged ) ); ?>
							</p>
						<?php else : ?>
							<p><?php esc_html_e( 'No scans yet. Run your first scan to get started.', 'content-guard-pro' ); ?></p>
						<?php endif; ?>
					</div>
					
					<div class="content-guard-pro-widget content-guard-pro-findings-summary">
						<h2><?php esc_html_e( 'Findings Summary', 'content-guard-pro' ); ?></h2>
						<p>
							<strong><?php esc_html_e( 'Critical:', 'content-guard-pro' ); ?></strong>
							<span class="content-guard-pro-count content-guard-pro-count-critical"><?php echo esc_html( number_format_i18n( $findings_counts['critical'] ) ); ?></span>
						</p>
						<p>
							<strong><?php esc_html_e( 'Suspicious:', 'content-guard-pro' ); ?></strong>
							<span class="content-guard-pro-count content-guard-pro-count-suspicious"><?php echo esc_html( number_format_i18n( $findings_counts['suspicious'] ) ); ?></span>
						</p>
						<p>
							<strong><?php esc_html_e( 'Review:', 'content-guard-pro' ); ?></strong>
							<span class="content-guard-pro-count content-guard-pro-count-review"><?php echo esc_html( number_format_i18n( $findings_counts['review'] ) ); ?></span>
						</p>
						<p>
							<strong><?php esc_html_e( 'Total Open:', 'content-guard-pro' ); ?></strong>
							<span class="content-guard-pro-count content-guard-pro-count-total"><?php echo esc_html( number_format_i18n( $findings_counts['total'] ) ); ?></span>
						</p>
					</div>
					
					<div class="content-guard-pro-widget content-guard-pro-quick-actions">
						<h2><?php esc_html_e( 'Quick Actions', 'content-guard-pro' ); ?></h2>
						<p>
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-scans' ) ); ?>" class="button button-primary">
								<?php esc_html_e( 'Run Scan', 'content-guard-pro' ); ?>
							</a>
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-findings' ) ); ?>" class="button">
								<?php esc_html_e( 'Review Findings', 'content-guard-pro' ); ?>
							</a>
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-settings' ) ); ?>" class="button">
								<?php esc_html_e( 'Settings', 'content-guard-pro' ); ?>
							</a>
						</p>
					</div>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Get last scan information from database.
	 *
	 * @since 1.0.0
	 * @return object|null Scan object or null if no scans found.
	 */
	private static function get_last_scan() {
		global $wpdb;

		$table_name = $wpdb->prefix . 'content_guard_pro_scans';

		// Get the most recent scan.
		$last_scan = $wpdb->get_row(
			"SELECT scan_id, started_at, finished_at, status, totals_checked, totals_flagged, mode
			FROM `{$table_name}`
			ORDER BY started_at DESC
			LIMIT 1"
		);

		return $last_scan;
	}

	/**
	 * Get findings counts by severity.
	 *
	 * @since 1.0.0
	 * @return array Counts array with keys: critical, suspicious, review, total.
	 */
	private static function get_findings_counts() {
		global $wpdb;

		$table_name = $wpdb->prefix . 'content_guard_pro_findings';

		// Get counts by severity for open findings.
		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT severity, COUNT(*) as count
				FROM `{$table_name}`
				WHERE status = %s
				GROUP BY severity",
				'open'
			)
		);

		// Initialize counts.
		$counts = array(
			'critical'   => 0,
			'suspicious' => 0,
			'review'     => 0,
			'total'      => 0,
		);

		// Valid severity levels.
		$valid_severities = array( 'critical', 'suspicious', 'review' );

		// Populate counts from results.
		if ( $results ) {
			foreach ( $results as $row ) {
				if ( in_array( $row->severity, $valid_severities, true ) ) {
					$counts[ $row->severity ] = absint( $row->count );
					$counts['total']         += absint( $row->count );
				}
			}
		}

		return $counts;
	}

	/**
	 * Format datetime for display.
	 *
	 * @since 1.0.0
	 * @param string $datetime MySQL datetime string.
	 * @return string Formatted datetime string.
	 */
	private static function format_datetime( $datetime ) {
		if ( empty( $datetime ) || '0000-00-00 00:00:00' === $datetime ) {
			return esc_html__( 'N/A', 'content-guard-pro' );
		}

		// Convert to timestamp.
		$timestamp = strtotime( $datetime );

		// Check if strtotime succeeded.
		if ( false === $timestamp ) {
			return esc_html__( 'Invalid date', 'content-guard-pro' );
		}

		// Format using WordPress date format.
		return date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $timestamp );
	}

	/**
	 * Get license status for dashboard display.
	 *
	 * @since 1.0.0
	 * @return array License status data.
	 */
	private static function get_license_status() {
		if ( class_exists( 'CGP_License_Manager' ) ) {
			return CGP_License_Manager::get_status_for_display();
		}

		// Fallback if license manager isn't loaded.
		return array(
			'tier'        => 'free',
			'tier_name'   => __( 'Free', 'content-guard-pro' ),
			'is_paid'     => false,
			'is_valid'    => true,
			'status'      => 'active',
			'upgrade_url' => 'https://contentguardpro.com/pricing',
		);
	}

	/**
	 * Render the license status banner.
	 *
	 * @since 1.0.0
	 * @param array $license_status License status data.
	 */
	private static function render_license_banner( $license_status ) {
		$tier      = $license_status['tier'];
		$tier_name = $license_status['tier_name'];
		$is_paid   = $license_status['is_paid'];

		// Determine badge color based on tier.
		$badge_classes = array(
			'free'         => 'cgp-tier-badge-free',
			'solo_guard'   => 'cgp-tier-badge-solo',
			'agency_lite'  => 'cgp-tier-badge-agency',
			'agency_pro'   => 'cgp-tier-badge-agency-pro',
			'enterprise'   => 'cgp-tier-badge-enterprise',
		);
		$badge_class = isset( $badge_classes[ $tier ] ) ? $badge_classes[ $tier ] : 'cgp-tier-badge-free';

		?>
		<div class="cgp-license-banner <?php echo esc_attr( $badge_class ); ?>">
			<div class="cgp-license-info">
				<span class="cgp-license-label"><?php esc_html_e( 'License:', 'content-guard-pro' ); ?></span>
				<span class="cgp-tier-badge <?php echo esc_attr( $badge_class ); ?>">
					<?php echo esc_html( $tier_name ); ?>
				</span>
				<?php if ( ! $is_paid ) : ?>
					<span class="cgp-license-hint">
						<?php esc_html_e( 'Upgrade to unlock all features', 'content-guard-pro' ); ?>
					</span>
				<?php endif; ?>
			</div>
			<div class="cgp-license-actions">
				<?php if ( ! $is_paid ) : ?>
					<a href="<?php echo esc_url( $license_status['upgrade_url'] ); ?>" class="button button-primary cgp-upgrade-button" target="_blank" rel="noopener noreferrer">
						<?php esc_html_e( 'Upgrade Now', 'content-guard-pro' ); ?>
					</a>
				<?php else : ?>
					<span class="cgp-license-active">
						<span class="dashicons dashicons-yes-alt"></span>
						<?php esc_html_e( 'Active License', 'content-guard-pro' ); ?>
					</span>
				<?php endif; ?>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=content-guard-pro-settings&tab=license' ) ); ?>" class="button">
					<?php esc_html_e( 'Manage License', 'content-guard-pro' ); ?>
				</a>
			</div>
		</div>
		
		<?php if ( ! $is_paid ) : ?>
			<?php self::render_free_tier_notice( $license_status ); ?>
		<?php endif;
	}

	/**
	 * Render a notice for free tier limitations.
	 *
	 * @since 1.0.0
	 * @param array $license_status License status data.
	 */
	private static function render_free_tier_notice( $license_status ) {
		$limits = isset( $license_status['limits'] ) ? $license_status['limits'] : array();
		
		$visible_findings = isset( $limits['visible_findings'] ) ? $limits['visible_findings'] : 10;
		$scans_per_day    = isset( $limits['manual_scans_per_day'] ) ? $limits['manual_scans_per_day'] : 1;
		?>
		<div class="cgp-free-tier-notice">
			<div class="cgp-notice-icon">
				<span class="dashicons dashicons-info-outline"></span>
			</div>
			<div class="cgp-notice-content">
				<strong><?php esc_html_e( 'Free Plan Limitations:', 'content-guard-pro' ); ?></strong>
				<ul>
					<li>
						<?php
						/* translators: %d: number of findings visible in free plan */
						printf( esc_html__( 'View up to %d findings (upgrade to see all)', 'content-guard-pro' ), absint( $visible_findings ) );
						?>
					</li>
					<li>
						<?php
						/* translators: %d: number of manual scans per day in free plan */
						printf( esc_html__( '%d manual scan per day', 'content-guard-pro' ), absint( $scans_per_day ) );
						?>
					</li>
					<li><?php esc_html_e( 'Quick scan mode only', 'content-guard-pro' ); ?></li>
					<li><?php esc_html_e( 'No email alerts or quarantine', 'content-guard-pro' ); ?></li>
				</ul>
				<a href="<?php echo esc_url( $license_status['upgrade_url'] ); ?>" class="cgp-learn-more" target="_blank" rel="noopener noreferrer">
					<?php esc_html_e( 'Compare Plans →', 'content-guard-pro' ); ?>
				</a>
			</div>
		</div>
		<?php
	}

	/**
	 * Display the setup wizard.
	 *
	 * Loads and delegates to CGP_Admin_Setup_Wizard class.
	 * This approach avoids hidden menu registration issues in PHP 8.x.
	 *
	 * Access via: admin.php?page=content-guard-pro&setup=1
	 *
	 * @since 1.0.2
	 */
	private static function display_setup_wizard() {
		// Load the setup wizard class if not already loaded.
		if ( ! class_exists( 'CGP_Admin_Setup_Wizard' ) ) {
			$wizard_file = CONTENT_GUARD_PRO_PATH . 'admin/pages/class-cgp-admin-setup-wizard.php';
			if ( file_exists( $wizard_file ) ) {
				require_once $wizard_file;
			}
		}

		// Display the wizard.
		if ( class_exists( 'CGP_Admin_Setup_Wizard' ) ) {
			CGP_Admin_Setup_Wizard::display();
		} else {
			wp_die( esc_html__( 'Setup wizard class not found.', 'content-guard-pro' ) );
		}
	}

	/**
	 * Get the setup wizard URL.
	 *
	 * Returns the dashboard URL with setup=1 parameter.
	 *
	 * @since 1.0.2
	 * @param int $step Optional. Step number to include in URL.
	 * @return string Setup wizard URL.
	 */
	public static function get_setup_wizard_url( $step = 1 ) {
		return add_query_arg(
			array(
				'page'  => 'content-guard-pro',
				'setup' => '1',
				'step'  => absint( $step ),
			),
			admin_url( 'admin.php' )
		);
	}
}

