

<?php
/**
 * Action Scheduler Cleanup Script
 * 
 * EMERGENCY SCRIPT: Removes all pending and past-due Content Guard Pro jobs.
 * 
 * ⚠️ USAGE:
 * 1. Navigate to: http://yourdomain.test/wp-admin/tools.php?page=action-scheduler
 * 2. Look for bulk actions at the top
 * 3. Use the built-in Action Scheduler UI to cancel jobs
 * 
 * OR use this script manually by loading WordPress and calling content_guard_pro_cleanup_action_scheduler()
 * 
 * @package Content_Guard_Pro
 * @since 1.0.1
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	die( 'Direct access not permitted.' );
}

/**
 * Clean up all Content Guard Pro Action Scheduler jobs.
 * 
 * This function will:
 * - Cancel all pending jobs
 * - Cancel all past-due jobs
 * - Leave completed jobs for audit trail
 * 
 * @return array Stats about cleanup.
 */
function content_guard_pro_cleanup_action_scheduler() {
	if ( ! class_exists( 'ActionScheduler' ) || ! function_exists( 'as_unschedule_all_actions' ) ) {
		return array(
			'success' => false,
			'message' => 'Action Scheduler not available',
		);
	}

	$stats = array(
		'success' => true,
		'message' => 'Cleanup completed',
		'actions_cancelled' => 0,
	);

	try {
		// Method 1: Cancel all actions in the content-guard-pro group.
		// This is the cleanest approach.
		$cancelled = as_unschedule_all_actions( '', array(), 'content-guard-pro' );
		$stats['actions_cancelled'] = $cancelled;

		cgp_log( sprintf(
			'Content Guard Pro: Cancelled %d Action Scheduler jobs',
			$cancelled
		) );

	} catch ( Exception $e ) {
		$stats['success'] = false;
		$stats['message'] = 'Error: ' . $e->getMessage();
		cgp_log( 'Content Guard Pro: Error during cleanup: ' . $e->getMessage() );
	}

	return $stats;
}

/**
 * Alternative: Direct database cleanup (nuclear option).
 * 
 * ⚠️ USE ONLY IF as_unschedule_all_actions() DOESN'T WORK!
 * 
 * @global wpdb $wpdb WordPress database abstraction.
 * @return array Stats about cleanup.
 */
function content_guard_pro_cleanup_action_scheduler_database() {
	global $wpdb;

	$actions_table = $wpdb->prefix . 'actionscheduler_actions';
	$groups_table = $wpdb->prefix . 'actionscheduler_groups';

	// Find the group_id for 'content-guard-pro'.
	$group_id = $wpdb->get_var(
		$wpdb->prepare(
			"SELECT group_id FROM `{$groups_table}` WHERE slug = %s",
			'content-guard-pro'
		)
	);

	if ( ! $group_id ) {
		return array(
			'success' => false,
			'message' => 'Group not found in database',
			'actions_deleted' => 0,
		);
	}

	// Delete all pending and past-due actions for this group.
	$deleted = $wpdb->query(
		$wpdb->prepare(
			"DELETE FROM `{$actions_table}` 
			WHERE group_id = %d 
			AND status IN ('pending', 'in-progress', 'running')",
			$group_id
		)
	);

	cgp_log( sprintf(
		'Content Guard Pro: Deleted %d actions from database',
		$deleted
	) );

	return array(
		'success' => true,
		'message' => 'Database cleanup completed',
		'actions_deleted' => $deleted,
	);
}

// If this script is loaded directly via require/include, run cleanup.
// Example: require_once( __DIR__ . '/cleanup-action-scheduler.php' );
if ( defined( 'ABSPATH' ) && function_exists( 'wp_get_current_user' ) ) {
	// Only allow admins to run this.
	if ( current_user_can( 'manage_options' ) ) {
		// Uncomment the line below to auto-run cleanup:
		// $result = content_guard_pro_cleanup_action_scheduler();
		// var_dump( $result ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_var_dump
	}
}

